// 	$Id: message_display.cc,v 1.25 1998/05/25 20:06:06 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : message_display.cc
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *
 * *  Functions for article viewing level.
 * *  
 * *
 * *
 * *
 ***************************************************************************/

#include <fstream.h>
#include <stdlib.h>   // abs()
#ifdef __WATCOMC__
#   include <io.h>    // F_OK
#else
#   include <unistd.h>
#endif
#include "jmr.hh"
#include "terminal_io.hh"
#include "menu.hh"
#include "mail.hh"
#include "menubar.hh"


static void save_thread_to_file (Msgthread *thread);
static void save_articles (Msgthread *thread, int flag, char *fname,
                           int filemode);


extern Terminal_screen Screen;
extern settings_t Settings;
extern Mail* mail;
extern Window* Wscript;


static const menuinfo article_menu_array[] = {
        {"[jmr]", 0 },
        {"Exit from menu", 0},
        {"Return to thread selection", QUIT_CMD},
        {"Quit jmr", FORCED_QUIT_CMD},
        {"@NEXT", 0 },
        {"[Moving]", 0},
        {"Next Unread", NEXT_UNREAD_CMD },
        {"First of thread", '<' },
        {"Last of thread", '>' },
        {"@NEXT", 0 },
        {"Followup", FOLLOWUP_CMD },
        {"@NEXT", 0 },
        {"Reply private", REPLY_CMD },
        {"@NEXT", 0 },
        {"@END", 0 }
};

/***************************************************************************
 * FUNCTION : save_thread_to_file
 ***************************************************************************
 *
 * description : Saves current thread or article to file.
 *
 *
 * in: Settings.extractdir, Screen
 * out:
 *
 * return :
 ***************************************************************************/
static void
save_thread_to_file (Msgthread *thread)
{
        ofstream file;
        String fullname;
        static char fname[MAXPATH];
        bool quit;
        int c;
        int flag = DEFAULT_FL;
        int filemode = 0;

        if ( !*fname ) {
                strcpy( fname, Settings.default_folder.c_str() );
        }
        Screen.gotoxy (0, Screen.get_ymax() -4 );
        for (int i = 0; i < 4; ++i ) {
                Screen.down();
                Screen.clr_eol();
        }
        if (ACCESS (Settings.extractdir.get(), F_OK)) {
                handle_error ("Cannot access extract directory");
                return;
        }
        Screen.up(1);
        show_option_str( "Save [@t@]hread [@a@]rticle or [@q@]uit ? @t@" );
        quit = false;
        do {
                c = Screen.get_ch();
                switch(c) {
                case 't':
                case SELECT_CMD:
                        flag = THREAD_FL;
                        quit = true;
                        break;
                case 'a':
                        flag = ARTICLE_FL;
                        Screen.addstr ("\ba" );
                        quit = true;
                        break;
                case QUIT_CMD:
                        return;
                }
        } while (quit == false);

        do {
                c = 0;
                Screen.bottom();
                Screen.clr_eol();
                Screen.reset_attr();
                Screen.addstr( "Input file name : " );
                Screen.textcolor(BOLD);
                Screen.getline (fname, 40);
                Screen.reset_attr();
                Screen.bottom();
                Screen.clr_eol();
                fullname = fname;
                if ( have_full_path( fullname ) == false && 
                     expand_path( fullname ) == false ) {
                        fullname = Settings.extractdir.get();
                        fullname += fname;
                } 
                if (  access( fullname.c_str(), R_OK ) == 0 ) {
                        Screen.bottom();
                        show_option_str("File exists. Select: [@o@]verwrite"
                                        " [@n@]ew file [@a@]ppend [@q@]uit ?"
                                        " @a@" );
                        Screen.clr_eol();
                        quit = false;
                        do {
                                c = Screen.get_ch();
                                switch (c) {
                                case 'o':
                                        filemode = ios::trunc;
                                        quit = true;
                                        break;
                                case 'n':
                                case QUIT_CMD:
                                        quit = true;
                                        break;
                                case 'a':
                                case SELECT_CMD:
                                        filemode = ios::app;
                                        quit = true;
                                        break;
                                }
                        } while (quit == false);
                        Screen.bottom();
                        Screen.clr_eol();
                } else {
                        filemode = ios::trunc;
                }
        } while (c == 'n');

        if (c != QUIT_CMD) {
                save_articles (thread, flag, fullname.get(), filemode);
        }
}

/***************************************************************************
 * FUNCTION : save_articles
 ***************************************************************************
 *
 * description : Saves article or thread to given file.
 *
 ***************************************************************************/
static void
save_articles (Msgthread *thread, // thread
              int flag,     // flag. if THREAD_FL, then saves whole thread
              char *fname,  // name of file
              int filemode) // mode. ios::trunc or ios::app
{
        ofstream file;
        Message *msg;
        
        file.open (fname, ios::out | filemode);
        if (file.fail()) {
                String err = "Cannot save message to file `";
                err += fname;
                err += "'";
                system_error ( err.c_str() );
                return;
        }
        
        if (flag & THREAD_FL) {
                thread->save_position();
                thread->first();
        }

        // write contents of thread to file
        do {
                msg = thread->get();
                assert( msg );
                msg->save_to_file (file, mail->get_bbs_name(),
                                   mail->get_group_name(),
                                   mail->get_group_number());

                if (!(flag & THREAD_FL)) {
                        break;
                }
        } while (thread->next() == true);
        
        if (flag & THREAD_FL) {
                thread->restore_position();
        }
        
        file.close();
}


//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: Msgdisplay
//
// description: Constructor for this class.
//
//
//**************************************************************************/
Msgdisplay::Msgdisplay ( Grpthread &groups_threads ) :
        threads( groups_threads), hist_ring(READ_HISTORY_SIZE)
{
        
        orig_article = 0;
        ymax = Screen.get_ymax() - 2;
        ymin = 6;
        ypos = ymin;
        lines_per_page = ymax - ymin;
        pages = 0;
        lines_in_article = 0;
        xmax = Screen.get_xmax();
        modefl = 0;
        if (Settings.is_wrapmode == true ) {
                modefl |= WRAPMODE;
        }
        menubar.init( article_menu_array );
        menubar.set_text( MENUBAR_TEXT );

        for (int i=0; i <= xmax; i++) {
                dashline += '-';
        }
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: enter
//
// description: Main function for class. Gets events from user.
//
// in:  Screen, Settings.is_wrapmode
// out: Settings.is_wrapmode
//
//
// returns: DEFAULT_FL or TAGGED_FL.
//
//**************************************************************************/
int
Msgdisplay::enter (
                   // pointer to flag of viewing mode
                   bool *showflag,
                   // if NEXT_UNREAD_FL, then search for next unread.
                   int flag)
{
        int c;
        int rc = DEFAULT_FL;
        bool quit = false;
        bool first_pass = false;
        show_all_articles = showflag;
        is_bottom = false;
        // go to next unread, or first message of thread
        if (flag & NEXT_UNREAD_FL) {
                threads.next_unread();
        } else {
                threads.get_thread()->first();
        }
        sline.init( Screen.get_xmax() + 1);
        
        update ();
        hist_ring.clear();
        hist_ring.add( orig_article );
        display (REFRESH_ALL_FL);
        iline.show_default();
        do {
                c = Screen.get_ch();
                iline.reset();
                switch (c) {
                case ESCAPE:
                        if ( Screen.get_xch() == ESCAPE ) {
                                iline.set( MENU_MSG );
                                iline.show();
                                c = menubar.select();
                                draw_header();
                                iline.reset();
                        } else if ( Screen.get_xch() == 'r' ) {
                                modefl ^= ROT13MODE;
                                display();
                        } else if ( Screen.get_xch() == 'w' ) {
                                modefl ^= WRAPMODE;
                                Settings.is_wrapmode =
                                        Settings.is_wrapmode ? false : true;
                                update();
                                display();
                        }
                        break;
                case MESSAGEWIN_CMD:
                        Wscript->enable();
                        iline.set ( MESSAGES_MSG );
                        iline.show();
                        Screen.get_ch();
                        iline.reset();
                        Wscript->disable();
                        break;
                case QUIT_CMD:
                case CODE_LEFT:
                        quit = true;
                        break;
                case CODE_DOWN:
                        down();
                        break;
                case CODE_NPAGE:
                case PAGE_DOWN_CMD:
                        if (is_bottom == true && first_pass == false) {
                                first_pass = true;
                                iline.set(AT_END_OF_ARTICLE_MSG);
                        } else if (is_bottom == true && first_pass == true) {
                                first_pass = false;
                                if (page_down(NEXT_FL) == false) {
                                        iline.set( NO_UNREAD_MSG, ERROR_FL );
                                }
                        } else {
                                page_down();
                        }
                        break;
                case CODE_PPAGE:
                case PAGE_UP_CMD:
                        page_up();
                        break;
                case CODE_UP:
                        up();
                        break;
                case CODE_HOME:
                        display (REFRESH_ALL_FL);
                        break;
                case CODE_END:
                        go_bottom();
                        break;
                case NEXT_UNREAD_CMD:
                case SELECT_CMD:
                        if (true == threads.next_unread()) {
                                update();
                                hist_ring.add( orig_article );
                                display (REFRESH_ALL_FL);
                        } else {
                                iline.set( NO_UNREAD_MSG, ERROR_FL );
                        }
                        break;
                case NEXT_CMD:
                        //case CODE_RIGHT:
                        if (true == next()) {
                                update();
                                hist_ring.add( orig_article );
                                display (REFRESH_ALL_FL);
                        } else {
                                quit = true;
                        }
                        break;
                case PREV_CMD:
                        if (true == prev()) {
                                update();
                                hist_ring.add( orig_article );
                                display (REFRESH_ALL_FL);
                        }
                        break;
                case EDIT_REPLY_CMD:
                        if(mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                mail->edit( orig_article );
                                update();
                                display( REFRESH_ALL_FL );
                        }
                        break;
                case 'E':
                        if(mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                mail->edit_header( orig_article );
                                update();
                                display( REFRESH_ALL_FL );
                        }
                        break;

                case REPLY_CMD:
                        mail->reply( orig_article );
                        display(REFRESH_ALL_FL);
                        break;
                case FOLLOWUP_CMD:
                        mail->followup( orig_article );
                        display(REFRESH_ALL_FL);
                        break;
                case SAVE_TO_FILE_CMD:
                        save_thread_to_file (threads.get_thread());
                        display(REFRESH_ALL_FL);
                        break;
                case TAG_CMD:
                        if (mail->get_group_number() < BASEGROUP_NUMBER) {
                                mail->tag_article( orig_article );
                                draw_header_status();
                                rc = TAGGED_FL;
                        }
                        break;
                case KILL_CMD:
                        if (mail->get_group_number() > PERSONAL_NUMBER) {
                                orig_article->invert_kill_flag();
                                draw_header_status();
                        }
                        break;
                case FORCED_QUIT_CMD:
#ifndef NO_EXCEPTIONS
                        throw quit_exception();
#else
                        Quit_flag = true;
                        break;
#endif
                case '<':
                case CODE_CTRL_A:
                        if (true == begin_of_thread()) {
                                update();
                                hist_ring.add( orig_article );
                                display( REFRESH_ALL_FL );
                        }
                        break;
                case '>':
                case CODE_CTRL_E:
                        if (true == end_of_thread()) {
                                update();
                                hist_ring.add( orig_article );
                                display( REFRESH_ALL_FL );
                        }
                        break;
                case HELP_CMD:
                        show_help();
                        display( REFRESH_ALL_FL );
                        break;
                case PREV_IN_HIST_CMD:
                        if (true == --hist_ring ) {
                                threads.go_message( hist_ring.get() );
                                update();
                                display( REFRESH_ALL_FL );
                        }
                        break;
                case NEXT_IN_HIST_CMD:
                        if (true == ++hist_ring) {
                                threads.go_message( hist_ring.get() );
                                update();
                                display( REFRESH_ALL_FL );
                        }
                        break;
                case 0:
                        break;
                default:
                        iline.set( INVALID_COMMAND_MSG );
                        break;
                }
                iline.show();
        } while (quit == false
#ifdef NO_EXCEPTIONS
                 && Quit_flag == false
#endif
                 );

        Screen.bottom();
        Screen.clr_eol();
        return rc;
}

//**************************************************************************/
// CLASS: Msgdisplay
// MEMBER FUNCTION: begin_of_thread
//**************************************************************************/ 
//
// Moves current position to first article of current thread.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
//
// RETURN: true if success
//**************************************************************************/
bool
Msgdisplay::begin_of_thread()
{
        return threads.get_thread()->first();
}

//**************************************************************************/
// CLASS: Msgdisplay
// MEMBER FUNCTION: end_of_thread
//**************************************************************************/ 
//
// Moves position within current thread to thread's last article
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
//
// RETURN: true if success
//**************************************************************************/
bool
Msgdisplay::end_of_thread()
{
        return threads.get_thread()->last();
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: next
//
// description: Moves position to next article in a current thread.
//              If thread ends, then switch to next thread.
//
// in:  
// out: 
//
// returns: true if success.
//
//**************************************************************************/
bool
Msgdisplay::next ()
{
        if ( threads.get_thread()->next() == true) {
                return true;
        }
        // uusi kokeilu, vanhassa ei ehk toiminut threadin vaihto oikein???
        if ( threads.next() == true ) {
                return threads.get_thread()->first();
        }
        return false;
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: prev
//
// description: Moves position to previous message of a current thread.
//              If already at start of thread, then moves to last message of
//              previous thread.
// in:  
// out: 
//
// returns: true if success.
//
//**************************************************************************/
bool
Msgdisplay::prev()
{
        bool rc = threads.get_thread()->prev();
        if (rc == false) {
                if (threads.prev() == true) {
                        rc = threads.get_thread()->last();
                }
        }
        return rc;
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: update
//
// description: Counts some necessary values and makes sure that lines
//              are read from file to memory. (Message::first_line()).
// FIXME: lis maininnat orig_articlen assosionnista...
//
// NOTE: This should be called only when position in thread is changed!!!!
//
//**************************************************************************/
void
Msgdisplay::update()
{
        xmax = Screen.get_xmax();
        orig_article = threads.get_article();
        orig_article->first_line();
        article = *orig_article;
        if ( modefl & WRAPMODE ) {
                article.wrap( xmax+1 );
        }
        lines_in_article = article.get_num_of_lines();
        first_of_page = 0;
        pages = lines_in_article / lines_per_page + 1;
}

//**************************************************************************/
//  CLASS: Msgdisplay
//  MEMBER FUNCTION: go_bottom
// 
// 
//  DESCRIPTION: Moves position to bottom of article
// 
// 
//**************************************************************************/
void
Msgdisplay::go_bottom()
{
        for (;;) {
                article.save_position();
                int i = lines_per_page - 1;
                while (i) {
                        article.next_line();
                        i--;
                }
                if (article.next_line() == true) {
                        first_of_page += lines_per_page;
                } else {
                        article.restore_position();
                        break;
                }
        }
        display();
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: display
//
// description: Displays everyhting. If flag is REFRESH_ALL_FL, then
//              moves top of article and redraws header and footer.
//
//
// in:  threadlist (number of articles)
// out: 
//
// returns: 
//
//**************************************************************************/
void
Msgdisplay::display (int flag)
{
        String str;
        int i;
        int y;
        int pros;
        int lastline = lines_in_article;
        int width = xmax + 1;
        Screen.reset_attr();
        if (flag & REFRESH_ALL_FL) {
                if (!(orig_article->get_status() & READ_ST) &&
                    mail->get_group_number() != PERSONAL_NUMBER &&
                    mail->get_group_number() != NEW_REPLIES_NUMBER) {
                        // Set articles status to readed
                        orig_article->set_status_bits (READ_ST);
                        threads.get_thread()->dec_unread();
                }
                Screen.erase();
                
                draw_header();
                if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                        iline.set_default( MSG_DISPLAY_NEW_REPLIES_KEYS_MSG);
                } else if (mail->get_group_number() >= BASEGROUP_NUMBER) {
                        iline.set_default( MSG_DISPLAY_BASEGROUP_KEYS_MSG);
                } else {
                        iline.set_default( MSG_DISPLAY_KEYS_MSG);
                }
                iline.show_default();
                first_of_page = 0;
                article.first_line();
        }
        // show group on status bar. If current group is some of the
        // basegroups, then display original group of the article
        str = "Group: [";
        str += orig_article->get_group_number();
        str += "] ";
        if (mail->get_group_number() >= BASEGROUP_NUMBER) {
                mail->save_group_pos();
                mail->go_group_number( orig_article->get_group_number());
                str +=  mail->get_group_name();
                mail->restore_group_pos();
        } else {
                str += mail->get_group_name();
        }

        if ((modefl & (ROT13MODE | WRAPMODE)) == (ROT13MODE | WRAPMODE)) {
                str += " (mode:ROT13,WRAP)";
        } else if ( modefl & ROT13MODE ) {
                str += " (mode:ROT13)";
        } else if ( modefl & WRAPMODE ) {
                str += " (mode:WRAP)";
        }
        DEBUG( "modefl = " << modefl << " " + str);
        sline.set_infotext( str.get() );
        
        // print message lines
        y = ymin;
        if (article.check_lines() == true) {
                do {
                        Screen.gotoxy (0,y);
                        Screen.clr_eol();
                        if ( article.getline().length() > width ) {
                                str.copy (article.getline(), 0, xmax);
                        } else {
                                str.copy (article.getline(), 0, width);
                        }
                        if (str.find_any_ch (">", 6) == true) {
                                Screen.textcolor (Settings.quote_color);
                        } else if (str.find_any_ch (QUOTECHARS, 2) == true) {
                                Screen.textcolor (Settings.quote_color);
                        } else {
                                Screen.textcolor (Settings.article_color);
                        }
                        if ( modefl & ROT13MODE ) {
                                str.rot13();
                        }
                        Screen.addstr( str );
                        if (article.getline().length() > width) {
                                Screen.textcolor (RED | BOLD);
                                if ( modefl & WRAPMODE ) {
                                        Screen.addch( Settings.wrap_indicator);
                                } else {
                                        Screen.addch ('$');
                                }
                        }
                        Screen.reset_attr();

                        if (++y > ymax)
                                break;
                } while (article.next_line() == true);
                lastline = y - ymin + first_of_page;
                pros = 100 * lastline;
                pros /= lines_in_article;
        } else {
                pros = 100;
        }
        i = y-ymin-1;

        // clear bottom of screen
        while (y <= ymax) {
                Screen.gotoxy(0,y);
                //Screen.empty_line();
                Screen.clr_eol();
                y++;
        }
        
        // Test, if current line is last line of message
        is_bottom = false;
        if (article.next_line() == true) {
                sline.set_percent( pros );
                article.prev_line();
                str = "L";
                str += lastline;
                str += "/";
                str += lines_in_article;
                sline.set_extra_info( str.get() );
        } else {
                Msgthread *tmp = threads.get_thread();
                is_bottom = true;
                if (tmp->next() == true) {
                        tmp->prev();
                        sline.set_extra_info( "Next Response" );
                } else {
                        sline.set_extra_info("Last Response");
                }
                if (lines_in_article <= ymax - ymin) {
                        sline.set_percent( ALL_PERCENT );
                } else {
                        sline.set_percent( BOT_PERCENT );
                }
        }

        // move pointer to first line of page
        while (i>0) {
                article.prev_line();
                i--;
        }
        sline.show();
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: draw_header
//
// description: Displays some interesting data at top of screen.
//
// NOTE: this displays data of original message 'orig_article' not
//       wrapped one!
//
//**************************************************************************/
void
Msgdisplay::draw_header()
{
        String str;

        menubar.show();
        Screen.reset_attr();
        for (int i=1; i <= 3; i++) {
                Screen.gotoxy( 0, i );
                Screen.empty_line();
        }
        Screen.gotoxy(0,1);
        Screen.print("   Date: %s, %s ", orig_article->get_date().c_str(),
                      orig_article->get_time().c_str() );

        draw_header_status();
        
        Screen.gotoxy (0,2);
        Screen.addstr( "   From: " );
        Screen.textcolor (Settings.header_author_color);
        Screen.addstr( str.copy (orig_article->get_writer(), 0, 40) );
        if (orig_article->get_writer().length() > 40) {
                Screen.addch( '$' );
        }
        Screen.reset_attr();
        Screen.gotoxy( 0, 3 );
        Screen.addstr( "     To: " );
        if (orig_article->get_receiver() == Settings.username) {
                Screen.textcolor (Settings.personal_color | BLINK);
        }
        Screen.addstr( str.copy (orig_article->get_receiver(), 0, 40) );
        if (orig_article->get_receiver().length() > 40) {
                Screen.addch( '$' );
        }
        Screen.reset_attr();
        
        Screen.gotoxy( Screen.get_xmax() - 30,2);
        Screen.print( "Msgno : %d", orig_article->get_number() );
        Screen.print( " Ref: %d", orig_article->get_reference_num() );
        Screen.gotoxy( Screen.get_xmax() - 30, 3);
        Screen.print( "Respno: %d/%d", orig_article->get_number_in_thread(),
                      threads.count_threads_articles() );
        
        Screen.gotoxy( 0, 4 );
        Screen.addstr( "Subject: " );
        if (Screen.is_color_term() == true) {
                Screen.textcolor ( Settings.header_subject_color );
        } else {
                Screen.textcolor ( BOLD );
        }
        Screen.addstr( str.copy (orig_article->get_subject(), 0, 72) );
        if (orig_article->get_subject().length() > 72) {
                Screen.addch( '$' );
        }
        Screen.gotoxy( 0, 5 );

        set_dashline_color();
        Screen.addstr( dashline );
        Screen.reset_attr();
}
//**************************************************************************/
// CLASS: Msgdisplay
// MEMBER FUNCTION: draw_header_status
//**************************************************************************/ 
//
// Display status information of current article.
// This also queries information directly from original message (orig_article)
// not from wrapped message, 'article'.
// 
//**************************************************************************/
void
Msgdisplay::draw_header_status()
{
        Screen.gotoxy( Screen.get_xmax() - 30, 1);
        Screen.reset_attr();
        Screen.clr_eol();
        Screen.addstr( "Status: " );
        if (orig_article->get_status() & REPLIED_ST) {
                Screen.textcolor (CYAN | BOLD | BLINK);
                Screen.addstr( "Replied" );
                Screen.reset_attr();
        } else if (orig_article->get_status() & PRIVATE_ST) {
                Screen.textcolor(MAGENTA);
                Screen.addstr( "Private" );
        } else {
                Screen.addstr( "Public " );
        }
        Screen.gotoxy( Screen.get_xmax() - 14, 1);
        if (orig_article->is_killed() == true) {
                Screen.textcolor (RED | BOLD | BLINK);
                Screen.addstr( "! Killed !" );
        } else if (orig_article->get_status() & TAGGED_ST) {
                Screen.textcolor (RED | BOLD);
                Screen.addstr( "- Tagged -" );
        }
        Screen.reset_attr();
}


//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: down
//
// description: Scrolls one line down
//
//
//**************************************************************************/
void
Msgdisplay::down()
{
        article.save_position();
        int i = lines_per_page - 1;
        while (i) {
                article.next_line();
                i--;
        }
        if (article.next_line() == true) {
                article.restore_position();
                article.next_line();
                first_of_page++;
                display();
        } else {
                article.restore_position();
        }
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: Up
//
// description: Scrolls one line up.
//
//
//**************************************************************************/
void
Msgdisplay::up()
{
        if (first_of_page > 0) {
                first_of_page--;
                article.prev_line();
                display();
        }
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: page_down
//
// description: Moves one page down, or if end of article and flag is
//              NEXT_FL then moves to next article.
//
//
// returns: false if no more articles
//
//**************************************************************************/
bool
Msgdisplay::page_down (int flag)
{
        bool rc = true;
        article.save_position();
        int i = lines_per_page - 1;
        while (i) {
                article.next_line();
                i--;
        }
        if (article.next_line() == true) {
                first_of_page += lines_per_page;
                display();
        } else if (flag & NEXT_FL) {
                //  if position is bottom of article, then switch to next
                //  article
                article.restore_position();
                if (true == (rc = threads.next_unread()) ) {
                        update();
                        display (REFRESH_ALL_FL);
                }
        }
        return rc;
}

//**************************************************************************/
//
// CLASS: Msgdisplay
// MEMBER FUNCTION: page_up
//
// description: Moves one page up within article.
//
//
//**************************************************************************/
void
Msgdisplay::page_up()
{
        int i = lines_per_page;
        while (i && first_of_page > 0) {
                first_of_page--;
                i--;
                article.prev_line();
        }
        display();
}
