// $Id: mail.hh,v 1.11 1998/05/25 20:13:08 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     mail.hh
 * *                  ---------
 * ***************************************************************************
 * *
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO. ALL RIGHTS RESERVED
 * ***************************************************************************
 * *
 * *      This module defines classes for handling groups,
 * *      messagedatabase and mail packets.
 * *
 *****************************************************************************/

#ifndef MAIL_HH
#define MAIL_HH

#include "../utilib/my_types.h"
#include "constants.h"
#include "../utilib/String.hh"
#include "../utilib/List.hh"
#include "messagedata.hh"
#include "replies.hh"
#include "tree.hh"
#include "datatypes.hh"


//**************************************************************************/
//
// CLASS: bbs_settings
//
// DERIVED FROM:
//
//        Class for reading bbs related settings from the resource
//        file. Name of that rc-file is <bbsid>.rc
//
//**************************************************************************/
class bbs_settings {
public:
        bbs_settings();
        ~bbs_settings(){};
        bool set (String& bbsid); // set up with values of <bbsid>.rc
        bool restore();           // restore original settings.
private:
        bool read_resources();
        bool parse_rcline(String&, int);
        
        String rcfname;

        key_tree trie;

        bool save_flag;
        settings_t saved;
};


//**************************************************************************/
//
// CLASS: Group
//
// DERIVED FROM:
//
// Object for handling contents of a single group.
//
//**************************************************************************/
class Group {
public:
        Group (const int& g_num, const String& g_name);
        const Group& operator = (const Group &); // not implemented
        Group (const Group&); // not implemented
        ~Group();

        int get_number() const;       // get numeber of group
        const String& get_numstr() const; // get number of group as String
        int count_articles() const;   // get amount of articles of group
        int get_sort_order() const;   // get sort order
        int count_unread() const;     // get amount of unread articles
        const String& get_name() const; // get name of group as a String
        bool is_used() const;         // return true if group contains articles

        bool is_tagged() const;
        void tag();
        
        void set_name( const String& s ) { name = s; }
        bool first_article();         // go to first article within group
        bool last_article();          // go to last article
        bool next_article();          // go to next article
        bool prev_article();          // go to previous article
        bool add (const Message *msg);// add new article to group
        void remove_article();        // remove current article from group
        void save_position();         // save positio within articles
        void restore_position();      // restore position
        Message* get_article() const; // get current article
        
        void set_mirror();            // set is_mirror flag on
        void sort (const int order);  // sort articles to given order
        void unload();            // unload lines of all articles (if possible)
        void update();                // update statistics of group
        void mark_all_read();         // mark all articles as read
private:
        bool is_mirror;  // true if contents of group is mirrored from some
                         // other groups (i.e. articles are only pointers)
        int status_;     // status flag.
        int number;      // Number of group
        int unread_msgs; // amount of unread articles
        int total_msgs;  // amount of all articles
        int sort_order;  // order of articles THREAD_SORTED, TIME_SORTED, ...
        String numberstr;       // number of group as a String
        String name;            // name of group
        List<Message> messages; // contents of group in list
};


//**************************************************************************/
//
// CLASS: Packet
//
// DERIVED FROM:
//
// Object for storing data of offline mail packet. This is abstract class.
//
//**************************************************************************/
class Packet {
public:
        Packet() : packet_articles(0) {};
        virtual ~Packet() { delete packet_articles; };
protected:
        String bbsname;
        String city;
        String phone;
        String sysop;
        String door;
        String time;
        String username;
        List<Group> *packet_articles; // all groups and articles in list
};


//**************************************************************************/
//
// CLASS: Mail
//
// DERIVED FROM: Packet
//
// Object for handling contents of mail packet and message database.
// This is heart of jmr offline mail reader. Class is abstract, and cannot
// used directly.
//
// Object must be init using static members Mail::create() and
// Mail::open_dead(). They creates right derived class for situation and
// returns pointer to it.
//
//**************************************************************************/
class Mail : public Packet {
public:
        static Mail* create (const char*, bool); // create new Mail-object
        static Mail* open_dead(); // create new Mail-object using killed one.
        
        Mail (const char* packet_name); // constructor. This must be called
                                        // in constructor of derived class
        virtual ~Mail();
        
        virtual void read_articles()=0; //every derived class should have this
        
        virtual void emergency_exit(); // Writes only new replies to
                                       // emergency file (e.g. if program is
                                       // killed)
        virtual void write_recovery_file(); // writes new replies to recovery
                                            // file
        
        virtual void read_deadjmr();   // reads killed session

        virtual bool go_group_number (const int num);
        virtual bool go_group_index (const int n);
        virtual bool first_group();    // go to first group
        virtual bool next_group();     // go to next group
        virtual bool prev_group();     // go to previous group
        virtual void save_group_pos();
        virtual void restore_group_pos();
        // get reference to group of given index
        virtual const Group& operator [] (int index); 

        virtual bool first_article();  // go first article of current group
        virtual bool next_article();   // next article of current group
        virtual bool prev_article();   // previous article of curr....
        virtual bool save_message_pos();
        virtual bool restore_message_pos();
        
        virtual Message* get_article() const; // get current article
        virtual void tag_article (Message* msg); // tag current article
        virtual void tag_group();
        
        virtual int count_articles() const; // articles in current group
        virtual int count_unread() const;   // unread articles in current...
        virtual int count_new_articles() const; // count of new articles 
        virtual int count_all_articles() const; // total articles

        virtual int count_tagged_groups() const;
        
        virtual bool is_group_used() const; // true if group is used
        virtual bool is_group_tagged() const;

        
        // unload lines of all articles of current group, if possible.
        virtual void unload_articles_of_group();
        virtual void catchup();                      // mark group as read
        virtual void sort (const int order);         // sort articles of group
        virtual void add_reply (Message* reply);     // add new reply
        virtual void kill_reply (const Message* msg);// kill reply
        
        virtual const String& get_bbs_name() const;
        virtual const String& get_bbs_sysop() const;
        virtual const String& get_bbsid() const;
        virtual const String& get_bbs_phone() const;
        virtual const String& get_packet_time() const;
        virtual const String& get_username() const;
        
        virtual int get_group_number() const; // get number of current group
        virtual const String& get_group_numstr() const;
        virtual const String& get_group_name() const; // get name of group
        
        virtual int get_sort_order() const;  // get sorting order
        virtual int get_num_groups() const;  // get amount of groups
        virtual bool have_replies() const;   // true if new replies exist
        virtual void update_groups_counts(); // update statistics of group

        // for writing new articles and editing exist reply
        virtual void followup (Message *msg);
        virtual void reply (Message *msg);
        virtual void edit (Message *msg);
        virtual void write();
        virtual void edit_header( Message *msg );
        
        virtual int search (char* pattern, int mode);
protected:
        String bbsid;    // Id string of bbs.
        bool dead_flag;  // true if session is killed
        void init();     // initializes Mail-object. Must call in end of
                         // constructors of derived classes
private:
        bool ask_replace() const;
        void move_messages_to_base();
        void set_personal_mail();  
        void thread_sort_articles ();
        void read_message_database();
        void read_replylog();
        void read_groups_from_db();
        void read_articles_from_db();
        void write_replylog();
        void write_log (const String&, const int, const int, 
                        bool keep_charset = false);
        void write_message_database();
        bool write_groups_to_db();
        bool write_articles_to_db();

        int  search_all(char*, int);
        int  search_current(char*, int);
        int  search_tagged(char*, int);
        
        Replyhandler *replyhandler; // object for writing replies
        bool have_new_replies; // true if new replies are written
        List<Group> grouplist; // all groups and articles in list
        
        int articles;        // total number of articles
        int new_articles;    // articles readed from packet

        int num_tagged_groups; // number of tagged groups
        
        String tmp_log_name; // name of temporaly file for replylog
        String tmp_db_name;  // name of temp. file for message database
        fstream* tmp_db_file; // handle to temp. file for database
        fstream* tmp_log_file;// handle to temp. file for replylog

        bbs_settings local_settings;
};



//         ******************************************************
//         **                                                  **
//         ** Inline functions of class Group                  **
//         **                                                  **
//         **                                                  **
//         ******************************************************




//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: first_article
//**************************************************************************/ 
//
// Go to first article of group
// 
// RETURN: false if group is empty
//**************************************************************************/
inline bool
Group::first_article()
{
        return messages.first();
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: next_article
//**************************************************************************/ 
//
// Go to next article.
// 
// RETURN: false if failed.
//**************************************************************************/
inline bool
Group::next_article()
{
        return messages.next();
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: prev_article
//**************************************************************************/ 
//
// Go to previous article.
// 
// RETURN: false if failed
//**************************************************************************/
inline bool
Group::prev_article()
{
        return messages.prev();
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: get_number
//**************************************************************************/ 
//
// Returns number of group
// 
//**************************************************************************/
inline int
Group::get_number() const
{
        return number;
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: get_article
//**************************************************************************/ 
//
// Returns pointer to current article
// 
//**************************************************************************/
inline Message*
Group::get_article() const
{
        assert (messages.check() == true);
        return messages.get();
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: get_sort_ordef
//**************************************************************************/ 
//
// Gets order of articles 
// 
//**************************************************************************/
inline int
Group::get_sort_order() const
{
        return sort_order;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: count_articles
//**************************************************************************/ 
//
// returns number of articles
// 
//**************************************************************************/
inline int
Group::count_articles() const
{
        return total_msgs;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: count_unread
//**************************************************************************/ 
//
// Returns number of unread articles
// 
//**************************************************************************/
inline int
Group::count_unread() const
{
        return unread_msgs;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: is_used
//**************************************************************************/ 
//
// Checks if group is used or if it is empty.
// 
// RETURN: true if group is used.
//**************************************************************************/
inline bool
Group::is_used() const
{
        //return unread_msgs > 0 ? true : false;
        return messages.check();
}


inline bool
Group::is_tagged() const
{
        return status_ & TAGGED_FL ? true : false;
}


inline void
Group::tag()
{
        status_ ^= TAGGED_FL;
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: set_mirror
//**************************************************************************/ 
//
// Sets group as mirrored. Use this if one or more articles of group are
// only pointers to data that is stored in somewhere else.
// NOTE: If one article is mirrored, then all others should also be. This
//       is because data of mirrored group is not deleted in destructor.
// 
//**************************************************************************/
inline void
Group::set_mirror()
{
        is_mirror = true;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: get_numstr
//**************************************************************************/ 
//
// Returns number of group in String
// 
//**************************************************************************/
inline const String&
Group::get_numstr() const
{
        return numberstr;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: get_name
//**************************************************************************/ 
//
// Returns name of group in String
// 
//**************************************************************************/
inline const String&
Group::get_name() const
{
        return name;
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: save_position
//**************************************************************************/ 
//
// Save current position within articles of group. Only latest saved postion
// can be restored, so be CAREFULLY with this!
// 
//**************************************************************************/
inline void
Group::save_position()
{
        messages.save_position();
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: restore_position
//**************************************************************************/ 
//
// Restores latest saved postion within articles of group
// 
//**************************************************************************/
inline void
Group::restore_position()
{
        messages.restore_position();
}





//         ******************************************************
//         **                                                  **
//         ** Inline functions of class Mail                   **
//         **                                                  **
//         **                                                  **
//         ******************************************************


//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: unload_articles_of_group
//**************************************************************************/ 
//
// Removes lines of these articles of current group, that have file stream
// handle and are stored to file.
// 
//
//**************************************************************************/
inline void
Mail::unload_articles_of_group()
{
        grouplist.get()->unload();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: update_groups_counts
//**************************************************************************/ 
//
// Updates statistics of current group
// 
//**************************************************************************/
inline void
Mail::update_groups_counts()
{
        grouplist.get()->update();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: have_replies
//**************************************************************************/ 
//
// Returns true if new replies is written within current session.
// 
//**************************************************************************/
inline bool
Mail::have_replies() const
{
        return have_new_replies;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: is_group_used
//**************************************************************************/ 
//
// Returns true if current group is not empty.
// 
//**************************************************************************/
inline bool
Mail::is_group_used() const
{
        return grouplist.get()->is_used();
}

inline bool
Mail::is_group_tagged() const
{
        return grouplist.get()->is_tagged();
}



//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_num_groups
//**************************************************************************/ 
//
// Return number of groups, including basegroups.
// 
//**************************************************************************/
inline int
Mail::get_num_groups() const
{
        return grouplist.count_items();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_sort_order
//**************************************************************************/ 
//
// Returns order of articles as a constant (THREAD_SORTED, ...)
// 
//**************************************************************************/
inline int
Mail::get_sort_order() const
{
        assert (grouplist.get() != 0);
        return grouplist.get()->get_sort_order();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: count_new_articles
//**************************************************************************/ 
//
// Returns number of new articles readed from QWK (or other) packet to
// database.
// 
//**************************************************************************/
inline int
Mail::count_new_articles() const
{
        return new_articles;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: count_all_articles
//**************************************************************************/ 
//
// Returns number of all articles, including new articles and contents of
// database.
// 
//**************************************************************************/
inline int
Mail::count_all_articles() const
{
        return articles;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: count_articles
//**************************************************************************/ 
//
// Returns number of articles in current group
// 
//**************************************************************************/
inline int
Mail::count_articles() const
{
        assert (grouplist.get() != 0);
        return grouplist.get()->count_articles();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: count_unread
//**************************************************************************/ 
//
// Returns number of unread articles in current group
// 
//**************************************************************************/
inline int
Mail::count_unread() const
{
        assert (grouplist.get() != 0);
        return grouplist.get()->count_unread();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_group_name
//**************************************************************************/ 
//
// Returns name of current group in String
// 
//**************************************************************************/
inline const String&
Mail::get_group_name() const
{
        assert (grouplist.check() == true);
        return grouplist.get()->get_name();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_group_numstr
//**************************************************************************/ 
//
// Returns number of current group in String
// 
//**************************************************************************/
inline const String&
Mail::get_group_numstr() const
{
        return grouplist.get()->get_numstr();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_username
//**************************************************************************/ 
//
// Returns username defined in mail packet (QWK, ...)
// 
//**************************************************************************/
inline const String&
Mail::get_username() const
{
        return username;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_bbs_name
//**************************************************************************/ 
//
// Returns name of bbs
// 
//**************************************************************************/
inline const String&
Mail::get_bbs_name() const
{
        return bbsname;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_bbs_sysop
//**************************************************************************/ 
//
// Returns name of sysop
// 
//**************************************************************************/
inline const String&
Mail::get_bbs_sysop() const
{
        return sysop;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// returns phone number of bbs
// 
//**************************************************************************/
inline const String&
Mail::get_bbs_phone() const
{
        return phone;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_bbsid
//**************************************************************************/ 
//
// Returns id string of bbs.
// 
//**************************************************************************/
inline const String&
Mail::get_bbsid() const
{
        return bbsid;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_packet_time
//**************************************************************************/ 
//
// Returns time of mail packet
// 
//**************************************************************************/
inline const String&
Mail::get_packet_time() const
{
        return time;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_group_number
//**************************************************************************/ 
//
// returns number of current group
// 
//**************************************************************************/
inline int
Mail::get_group_number() const
{
        return grouplist.get()->get_number();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: save_group_pos
//**************************************************************************/ 
//
// Saves current position within groups. Only latest saved position is
// stored, so be carefully.
// 
//**************************************************************************/
inline void
Mail::save_group_pos()
{
        grouplist.save_position();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: restore_group_pos
//**************************************************************************/ 
//
// Restores latest saved position within groups.
// 
//**************************************************************************/
inline void
Mail::restore_group_pos()
{
        grouplist.restore_position();

}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: first_article
//**************************************************************************/ 
//
// Go to first article within current group.
// 
// RETURN: false if group is empty
//**************************************************************************/
inline bool
Mail::first_article()
{
        return grouplist.get()->first_article();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: save_message_pos
//**************************************************************************/ 
//
// Saves current position within articles of current group. Only latest
// saved position will be stored.
// 
// RETURN: true
//**************************************************************************/
inline bool
Mail::save_message_pos()
{
        grouplist.get()->save_position();
        return true;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: restore_message_pos
//**************************************************************************/ 
//
// Restores position within articles of current group.
// 
// RETURN: true
//**************************************************************************/
inline bool
Mail::restore_message_pos()
{
        grouplist.get()->restore_position();
        return true;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: next_article
//**************************************************************************/ 
//
// Go to next article of current group.
// 
// RETURN: false if group is empty, or if theres no next article
//**************************************************************************/
inline bool
Mail::next_article()
{
        return grouplist.get()->next_article();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: prev_article
//**************************************************************************/ 
//
// Go to previous article of curretn group.
// 
// RETURN: true if OK.
//**************************************************************************/
inline bool
Mail::prev_article()
{
        return grouplist.get()->prev_article();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: get_article
//**************************************************************************/ 
//
// Returns pointer to current article of current group
// 
//**************************************************************************/
inline Message*
Mail::get_article() const
{
        return grouplist.get()->get_article();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: catchup
//**************************************************************************/ 
//
// Marks all articles of current group as read
// 
//**************************************************************************/
inline void
Mail::catchup()
{
        grouplist.get()->mark_all_read();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: reply
//**************************************************************************/ 
//
// Write new reply using given article as a reference.
// 
//**************************************************************************/
inline void
Mail::reply (Message* msg)
{
        replyhandler->reply (msg);

}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: followup
//**************************************************************************/ 
//
// Write followup to given article
// 
//**************************************************************************/
inline void
Mail::followup (Message* msg)
{
        replyhandler->followup (msg);

}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: edit
//**************************************************************************/ 
//
// Edit given reply.
// 
//**************************************************************************/
inline void
Mail::edit (Message* msg)
{
        replyhandler->edit (msg);

}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write
//**************************************************************************/ 
//
// Write new article to current group
// 
//**************************************************************************/
inline void
Mail::write()
{
        replyhandler->write();

}

inline void
Mail::edit_header( Message* msg)
{
        replyhandler->edit_header( msg );
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: first_group
//**************************************************************************/ 
//
// Sets position in list of groups to first group.
// 
// RETURN: false if list is empty.
//**************************************************************************/ 
inline bool
Mail::first_group()
{
        return grouplist.first();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: next_group
//**************************************************************************/ 
//
// Moves position in list of groups to next group.
// 
// RETURN: false if failed
//**************************************************************************/
inline bool
Mail::next_group ()
{
        return grouplist.next();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: prev_group
//**************************************************************************/ 
//
// Moves position in list of groups to previous group.
// 
// RETURN: false if failed
//**************************************************************************/
inline bool
Mail::prev_group()
{
        return grouplist.prev();
}



#endif
