// $Id: jmrmisc.cc,v 1.3 1998/05/25 20:06:04 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     jmrmisc.cc
 * *                  ----------
 * *
 * *
 * * Revision  : $Revision: 1.3 $
 * * Date(UTC) : $Date: 1998/05/25 20:06:04 $
 * * Source    : $Source: /usr/local/cvsroot/jmr/src/jmrmisc.cc,v $
 * * Author    : $Author: jvuokko $
 * *
 * ***************************************************************************
 * *    This file is part of jmr offline mail reader.
 * *
 * *    Copyright (C) 1998 Jukka Vuokko <jvuokko@iki.fi>
 * *
 * *    This program is free software; you can redistribute it and/or modify
 * *    it under the terms of the GNU General Public License as published by
 * *    the Free Software Foundation; either version 2 of the License, or
 * *    (at your option) any later version.
 * *
 * *    This program is distributed in the hope that it will be useful,
 * *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 * *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * *    GNU General Public License for more details.
 * *
 * *    You should have received a copy of the GNU General Public License
 * *    along with this program; if not, write to the Free Software
 * *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***************************************************************************
 * *
 * *      This module contains miscallenous helper functions for jmr
 * *
 *****************************************************************************/

#include <stdio.h>      // sprintf(), rand(), srand(), tmpnam()
#include <time.h>       // time()
#include <stdlib.h>     // EXIT_FAILURE, EXIT_SUCCESS
#include <signal.h>
#include <string.h>     // strerror
#include <errno.h>

#ifdef __WATCOMC__      // Watcom stuff
#   include <io.h>         // access()
#   include <direct.h>
#elif defined (__EMX__)  // EMX-stuff
#   include <unistd.h>   // F_OK
#   include <sys/types.h>
#   include <dirent.h>
#else                    // UNIX
#   include <sys/types.h>
#   include <sys/wait.h>    // wait()
#   include <sys/utsname.h> // uname()
#   include <unistd.h>   // F_OK
#   include <dirent.h>
#   include <fcntl.h>    // open, write, creat, read
#endif

#include "constants.h"
#include "jmr.hh"
#include "time.hh"
#include "terminal_io.hh"
#include "linereader.hh"

//-------------------------------
// for set current diskdrive and copying file
#ifdef OS2
//#   define INCL_DOSERRORS
#   define INCL_ERRORS
#   define INCL_DOSFILEMGR
#   include <os2.h>
#endif

#if defined (WIN32) || defined (DOS)
#   include <dos.h>
#endif

#ifdef __unix__
#   include <unistd.h>  // getcwd, chdir
#endif

//****************** GLOBAL VARIABLES *************************************/
extern settings_t Settings;
extern Terminal_screen Screen;
extern Window *Wscript;

static const char* help_text[] = {
        "@COMMON MOVING COMMANDS:",
        "",
        "  HOME, ESC <    Beginning of list/article",
        "  END, ESC >     End of list/article",
        "  PGUP, Ctrl-v   Page up",
        "  PGDN, ESC v    Page down",
        "  UP ARROW       Line up",
        "  DOWN ARROW     Line down",
        "  LEFT ARROW     Back to previous level",
        "  RIGHT ARROW    Enter to next level",
        "",
        "@COMMON EDITING COMMANDS:",
        "",
        "  Ctrl-a, Ctrl-e Move to beginning or end of the line",
        "  Ctrl-k         Delete all characters, from cursor to end of line",
        "  Ctrl-h         Delete the previous character",
        "  Ctrl-b, Ctrl-f Move backward of forward",
        "",
        "@MAIL PACKET AND DATABASE SELECTION:",
        "",
        "  r              Refresh file or database list",
        "  n              Next packet or database",
        "  p              Previous packet or database",
        "  ENTER          Open packet or database",
        "  l              Switch to list of available databases or list of",
        "                   available mail packets",
        "  q, Q           Quit jmr",
        "",
        "@GROUP SELECTION:",
        "",
        "  n              Next group",
        "  p              Previous group",
        "  ENTER, ->      Enter to group",
        "  TAB            Enter to next group with unread articles",
        "  w              Write an article to current group",
        "  y              Switch between all groups and groups with unread articles",
        "  c              Catchup. Mark all articles in current group as read",
        "  C              Catchup without questions",
        "  t              Tag/untag current group",
        "  T              Tag/untag all groups",
        "  /              Search from current, tagged or all groups",
        "  q              Return to packet selection level",
        "",
        "@THREAD SELECTION:",
        "",
        "  n              Next thread",
        "  p              Previous thread",
        "  TAB            View next unread article within all threads",
        "  ENTER, ->      View first article of current thread",
        "  w              Write an article to current group",
        "  f              Write followup to current article or first article",
        "                   of current thread, if reading mode is by threads",
        "  r              Write private reply to current article (or first",
        "                   article of current thread)",
        "  y              Show all threads or only threads with unread articles",
        "  t              Tag/untag current article (does not work in thread mode)",
        "  T              Tag/untag all articles (not in thread mode)",
        "  s              Sort articles",
        "  c              Catchup. Mark all articles within current group as",
        "                   a read",
        "  C              Forced catchup",
        "  e              Edit current reply. Works only in group 'New Replies'",
        "  E              Edit headers of a current reply.",
        "  k              Kill/unkill article. Works only in groups Replylog,",
        "                   New Replies and Tagged Articles",
        "  q, <-          Return to group selection level",
        "",
        "@MESSAGE VIEWER:",
        "",
        "  TAB, ENTER     View next unread article within current group",
        "  n              View next article",
        "  p              View previous article",
        "  <, Ctrl-a      View first article of a current thread",
        "  >, Ctrl-e      View last article of a current thread",
        "  N              Next article in history",
        "  P              Previous article in history",
        "  f              Write followup to current article",
        "  r              Write private reply to current article",
        "  e              Edit current reply. Works only in group 'New Replies'",
        "  E              Edit headers of a current reply.",
        "  t              Tag/untag current article",
        "  k              Kill/unkill current article. Only in groups:",
        "                   Replylog, New Replies and Tagged Articles",
        "  s              Save current article or thread to file",
        "  SPC            Page down, or if end of current article, then",
        "                   view next unread article",
        "  ESC-r          View article using Rot13",
        "  ESC-w          Toggle line wrapping mode on/off",
        "  q, LEFT ARROW  Return to thread selection level",
        "",
        "@MISC",
        "",
        "  Ctrl-l         Redraw screen",
        "  m              Show message buffer",
        "  Q              Quit jmr",
        "@END" // This must be last item of the array!
};



//*************************************************************************/
// FUNCTION: read_tagline
//*************************************************************************/
//
// Function reads random line from given file.
//
//
// RETURN: line as String object.
//*************************************************************************/
String read_tagline (String const &file)
{
        List<String> *taglines;
        Linereader l( file );

        // read lines from file of taglines to linked list
        taglines = l.read_all_lines();

        // get random line
        String tmp;
        DEBUG( "read_tagline: Lines = " << taglines->count_items());
        int index = rand() % taglines->count_items();
        DEBUG( "read_tagline: random is: " << index );
        if ( taglines->first() == true ) {
                while ( index ) {
                        taglines->next();
                        --index;
                }
                tmp = *taglines->get();
        }
        delete taglines;
        return tmp;
}


//*************************************************************************/
// FUNCTION: get_tagline
//*************************************************************************/
//
// Function gets tagline from file of taglines, or from resource file.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : Settings.jmrdir, Settings.tagline
//   OUT: 
//
// RETURN: tagline as String
//*************************************************************************/
String
get_tagline ()
{
        // default tagline
        String tag = Settings.tagline;
        tag.cut_tail();
        // read random tagline from file.
        if (Settings.tagline.nget (0) == '@') {
                String file;
                String tmp;
                tmp.copy (Settings.tagline, 1);
                correct_path (tmp);  // expand possible ~
                tmp.ncut_end (1);    // cut trailing '/' or '\'
                file = tmp;
                if (ACCESS (file.get(), F_OK) == 0) {
                        tag = read_tagline (file);
                } else {
                        file = "/usr/local/lib/jmr/" + tmp;
                        if (ACCESS (file.get(), F_OK) == 0) {
                                tag = read_tagline (file);
                        } else {
                                file = Settings.jmrdir + tmp;
                                if (ACCESS (file.get(), F_OK) == 0) {
                                        tag = read_tagline (file);
                                } else {
                                        handle_error ("Cannot access file of taglines.");
                                        tag = "";
                                }
                        }
                }
        }

           // Remove possible ^M from the tagline. Seems like that
           // usually tagline file is in MS-DOG format.... 
           // (where is good old dos2unix?)
        tag.strip_ch( '\r' );
        return tag;
}

/***************************************************************************
 * FUNCTION: clear_workdir
 ***************************************************************************
 *
 * DESCRIPTION: Removes all files from working directory
 *
 * EXTERNAL REFERENCES
 * IN : Settings.workdir
 * OUT:
 *
 ***************************************************************************/
void
clear_workdir()
{
        String path;
        String file;
        String tmp;
        
        path = Settings.workdir;
        correct_path (path);
        
        file = path + "*.[nN][dD][xX]";
        rm_files( file );

        file = path + "*.[dD][aA][tT]";
        rm_files( file );

        file = path + "*.[iI][dD]";
        rm_files( file );

        file = path + "*.[mM][sS][gG]";
        rm_files( file );
        
        // remove files with exact name.
        // Under Os/2, readdir() returns uppercase name, if file is
        // stored to FAT file system. That's why in Os/2 it is needed
        // also check for uppercase names...
        tmp = DEADJMR_FILENAME;
        file = path + tmp;
        rm_files( file );

#ifdef OS2
        file = path + tmp.upcase();
        rm_files( file );
#endif
        
        tmp = DEADJMR_ID_FILENAME;
        file = path + tmp;
        rm_files( file );
#ifdef OS2
        file = path + tmp.upcase();
        rm_files( file );
#endif

        tmp = REPLY_TEMP_FILENAME;
        file = path + tmp;
        rm_files( file );
#ifdef OS2
        file = path + tmp.upcase();
        rm_files( file );
#endif  
}

/***************************************************************************
 * FUNCTION: handle_error
 ***************************************************************************
 *
 * DESCRIPTION: Shows error message, and if flag is HALT_FL, then program 
 *              is halted. 
 *
 *              flag:
 *              DEFAULT_FL     Show error message and wait for key
 *              REPORT_FL      Print error message to stderr, and continue
 *              HALT_FL        Print error message to stderr and halt.
 *
 * EXTERNAL REFERENCES
 * IN : Settings.currentdir
 * OUT:
 *
 ***************************************************************************/
void
handle_error (const char *msg, int flag)
{
        Screen.enable();
        Wscript->reset_attr();
        Wscript->textcolor (BOLD | RED);
        Wscript->enable();
        Wscript->print("\n*** ERROR: ");
        Wscript->reset_attr();
        Wscript->print("%s", msg );
        if (flag != HALT_FL && flag != REPORT_FL) {
                Wscript->print( "\nPress any key to continue.." );
                Screen.get_ch();
                Wscript->disable();
                Wscript->print("\n -----------------------------");
        } else if ( flag == HALT_FL) {
                Wscript->print("\n*** Program halted.\n" );
                Screen.refresh();
                if (Settings.currentdir[0]) {
                        change_dir (Settings.currentdir);
                }
                exit (EXIT_FAILURE);
        }
        Screen.refresh();
}

void
system_error ( const char *msg, int flag )
{
        String str = msg;
        str += " -- ";
        str += strerror( errno );
        handle_error( str.c_str(), flag );
}

void
fatal_error( const char *msg )
{
        handle_error( msg, HALT_FL );
}

/***************************************************************************
 * FUNCTION: convert_to_qwk_date
 ***************************************************************************
 *
 * DESCRIPTION: Converts date from format mm-dd-yyyy to QWK style mm-dd-yy
 *              (and wait year 2000 happy!)
 *
 *
 * RETURN : Pointer to c-string that contains date. Copy it immediately to
 *          the final location
 ***************************************************************************/
char*
convert_to_qwk_date (char *s)
{
        static char buf[12];
        strncpy (buf, s, 6);
        buf[6] = s[8];
        buf[7] = s[9];
        buf[8] = '\0';
        return buf;
}

/***************************************************************************
 * FUNCTION: convert_to_jmr_date
 ***************************************************************************
 *
 * DESCRIPTION: Converts date from stubid QWK format to better mm-dd-yyyy
 *              format
 *
 *
 * RETURN : pointer to static c-string that contains new date.
 ***************************************************************************/
char *
convert_to_jmr_date (char *s)
{
        Time_handler Time;
        static char buf[12];
        char *tmp;
        tmp = Time.get_date();
        strcpy (buf, s);
        buf[6] = tmp[6];
        buf[7] = tmp[7];
        buf[8] = s[6];
        buf[9] = s[7];
        buf[10] = '\0';
        return buf;
}
//*************************************************************************/
// FUNCTION: get_month
//*************************************************************************/
//
// Returns number of month from date that is in jmr's format (mm-dd-yyyy)
//
//*************************************************************************/
int
get_month (char *s)
{
        return (int) (10 * ((*s) - 48) + *(s+1) - 48);
}
//*************************************************************************/
// FUNCTION: get_day
//*************************************************************************/
//
// Returns number of day from date string that is in jmr's date format.
//
//*************************************************************************/
int
get_day (char *s)
{
        return (int) (10 * (*(s+3) - 48) + *(s+4) -48);
}
//*************************************************************************/
// FUNCTION: get_year
//*************************************************************************/
//
// Returns number of year from date string 
//
//*************************************************************************/
int
get_year (char *s)
{
        return atoi (s+6);
}


/***************************************************************************
 * FUNCTION: strip_re_prefix
 ***************************************************************************
 *
 * DESCRIPTION: Checks if prefix 're:' exist. If so, then remove prefix.
 *
 *
 * RETURN : true, if prefix 're:' found.
 ***************************************************************************/
bool
strip_re_prefix (String& str)
{
        bool rc = false;
        String tmp = str;
        
        tmp.cut_first();
        if (tmp.ifind ("re:") == 0) {
                rc = true;
                str.copy (tmp, 3);
        }

        str.cut_first();
        str.cut_tail();
        return rc;
}


/***************************************************************************
 * FUNCTION: get_yesno
 ***************************************************************************
 *
 * DESCRIPTION: Reads answer to question y/n. If user just press enter, then
 *              default_value is token as answer.
 *
 *
 * RETURN : ascii code of answer.
 ***************************************************************************/
int
get_yesno (int default_value)
{
        int c;
        do {
                c = Screen.get_ch();
        } while (c != 'y' && c != SELECT_CMD && c != 'n');
        if (c == SELECT_CMD) {
                c = default_value;
        }
        return c;
}

//*************************************************************************/
// FUNCTION: show_help
//*************************************************************************/
//
// This function shows help.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : Screen
//   OUT: 
//
// RETURN: 
//*************************************************************************/
void show_help()
{
        int y;
        int ymax = Screen.get_ymax() - 3;
        int index;
  loop:
        index = 0;
        int c;
        y = 0;
        Screen.clear();
        do {
                if (*help_text[index] == '@') {
                        Screen.textcolor( BOLD );
                        Screen.print( "%s\n", help_text[index]+1 );
                        //cout << help_text[index]+1 << "\n";
                        Screen.reset_attr();
                } else {
                        Screen.print( "%s\n", help_text[index] );
                        //cout << help_text[index] << "\n";
                }
                ++index;
                ++y;
                if (y == ymax) {
                        Screen.gotoxy( 0, ymax + 3);
                        Screen.print( "q to quit, ? to start over or any other key to continue" );
                        c = Screen.get_ch();
                        if (c == QUIT_CMD) {
                                return;
                        }
                        if (c == HELP_CMD) {
                                goto loop;
                        }
                        Screen.clear();
                        y = 0;
                        
                }
        } while ( strcmp( help_text[index], "@END" ) );
        Screen.gotoxy(0, ymax + 3);
        Screen.print( "? to start over, any other to exit from help" );
        c = Screen.get_ch();
        if (c == HELP_CMD) {
                goto loop;
        }
}

//**************************************************************************/
// CLASS: Infoline
// MEMBER FUNCTION: reset
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/

void
Infoline::reset()
{
        if (true == active) {
                Screen.bottom();
                _msg = "";
                show_option_str( default_msg.get(), Settings.iline_color );
                Screen.clr_eol();
                active = false;
        }
}

//**************************************************************************/
// CLASS: Infoline
// MEMBER FUNCTION: show
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/

void
Infoline::show( int flag)
{
        if (true == active || flag & REFRESH_ALL_FL) {
                int color = is_error ? Settings.iline_err_color
                        : Settings.iline_color;
                Screen.bottom();

                if (_msg.is_empty() == false) {
                        show_option_str( _msg.c_str(), color );
                } else {
                        show_option_str( default_msg.c_str(), color );
                }
                Screen.clr_eol();
        }
}
//**************************************************************************/
// CLASS: Infoline
// MEMBER FUNCTION: show_default
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Infoline::show_default()
{
        Screen.bottom();
        show_option_str( default_msg.c_str(), Settings.iline_color | BOLD,
                         Settings.iline_color);
        Screen.clr_eol();
}


//**************************************************************************/
// CLASS: Infoline
// MEMBER FUNCTION: set
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Infoline::set( const char* msg, int fl )
{
        active = true;
        if (fl & ERROR_FL ) {
                _msg = "*** ";
                is_error = true;
        } else {
                _msg = "";
                is_error = false;
        }
        _msg += msg;

}

void
Sline::show()
{
        Screen.gotoxy( 0, Screen.get_ymax() - 1 );
        Screen.reset_attr();
        if( Screen.is_color_term() == true ) {
                Screen.textcolor( Settings.sline_color );
                Screen.back_color( Settings.slineback_color );
        } else {
                Screen.inverse();
        }
        Screen.addstr( get_sline() );
        Screen.reset_attr();
}



//*************************************************************************/
// FUNCTION: set_dashline_color
//*************************************************************************/
//
// Sets colors of dashed line on.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : Settings.dash_color, Settings_dashback_color
//   OUT: Screen
//
// RETURN: 
//*************************************************************************/
void
set_dashline_color()
{
        Screen.reset_attr();
        if (Settings.dash_color != DEFAULT_FCOLOR) {
                Screen.textcolor( Settings.dash_color );
        }
        if (Settings.dashback_color != DEFAULT_BCOLOR) {
                Screen.back_color( Settings.dashback_color );
        } 
}
//*************************************************************************/
// FUNCTION: show_option_str
//*************************************************************************/
//
// Displays given string using given 'bold_color' in all text that are
// between '@' marks. Color for other text is 'color'.
//
// e.g. If msg is "Press @Q@ for quit", is letter Q highlighted using
// color 'bold_color' and other text is drawed using color 'color'.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : -
//   OUT: -
//
// RETURN: -
//*************************************************************************/
void
show_option_str( const char *msg, int bold_color, int color)
{
        bool bold = false;
        const char *ptr = msg;
        Screen.reset_attr();
        if (color != DEFAULT_FCOLOR) {
                DEBUG("Settings option color : " << color );
                Screen.textcolor( color );
        }
        while (*ptr) {
                if ( *ptr == '@' ) {
                        bold = bold == false ? true : false;
                        if (bold == true ) {
                                if (bold_color == DEFAULT_FCOLOR) {
                                        Screen.bold();
                                } else {
                                        Screen.textcolor( bold_color );
                                }
                        } else {
                                Screen.reset_attr();
                                if (color != DEFAULT_FCOLOR ) {
                                        Screen.textcolor( color );
                                }
                        }
                        if (! *++ptr ) {
                                break;
                        }
                }
                Screen.addch( *ptr );
                ++ptr;
        }
        Screen.reset_attr();
}

//*************************************************************************/
// FUNCTION: get_system_info
//*************************************************************************/
//
// Gets name of system, in long or short form. If flag is ALL_FL, then
// long form is used, i.e. '[processor/system_name]'. Short format is
// '[system_name]'.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : -
//   OUT: -
//
// RETURN: System name in string, between square brackets.
//*************************************************************************/
String
get_system_info( int flag )
{
        String str;
#ifdef DO_NOT_USE_UNAME
        if (flag == ALL_FL ) {
                str =  LONG_SYSTEM ;
        } else {
                str =  SYSTEM ;
        }
#else
        struct utsname sysdata;
        if ( uname( &sysdata ) <= 0 ) {
                str =  SYSTEM ;
        } else {
                str =  "[";
                if( flag == ALL_FL ) {
                        str += sysdata.machine;
                        str += "/";
                }
                str += sysdata.sysname;
                str +="]";
        }
#endif
        return str;
}

int jmrsystem( const char* cmd )
{
        Screen.set_origmode();
#ifdef __unix__
        struct sigaction act, save;
        act.sa_flags = 0;
        act.sa_handler = SIG_DFL;
        sigemptyset( &act.sa_mask );

        sigaction( SIGTSTP, &act, &save );
        int i = _system( cmd );
        sigaction( SIGTSTP, &save, NULL );
#else
        int i = _system( cmd );
#endif
        Screen.set_rawmode();
        return i;
}

/***************************************************************************
 * FUNCTION : get_integer
 ***************************************************************************
 *
 * description : Gets integer value from string in byte_t array.
 *
 * Example: Array = { 1,5,7,9 } ---> returned value is 1579
 *
 * return : value readed from array.
 ***************************************************************************/
int
get_integer (
            byte_t *str,   // pointer to first number
            int len)     // amount of numbers
{
        int i;
        char *tmp = new char[len+1];

        for (i=0; i < len; i++) {
                tmp[i] = (char) *(str+i);
        }
        tmp[i] = 0;
        i = atoi (tmp);
        delete[] tmp;        
        return i;
}

/***************************************************************************
 * FUNCTION : get_ushort
 ***************************************************************************
 *
 * description : returns value of two byte little-endian number
 *
 ***************************************************************************/
int
get_ushort (
           byte_t *str) // array of two bytes.
{
        int i = (int) str[0] + 256*(int) str[1];
        return i;
}
