// 	$Id: filelist.cc,v 1.13 1998/07/18 19:49:17 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : filelist.cc
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *
 * *      This module contains methods for handling file lists
 * *      (expand wildcards, select file etc...)
 * *
 * *
 * *
 ***************************************************************************/

#include <stdio.h> // sprintf()
#ifdef __EMX__
#   include <sys/types.h>
#endif 
#ifdef __WATCOMC__
#   include <direct.h>
#else
#   include <dirent.h>
#endif
#include "jmr.hh"
#include "filelist.hh"
#include "../utilib/String.hh"
#include "../utilib/List.hh"
#include "mail.hh"

extern Terminal_screen Screen;
extern settings_t Settings;
extern Window* Wscript;

static const menuinfo filelist_menu_array[] = {
        {"[jmr]", 0 },
        {"Exit from menu", 0},
        {"Quit jmr", FORCED_QUIT_CMD},
        {"@NEXT", 0 },
        {"@END", 0 }
};



Filelist::Filelist()
{
        menubar.init( filelist_menu_array );
        menubar.set_text( MENUBAR_TEXT );
        for( int i = Screen.get_xmax(); i >= 0; i--) {
                dashline += '-';
        }
}

/***************************************************************************
 * FUNCTION: compare_name
 ***************************************************************************
 *
 * DESCRIPTION: compares two strings. This is for qsort().
 *
 * EXTERNAL REFERENCES
 * IN : 
 * OUT: 
 *
 * RETURN: <0, 0 or >0
 ***************************************************************************/
int compare_name (const void *a, const void *b)
{
        String **aa, **bb;
        aa = (String**) a;
        bb = (String**) b;
        return (*aa)->compare (*(*bb));
}


Mail*
Filelist::open_file(const String& bbsid)
{
        Mail *mail;
        open_new = false;
        int c;
		if (bbsid.is_empty()) {
        for (;;) {
                mail = 0;
                if (open_new == true) {
                        read_files (Settings.qwkpath);
                        pattern = Settings.searchpattern;
                } else {
                        read_files (Settings.datadir);
                        pattern = "*.[gG][dD][bB]";
                }
                
                // keep only entries, that matches with pattern
                search_matches();
                
                c = select ();
                Screen.bottom();
                Screen.clr_eol();
                //Screen.clear();
                if (c == QUIT_CMD || c == FORCED_QUIT_CMD) {
                        break;
                } else if (c == LOAD_CMD) {
                        open_new = open_new == true ? false : true;
                } else if (c == SELECT_CMD) {
                        mail = Mail::create (filelist.get()->get(), open_new);
                        break;
                }
        }
		} else {
			mail = 0;
			read_files(Settings.qwkpath);
			pattern = bbsid + ".[qQ][wW][kK]";
			if (search_matches()) {
				mail = Mail::create (filelist.get()->get(), true);
			} else {
				filelist.destroy();
				read_files (Settings.datadir);
				pattern = bbsid + ".[gG][dD][bB]";
				if (search_matches()) {
					mail = Mail::create(filelist.get()->get(), false);
				}
			}
		}
        filelist.destroy();
        return mail;
}


//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: read_files
// 
// 
//  DESCRIPTION: Reads all names of files from directory path
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//**************************************************************************/
void
Filelist::read_files (const String& path)
{
        DIR *dir;
        struct dirent *ent;
        String *str;

        filelist.destroy();
        if ( (dir = opendir ( path.get() )) == NULL) {
                String tmp = "Unable to open mail directory: " + path;
                handle_error (tmp.get(), HALT_FL);
        }
        while ((ent = readdir(dir)) != NULL) {
                if (ent->d_name[0] != '.') {
                        str = new String;
                        str->put(ent->d_name);
                        filelist.add (str);
                }
        }

        filelist.sort (compare_name);

        if (closedir(dir) != 0)
                handle_error("Unable to close mail directory", HALT_FL);

}

//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: search_matches
// 
// 
//  DESCRIPTION: Removes all files that does not match with pattern from list
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: false if list is empty
// 
//**************************************************************************/
bool
Filelist::search_matches()
{
        bool rc;
        String *str;
                
        if (filelist.first() == false) {
                return false;
        }
        
        do {
                str = filelist.get();
                if (check_matches (str->get(), pattern.get()) == false) {
                        rc = filelist.remove();
                } else {
                        rc = filelist.next();
                }
        } while (rc == true);

        return filelist.first();
}


//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: select
// 
// 
//  DESCRIPTION: 'User interface'. 
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: selected event.
// 
//**************************************************************************/
int
Filelist::select()
{
        int c;
        first = 0;
        current = 0;
        ypos = ymin = 6;
        ymax = Screen.get_ymax() - 5;
        lines = ymax - ymin;
        ++lines;
        //Screen.clear();
        sline.init( Screen.get_xmax() + 1);

        if (open_new == true) {
                iline.set_default( PACKET_LIST_KEYS_MSG );
        } else {
                iline.set_default( DATABASE_LIST_KEYS_MSG );
        }
        
        display();
        iline.show_default();
        
        while(1) {
                Screen.refresh();
                c = Screen.get_ch();
                iline.reset();
                if (c == ESCAPE ) {
                        iline.set( MENU_MSG );
                        iline.show();
                        c = menubar.select();
                        display();
                        iline.reset();
                }
                switch (c) {
                case MESSAGEWIN_CMD:
                        Wscript->enable();
                        iline.set ( MESSAGES_MSG );
                        iline.show();
                        Screen.get_ch();
                        iline.reset();
                        Wscript->disable();
                        break;
                case CODE_UP:
                case PREV_CMD:
                        prev();
                        break;
                case CODE_DOWN:
                case NEXT_CMD:
                        next();
                        break;
                case CODE_NPAGE:
                case PAGE_DOWN_CMD:
                        page_down();
                        break;
                case CODE_PPAGE:
                case PAGE_UP_CMD:
                        page_up();
                        break;
                        //case CODE_CTRL_L:
                        //display();
                        //break;
                case CODE_HOME:
                        first = 0;
                        current = 0;
                        ypos = ymin;
                        display();
                        break;
                case CODE_END:
                        current = filelist.count_items()-1;
                        // next line is not a bug! It's the only
                        // way to calc first index of page...
                        first = (current / lines) * lines ;
                        ypos = ymin + (current - first);
                        display();
                        break;
                case SELECT_CMD:
                case CODE_RIGHT:
                        if (filelist.check() == true) {
                                return SELECT_CMD;
                        }
                        break;
                case FORCED_QUIT_CMD:
#ifndef NO_EXCEPTIONS
                        throw quit_exception();
#endif
                case QUIT_CMD:
                        return QUIT_CMD;
                case REFRESH_CMD:
                        return REFRESH_CMD;
                case LOAD_CMD:
                        return LOAD_CMD;
                case HELP_CMD:
                        show_help();
                        display();
                        break;
                case 0:
                        break;
                default:
                        iline.set( INVALID_COMMAND_MSG );
                };
                iline.show();
        }
}


//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: display
// 
// 
//  DESCRIPTION: display files.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Filelist::display()
{
        int i = 0;
        int y;
        Screen.erase();
        menubar.show();
        display_header();
        if (open_new == true) {
                sline.set_infotext( "List of QWK packets" );
        } else {
                sline.set_infotext ("List of databases" );
        }
        if (filelist.first() == false) {
                Screen.gotoxy (0, ymin);
                if (open_new == true) {
                        Screen.print( "No files matches with pattern \'%s\' "
                                      "in a directory : \'%s\'",
                                      pattern.c_str(),
                                      Settings.qwkpath.c_str() );
                } else {
                        Screen.print( "No message databases available." );
                }
                return;
        }
        
        while (i < first) {
                filelist.next();
                ++i;
        }

        i = first;
        y = ymin;

        do {
                        
                Screen.gotoxy (0,y);
                if (i == current) {
                        Screen.inverse();
                        ypos = y;

                } 
                draw_current();
                Screen.reset_attr();
                ++y;
                ++i;
        } while (filelist.next() == true && y <= ymax);


        if (filelist.count_items() <= ymax - ymin) {
                sline.set_percent( ALL_PERCENT );
        } else if (0 == first) {
                sline.set_percent( TOP_PERCENT );
        } else {
                sline.set_percent( (100*i)/filelist.count_items() );
        }
        sline.show();

        filelist.first();
        i = current;
        while (i) {
                filelist.next();
                --i;
        }
        iline.show( REFRESH_ALL_FL );
        Screen.gotoxy (0,ypos);
}

//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: draw_current
// 
// 
//  DESCRIPTION: Displays name of the current file.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Filelist::draw_current()
{
        Screen.empty_line();
        Screen.goto_x( 8 );
        Screen.print( "%s", filelist.get()->c_str() );
}

//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: display_header
// 
// 
//  DESCRIPTION: Displays header of screen...
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Filelist::display_header()
{
        Screen.reset_attr();
        Screen.gotoxy( 0, 1 );
        Screen.textcolor( Settings.header_subject_color );
        Screen.print( "%s -- A QWK compatible offline mail reader",
                      Settings.version_str.c_str() );
        Screen.textcolor( Settings.header_author_color );
        Screen.print("\nCopyright (c) 1997,1998 Jukka Vuokko. All rights reserved.");
        Screen.reset_attr();
        Screen.print( "\nThis version is compiled at %s" , __DATE__ );
        Screen.print( "\nOfficial jmr homepage: http://www.iki.fi/"
                      "jvuokko/jmr/");
        Screen.gotoxy( 0, 5 );
        set_dashline_color();
        Screen.print( dashline.c_str() );
        Screen.gotoxy( 7, 5 );
        if (open_new == true) {
                Screen.print( " packets: " );
        } else {
                Screen.print( " databases: " );
        }
        Screen.reset_attr ();

}
        
//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: next
// 
// 
//  DESCRIPTION: Moves to next file.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Filelist::next()
{
        if (current < (filelist.count_items()-1)) {
                if (ypos == ymax) {
                        first = ++current;
                        display();
                } else {
                        Screen.gotoxy( 0, ypos );
                        Screen.empty_line();
                        draw_current();
                        Screen.gotoxy( 0, ypos+1 );
                        filelist.next();
                        Screen.inverse();
                        draw_current ();
                        Screen.reset_attr();
                        ++current;
                        ++ypos;
                }
        } else {
                iline.set( AT_TOP_MSG );
        }

}

//**************************************************************************/
//  CLASS: Filelist
//  MEMBER FUNCTION: prev
// 
// 
//  DESCRIPTION: Moves to previous file.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Filelist::prev()
{
        if (current > 0) {
                if (ypos == ymin && first > 0) {
                        --current;
                        first -= lines;
                        display();
                } else {
                        Screen.gotoxy( 0, ypos );
                        Screen.empty_line();
                        draw_current ();
                        Screen.gotoxy( 0, ypos-1);
                        Screen.inverse();
                        filelist.prev();
                        draw_current ();
                        Screen.reset_attr();
                        --current;
                        --ypos;
                }
        } else {
                iline.set( AT_BOTTOM_MSG );
        }
}

void
Filelist::page_down()
{
        int max = filelist.count_items()-1;
        if (current >= max) {
                iline.set( AT_LAST_PAGE_MSG );
                return;
        }
        if (ypos == ymax) {
                first = ++current;
                for (int i=1; i < lines && current < max; i++, current++);
                ypos = ymin + (current - first);
        } else {
                while (current < max && ypos < ymax) {
                        ++current;
                        ++ypos;
                }
        }
        display();
}

void
Filelist::page_up()
{
        int i = lines;
        if (current == 0) {
                iline.set( AT_FIRST_PAGE_MSG );
                return;
        }
        if (ypos == ymin) {
                while (i && first > 0) {
                        --first;
                        --current;
                        --i;
                }
        } else {
                while (ypos > ymin) {
                        --ypos;
                        --current;
                }
        }
        display();
}



