#include <ctype.h>
#include <curses.h>
#include <dirent.h>
#include <menu.h>
#include <panel.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include "doorutil.h"
#include "vars.h"
#include "funcs.h"

int main(int argc, char *argv[])
{
	WINDOW *win;
	int j;

	// Initialize Falken Routines
	util_init();

	// Get the UID of the BBS user
	if (strlen(cfg->bbs_owner_name))
		bbs_uid = getpwnam(cfg->bbs_owner_name)->pw_uid;
	else {
		printf("The BBS Username is currently NOT setup.  Memory statistics\n");
		printf("will not be accurate.\n");
		printf("Please wait ");
		bbs_uid = 0;
		for (j = 0; j < 5; j++)
		{
			sleep(1);
			printf(".");
		}
	}
		
	// Read in the max users info from disk
	read_max();

	// Trap signals so we can get rid of curses on ctrl-c etc...
	for (j=SIGHUP;j<=SIGTERM;j++)
		if (j != SIGSEGV)
			if (signal(j,SIG_IGN)!=SIG_IGN)
				signal(j,onsig);
	
	//ESCDELAY=10;
	
	tabs[0].func = do_users;
	tabs[1].func = acct_editor;
	tabs[2].func = do_stats;
	tabs[3].func = do_strings;
	tabs[4].func = do_config;
	tabs[5].func = do_ops;
	tabs[6].func = do_monitor;
	tabs[7].func = do_ranks;

	win = initscr();
	if ((LINES < 24) || (COLS < 80))
	{
		endwin();
		printf("You must have at least 24 lines and 80 columns to use BBSConsole\n");
		printf("You currently have %d lines and %d columns\n", LINES, COLS);
		exit(0);
	}
	start_color();
	noecho();
	curs_set(0);
	cbreak();
	
	// Main program
	do_console(win);
	
	// Shut everything down
	curs_set(1);
	echo();
	endwin();
	printf("\n\n");
	exit(0);
}

void onsig(int n)
{
	int j;

	if (!screenlocked)
	{
		curs_set(1);
		echo();
		endwin();
		printf("\n\nExiting on signal %d\n", n);
		if (n == SIGSEGV)
		{
			signal(n, SIG_DFL);
			kill(getpid(), SIGSEGV);
		}
		exit(1);
	} else {
		for (j=SIGHUP;j<=SIGTERM;j++)
			if (j != SIGSEGV)
				if (signal(j,SIG_IGN)!=SIG_IGN)
					signal(j,onsig);
	}
}

void do_console(WINDOW *win)
{
	int ctr, cur_win, prev_win;
	int key=-1;
	WINDOW *menu_win[8], *info_win;
	time_t curtime = time(NULL);
	int lap=0, bbs_mem;
	char now[13];
	WINDOW *tab_win[8];
	PANEL  *tab_pan[8];
	
	// Setup the color pairs for the windows
	init_pair(1, COLOR_WHITE, COLOR_BLUE);
	init_pair(2, COLOR_YELLOW, COLOR_BLUE);
	init_pair(3, COLOR_WHITE, COLOR_BLACK);
	init_pair(4, COLOR_WHITE, COLOR_RED);
	init_pair(5, COLOR_BLACK, COLOR_BLACK);
	init_pair(6, COLOR_BLACK, COLOR_WHITE);
	init_pair(7, COLOR_BLACK, COLOR_BLUE);
	init_pair(8, COLOR_RED, COLOR_BLACK);
	
	// Resize the main window and set the color, then display
	wattron(win, COLOR_PAIR(1) | A_BOLD);
	wresize(win, LINES, COLS-14);
	box(win, 0, 0);
	mvwaddstr(win, 0, 4, "[ BBSConsole ]");
	refresh();
	nodelay(win, TRUE);
	keypad(win, TRUE);
	refresh();
	
	// Make the different panels for the actual display screens
	for (ctr = 0; ctr < 8; ctr++)
	{
		tab_win[ctr] = newwin(LINES-2, COLS-16, 1, 1);
		tab_pan[ctr] = new_panel(tab_win[ctr]);
		clear_window(tab_win[ctr], 1);
		nodelay(tab_win[ctr], TRUE);
		keypad(tab_win[ctr], TRUE);
	}
	
	clear_window(tab_win[1], 3);  // Leave the acct edit window black
	load_user_account(&account.cur_acct, 0); // Go ahead and load the first acct
	                                      	// into the editor
	
	
	
	// Setup the INFO window underneath the tabs
	info_win = newwin(0, 0, 18, COLS-14);
	box(info_win, 0, 0);
	nodelay(info_win, TRUE);
	keypad(info_win, TRUE);
	wnoutrefresh(info_win);
	
	// Make the tabs for the menu, set win attributes, etc...
	for (ctr = 0; ctr < 8; ctr++)
	{
		menu_win[ctr] = newwin(3, 0, ctr*2+1, COLS-15);
		wattron(menu_win[ctr], COLOR_PAIR(1) | A_BOLD);
		background(menu_win[ctr], ctr);
		nodelay(menu_win[ctr], TRUE);
		keypad(menu_win[ctr], TRUE);
		wnoutrefresh(menu_win[ctr]);
	}
	
	// Start the menu on the first tab
	cur_win = 0;
	prev_win = 0;
	foreground(menu_win[cur_win], cur_win);
	
	// Main loop
	while ((toupper((char)key) != 'Q') || (key == 27))
	{
		usleep(100000);
		if (curtime != time(NULL))
		{
			int i;
			i = num_users();
			if (i > max_info.users)
			{
				max_info.users = i;
				write_max();
			}
			curtime = time(NULL);
			if (screenlocked)
				mvwprintw(info_win, 0, 2, "[LOCKED]");
			else
				mvwhline(info_win, 0, 2, ACS_HLINE, 8);
			strftime(now, 12, "%a %b %d", localtime(&curtime));
			mvwprintw(info_win, 1, 1, "%-12.12s", now);
			strftime(now, 12, "%H:%M:%S", localtime(&curtime));
			mvwprintw(info_win, 2, 1, "%-12.12s", now);
			mvwprintw(info_win, 3, 1, "Online: %d", i);
			mvwprintw(info_win, 4, 1, "Max   : %d", max_info.users);
			wnoutrefresh(info_win);
		}
		key = wgetch(tab_win[cur_win]);
		if ((key != -1) && screenlocked)
		{
			unlock();
			key = -1;
		} 
		if (key == CTRL('p'))
		{
			screenlocked=1;
			continue;
		}
		if ((key >= '1') && (key <= '8'))
		{
			prev_win = cur_win;
			background(menu_win[cur_win], cur_win);
			cur_win = key - '1';
			nocbreak();
			cbreak();
			nodelay(tab_win[cur_win], TRUE);
			foreground(menu_win[cur_win], cur_win);
			if (panel_hidden(tab_pan[cur_win]))
			{
				top_panel(tab_pan[cur_win]);
				wnoutrefresh(tab_win[cur_win]);
			}
			tabs[cur_win].func(tab_win[cur_win], tab_pan[cur_win], key);
		}
		if (key == 9)	// TAB
		{
			background(menu_win[cur_win], cur_win);
			prev_win = cur_win;
			if (cur_win == 7) 
				cur_win = -1;
			cur_win++;
			nocbreak();
			cbreak();
			nodelay(tab_win[cur_win], TRUE);
			foreground(menu_win[cur_win], cur_win);
			key = -1;
		}
		if ((key == 10) && (cur_win == 0))  // Enter pressed on users screen
		{
			if (acnt[cur_user].acctname[0] != 0)
			{
				prev_win = cur_win;
				background(menu_win[cur_win], cur_win);
				cur_win = 1;
				load_user_account(&account.cur_acct, atoi(acnt[cur_user].acctnum)-1);
				foreground(menu_win[cur_win], cur_win);
			}
			key = -1;
		}
		if (key == CTRL('l'))
		{
			touchwin(curscr);
			wrefresh(curscr);
		}
		// Refresh the current window
		if (panel_hidden(tab_pan[cur_win]))
		{
			top_panel(tab_pan[cur_win]);
			wnoutrefresh(tab_win[cur_win]);
		}
		tabs[cur_win].func(tab_win[cur_win], tab_pan[cur_win], key);
		if ((lap == 300) && (cur_win != 2)) // Get the mem info every 30 seconds 
		{
			get_proc_info();
			bbs_mem = get_bbs_mem();
			if (bbs_mem > max_info.memory)
			{
				max_info.memory = bbs_mem;
				write_max();
			}
			lap=0;
		} else
			lap++;
		//if (key != -1) wprintw(win, "%d\n", key);
		doupdate();
	}
}

void background(WINDOW *win, int num)  // Put a tab in the background
{
	int ctr;

	wattrset(win, 0);
	for (ctr = 0; ctr < 3; ctr++)
		mvwaddch(win, ctr, 14, ' ');
	wrefresh(win);
	wattron(win, COLOR_PAIR(7) | A_BOLD);
	wresize(win, 3, 14);
	if (num == 0)
		wborder(win, 0, 0, 0, 0, ACS_LTEE, ACS_URCORNER, ACS_LTEE,  ACS_RTEE);
	else if (num == 7)
		wborder(win, 0, 0, 0, 0, ACS_LTEE, ACS_RTEE, ACS_LTEE,  ACS_LRCORNER);
	else
		wborder(win, 0, 0, 0, 0, ACS_LTEE, ACS_RTEE, ACS_LTEE,  ACS_RTEE);
	wattron(win, COLOR_PAIR(1) | A_BOLD);
	for (ctr = 0; ctr < 3; ctr++)
		mvwaddch(win, ctr, 0, ACS_VLINE);
	wattrset(win, COLOR_PAIR(1));
	mvwprintw(win, 1, 1, "%-12.12s", tab_opt[num]);
	wnoutrefresh(win);
}

void foreground(WINDOW *win, int num)  // Put a tab in the foreground
{
	wattron(win, COLOR_PAIR(1) | A_BOLD);
	wresize(win, 3, 15);
	wborder(win, ' ', 0, 0, 0, ACS_LLCORNER, 0, ACS_ULCORNER,  0);
	wattron(win, COLOR_PAIR(2) | A_BOLD);
	mvwprintw(win, 1, 1, " %-12.12s", tab_opt[num]);
	wnoutrefresh(win);
}

int num_users(void)  // Get the total number of users
{
	int ctr;
	int count=0;

	for (ctr = 0; ctr < numlines; ctr++)
	{
		if (acnt[ctr].acctname[0] != 0)
			count++;
	}
	return(count);
}

void do_stats(WINDOW *win, PANEL *pan, int key)
{
	char loadaverage[80];
	int num_procs=0, bbs_mem;
	
	get_mem_info();
	
	bbs_mem = get_bbs_mem();
	num_procs = get_proc_info();
	if (bbs_mem > max_info.memory)
	{
		max_info.memory = bbs_mem;
		write_max();
	}
	mvwprintw(win, 1, 2, "Total # of Calls     : %d     ", get_number_calls());
	mvwprintw(win, 2, 2, "Daemon Started at    : %s     ", shm->laststartup);
	mvwprintw(win, 3, 2, "System Load Averages : %s     ", get_load_avg(loadaverage));
	mvwprintw(win, 4, 2, "Total # of Accts     : %d     ", tot_accts());
	mvwhline(win, 6, 1, ACS_HLINE, 60);
	mvwaddch(win, 6, 1, ACS_ULCORNER);
	mvwaddch(win, 6, 9, ACS_TTEE);
	mvwaddch(win, 6, 15, ACS_TTEE);
	mvwaddch(win, 6, 24, ACS_TTEE);
	mvwaddch(win, 6, 33, ACS_TTEE);
	mvwaddch(win, 6, 42, ACS_TTEE);
	mvwaddch(win, 6, 51, ACS_TTEE);
	mvwaddch(win, 6, 60, ACS_URCORNER);
	mvwprintw(win, 6, 3, "[ Memory Usage in kilobytes ]");
	mvwprintw(win, 7, 1, "|       |  #  | Total  | Used   | Free   | Res    | Shared |");
	mvwprintw(win, 8, 1, "|  All  | %-3d | %-6d | %-6d | %-6d | N/A    | %-6d |", num_procs, mem_info.total/1024, mem_info.used/1024, mem_info.free/1024, mem_info.shared/1024);
	mvwprintw(win, 9, 1, "|  Swap | N/A | %-6d | %-6d | %-6d | N/A    | N/A    |", mem_info.swaptotal/1024, mem_info.swapused/1024, mem_info.swapfree/1024);
	mvwprintw(win, 10, 1, "|  BBS  | %-3d | N/A    | %-6d | N/A    | %-6d | %-6d |", num_bbs_procs(), bbs_mem, get_bbs_resident(), get_bbs_shared());
	mvwvline(win, 7, 1, ACS_VLINE, 4);
	mvwvline(win, 7, 9, ACS_VLINE, 4);
	mvwvline(win, 7, 15, ACS_VLINE, 4);
	mvwvline(win, 7, 24, ACS_VLINE, 4);
	mvwvline(win, 7, 33, ACS_VLINE, 4);
	mvwvline(win, 7, 42, ACS_VLINE, 4);
	mvwvline(win, 7, 51, ACS_VLINE, 4);
	mvwvline(win, 7, 60, ACS_VLINE, 4);
	mvwhline(win, 11, 1, ACS_HLINE, 60);
	mvwaddch(win, 11, 1, ACS_LLCORNER);
	mvwaddch(win, 11, 9, ACS_BTEE);
	mvwaddch(win, 11, 15, ACS_BTEE);
	mvwaddch(win, 11, 24, ACS_BTEE);
	mvwaddch(win, 11, 33, ACS_BTEE);
	mvwaddch(win, 11, 42, ACS_BTEE);
	mvwaddch(win, 11, 51, ACS_BTEE);
	mvwaddch(win, 11, 60, ACS_LRCORNER);
	mvwprintw(win, 12, 2, "Maximum Mem Usage   : %d kb     ", max_info.memory);
	wnoutrefresh(win);

	// The stats window needs to block for 5 seconds or it sucks BIG CPU
	// We do this AFTER it's displayed the first time
	halfdelay(50);
}

void do_users(WINDOW *win, PANEL *pan, int key)
{
	int ctr, tmp;
	char buffer[1024];
	static WINDOW *online_win=0;
	static WINDOW *cool_win=0;
	static int first_user=0;
	static int lap=0;

	if (!online_win)
	{
		online_win = subwin(win, 16, 40, 3, 4);
		outline_window(win, online_win);
		nodelay(online_win, TRUE);
		keypad(online_win, TRUE);
		clear_window(online_win, 3);
	}
	if (!cool_win)
	{
		cool_win = subwin(win, 1, COLS-22, LINES-3, 4);
		outline_window(win, cool_win);
		nodelay(cool_win, TRUE);
		keypad(cool_win, TRUE);
		clear_window(cool_win, 3);
		wnoutrefresh(cool_win);
	}
	lap++;
	if (lap % TICKER_DELAY)
		lap = update_ticker(cool_win, lap);
	wmove(online_win, 0, 0);
	switch (key)
	{
		case CTRL('h'):
			dialog(11, 60, 5, 3, "[ Help ]", " CTRL-L       = Redraw Screen\n Up/Down      = Select Line/User\n Page Up/Down = Up/Down 1 Page\n Enter        = Load Selected User into Account Editor\n CTRL-K       = Kill Selected User\n CTRL-T       = Send a message to selected user\n CTRL-F       = Fake Line Noise (HA!)\n Q / Escape   = Quit");
			wrefresh(win);
			break;
		case CTRL('k'):
			if (dialog_ny(4, 25, 5, 10, "[ Kill User ]", "Are you sure?      "))
				bbs_logoff(cur_user);
			break;
		case CTRL('f'):
			buffer[0] = 0;
			tmp = random_number(10, 20);
			for (ctr = 0; ctr < tmp; ctr++)
				sprintf(buffer, "%s%c", buffer, random_number(32, 254));
			qnprintf(cur_user, "%s", buffer);
			break;
		case CTRL('t'):
			buffer[0] = 0;
			gets_dialog(3, 62, 5, 2, "[ Send Message ]", ":", buffer, 59, 59, 1);
			if (strlen(buffer))
				qnprintf(cur_user, "\rMessage from Console\r%s\r", buffer);
			break;
		case 'j':
		case KEY_DOWN:
			cur_user++;
			if (cur_user >= numlines)			cur_user--;
			if (cur_user > first_user+15)
			{
				first_user+=8;
				scrollok(online_win, TRUE);
				wscrl(online_win, 8);
				scrollok(online_win, FALSE);
			}
			break;
		case CTRL('d'):
		case KEY_NPAGE:
			cur_user+=8;
			first_user+=8;
			scrollok(online_win, TRUE);
			wscrl(online_win, 8);
			scrollok(online_win, FALSE);
			wnoutrefresh(online_win);
			if (first_user >= numlines-16) first_user = numlines-16;
			if (cur_user >= numlines)			cur_user = numlines-1;
			break;
		case KEY_HOME:
			cur_user = 0;
			first_user = 0;
			break;
		case KEY_END:
			cur_user = numlines - 1;
			first_user = numlines-16;
			break;
		case ('k'):
		case KEY_UP:
			cur_user--;
			if (cur_user == -1) 					cur_user=0;
			if (cur_user < first_user) 		first_user-=8;
			if (first_user < 0) 					first_user = 0;
			break;
		case CTRL('u'):
		case KEY_PPAGE:
			cur_user-=16;
			first_user-=16;
			if (cur_user < 0)							cur_user=0;
			if (first_user < 0) 					first_user = 0;
			break;
		case CTRL('r'):
			if (graphics_on)
				graphics_on = 0;
			else
				graphics_on = 1;
			break;
		default:
			break;
	}
	for (ctr = first_user; ctr < first_user + 16; ctr++)
	{
		if (ctr < numlines)
		{
			if (acnt[ctr].acctname[0] != 0)
			{
				if (cur_user == ctr)
					wattron(online_win, COLOR_PAIR(4) | A_BOLD);
				else
					wattron(online_win, COLOR_PAIR(3) | A_BOLD);
				sprintf(buffer, " %-2.2d: %s in %s", ctr, acnt[ctr].acctname, user[ctr].doors_id);
				mvwprintw(online_win, ctr - first_user, 0, "%-40.40s\n", buffer);
			} else {
				if (cur_user == ctr)
					wattron(online_win, COLOR_PAIR(4) | A_BOLD);
				else
					wattron(online_win, COLOR_PAIR(3) | A_BOLD);
				sprintf(buffer, " %-2.2d: Waiting for Call", ctr);
				mvwprintw(online_win, ctr - first_user, 0, "%-40.40s\n", buffer);
			}
		} else {
				sprintf(buffer, " %-2.2d: Line not configured", ctr);
				mvwprintw(online_win, ctr - first_user, 0, "%-40.40s\n", buffer);
		}
	}
	wnoutrefresh(online_win);
	mvwprintw(win, 2, 45, "Account #: %-8.8s", acnt[cur_user].acctnum); 
	mvwprintw(win, 3, 45, "Time On  : %-3dmin", user[cur_user].time_on_thiscall);
	if (user[cur_user].idletimer)
		mvwprintw(win, 4, 45, "Idle Time: %-2d min  ", user[cur_user].u_idletime - user[cur_user].idletimer); 
	else
		mvwprintw(win, 4, 45, "Idle Time: Disable ", user[cur_user].u_idletime - user[cur_user].idletimer); 
	mvwprintw(win, 5, 45, "ANSI     : %-8.8s", (user[cur_user].u_ansi) ? "On" : "Off"); 
	mvwprintw(win, 6, 45, "Hotkeys  : %-8.8s", (user[cur_user].hotkeys) ? "On" : "Off"); 
	mvwprintw(win, 7, 45, "Busy     : %-8.8s", (user[cur_user].busyflag) ? "On" : "Off"); 
	mvwprintw(win, 8, 45, "Stealth  : %-8.8s", (user[cur_user].stealth) ? "On" : "Off"); 
	if (shm->linctrl[cur_user].active)
		mvwprintw(win, 9, 45, "PID      : %-8d", shm->linctrl[cur_user].pid);
	else 
		mvwprintw(win, 9, 45, "PID      :       ");
	mvwprintw(win, 10, 45, "TTY      : %-8.8s", (shm->linctrl[cur_user].active) ? shm->linctrl[cur_user].ttyname+5 : "");
	mvwprintw(win, 11, 45, "");
	mvwprintw(win, 12, 45, "Logons   : %-8d", acnt[cur_user].totlogons); 
	mvwprintw(win, 13, 45, "Uploads  : %-8d", acnt[cur_user].uploads); 
	mvwprintw(win, 14, 45, "UL KB    : %-8d", acnt[cur_user].upload_k); 
	mvwprintw(win, 15, 45, "Downloads: %-8d", acnt[cur_user].dloads); 
	mvwprintw(win, 16, 45, "DL KB    : %-8d", acnt[cur_user].download_k); 
	wnoutrefresh(win);
}


void do_strings(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the strings screen eventually");
	wnoutrefresh(win);
}

void do_config(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the config screen eventually");
	wnoutrefresh(win);
}

void do_ops(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the ops screen eventually");
	wnoutrefresh(win);
}

void do_monitor(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the monitor screen eventually");
	wnoutrefresh(win);
}

void do_fido(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the fido screen eventually");
	wnoutrefresh(win);
}

void do_ranks(WINDOW *win, PANEL *pan, int key)
{
	mvwprintw(win, 1, 2, "This will be the ranks screen eventually");
	wnoutrefresh(win);
}

unsigned int get_number_calls(void)
{
	FILE *f;
	unsigned int total_calls;

	if ((f = fopen("/home/bbs/callcnt.bbs", "r")) == 0)
	{
		return(0);
	} else {
		fread(&total_calls, sizeof(int), 1, f);
		fclose(f);
		return(total_calls);
	}
}

char *get_load_avg(char *loadaverage)
{
	FILE *infile;
	char string[81];
	char *str;

	if((infile = fopen("/proc/loadavg", "rb")) == NULL)
		strcpy(loadaverage, "0.00     ");
	else { 
		fgets(string, 80, infile);
		fclose(infile);
		// Get the 1 minute average
		if((str = strtok(string,  " ")) == NULL)
			strcpy(loadaverage, "1: 0.00     ");
		else 
			sprintf(loadaverage, "1: %s  ", str);
		// Get the 5 minute average
		if((str = strtok(NULL,  " ")) == NULL)
			sprintf(loadaverage, "%s 5: 0.00 ", loadaverage);
		else 
			sprintf(loadaverage, "%s 5: %s ", loadaverage, str);
		// Get the 15 minute average
		if((str = strtok(NULL,  " ")) == NULL)
			sprintf(loadaverage, "%s 15: 0.00 ", loadaverage);
		else 
			sprintf(loadaverage, "%s 15: %s ", loadaverage, str);
	}
	return(loadaverage);
}

void get_mem_info()
{
	FILE *f;
	char buffer[128];
	char *str;
	int ctr;

	if ( (f = fopen("/proc/meminfo","r")) == 0)
	{
		memset(&mem_info, 0, sizeof(struct _mem_info));
		return;
	} else {
		fgets(buffer, 79, f);
		fgets(buffer, 79, f);
		str = strtok(buffer, " ");
		for (ctr = 0; ctr < 5; ctr++)
		{
			str = strtok(NULL, " ");
			if (str == NULL)
				break;
			switch (ctr)
			{
				case 0: 
					mem_info.total=atoi(str);
					break;
				case 1: 
					mem_info.used=atoi(str);
					break;
				case 2: 
					mem_info.free=atoi(str);
					break;
				case 3: 
					mem_info.shared=atoi(str);
					break;
				case 4: 
					mem_info.buffers=atoi(str);
					break;
				default:
					break;
			}
		}	
		fgets(buffer, 79, f);
		str = strtok(buffer, " ");
		for (ctr = 0; ctr < 3; ctr++)
		{
			str = strtok(NULL, " ");
			if (str == NULL)
				break;
			switch (ctr)
			{
				case 0: 
					mem_info.swaptotal=atoi(str);
					break;
				case 1: 
					mem_info.swapused=atoi(str);
					break;
				case 2: 
					mem_info.swapfree=atoi(str);
					break;
				default:
					break;
			}
		}
		fclose(f);
	}
}

void acct_editor(WINDOW *win, PANEL *pan, int key)
{
	int ctr, loop;
	static WINDOW *ae_inst_win=0;
	static WINDOW *acct_edit_win=0;
	static int first_field=0;
	static int cur_field=1;
	static int prev_field=0;
	char buffer[1024];
	char buffer2[1024];
	long li;
	int *i;
	time_t *t;
	char *c;
	unsigned char *uc;

	if (!ae_inst_win)
	{
		ae_inst_win = subwin(win, 1, COLS-16, LINES-2, 1);
		outline_window(win, ae_inst_win);
		clear_window(ae_inst_win, 1);
		nodelay(ae_inst_win, TRUE);
		keypad(ae_inst_win, TRUE);
		waddstr(ae_inst_win, " CTRL-H for help - Arrow Keys or VI commandset to move");
		wnoutrefresh(ae_inst_win);
	}
	if (!acct_edit_win)
	{
		acct_edit_win = subwin(win, LINES-4, COLS-16, 1, 1);
		clear_window(acct_edit_win, 3);
		nodelay(acct_edit_win, TRUE);
		keypad(acct_edit_win, TRUE);
		wnoutrefresh(acct_edit_win);
	}
	switch (key)
	{
		case 'g':
		case 'G':
			sprintf(buffer, "%d", atoi(account.cur_acct.acctnum));
			gets_dialog(3, 40, 8, 10, "[ Goto Account ]", "Account #:", buffer, 5, 5, 1);
			load_user_account(&account.cur_acct, atoi(buffer)-1);
			wrefresh(win);
			first_field=0;
			cur_field=1;
			break;
		case 's':
		case 'S':
			if (dialog_yn(5, 30, 5, 10, "[ Save Account ]", " Save this account? "))
				if (save_user_account(&account.cur_acct))
					dialog(4, 40, 6, 12, "[ Saving Account ]", " Account Information is saved ");
				else
					dialog(4, 40, 6, 12, "[ Saving Account ]", " Account Information NOT saved ");
			break;
		case 'f':
		case 'F':
			//sprintf(buffer, "%s", account.cur_acct.acctname);
			buffer[0] = 0;
			gets_dialog(3, 50, 8, 8, "[ Search ]", "Search for: ", buffer, uidlen, uidlen, 1);
			if ( (ctr = search_username(buffer, atoi(account.cur_acct.acctnum))) < 0)
			{
				sprintf(buffer2, " No user found with:\n  %s\n in the account name", buffer);
				dialog(6, 60, 9, 3, "[ Search ]", buffer2);
			} else
				load_user_account(&account.cur_acct, ctr);
			wrefresh(win);
			first_field=0;
			cur_field=1;
			break;
		case CTRL('h'):
			dialog(11, 52, 5, 7, "[ Help ]", " CTRL-L       = Redraw Screen\n S            = Save Account\n F            = Find\n G            = Go to Acct #\n Up/Down      = Select Field\n Left/Right   = Forward/Back 1 Record\n Page Up/Down = Up/Down 1 Page\n Enter        = Toggle or Modify selected setting");
			wrefresh(win);
			break;
		case 'l':
		case KEY_RIGHT:
			if (atoi(account.cur_acct.acctnum) == tot_accts())
				load_user_account(&account.cur_acct, 0);
			else
				load_user_account(&account.cur_acct, atoi(account.cur_acct.acctnum));
			first_field=0;
			cur_field=1;
			break;
		case 'h':
		case KEY_LEFT:
			if (atoi(account.cur_acct.acctnum) != 1)
				load_user_account(&account.cur_acct, atoi(account.cur_acct.acctnum)-2);
			else
				load_user_account(&account.cur_acct, tot_accts()-1);
			first_field=0;
			cur_field=1;
			break;
		case 'j':
		case KEY_DOWN:
			prev_field=cur_field;
			cur_field++;
			if (cur_field >= num_opts)			
				cur_field--;
			if (acct_opts[cur_field].name[0] == '.') 
				cur_field++;
			if (cur_field >= first_field+LINES-4)
			{
				first_field=cur_field-LINES+5;;
				scrollok(acct_edit_win, TRUE);
				wscrl(acct_edit_win, 1);
				scrollok(acct_edit_win, FALSE);
			}
			if (acct_opts[cur_field].name[0] == '.') 
				cur_field++;
			break;
		case CTRL('d'):
		case KEY_NPAGE:
			prev_field=cur_field;
			cur_field+=(LINES-4)/2;
			first_field+=(LINES-4)/2;
			scrollok(acct_edit_win, TRUE);
			wscrl(acct_edit_win, (LINES-4)/2);
			scrollok(acct_edit_win, FALSE);
			if (first_field > num_opts-LINES+4) 
				first_field = num_opts-LINES+4;
			if (cur_field >= num_opts)			
				cur_field = num_opts-1;
			if (acct_opts[cur_field].name[0] == '.') 
				cur_field++;
			break;
		case KEY_HOME:
			first_field = 0;
			cur_field = 0;
			if (acct_opts[cur_field].name[0] == '.') 
				cur_field++;
			break;
		case KEY_END:
			first_field = num_opts-LINES+4;
			cur_field = num_opts-1;
			break;
		case 'k':
		case KEY_UP:
			prev_field=cur_field;
			cur_field--;
			if (cur_field == -1) 						cur_field=0;
			if (acct_opts[cur_field].name[0] == '.') 
			{
				if (cur_field != 0)
					cur_field--;
				else
					cur_field++;
			}
			if (cur_field < first_field) 		first_field=cur_field;
			if (first_field < 0) 						first_field = 0;
			if (cur_field == -1) 						cur_field=0;
			break;
		case CTRL('u'):
		case KEY_PPAGE:
			prev_field=cur_field;
			cur_field-=(LINES-4)/2;
			first_field-=(LINES-4)/2;
			if (cur_field < 0)							cur_field=0;
			if (first_field < 0) 						first_field = 0;
			if (acct_opts[cur_field].name[0] == '.') 
			{
				if (cur_field != 0)
					cur_field--;
				else
					cur_field++;
			}
			break;
		case 10:
			switch (acct_opts[cur_field].type)
			{
				case E_STRING:
					e_mvwgetnstrb(acct_edit_win, cur_field-first_field, 24, acct_opts[cur_field].value, acct_opts[cur_field].len, acct_opts[cur_field].len, 1, 1);
					break;
				case PASSWORD:
					e_mvwgetnstrb(acct_edit_win, cur_field-first_field, 24, acct_opts[cur_field].value, 30, 30, 1, 1);
					break;
				case INT:
					{
						char buffer[20];
						i = acct_opts[cur_field].value;
						sprintf(buffer, "%d", *i);
						e_mvwgetnstrb(acct_edit_win, cur_field-first_field, 24, buffer, 10, 10, 1, 1);
						*i = atoi(buffer);
					}
					break;
				case CHAR:
					{
						char buffer[20];
						uc = acct_opts[cur_field].value;
						sprintf(buffer, "%d", *uc);
						e_mvwgetnstrb(acct_edit_win, cur_field-first_field, 24, buffer, 3, 3, 1, 1);
						*uc = (unsigned char)atoi(buffer);
					}
					break;
				case SEX:
					i = acct_opts[cur_field].value;
					if (*i)
						*i = 0;
					else
						*i = 1;
					break;
				case BOOL_1:
					i = acct_opts[cur_field].value;
					if (*i)
						*i = 0;
					else
						*i = 1;
					break;
				case BOOL_3:
					i = acct_opts[cur_field].value;
					if (*i == 'y')
						*i = 'n';
					else
						*i = 'y';
					break;
				case A_FLAG:
					li = (long int)acct_opts[cur_field].value;
					if (account.cur_acct.accessflags & li) { 
						li = ~((long int)acct_opts[cur_field].value);
						account.cur_acct.accessflags &= li; 
					} else {  
						li = (long int)acct_opts[cur_field].value;
						account.cur_acct.accessflags |= li;
					}
					break;
				case M_FLAG:
					li = (long int)acct_opts[cur_field].value;
					if (account.cur_acct.menu_flags & li) { 
						li = ~((long int)acct_opts[cur_field].value);
						account.cur_acct.menu_flags &= li; 
					} else {  
						li = (long int)acct_opts[cur_field].value;
						account.cur_acct.menu_flags |= li;
					}
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
	for (ctr = first_field; ctr < first_field+LINES-4; ctr++)
	{
		wmove(acct_edit_win, ctr-first_field, 1);
		wclrtoeol(acct_edit_win);
		if (acct_opts[ctr].type == HEADER)
		{
			wmove(acct_edit_win, ctr-first_field, 1);
			for (loop=0; loop < ((COLS-15)/2)-(strlen(acct_opts[ctr].name)/2)-2; loop++)
				waddch(acct_edit_win, ACS_HLINE);
			waddch(acct_edit_win, '[');
			waddstr(acct_edit_win, acct_opts[ctr].name+1);
			waddch(acct_edit_win, ']');
			for (loop=((COLS-15)/2)-(strlen(acct_opts[ctr].name)/2)+2+strlen(acct_opts[ctr].name+1); loop < COLS-16; loop++)
				waddch(acct_edit_win, ACS_HLINE);
			wclrtoeol(acct_edit_win);
		} else {
			switch (acct_opts[ctr].type)
			{
				case C_STRING:
				case E_STRING:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s", acct_opts[ctr].name, (char *)acct_opts[ctr].value);
					break;
				case INT:
					i = acct_opts[ctr].value;
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %d", acct_opts[ctr].name, *i);
					break;
				case CHAR:
					c = acct_opts[ctr].value;
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %d", acct_opts[ctr].name, c[0]);
					break;
				case DATE:
					t = acct_opts[ctr].value;
					if (*t)
						mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s", acct_opts[ctr].name, ctime(t));
					else
						mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: N/A", acct_opts[ctr].name);
					break;
				case PASSWORD:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: ********", acct_opts[ctr].name);
					break;
				case SEX:
					i = acct_opts[ctr].value;
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s",  acct_opts[ctr].name, (*i) ? "Male" : "Female");
					break;
				case BOOL_1:
					i = acct_opts[ctr].value;
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s",  acct_opts[ctr].name, (*i) ? "On" : "Off");
					break;
				case BOOL_2:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s",  acct_opts[ctr].name, (acct_opts[ctr].value) ? "True" : "False");
					break;
				case BOOL_3:
					i = acct_opts[ctr].value;
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s",  acct_opts[ctr].name, (*i == 'y') ? "Yes" : "No");
					break;
				case A_FLAG:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s", acct_opts[ctr].name, (account.cur_acct.accessflags & (int)acct_opts[ctr].value) ? "On" : "Off");
					break;
				case M_FLAG:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: %s", acct_opts[ctr].name, (account.cur_acct.menu_flags & (int)acct_opts[ctr].value) ? "On" : "Off");
					break;
				default:
					mvwprintw(acct_edit_win, ctr-first_field, 1, " %-20.20s: -=*[Not Implemented]*=-", acct_opts[ctr].name);
					break;
			}
		}
	}
	wattron(acct_edit_win, COLOR_PAIR(4) | A_BOLD);
	mvwprintw(acct_edit_win, cur_field-first_field, 1, " %-20.20s", acct_opts[cur_field].name);
	wattron(acct_edit_win, COLOR_PAIR(3) | A_BOLD);
	wnoutrefresh(acct_edit_win);
	return;
}

int update_ticker(WINDOW *win, int lap)
{
	static char text[80];
	char loadaverage[80];
	char fname[128];
	static int direction=0;
	static int message=0;
	static int num_iterations=0;
	struct last20_str l20;
	int len = strlen(text);
	int i=0;
	int y, x;
	int ctr;
	FILE *f;
	
	if (num_iterations == 1)
	{
		num_iterations=0;
		clear_window(win, 3);
		if (++message == 4)
			message=0;
		i=1;
	}
	if (i)
	{
		switch (message)
		{
			case 0:
				strcpy(text, "BBSConsole (c) 1998 RDS - kEwjhOe");
				break;
			case 1:
				sprintf(text, "Load Average: %s", get_load_avg(loadaverage));
				break;
			case 2:
				sprintf(fname, "%s/callcnt.bbs", cfg->bbs_path);
				if ( ( f = fopen(fname, "r")) != NULL)
				{
					fseek(f, sizeof(struct last20_str) * -1, SEEK_END);
					if (fread(&l20, sizeof(struct last20_str), 1, f))
					{
						sprintf(text, "Last Caller: %s - %s", l20.l2_name, l20.l2_time);
						text[(win)->_maxx-1] = 0;
					} else {
						sprintf(text, "Last Caller: Information Not Available (1)");
					}
					fclose(f);
				} else
					sprintf(text, "Last Caller: Information Not Available (2)");
				break;
			case 3:
				strcpy(text, "Falken Website - http://www.FalkenBBS.com");
				break;
			default:
				strcpy(text, "Falken BBS Software For Linux");
				break;
		}
	}
	len = strlen(text);
	getmaxyx(win, y, x);
	wmove(win, 0, (x/2)-(len/2));
	if (graphics_on)
	{
		if (direction)
		{
			for (ctr = 0; ctr < len; ctr++)
			{
				i = len - ((lap / TICKER_DELAY) - len);
				if ((ctr >= i - 11)  && (ctr <= i - 7))
				{
					wattrset(win, COLOR_PAIR(8) | A_BOLD);
					waddch(win, text[ctr]);
				} else if ((ctr >= i - 6)  && (ctr <= i - 3)) {
					wattrset(win, COLOR_PAIR(3));
					waddch(win, text[ctr]);
				} else if ((ctr >= i - 2) && (ctr <= i + 2)) {
					wattrset(win, COLOR_PAIR(3) | A_BOLD);
					waddch(win, text[ctr]);
				} else if ((ctr >= i + 3) && (ctr <= i + 6)) {
					wattrset(win, COLOR_PAIR(3));
					waddch(win, text[ctr]);
				} else if ((ctr >= i + 7) && (ctr <= i + 11)) {
					wattrset(win, COLOR_PAIR(8) | A_BOLD);
					waddch(win, text[ctr]);
				} else {
					wattrset(win, COLOR_PAIR(8));
					waddch(win, text[ctr]);
				}
			}
		} else {
			for (ctr = 0; ctr < len; ctr++)
			{
				if ((ctr >= (lap / TICKER_DELAY) - 11) && (ctr <= (lap / TICKER_DELAY) - 7))
				{
					wattrset(win, COLOR_PAIR(8) | A_BOLD);
					waddch(win, text[ctr]);
				} else if ((ctr >= (lap / TICKER_DELAY) - 6) && (ctr <= (lap / TICKER_DELAY) - 3)) {
					wattrset(win, COLOR_PAIR(3));
					waddch(win, text[ctr]);
				} else if ((ctr >= (lap / TICKER_DELAY) - 2) && (ctr <= (lap / TICKER_DELAY) + 2)) {
					wattrset(win, COLOR_PAIR(3) | A_BOLD);
					waddch(win, text[ctr]);
				} else if ((ctr >= (lap / TICKER_DELAY) + 3) && (ctr <= (lap / TICKER_DELAY) + 6)) {
					wattrset(win, COLOR_PAIR(3));
					waddch(win, text[ctr]);
				} else if ((ctr >= (lap / TICKER_DELAY) + 7) && (ctr <= (lap / TICKER_DELAY) + 11)) {
					wattrset(win, COLOR_PAIR(8) | A_BOLD);
					waddch(win, text[ctr]);
				} else {
					wattrset(win, COLOR_PAIR(8));
					waddch(win, text[ctr]);
				}
			}
		}
	} else {
		wattrset(win, COLOR_PAIR(3));
		waddstr(win, text);
	}
	wnoutrefresh(win);
	if ((lap / TICKER_DELAY) == len)
		direction = 1;
	if ((lap / TICKER_DELAY) == len*2)
	{
		lap = 0; 
		num_iterations++;
		direction = 0;
	}
	return(lap);
}

int tot_accts(void)
{
	char fname[128];
	
	sprintf(fname, "%s/users.dat", cfg->data_path);
	return(file_size(fname) / sizeof(struct acct_rec));
}

int get_proc_info(void) // This is a cpu hog.. don't use it too much!
{
	DIR *proc_fs;
	FILE *proc_file;
	int loop=0;
	struct dirent *ent;
	char proc_line[1024];
	char fname[128];
	struct proc_s *cur_proc = NULL;
	char* tmp; 

	if ( (proc_fs = opendir("/proc")) == NULL)
	{
		fprintf(stderr, "Couldn't open proc directory\n");
		return(loop);
	}
	clear_proc_list();
	while ( (ent = readdir(proc_fs)) != NULL)
	{
		if ((ent->d_name[0] >= '0') && (ent->d_name[0] <= '9'))
		{
			loop++;
			sprintf(fname, "/proc/%s/status", ent->d_name);
			if ( (proc_file = fopen(fname, "r")) != NULL)
			{
				if (first_proc == NULL)
				{
					//fprintf(stderr, "loop %d lap %d- Created first!\n", loop, lap);
					first_proc = malloc(sizeof(struct proc_s));
					cur_proc = first_proc;
					memset(cur_proc, 0, sizeof(struct proc_s)); 
				} else if (cur_proc->next == NULL) {
					cur_proc->next = malloc(sizeof(struct proc_s));
					//fprintf(stderr, "%d -> %d - ", (int)cur_proc, (int)cur_proc->next);
					cur_proc = cur_proc->next;
					memset(cur_proc, 0, sizeof(struct proc_s)); 
				} 
				fgets(proc_line, 1023, proc_file);
				fgets(proc_line, 1023, proc_file);
				fgets(proc_line, 1023, proc_file);
				fgets(proc_line, 1023, proc_file);
				fgets(proc_line, 1023, proc_file);
				sscanf(proc_line, "Uid: %d %d", &cur_proc->euid, &cur_proc->uid);
				fclose(proc_file);
			}
			if (cur_proc->uid == bbs_uid)  // Skip all this CPU intense bullshit if we don't need it
			{
				sprintf(fname, "/proc/%s/stat", ent->d_name);
				if ( (proc_file = fopen(fname, "r")) != NULL)
				{
					fgets(proc_line, 1023, proc_file);
					tmp = strrchr(proc_line, ')');
					*tmp = '\0';
					memset(cur_proc->cmd, 0, sizeof(cur_proc->cmd));
					sscanf(proc_line, "%d (%39c", &cur_proc->pid, cur_proc->cmd);
					sscanf(tmp + 2,     /* skip space after ')' too */
					"%c %d %d %d %d %d %lu %lu %lu %lu %lu %ld %ld %ld %ld %d "
					"%d %lu %lu %ld %lu %lu %lu %lu %lu %lu %lu %lu %d %d %d %d %lu",
					&cur_proc->state, &cur_proc->ppid, &cur_proc->pgrp, &cur_proc->session, &cur_proc->tty, &cur_proc->tpgid,
					&cur_proc->flags, &cur_proc->min_flt, &cur_proc->cmin_flt, &cur_proc->maj_flt, &cur_proc->cmaj_flt,
					&cur_proc->utime, &cur_proc->stime, &cur_proc->cutime, &cur_proc->cstime, &cur_proc->priority, &cur_proc->nice,
					&cur_proc->timeout, &cur_proc->it_real_value, &cur_proc->start_time, &cur_proc->vsize, &cur_proc->rss,
					&cur_proc->rss_rlim, &cur_proc->start_code, &cur_proc->end_code, &cur_proc->start_stack,
					&cur_proc->kstk_esp, &cur_proc->kstk_eip, &cur_proc->signal, &cur_proc->blocked, &cur_proc->sigignore,
					&cur_proc->sigcatch, &cur_proc->wchan);
					//fprintf(stderr, " - %s", cur_proc->cmd);
					fclose(proc_file);
				}
				sprintf(fname, "/proc/%s/statm", ent->d_name);
				if ( (proc_file = fopen(fname, "r")) != NULL)
				{
					fgets(proc_line, 1023, proc_file);
					sscanf(proc_line, "%ld %ld %ld %ld %ld %ld %ld",
						&cur_proc->size, &cur_proc->resident, &cur_proc->share,
						&cur_proc->trs, &cur_proc->lrs, &cur_proc->drs, &cur_proc->dt);
					fclose(proc_file);
				}
			}
		}
	}
	closedir(proc_fs);
	return(loop);
}

int num_bbs_procs(void)
{
	struct proc_s *cur_proc = first_proc;
	int loop=0;

	while (cur_proc != NULL)
	{
		if ((cur_proc->uid == bbs_uid) || (strcmp(cur_proc->cmd, "mainbbs") == 0))
		{
			//fprintf(stderr, "%d %s\n", loop, cur_proc->cmd);
			loop++;
		}
		cur_proc = cur_proc->next;
	}
	return(loop);
}

int get_bbs_mem(void)
{
	struct proc_s *cur_proc = first_proc;
	int total_pages=0;
	
	while (cur_proc)
	{
		if ((cur_proc->uid == bbs_uid) || !strcmp(cur_proc->cmd, "mainbbs"))
			total_pages+=cur_proc->size;
		cur_proc = cur_proc->next;
	}
	return(total_pages * 4);
}

int get_bbs_resident(void)
{
	struct proc_s *cur_proc = first_proc;
	int total_pages=0;
	
	while (cur_proc)
	{
		if ((cur_proc->uid == bbs_uid) || !strcmp(cur_proc->cmd, "mainbbs"))
			total_pages+=cur_proc->resident;
		cur_proc = cur_proc->next;
	}
	return(total_pages * 4);
}

int get_bbs_shared(void)
{
	struct proc_s *cur_proc = first_proc;
	int total_pages=0;
	
	while (cur_proc)
	{
		if ((cur_proc->uid == bbs_uid) || !strcmp(cur_proc->cmd, "mainbbs"))
			total_pages+=cur_proc->share;
		cur_proc = cur_proc->next;
	}
	return(total_pages * 4);
}

int clear_proc_list(void)
{
	struct proc_s *next_rec, *cur_rec = first_proc;
	
	while (cur_rec != NULL)
	{
		next_rec = cur_rec->next;
		free(cur_rec);
		cur_rec = next_rec;
	}
	first_proc = NULL;
	return(0);
}

void write_max(void)
{
	FILE *f;
	
	if ( (f = fopen("max.dat", "w")) != NULL)
	{
		fwrite(&max_info, sizeof(struct _max_info), 1, f);
		fclose(f);
	}
}

void read_max(void)
{
	FILE *f;
	
	if ( (f = fopen("max.dat", "r")) != 0)
	{
		fread(&max_info, sizeof(struct _max_info), 1, f);
		fclose(f);
	}
	return;
}


int random_number(int min, float max)
{
	return(min+(int) (max*rand()/(RAND_MAX+1.0)));
}

// -----------------------------------------------------------------------
// NCurses utilities
// -----------------------------------------------------------------------

int clear_window(WINDOW *win, int color)
{
	char spaces[1024];
	int x2, y2, ctr;

	wattron(win, COLOR_PAIR(color) | A_BOLD);
	getmaxyx(win, y2, x2);

	memset(spaces, ' ', x2);
	spaces[x2] = 0;
	
	wmove(win, 0, 0);
	for (ctr = 0; ctr < y2; ctr++)
		wprintw(win, "%s", spaces);
	wmove(win, 0, 0);
	return(0);
}
	
void outline_window(WINDOW *parent, WINDOW *win)
{
	int x1, y1, x2, y2, ctr;
	
	wattron(parent, COLOR_PAIR(1) | A_BOLD);
	getbegyx(win, y1, x1);
	getmaxyx(win, y2, x2);
	y1--; x1--;
	y2+=y1-1; x2+=x1-1;
	mvwaddch(parent, y1-1, x1-1, ACS_ULCORNER);
	for (ctr = x1; ctr <= x2; ctr++)
		mvwaddch(parent, y1-1, ctr, ACS_HLINE);
	mvwaddch(parent, y1-1, x2+1, ACS_URCORNER);
	for (ctr = y1; ctr <= y2; ctr++)
		mvwaddch(parent, ctr, x1-1, ACS_VLINE);
	for (ctr = y1; ctr <= y2; ctr++)
		mvwaddch(parent, ctr, x2+1, ACS_VLINE);
	mvwaddch(parent, y2+1, x1-1, ACS_LLCORNER);
	for (ctr = x1; ctr <= x2; ctr++)
		mvwaddch(parent, y2+1, ctr, ACS_HLINE);
	mvwaddch(parent, y2+1, x2+1, ACS_LRCORNER);
}

void e_mvwgetnstrb(WINDOW *win, int y, int x, char *s, int max, int barsize, int color, int echo)
{
	char buffer[max+1];
	int ks=0; char fs[20];
	int curpos = strlen(s);
	int i, ctr;

	curs_set(1);
	wattron(win, COLOR_PAIR(color) | A_BOLD);
	strcpy(buffer, s);
	sprintf(fs, "%%-%d.%ds", barsize, barsize);
	wmove(win, y, x);
	wprintw(win, fs, s);
	wmove(win, y, x+strlen(s));
	i = (win)->_delay;  // This should block REGARDLEESS of what the user thinks
	(win)->_delay = -1; // we'll be nice enough to set it back later
	while (ks != 10)
	{
		ks = wgetch(win);
		if (ks == -1) continue;
		switch (ks)
		{
			case KEY_LEFT:
				if (curpos > 0)
				{
					curpos--;
					wmove(win, (win)->_cury, (win)->_curx-1);
				}
				break;
			case KEY_RIGHT:
				if (curpos < strlen(buffer))
				{
					curpos++;
					wmove(win, (win)->_cury, (win)->_curx+1);
				}
				break;
			case 330:  // Delete
				if (curpos < strlen(buffer))
				{
					memmove(buffer+curpos, buffer+curpos+1, strlen(buffer)-curpos+1);
					wmove(win, y, x);
					if (echo)
						wprintw(win, fs, buffer);
					else
						for (ctr = 0; ctr < barsize; ctr++)
							if (ctr < strlen(buffer))
								waddch(win, '*');
							else
								waddch(win, ' ');
					wmove(win, y, x+curpos);
				} else {
					beep();
					wrefresh(win);
				}
				break;
			case CTRL('y'):
			case CTRL('k'):
				buffer[0] = 0;
				curpos = 0;
				wmove(win, y, x);
				wprintw(win, fs, buffer);
				wmove(win, y, x);
				break;
			case 262:  // END key
				curpos = 0;
				wmove(win, y, x);
				break;
			case 360:  // END key
				curpos = strlen(buffer);
				wmove(win, y, x+curpos);
				break;
			case 263:    // Backspace
			case 8:    // Backspace
				if (curpos > 0)
				{
					if (curpos < strlen(buffer))
					{
						memmove(buffer+curpos-1, buffer+curpos, strlen(buffer)-curpos+1);
						curpos--;
					} else
						buffer[--curpos] = 0;
					wmove(win, y, x);
					if (echo)
						wprintw(win, fs, buffer);
					else
						for (ctr = 0; ctr < barsize; ctr++)
							if (ctr < strlen(buffer))
								waddch(win, '*');
							else
								waddch(win, ' ');
					wmove(win, y, x+curpos);
				} else {
					beep();
					wrefresh(win);
				}
				break;
			case 10:
				break;
			default:
				if ((ks >= 32) && (ks <= 127))
				{
					if (strlen(buffer) < max)
					{
						if (curpos < strlen(buffer))
						{
							memmove(buffer+curpos+1, buffer+curpos, strlen(buffer)-curpos+1);
							wmove(win, y, x);
							if (echo)
								wprintw(win, fs, buffer);
							else
								for (ctr = 0; ctr < barsize; ctr++)
									if (ctr < strlen(buffer))
										waddch(win, '*');
									else
										waddch(win, ' ');
							wmove(win, y, x+curpos);
						}
						if (curpos == strlen(buffer))
							buffer[curpos+1] = 0;
						buffer[curpos++] = ks;
						if (echo)
							waddch(win, ks);
						else
							waddch(win, '*');
						wrefresh(win);
					}
				} else {
					//wprintw(win, "%d", ks);
					beep();
					wrefresh(win);
				}
				break;
		}
	}
	(win)->_delay = i; // see.. told ya i was a nice guy;
	strcpy(s, buffer);
	curs_set(0);
	wattron(win, COLOR_PAIR(3) | A_BOLD);
}

char *gets_dialog(int nlines, int ncols, int beginy, int beginx, char *title, char *prompt, char *text, int max, int barsize, int echo)
{
	WINDOW *dialog_win;
	PANEL *dialog_pan;

	dialog_win = newwin(nlines, ncols, beginy, beginx);
	dialog_pan = new_panel(dialog_win);
	clear_window(dialog_win, 1);
	nodelay(dialog_win, TRUE);
	keypad(dialog_win, TRUE);
	box(dialog_win, 0, 0);
	mvwaddstr(dialog_win, 0, 4, title);
	mvwaddstr(dialog_win, 1, 1, prompt);
	wrefresh(dialog_win);
	e_mvwgetnstrb(dialog_win, 1, strlen(prompt)+1, text, max, barsize, 3, echo);
	hide_panel(dialog_pan);
	delwin(dialog_win);
	del_panel(dialog_pan);
	return(text);
}

void dialog(int nlines, int ncols, int beginy, int beginx, char *title, char *text)
{
	WINDOW *dialog_win;
	PANEL *dialog_pan;
	int ctr, i;

	dialog_win = newwin(nlines, ncols, beginy, beginx);
	dialog_pan = new_panel(dialog_win);
	clear_window(dialog_win, 1);
	nodelay(dialog_win, TRUE);
	keypad(dialog_win, TRUE);
	box(dialog_win, 0, 0);
	mvwaddstr(dialog_win, 0, 4, title);
	wmove(dialog_win, 1, 1);
	for (ctr = 0; ctr < strlen(text); ctr++)
		if (text[ctr] == '\n')
			wmove(dialog_win, (dialog_win)->_cury+1, 1);
		else
			waddch(dialog_win, text[ctr]);
	wattron(dialog_win, COLOR_PAIR(6));
	mvwaddstr(dialog_win, (dialog_win)->_maxy-1, ((dialog_win)->_maxx/2)-4, " OK ");
	wattron(dialog_win, COLOR_PAIR(1) | A_BOLD);
	wrefresh(dialog_win);
	i = (dialog_win)->_delay;  // This should block REGARDLEESS of what the user thinks
	(dialog_win)->_delay = -1; // we'll be nice enough to set it back later
	while (wgetch(dialog_win) != 10);
	(dialog_win)->_delay = i; 
	hide_panel(dialog_pan);
	delwin(dialog_win);
	del_panel(dialog_pan);
}

int dialog_yesno(int nlines, int ncols, int beginy, int beginx, char *title, char *text, int start)
{
	WINDOW *dialog_win;
	PANEL *dialog_pan;
	int ctr;
	int selection=start;
	int ks = 0, i;

	dialog_win = newwin(nlines, ncols, beginy, beginx);
	dialog_pan = new_panel(dialog_win);
	clear_window(dialog_win, 1);
	nodelay(dialog_win, TRUE);
	keypad(dialog_win, TRUE);
	box(dialog_win, 0, 0);
	mvwaddstr(dialog_win, 0, 4, title);
	wmove(dialog_win, 1, 1);
	i = (dialog_win)->_delay;  // This should block REGARDLEESS of what the user thinks
	(dialog_win)->_delay = -1; // we'll be nice enough to set it back later
	for (ctr = 0; ctr < strlen(text); ctr++)
		if (text[ctr] == '\n')
			wmove(dialog_win, (dialog_win)->_cury+1, 1);
		else
			waddch(dialog_win, text[ctr]);
	wmove(dialog_win, (dialog_win)->_maxy-1, (dialog_win)->_maxx/2 - 8);
	while (ks != 10)
	{
		switch (selection)
		{
			case 0:
				wmove(dialog_win, (dialog_win)->_maxy-1, (dialog_win)->_maxx/2 - 8);
				wattron(dialog_win, COLOR_PAIR(1) | A_BOLD);
				wprintw(dialog_win, " OK  ");
				wrefresh(dialog_win);
				wattron(dialog_win, COLOR_PAIR(6));
				wprintw(dialog_win, " Cancel ");
				break;
			case 1:
				wmove(dialog_win, (dialog_win)->_maxy-1, (dialog_win)->_maxx/2 - 8);
				wattron(dialog_win, COLOR_PAIR(6));
				wprintw(dialog_win, " OK ");
				wrefresh(dialog_win);
				wattron(dialog_win, COLOR_PAIR(1) | A_BOLD);
				wprintw(dialog_win, "  Cancel ");
				break;
		}
		wrefresh(dialog_win);
		ks = wgetch(dialog_win);
		switch (ks)
		{
			case KEY_LEFT:
			case KEY_RIGHT:
			case 9:						// TAB
				if (selection == 1)
					selection = 0;
				else
					selection = 1;
				break;
			case 'c':
			case 'C':
				selection = 0;
				ks = 10;
				break;
			case 'o':
			case 'O':
				selection = 1;
				ks = 10;
				break;
		}
	}
	(dialog_win)->_delay = i; // we'll be nice enough to set it back later
	hide_panel(dialog_pan);
	delwin(dialog_win);
	del_panel(dialog_pan);
	return(selection);
}

int search_username(char *name, int startpoint)
{
	int ctr;
	int end;
	char buffer[uidlen+1];
	struct acct_rec sacct;
	
	strcpy(buffer, name);
	strupr(buffer);
	memset(&sacct, 0, sizeof(struct acct_rec));
	end = tot_accts() - 1;
	for (ctr = startpoint;  ctr <= end; ctr++)
	{
		load_user_account(&sacct, ctr); 
		strupr(sacct.acctname);
		if (strstr(sacct.acctname, buffer) != NULL)
			return(atoi(sacct.acctnum)-1);
	}
	dialog_yn(6, 50, 9, 4, "[ Search ]", "Search reached the end of the userfile\nProceed from the beginning?");
	for (ctr = 0;  ctr < startpoint-1; ctr++)
	{
		load_user_account(&sacct, ctr); 
		strupr(sacct.acctname);
		if (strstr(sacct.acctname, buffer) != NULL)
		{
			return(atoi(sacct.acctnum)-1);
		}
	}
	return(-1);
}

int unlock(void)
{
	char buffer[32];

	buffer[0] = 0;
	gets_dialog(3, 45, 5, 8, "[ Screen Lock ]", "Password: ", buffer, 32, 32, 0);
	if (strcmp(cfg->sysop_password, buffer) == 0)
	{
		screenlocked = 0;
		return(0);
	} else {
		return(1);
	}
}

