/*
 * Copyright (C) 1995  Peter L Jones
 * See file COPYING before you even think about using this program.
 */
static char version[] = "$Id: usermod.c,v 1.1 1995/12/19 19:44:57 thanatos Exp $";
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <dirent.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*
 * I have not written supplementary groups code yet
 * - please update /etc/group by hand :-(
 * (options are accepted and ignored)
 */
#define NO_SUPPGROUPS

#define PASSWD_FILE "/etc/passwd"
#define GROUP_FILE "/etc/group"
const char SHADOW_SYSV[] = "/etc/shadow";
const char SHADOW_BSD[] = "/etc/master.passwd";	/* I got this name from "adduser" */

/*
 * Private local functions
 */
static int Shadow(void);
static int Parse(int argc, char *argv[]);
static void Unlock(void);
static int Validate(void);
static int Update(void);

static void Usage(void);
static void Long_usage(void);
static int LockPW(void);
static int BackupPW(void);
static int ModifyUser(void);
#ifndef NO_SUPPGROUPS
static int LockGrp(void);
static int BackupGrp(void);
static int AddGroups(void);
#endif

static struct mainOptions {
	gid_t	num_gid;
	const char*	shell;
	const char*	group;
	const char*	homedir;
	const char*	comment;
	const char*	suppgroups;
	const char*	username;
	const char*	pname;
} Opt = { 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, };


int main(int argc, char *argv[]) {
	Opt.pname = argv[0];
	if (atexit(Unlock)) {
		perror(argv[0]);
		return 1;
	}
	if (Shadow()) return 2;
	if (Parse(argc, argv)) return 3;
	if (Validate()) return 4;
	if (Update()) return 5;
	return 0;
}


static int Update(void) {
	if (LockPW()) return 1;
	if (BackupPW()) return 1;
	if (ModifyUser()) return 1;
#ifndef NO_SUPPGROUPS
	if (Opt.suppgroups) {
		if (LockGrp()) return 1;
		if (BackupGrp()) return 1;
		if (ChangeGroups()) return 1;
	}
#endif
	return 0;
}

static int ModifyUser(void) {
	FILE *PASSWD_IN;
	FILE *PASSWD_OUT;
	static struct passwd pw, *pwp;
	
	PASSWD_IN = fopen(PASSWD_FILE ".OLD", "r");
	if (!PASSWD_IN) {
		fprintf(stderr,"%s: could not read from %s: %s\n", Opt.pname, PASSWD_FILE ".OLD", strerror(errno));
		fclose(PASSWD_IN);
		return 1;
	}
	PASSWD_OUT = fopen(PASSWD_FILE, "w");
	if (!PASSWD_OUT) {
		fprintf(stderr,"%s: could not write to %s: %s\n", Opt.pname, PASSWD_FILE, strerror(errno));
		fclose(PASSWD_IN);
		return 1;
	}
	while ((pwp = fgetpwent(PASSWD_IN)) != NULL) {
		if (strcmp(pwp->pw_name,Opt.username)) {
			putpwent(pwp, PASSWD_OUT);
		} else {
			memcpy(&pw, pwp, sizeof pw);
			if (Opt.shell) pw.pw_shell = strdup(Opt.shell);
			if (Opt.group) pw.pw_gid = Opt.num_gid;
			if (Opt.homedir) pw.pw_dir = strdup(Opt.homedir);
			if (Opt.comment) pw.pw_gecos = strdup(Opt.comment);
			putpwent(&pw, PASSWD_OUT);
			if (Opt.shell) free(pw.pw_shell);
			if (Opt.homedir) free(pw.pw_dir);
			if (Opt.comment) free(pw.pw_gecos);
		}
	}
	fclose(PASSWD_IN);
	fclose(PASSWD_OUT);
	return 0;
}

#ifndef NO_SUPPGROUPS
static int ChangeGroups(void) {
	return 0;
}

static int BackupGrp(void) {
	if (system("/bin/cp " GROUP_FILE " " GROUP_FILE ".OLD")) return 1;
	chmod(GROUP_FILE ".OLD", 0);
	return 0;
}

static int LockGrp(void) {
	static int fd;
	if ((fd = open(GROUP_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create group file lock: %s\n", Opt.pname, strerror(errno));
		remove(PASSWD_FILE "~");
		return 1;
	} else close(fd);
	return 0;
}
#endif

static int Validate() {
	static char *p;
	static long id;

	if (!(Opt.homedir || Opt.shell || Opt.comment || Opt.group || Opt.suppgroups)) {
		fprintf(stderr,"%s: no changes requested.\n",Opt.pname);
		return 1;
	}

	if (Opt.group) {
		id = strtol(Opt.group,&p,0);

		if (p && (*p == '\0')) {
			/* numeric */
			if (!getgrgid(id)) {
				fprintf(stderr,"%s: '%s' is not an existing group number.\n", Opt.pname, Opt.group);
				return 1;
			}
			Opt.num_gid = id;
		} else {
			/* non-numeric */
			static struct group *g;
			if (!(g = getgrnam(Opt.group))) {
				fprintf(stderr,"%s: '%s' is not an existing group name.\n", Opt.pname, Opt.group);
				return 1;
			}
			Opt.num_gid = g->gr_gid;
		}
	}

#ifndef NO_SUPPGROUPS
	if (Opt.suppgroups) {
		/* this should be a comma-separated list of existing groups */
		/* It's probably a good idea to split these out into a more useable format at this stage */
	}
#endif

	if (!getpwnam(Opt.username)) {
		fprintf(stderr,"%s: '%s' is not a valid user name.\n", Opt.pname, Opt.username);
		return 1;
	}
	
	return 0;
}

static int Parse(int argc, char *argv[]) {
	int c;
	char short_options[] = "hVd:s:c:g:G:";
	struct option long_options[] = {
		{"help",		no_argument,       0, 'h'},
		{"version",		no_argument,       0, 'V'},
		{"directory",		required_argument, 0, 'd'},
		{"shell",		required_argument, 0, 's'},
		{"comment",		required_argument, 0, 'c'},
		{"group",		required_argument, 0, 'g'},
		{"suppgroups",		required_argument, 0, 'G'},
		{0, 0, 0, 0}
	};

	while ((c = getopt_long(argc, argv, short_options,
			long_options, NULL)) != -1) switch (c) {
		case 'd':	Opt.homedir=optarg; break;
		case 's':	Opt.shell=optarg; break;
		case 'c':	Opt.comment=optarg; break;
		case 'g':	Opt.group=optarg; break;
		case 'G':	Opt.suppgroups=optarg; break;

		case 'V':
			printf("%s\n",version);
			exit(0);

		case 'h':
			Long_usage();
			exit(0);

		default:
			Usage();
			return 1;
	}

	if (optind != argc-1) {
		fprintf(stderr,"%s: user name not specified\n", Opt.pname);
		Usage();
		return 1;
	}

	Opt.username=argv[optind];

	return 0;
}

static void Usage(void) {
	fprintf(stderr,"Try `%s --help' for more information.\n",Opt.pname);
}

static void Long_usage(void) {
	fprintf(stderr,"Usage: %s [option ...] username\n", Opt.pname);
	fprintf(stderr,
		"where 'option' is:\n"
		"  -h,          --help\n"
		"  -V,          --version\n"
		"  -s progname, --shell=progname\n"
		"  -g gid,      --group=gid\n"
		"  -d homedir,  --directory=homedir\n"
		"  -c realname, --comment=realname\n"
#ifndef NO_SUPPGROUPS
		"  -G list,     --suppgroups=list\n"
#endif
	);
}

static int Shadow(void) {
	/* if either /etc/shadow or /etc/master.passwd exists, this is shadow system - we don't work! */
	if (!access(SHADOW_SYSV,F_OK) || !access(SHADOW_BSD,F_OK)) {
		fprintf(stderr,"%s: Detected a SHADOW PASSWORD file on this system.  "
			"If you are not using shadow passwords, please remove the shadow file.\n", Opt.pname);
		return 1;
	}
	return 0;
}

static int BackupPW(void) {
	if (system("/bin/cp " PASSWD_FILE " " PASSWD_FILE ".OLD")) return 1;
	chmod(PASSWD_FILE ".OLD", 0);
	return 0;
}

static int LockPW(void) {
	/* lock the password and group files */
	static int fd;
	if ((fd = open(PASSWD_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create password lock: %s\n", Opt.pname, strerror(errno));
		return 1;
	} else close(fd);
	return 0;
}

static void Unlock(void) {
	remove(PASSWD_FILE "~");
	remove(GROUP_FILE "~");
}
