/*
 * Citadel/UX
 *
 * commands.c - front end for Citadel
 *
 * This version is the traditional command parser for room prompts.
 *
 */

#include "sysdep.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <sys/time.h>

#ifdef POSIX_TERMIO
#include <termios.h>
#else
#include <sgtty.h>
#endif

#ifdef NEED_SELECT_H
#include <sys/select.h>
#endif


#include <signal.h>
#include <errno.h>
#include "citadel.h"

struct citcmd {
	struct citcmd *next;
	int c_cmdnum;
	int c_axlevel;
	char c_keys[5][64];
	};

#define IFNEXPERT if ((userflags&US_EXPERT)==0)


extern unsigned room_flags;
extern char room_name[];
extern struct serv_info serv_info;
extern char axlevel;
extern char is_room_aide;
extern unsigned userflags;
extern char sigcaught;
extern char editor_path[];
extern char printcmd[];
extern char have_xterm;
extern char rc_username[];
extern char rc_password[];
extern char rc_floor_mode;
char rc_exp_beep;
extern int lines_printed;
extern char express_msgs;

char *gl_string;

struct citcmd *cmdlist = NULL;

void sighandler();
void logoff();
int struncmp();
void formout();
int room_prompt();
void back();
int checkpagin();
void color();


/* these variables are local to this module */
char keepalives_enabled = KA_YES;	/* send NOOPs to server when idle */
int ok_to_interrupt = 0;		/* print express msgs asynchronously */


/*
 * print_express()  -  print express messages if there are any
 */
void print_express() {
	char buf[256];

	if (express_msgs == 0) return;
	express_msgs = 0;
	serv_puts("PEXP");
	serv_gets(buf);
	if (buf[0]!='1') return;

	if (rc_exp_beep) {
		putc(7,stdout);
		}
	color(1);
	printf("---\n");
	while (serv_gets(buf), strcmp(buf,"000")) {
		printf("%s\n",buf);
		}
	printf("---\n");
	color(7);
	}


void set_keepalives(s)
int s; {
	keepalives_enabled = (char)s;
	}

/* 
 * This loop handles the "keepalive" messages sent to the server when idling.
 */
void do_keepalive() {
	char buf[256];
	static long idlet = 0L;
	long now;

	time(&now);
	if ((now - idlet) < ((long)S_KEEPALIVE)) return;
	time(&idlet);

	if (keepalives_enabled != KA_NO) {
		serv_puts("NOOP");
		if (keepalives_enabled == KA_YES) {
			serv_gets(buf);
			if (buf[3]=='*') {
				express_msgs = 1;
				if (ok_to_interrupt == 1) {
					printf("\r                    \r");
					print_express();
					printf("%s%c ",room_name,
						room_prompt(room_flags));
					fflush(stdout);	
					}
				}
			}
		}
	}


int inkey() {		/* get a character from the keyboard, with   */
	int a;		/* the watchdog timer in effect if necessary */
        fd_set rfds;
        struct timeval tv;
	long start_time, now;
	char inbuf[2];

	time(&start_time);
	
	do {

		/* This loop waits for keyboard input.  If the keepalive
		 * timer expires, it sends a keepalive to the server if
		 * necessary and then waits again.
		 */
		do {
			do_keepalive();
			fflush(stdout);
			FD_ZERO(&rfds);
			FD_SET(0,&rfds);
			tv.tv_sec = S_KEEPALIVE;
			tv.tv_usec = 0;

			time(&now);
			if (((now-start_time) > SLEEPING)
			   && (SLEEPING != 0) && (getppid()==1)) {
				printf("Sleeping? Call again.\n");
				logoff(SIGALRM);
				}

			select(1, &rfds, NULL, NULL, &tv);
			} while (!FD_ISSET(0, &rfds));




		/* At this point, there's input, so fetch it.
		 * (There's a hole in the bucket...)
		 */
		read(0, inbuf, 1);
		a = inbuf[0];
		if (a==127) a=8;
		if (a>126) a=0;
		if (a==10) a=13;
		if (((a!=4)&&(a!=13)&&(a!=8)&&(a!=NEXT_KEY)&&(a!=STOP_KEY))
			&& ((a<32)||(a>126))) a=0;
		} while(a==0);
	return(a);
	}

void getline(string,lim)	/* Gets a line from the terminal */
char string[]; 		/* Pointer to string buffer */
int lim;		/* Maximum length - if negative, no-show */
{
	int a,b;
	char flag = 0;

	if (lim<0) { lim=(0-lim); flag=1; }
	strcpy(string,"");
	gl_string = string;
GLA:	a=inkey(); a=(a&127);
	if ((a==8)&&(strlen(string)==0)) goto GLA;
	if ((a!=13)&&(a!=8)&&(strlen(string)==lim)) goto GLA;
	if ((a==8)&&(string[0]!=0)) {
		string[strlen(string)-1]=0;
		putc(8,stdout); putc(32,stdout); putc(8,stdout); goto GLA; }
	if ((a==13)||(a==10)) {
		putc(13,stdout);
		putc(10,stdout);
		return;
		}
	if (a<32) a='.';
	b=strlen(string);
	string[b]=a;
	string[b+1]=0;
	if (flag==0) putc(a,stdout);
	if (flag==1) putc('*',stdout);
	goto GLA;
	}


/*
 * strprompt()  -  prompt for a string, print the existing value and
 *                 allow the user to press return to keep it...
 */
void strprompt(prompt,str,len)
char *prompt;
char *str;
int len; {
	char buf[128];
	print_express();
	color(3);
	printf("%s [", prompt);
	color(1);
	printf("%s", str);
	color(3);
	printf("]: ");
	color(2);
	getline(buf,len);
	color(7);
	if (buf[0]!=0) strcpy(str,buf);
	}

/* 
 * intprompt()  -  like strprompt(), except for an integer
 *                 (note that it RETURNS the new value!)
 */
int intprompt(prompt,ival,imin,imax)
char *prompt;
int ival;
int imin;
int imax; {
	char buf[16];
	int i;
	i = ival;
	do {
		sprintf(buf,"%d",i);
		strprompt(prompt,buf,15);
		i=atoi(buf);
		if (i<imin) printf("*** Must be no less than %d.\n",imin);
		if (i>imax) printf("*** Must be no more than %d.\n",imax);
		} while((i<imin)||(i>imax));
	return(i);
	}

/* 
 * newprompt()  -  prompt for a string with no existing value
 *                 (clears out string buffer first)
 */
void newprompt(prompt,str,len)
char *prompt;
char *str;
int len; {
	color(3);
	printf("%s",prompt);
	color(2);
	getline(str,len);
	color(7);
	}


int lkey() {	/* returns a lower case value */
	int a;
	a=inkey();
	if (isupper(a)) a=tolower(a);
	return(a);
	}

/*
 * parse the citadel.rc file
 */
void load_command_set() {
	FILE *ccfile;
	char buf[256];
	struct citcmd *cptr;
	int a,d;
	int b = 0;


	/* first, set up some defaults for non-required variables */

	strcpy(editor_path,"");
	strcpy(printcmd,"");
	strcpy(rc_username,"");
	strcpy(rc_password,"");
	rc_floor_mode = 0;
	rc_exp_beep = 1;

	/* now try to open the citadel.rc file */

	ccfile = NULL;
	if (getenv("HOME") != NULL) {
		sprintf(buf,"%s/.citadelrc",getenv("HOME"));
		ccfile = fopen(buf,"r");
		}
	if (ccfile==NULL) {
		ccfile = fopen("/usr/local/lib/citadel.rc","r");
		}
	if (ccfile==NULL) {
		sprintf(buf,"%s/citadel.rc",BBSDIR);
		ccfile = fopen(buf,"r");
		}
	if (ccfile==NULL) {
		perror("commands: cannot open citadel.rc");
		logoff(errno);
		}

	while (fgets(buf, 256, ccfile) != NULL) {
	    while (isspace(buf[strlen(buf)-1]))
		buf[strlen(buf)-1] = 0;

	    if (!struncmp(buf,"editor=",7))
		strcpy(editor_path,&buf[7]);

	    if (!struncmp(buf,"printcmd=",9))
		strcpy(printcmd,&buf[9]);

	    if (!struncmp(buf,"local_screen_dimensions=",24))
		have_xterm = (char)atoi(&buf[24]);

	    if (!struncmp(buf,"use_floors=",11)) {
		if (!strucmp(&buf[11],"yes")) rc_floor_mode = RC_YES;
		if (!strucmp(&buf[11],"no")) rc_floor_mode = RC_NO;
		if (!strucmp(&buf[11],"default")) rc_floor_mode = RC_DEFAULT;
		}

	    if (!struncmp(buf,"beep=",5)) {
		rc_exp_beep = atoi(&buf[5]);
		}

	    if (!struncmp(buf,"username=",9))
		strcpy(rc_username,&buf[9]);

	    if (!struncmp(buf,"password=",9))
		strcpy(rc_password,&buf[9]);

	    if (!struncmp(buf,"cmd=",4)) {
		strcpy(buf,&buf[4]);

		cptr = (struct citcmd *) malloc (sizeof(struct citcmd));
		
		cptr->c_cmdnum = atoi (buf);
		for (d=strlen(buf); d>=0; --d)
			if (buf[d]==',') b=d;
		strcpy (buf, &buf[b+1]);

		cptr->c_axlevel = atoi (buf);
		for (d=strlen(buf); d>=0; --d)
			if (buf[d]==',') b=d;
		strcpy (buf, &buf[b+1]);

		for (a=0; a<5; ++a) cptr->c_keys[a][0] = 0;

		a = 0;  b = 0;
		buf[strlen(buf)+1] = 0;
		while (strlen(buf) > 0) {
			b = strlen(buf);
			for (d=strlen(buf); d>=0; --d)
				if (buf[d]==',') b=d;
			strncpy(cptr->c_keys[a],buf,b);
			cptr->c_keys[a][b] = 0;
			if (buf[b]==',')
				strcpy(buf,&buf[b+1]);
			else
				strcpy(buf,"");
			++a;
			}

		cptr->next = cmdlist;
		cmdlist = cptr;

		}
	    }
	fclose(ccfile);
	}



/*
 * return the key associated with a command
 */
char keycmd(cmdstr)
char cmdstr[]; {
	int a;

	for (a=0; a<strlen(cmdstr); ++a)
		if (cmdstr[a]=='&')
			return(tolower(cmdstr[a+1]));
	return(0);
	}


/*
 * Output the string from a key command without the ampersand
 * "mode" should be set to 0 for normal or 1 for <C>ommand key highlighting
 */
char *cmd_expand(strbuf,mode)
char strbuf[];
int mode; {
	int a;
	static char exp[64];
	char buf[256];
		
	strcpy(exp,strbuf);
	
	for (a=0; a<strlen(exp); ++a) {
		if (strbuf[a] == '&') {

		    if (mode == 0) {
			strcpy(&exp[a],&exp[a+1]);
			}

		    if (mode == 1) {
			exp[a] = '<';
			strcpy(buf,&exp[a+2]);
			exp[a+2] = '>';
			exp[a+3] = 0;
			strcat(exp,buf);
			}

		    }

		if (!strncmp(&exp[a],"^r",2)) {
			strcpy(buf,exp);
			strcpy(&exp[a],room_name);
			strcat(exp,&buf[a+2]);
			}

		if (!strncmp(&exp[a],"^c",2)) {
			exp[a] = ',';
			strcpy(&exp[a+1],&exp[a+2]);
			}

		}

	return(exp);
	}



/*
 * Comparison function to determine if entered commands match a
 * command loaded from the config file.
 */
int cmdmatch(cmdbuf,cptr,ncomp)
char cmdbuf[];
struct citcmd *cptr;
int ncomp; {
	int a;
	int cmdax;

	cmdax = 0;
	if (is_room_aide) cmdax = 1;
	if (axlevel >=6) cmdax = 2;

	for (a=0; a<ncomp; ++a) {
		if ( (tolower(cmdbuf[a]) != keycmd(cptr->c_keys[a]))
		   || (cptr->c_axlevel > cmdax) )
			return(0);
		}
	return(1);
	}


/*
 * This function returns 1 if a given command requires a string input
 */
int requires_string(cptr,ncomp)
struct citcmd *cptr;
int ncomp; {
	int a;
	char buf[64];
	
	strcpy(buf,cptr->c_keys[ncomp-1]);
	for (a=0; a<strlen(buf); ++a) {
		if (buf[a]==':') return(1);
		}
	return(0);
	}


/*
 * Input a command at the main prompt.
 * This function returns an integer command number.  If the command prompts
 * for a string then it is placed in the supplied buffer.
 */
int getcmd(argbuf)
char *argbuf;  {
	char cmdbuf[5];
	int cmdspaces[5];
	int cmdpos;
	int ch;
	int a;
	int got;
	struct citcmd *cptr;

	/* if we're running in idiot mode, display a cute little menu */
	IFNEXPERT formout("mainmenu");

	print_express(); /* print express messages if there are any */
	strcpy(argbuf, "");
	cmdpos = 0;
	for (a=0; a<5; ++a) cmdbuf[a]=0;
	/* now the room prompt... */
	ok_to_interrupt = 1;
	printf("\n%s%c ",room_name,room_prompt(room_flags));
	fflush(stdout);
	
	while(1) {
		ch = inkey();
		ok_to_interrupt = 0;

		if ( (ch == 8) && (cmdpos > 0) ) {
			back(cmdspaces[cmdpos-1] + 1);
			cmdbuf[cmdpos] = 0;
			--cmdpos;
			}

		cmdbuf[cmdpos] = tolower(ch);

		for (cptr = cmdlist; cptr != NULL; cptr = cptr->next) {
			if (cmdmatch(cmdbuf,cptr,cmdpos+1)) {
				
				printf("%s",cmd_expand(cptr->c_keys[cmdpos],0));
				cmdspaces[cmdpos] = strlen(
					cmd_expand(cptr->c_keys[cmdpos],0) );
				if (cmdpos<4) 
					if ((cptr->c_keys[cmdpos+1]) != 0)
						putc(' ',stdout);
				++cmdpos;
				}
			}

		for (cptr = cmdlist; cptr != NULL; cptr = cptr->next) {
			if (cmdmatch(cmdbuf,cptr,5)) {
				if (requires_string(cptr,cmdpos)) {
					getline(argbuf,32);
					}
				else {
					printf("\n");
					}
				return(cptr->c_cmdnum);
				}
			}

		if (ch == '?') {
			printf("\rOne of ...                         \n");
			for (cptr = cmdlist; cptr != NULL; cptr = cptr->next) {
			    if (cmdmatch(cmdbuf,cptr,cmdpos)) {
				for (a=0; a<5; ++a) {
				    printf("%s ",cmd_expand(cptr->c_keys[a],1));
				    }
				printf("\n");
				}
			    }

			printf("\n%s%c ",room_name,room_prompt(room_flags));
			got = 0;
			for (cptr = cmdlist; cptr != NULL; cptr = cptr->next) {
			    if ((got==0)&&(cmdmatch(cmdbuf,cptr,cmdpos))) {
				for (a=0; a<cmdpos; ++a) {
				    printf("%s ",
					cmd_expand(cptr->c_keys[a],0));
				    }
				got = 1;
				}
			    }
			}

		}

	}





/*
 * set tty modes.  commands are:
 * 
 * 0 - set to bbs mode, intr/quit disabled
 * 1 - set to bbs mode, intr/quit enabled
 * 2 - save current settings for later restoral
 * 3 - restore saved settings
 */
#ifdef POSIX_TERMIO
void sttybbs(cmd) 		/* SysV version of sttybbs() */
int cmd; {
	struct termios live;
	static struct termios saved_settings;

	if ( (cmd == 0) || (cmd == 1) ) {
		tcgetattr(0,&live);
		live.c_iflag=ISTRIP|IXON|IXANY;
		live.c_oflag=OPOST|ONLCR;
		live.c_lflag=NOFLSH;
		if (cmd==1) live.c_lflag=ISIG|NOFLSH;

		if (cmd==SB_YES_INTR) {
			live.c_cc[VINTR]=NEXT_KEY;
			live.c_cc[VQUIT]=STOP_KEY;
			signal(SIGINT,sighandler);
			signal(SIGQUIT,sighandler);
			}
		else {
			signal(SIGINT,SIG_IGN);
			signal(SIGQUIT,SIG_IGN);
			live.c_cc[VINTR]=(-1);
			live.c_cc[VQUIT]=(-1);
			}

		/* do we even need this stuff anymore? */
		/* live.c_line=0; */
		live.c_cc[VERASE]=8;
		live.c_cc[VKILL]=24;
		live.c_cc[VEOF]=1;
		live.c_cc[VEOL]=255;
		live.c_cc[VEOL2]=0;
		live.c_cc[VSTART]=0;
		tcsetattr(0,TCSADRAIN,&live);
		}
	if (cmd == 2) {
		tcgetattr(0,&saved_settings);
		}
	if (cmd == 3) {
		tcsetattr(0,TCSADRAIN,&saved_settings);
		}
	}
#else
void sttybbs(cmd) 		/* BSD version of sttybbs() */
int cmd; {
	struct sgttyb live;
	static struct sgttyb saved_settings;

	if ( (cmd == 0) || (cmd == 1) ) {
		gtty(0,&live);
		live.sg_flags |= CBREAK;
		live.sg_flags |= CRMOD;
		live.sg_flags |= NL1;
		live.sg_flags &= ~ECHO;
		if (cmd==1) live.sg_flags |= NOFLSH;
		stty(0,&live);
		}
	if (cmd == 2) {
		gtty(0,&saved_settings);
		}
	if (cmd == 3) {
		stty(0,&saved_settings);
		}
	}
#endif


/*
 * display_help()  -  help file viewer
 */
void display_help(name)
char name[]; {
	formout(name);
	}


/*
 * fmout()  -  Citadel text formatter and paginator
 */
int fmout(width,fp,pagin,height,starting_lp,subst)
int width;		/* screen width to use */
FILE *fp;		/* file to read from, or NULL to read from server */
char pagin;		/* nonzero if we should use the paginator */
int height;		/* screen height to use */
int starting_lp;	/* starting value for lines_printed, -1 for global */
char subst;		/* nonzero if we should use hypertext mode */
	{
	int a,b,c,d,old;
	int real = (-1);
	char aaa[140];
	char buffer[512];
	int eof_flag = 0;

	if (starting_lp >= 0) {	
		lines_printed = starting_lp;
		}
	strcpy(aaa,""); old=255;
	strcpy(buffer,"");
	c=1; /* c is the current pos */

	sigcaught = 0;
	sttybbs(1);

FMTA:	while ( (eof_flag==0) && (strlen(buffer)<126) ) {
		if (sigcaught) goto OOPS;
		if (fp!=NULL) { /* read from file */
			if (feof(fp)) eof_flag = 1;
			if (eof_flag==0) {
				a=getc(fp);
				buffer[strlen(buffer)+1] = 0;
				buffer[strlen(buffer)] = a;
				}
			}
		else {		/* read from server */
			d=strlen(buffer);
			serv_gets(&buffer[d]);
while ( (!isspace(buffer[d])) && (isspace(buffer[strlen(buffer)-1])) )
	buffer[strlen(buffer)-1]=0;
			if (!strcmp(&buffer[d],"000")) {
				buffer[d] = 0;
				eof_flag = 1;
				while(isspace(buffer[strlen(buffer)-1]))
					buffer[strlen(buffer)-1] = 0;
				}
			d=strlen(buffer);
			buffer[d] = 10;
			buffer[d+1] = 0;
			}
		}

	buffer[strlen(buffer)+1] = 0;
	a=buffer[0];
	strcpy(buffer,&buffer[1]);
	
	old=real;
	real=a;
	if (a<=0) goto FMTEND;
	
	if ( ((a==13)||(a==10)) && (old!=13) && (old!=10) ) a=32;
	if ( ((old==13)||(old==10)) && (isspace(real)) ) {
		printf("\n");
		++lines_printed;
		lines_printed = checkpagin(lines_printed,pagin,height);
		c=1;
		}
	if (a>126) goto FMTA;

	if (a>32) {
	if ( ((strlen(aaa)+c)>(width-5)) && (strlen(aaa)>(width-5)) ) {
		printf("\n%s",aaa); c=strlen(aaa); aaa[0]=0;
		++lines_printed;
		lines_printed = checkpagin(lines_printed,pagin,height);
		}
	    b=strlen(aaa); aaa[b]=a; aaa[b+1]=0;
	    }
	if (a==32) {
		if ((strlen(aaa)+c)>(width-5)) { 
			c=1;
			printf("\n");
			++lines_printed;
			lines_printed = checkpagin(lines_printed,pagin,height);
			}
		printf("%s ",aaa); ++c; c=c+strlen(aaa);
		strcpy(aaa,"");
		goto FMTA;
		}
	if ((a==13)||(a==10)) {
		printf("%s\n",aaa);
		c=1;
		++lines_printed;
		lines_printed = checkpagin(lines_printed,pagin,height);
		strcpy(aaa,"");
		goto FMTA;
		}
	goto FMTA;

	/* signal caught; drain the server */
OOPS:	do {
		serv_gets(aaa);
		} while(strcmp(aaa,"000"));

FMTEND:	printf("\n");
	++lines_printed;
	lines_printed = checkpagin(lines_printed,pagin,height);
	return(sigcaught);
}


/*
 * support ANSI color if defined
 */
void color(colornum)
int colornum;  {

#ifdef ANSI_COLOR
	fflush(stdout);
	printf("%c[3%dm", 27, colornum);
	fflush(stdout);
#endif
	}

void cls() {
#ifdef ANSI_COLOR
	fflush(stdout);
	printf("%c[2J%c[H", 27, 27);
	fflush(stdout);
#endif
	}
