// xfiles - synchronize and cross-validate directory trees across a network
// Copyright (C) 1999  j.p.lewis  
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Primary author contact info: www.idiom.com/~zilla  zilla@computer.org

import java.io.*;

import javax.swing.*;
import javax.swing.tree.*;

class Xfiletree extends DefaultMutableTreeNode
{
  static final boolean	gDebug = false;

  Xfiletree		parent;
  File			file;
  boolean		isDir;
  Xfiletree[] 		children;
  boolean		isLink;
  Object		userObject;

  //----------------------------------------------------------------

  static boolean	nativeLoaded = false;
  static boolean	haveNative = false;
  static nativeFileInterface nativeFile = null;

  /**
   * Load native code to distinguish files from links.
   * If nativeFile.class is not found then haveNative 
   * remains false and the Absolute/Canonical comparison
   * is used.
   */
  private static void loadNative()
  {
    nativeLoaded = true;
    try {
      Class c = Class.forName("nativeFile");
      nativeFileInterface nativeFile
	= (nativeFileInterface)c.newInstance();
      haveNative = true;
      System.out.println("using nativeFile");
    }
    catch (Exception ex) {
      System.out.println("using portable link detection");
    }
  } //loadNative

  private boolean computeIsLink()
  {
    try {
      if (haveNative && nativeFile != null) 
	return nativeFile.isLink(file.getPath());
      else 
	return XfilesCommon.isLink(file);
    }
    catch (Exception ex) {
      System.err.println(ex);
      ex.printStackTrace();
      return false;
    }
  } //computeIsLink

  //----------------------------------------------------------------

  // Want all tree modifications to occur in the gui thread.
  // Currently this is not true: userObject is set directly;
  // createChildren is also called by client thread.

  static class guiCB implements Runnable
  {
    Xfiletree parent, child;

    guiCB(Xfiletree parent, Xfiletree child)
    {
      //System.out.println("guiCB.new " + parent +" "+ child);
      this.parent = parent;
      this.child = child;
    }

    public void run()
    {
      //System.out.println("guiCB.run " + parent +" "+ child+"\n");

      // this tells the JTree to add us as a child.
      if (parent != null)
	parent.add(child);

      //XfilesGui.frame.repaint();
    }
  } //guiCB

  public Xfiletree(Xfiletree parent, String path,
		   boolean recurse, boolean notifygui)
    throws IOException
  {
    System.out.println("new Xfiletree: " + path +" "+ recurse);

    if (!nativeLoaded) loadNative();

    userObject = null;
    this.parent = parent;
    file = new File(path);
    XfilesCommon.assert(file != null);
    isLink = computeIsLink();
    isDir = (file.isDirectory() && !isLink);

    if (notifygui)
      {
	Runnable t = new guiCB(parent, this);
	SwingUtilities.invokeLater(t);
      }

    if (recurse)
      {
	if (isDir)
	  createChildren(recurse, notifygui);
      }
  } //Xfiletree


  public String toString()
  {
    if (file != null) {
      String name = file.getName();
      if (isLink) {
	if (isDir)
	  name += "/";
	name += "@";
      }
      return name;
    }
    return "---";
  } //toString

  // For a link that points to a file or directory,
  // File.isFile/isDirectory describe the pointed-to object.
  // getCannonicalPath evidently changes to the actual
  // path rather than the link path.
  // For a link that points to nothing, neither isFile/isDir
  // are true.
  // Thus need to return !isDirectory rather than isFile
  // so that a dangling link is assumed to point to
  // a non-existent file rather than to a non-existent directory.
  // (either is valid, but unix ls guesses a file)
  public boolean isLeaf()
  {
    return (children == null);
    //      if (file != null)
    //	return !file.isDirectory();
    //      return true;
  }

  public boolean getAllowsChildren()
  {
    return isDir;
  }


  Xfiletree[] createChildren(boolean recurse, boolean notifygui)
  {
    //if (children != null)  return children;

    try {
      String[] files = file.list();
      if (files != null) {
	children = new Xfiletree[files.length];
	String path = file.getPath();
	for( int i=0; i < files.length; i++ ) {
	  String fullpath = path + "/" + files[i];
	  children[i] = new Xfiletree(this, fullpath, recurse, notifygui);
	}
      }
    }
    catch (Exception ex) {
      System.err.println(ex);
      ex.printStackTrace();
    }
    return children;
  } //createChildren


  /**
   * Called by XfilesCommon.delete.
   * XfilesClient notifies the JTree.
   */
  public void deleteChild(Xfiletree child)
  {
    if (children != null) {
      int nold = children.length;
      if (nold == 0)
	children = null;
      else {
	int nnew = nold - 1;
	Xfiletree[] newchildren = new Xfiletree[nnew];
	int ni=0;
	for( int oi=0; oi < nold; oi++ ) {
	  if (children[oi] != child)
	    newchildren[ni++] = children[oi];
	}
	children = newchildren;
      }
    }
  } //deleteChild

  //----------------------------------------------------------------

  public interface traverseCB {
    public boolean cb(Xfiletree f) throws Exception;
  }

  public void traverse(traverseCB cb)
    throws Exception
  {
    if (isLink) return;
    //System.out.println("traverse[ " + file.getPath());
    if (cb.cb(this)) {
      //System.out.println("traverse> " + file.getPath());
      if (children != null) {
	for( int i=0; i < children.length; i++ ) {
	  children[i].traverse(cb);
	}
      }
    }
    //System.out.println("traverse] " + file.getPath());
  } //traverse
  
} //Xfiletree


