// xfiles - synchronize and cross-validate directory trees across a network
// Copyright (C) 1999  j.p.lewis  
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Primary author contact info: www.idiom.com/~zilla  zilla@computer.org


import java.util.*;
import java.io.*;
import java.rmi.*;
import java.rmi.server.*;
import java.rmi.registry.LocateRegistry;

import XfilesClient.*;

  class XfilesServer extends UnicastRemoteObject
    implements XfilesInterface
  {
    static final boolean gDebug = false;

    Xfiletree 			tree;
    Hashtable 			htab;

    BufferedInputStream 	gFin;
    BufferedOutputStream 	gFout;
    String			gFoutpath;

    //----------------------------------------------------------------

    XfilesServer()
      throws RemoteException
    {
      //htab = new Hashtable();
    }

    //----------------------------------------------------------------

    void buildHashtable()
    {
      htab = new Hashtable();
      try {
	tree.traverse(new hashCB());
      } catch(Exception ex) { System.err.println(ex); }
    } //buildHashtable

    //----------------------------------------------------------------

    class hashCB implements Xfiletree.traverseCB
    {
      public boolean cb(Xfiletree f)
      {
	htab.put(f.file.getPath(), f);
	return true;
      }
    }

    //----------------------------------------------------------------

    // this is now just a sanity check - make sure the server root exists
     public void setRootPath(File path)
       throws RemoteException, XFException
     {
       String spath = path.getPath();
       if (!path.exists())	// does not exist or is dangling link
	 throw new XFExServerRootIsBad(spath);
       if (XfilesCommon.isLink(path))	// is symbolic link
	 throw new XFExServerRootIsLink(spath);

       tree = null;
     } //setRoot

    //----------------------------------------------------------------
    
    public boolean fileExists(File path)
      throws RemoteException
    {
      System.err.println("exists? " + path.getPath());
      if (tree == null) {
	// called before readtree
	//System.err.println("exists -> " + path.exists());
	return path.exists();
      }
      else {
	// this branch is not used currently
	Xfiletree f = (Xfiletree)htab.get(path.getPath());
	System.err.println("exists -> " + (f != null));
	return f != null;
      }
    } //exists

    //----------------------------------------------------------------
    
    public void readTree(File path)
      throws RemoteException, XFException
    {
      try {
	String spath = path.getPath();
	System.out.println("server starting on " + spath);
 	//root = path; 	rootLen = path.length();
	tree = new Xfiletree(null, spath, true, false);
	buildHashtable();
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    } //readTree

    //----------------------------------------------------------------

    /** return a string of paths that end in this filename
     * Why a string: because we are only going to display the results.
     */
    public String lookup(String fileName)
      throws RemoteException, XFException
    {
      //fileName = "/" + fileName;
      System.out.println("remote.lookup " + fileName);
      return XfilesCommon.htabLookup(htab, fileName);
    } //lookup

    //----------------------------------------------------------------

/****************
    public void readSubTree(String path)
      throws RemoteException, XFException
    {
      try {
	System.out.println("server starting on subtree " + path);
	Xfiletree subtree = (Xfiletree)htab.get(path);
	if (subtree != null) {
	  Xfiletree parent = subtree.parent;
	  if (parent.replaceChild(path)) {
	    // rebuild the htab from scratch,
	    // otherwise it will contain the old entries as well as the new.
	    buildHashtable();
	  }
	  else
	    System.err.println("subtree " + path + " could not be replaced.");
	}
	else {
	  System.err.println("readSubTree could not lookup " + path);
	}
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    } //readSubTree
****************/

    //----------------------------------------------------------------

    public boolean isDir(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      Xfiletree f = (Xfiletree)htab.get(spath);
      if (f == null) throw new XFExnoremote(spath);
      XfilesCommon.assert(f.file != null);
      return (f.file.isDirectory() && !f.isLink);
    } //isDir

    //----------------------------------------------------------------

    public boolean isLink(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      Xfiletree f = (Xfiletree)htab.get(spath);
      if (f == null) throw new XFExnoremote(spath);
      return f.isLink;
    } //isLink

    //----------------------------------------------------------------

    public String[] readDir(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      System.out.println("server readdir " + spath);
      Xfiletree f = (Xfiletree)htab.get(spath);
      if (f == null) throw new XFExnoremote(spath);

      int nchildren = f.children.length;
      if (nchildren > 0) {
	int nchildren_nolink = 0;
	for( int i=0; i < nchildren; i++ ) {
	  if (!f.children[i].isLink) {
	    nchildren_nolink++;
	  }
	}
	String[] rval = new String[nchildren_nolink];
	int inolink = 0;
	for( int i=0; i < nchildren; i++ ) {
	  if (!f.children[i].isLink) {
	    rval[inolink] = f.children[i].file.getName();
	    inolink++;
	  }
	}
	return rval;
      }

      return null;
    } //readDir

    //----------------------------------------------------------------

    public XfilesClient.checksum checksumFile(File path)
      throws RemoteException, XFException
    {
      XfilesCommon.assert(htab != null);
      String spath = path.getPath();
      if (htab.get(spath) == null)
	throw new XFExnoremote(spath);
      try {
	XfilesClient.checksum cs = XfilesCommon.checksum(spath);
	return cs;
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    } //checksumFile

    //----------------------------------------------------------------
    
    public void readFile(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      System.out.println("server examining " + spath);
      XfilesCommon.assert(htab != null);
      if (htab.get(spath) == null)
	throw new XFExnoremote(spath);
      try {
	gFin = XfilesCommon.readFile(spath);
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    } //readFile

    //----------------------------------------------------------------

    public bytebuf readbuf(int wantlen)
      throws RemoteException, XFException
    {
      try {
	XfilesCommon.assert(gFin != null);
	bytebuf bbuf = new bytebuf(wantlen);
	bbuf.len = XfilesCommon.readbuf(gFin, bbuf.buf);
	return bbuf;
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    } //readbuf

    //----------------------------------------------------------------

    public long getLength(File path)
      throws RemoteException, XFException
    {
      // use hashtable - avoid creating new path,file objects
      String spath = path.getPath();
      Xfiletree f = (Xfiletree)htab.get(spath);
      if (f == null) throw new XFExnoremote(spath);
      XfilesCommon.assert(f.file != null);
      return f.file.length();
    }

    //----------------------------------------------------------------

    public long lastModified(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      XfilesCommon.assert(spath != null);
      XfilesCommon.assert(htab != null);

      Xfiletree f = (Xfiletree)htab.get(spath);
      if (f == null) throw new XFExnoremote(spath);
      XfilesCommon.assert(f.file != null);
      long ldate = f.file.lastModified();

      System.out.println("remote.lastModified " + spath +
			 " " + ldate + " " + new Date(ldate));
      return ldate;
    }

    //----------------------------------------------------------------

    public void mkDir(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      System.out.println("mkDir " + spath);

      try {
	XfilesCommon.mkDir(spath);
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }

    } //mkDir

    //----------------------------------------------------------------

    public void writeFile(File path)
      throws RemoteException, XFException
    {
      String fullpath = path.getPath();
      System.out.println("writeFile " + fullpath);

      XfilesScript.preCopyScript(fullpath);

      try {
	gFoutpath = fullpath;
	gFout = XfilesCommon.writeFile(fullpath);
      }
      catch(IOException ex) {
	System.err.println("writefile i/o error " + ex.getMessage());
	throw new XFException(ex.getMessage());
      }

    } //writeFile

    //----------------------------------------------------------------

    public void writebuf(byte[] buf, int count)
      throws RemoteException, XFException
    {
      try {
	gFout.write(buf, 0, count);
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }
    }

    //----------------------------------------------------------------

    public void writeClose()
      throws RemoteException, XFException
    {
      try {
	gFout.close();
      }
      catch(IOException ex) { throw new XFException(ex.getMessage()); }

      XfilesScript.postCopyScript(gFoutpath);
      gFoutpath = null;
    } //writeClose()

    //----------------------------------------------------------------

    public void delete(File path)
      throws RemoteException, XFException
    {
      String spath = path.getPath();
      System.out.println("delete " + spath);
      try {
	XfilesCommon.delete(spath, htab);
      }
      catch(Exception ex) { throw new XFException(ex.getMessage()); }
    } //delete

    //----------------------------------------------------------------
    //----------------------------------------------------------------

    public static void main(String[] args)
    {
      try {
	XfilesScript.scriptSetup();
        LocateRegistry.createRegistry(9753);
	XfilesServer server = new XfilesServer();
	String name = System.getProperty("XFILES", "XF");
	Naming.rebind(name, server);
	System.out.println("XfilesServer is running.");
      }
      catch (java.rmi.ConnectIOException e) {
	System.err.println("could not setup server socket - ");
	System.err.println("tcp networking probably needs to be turned on.");
	System.exit(1);
      }
      catch (Exception e) {
	System.err.println("ERROR");
	System.err.println(e);
	e.printStackTrace();
	System.err.println("Usage: java [-DXFILES=<name>] XfilesServer");
	System.exit(1); // Force an exit because there might be other threads
      }
      catch (Throwable t) {
	System.err.println("ERROR: uncaught exception: " + t);
	t.printStackTrace();
      }
    } //main

  } //XfilesServer
