// xfiles - synchronize and cross-validate directory trees across a network
// Copyright (C) 1999  j.p.lewis  
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Primary author contact info: www.idiom.com/~zilla  zilla@computer.org

import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;

import Xfiletree.*;

public class XfilesGui 
{
  static public int 		CMD;
  static final public int 	CMDnone		= 200;
  static final public int 	CMDcompare	= 201;
  static final public int 	CMDcopytoserver	= 202;
  static final public int 	CMDcopytoclient	= 203;
  static final public int 	CMDclientdelete	= 204;
  static final public int 	CMDserverdelete	= 205;
  static final public int 	CMDskip		= 206;
  static final public int 	CMDdone		= 207;
  static final public int 	CMDstop		= 208;

  final static int		XSIZE		= 800;
  final static int		YSIZE		= 600;

  static JFrame 		frame;
  static JTree			clientTree;
  static JTextArea		compareArea;
  static JTextArea		clientArea;
  static JTextArea		serverArea;
  static JTextArea		statusArea;
  static JButton		startButton;	

  static Xfiletree		root;

  //----------------------------------------------------------------
  // "condition" is something like the a file not existing
  // "error" is a program bug or something like a host going down

  public static void showAlert(String msg)
  {
    System.err.println(msg);
    JOptionPane.showMessageDialog(frame, msg, msg,
				  JOptionPane.ERROR_MESSAGE);
  }

  public static void showCondition(String msg)
  {
    System.err.println(msg);
    setStatus(msg);
  }

  public static void showError(Exception ex)
  {
    System.err.println(ex.getMessage());
    ex.printStackTrace();
  }

  //----------------------------------------------------------------

  public static void deleteTreeFile(final Xfiletree f)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	DefaultTreeModel model = (DefaultTreeModel)clientTree.getModel();
	model.removeNodeFromParent(f);
      }
    });
  } //deleteTreeFile

  public static void selectTreeFile(final Xfiletree f)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	DefaultTreeModel model = (DefaultTreeModel)clientTree.getModel();
	TreePath path = new TreePath(model.getPathToRoot(f));
	if (path != null) {
	  clientTree.scrollPathToVisible(path);
	  clientTree.setSelectionPath(path);
	  clientTree.expandPath(path);
	  clientTree.makeVisible(path);
	  frame.repaint();
	}
	else System.err.println("warning: selectTreeFile failed");
      }
    });
  } //selectTreeFile

  //----------------

  public static void clearCompareText()
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	compareArea.setText("");
      }
    });
  }

  public static void addCompareText(final String msg)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	compareArea.append(msg);
      }
    });
  }

  //----------------

  static final Color highlightColor = new Color(0.f, 0.f, 1.f);

  public static void setClientHighlight(boolean highlight)
  {
    clientArea.setForeground(highlight ? highlightColor : Color.black);
  }

  public static void setClientText(final String msg)
  {
    setClientText(msg, false);
  }

  public static void addClientText(final String msg)
  {
    setClientText(msg, true);
  }

  public static void setClientText(final String msg, final boolean append)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	if (append) {
	  clientArea.append(msg);
	}
	else
	  clientArea.setText(msg);
      }
    });
  }

  //----------------

  public static void setServerHighlight(boolean highlight)
  {
    serverArea.setForeground(highlight ? highlightColor : Color.black);
  }

  public static void setServerText(final String msg)
  {
    setServerText(msg, false);
  }

  public static void addServerText(final String msg)
  {
    setServerText(msg, true);
  }

  public static void setServerText(final String msg, final boolean append)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	if (append) {
	  serverArea.append(msg);
	}
	else
	  serverArea.setText(msg);
      }
    });
  }

  //----------------

  public static void setStatus(final String msg)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
	statusArea.setText(msg);
      }
    });
  }

  //----------------------------------------------------------------

  static void setupForEditing()
  {
    // TODO:
    // implement a treemodellisteer (treeNodesChanged, etc)
    // on nodesChanged call the underlying Xfiletree on the client,
    // also pop up a dialog asking whether to do the change on the server too.
    // I do not see how nodes are inserted/removed triggered by the
    // Jtree itself
    clientTree.setEditable(true);
    TreeModel model = clientTree.getModel();
    model.addTreeModelListener(new TreeModelListener () {
      public void treeNodesChanged(TreeModelEvent e) {
	System.out.println("nodes changed");
      }
      public void treeNodesInserted(TreeModelEvent e) {
	System.out.println("nodes inserted");
      }
      public void treeNodesRemoved(TreeModelEvent e) {
	System.out.println("nodes removed");
      }
      public void treeStructureChanged(TreeModelEvent e) {
	System.out.println("tree changed");
      }
    });
  } //setupForEditing

  //----------------------------------------------------------------

  static String GPLmessage =
    "\nXfiles version 1.3.1 23apr99, Copyright (C) 1999 j.p.lewis\n" + 
    "Xfiles comes with ABSOLUTELY NO WARRANTY.\n" +
    "This is free software, and you are welcome\n" +
    "to redistribute it under the GNU General Public Licence;\n" +
    "see www.gnu.org or the file LICENSE.txt for details.\n\n" +
    "JPython (Version 1.0.x) is made available subject to the\n" +
    "terms and conditions in CNRI's License Agreement.\n" +
    "See the file LICENSE.txt for details.\n";

  public static void main(String[] args)
  {
    System.err.println(GPLmessage);

    if (args.length != 2) {
      System.err.println("Usage:");
      System.err.println("java -DXFILES=rmi://hostname:9753/XF XfilesGui clientroot serverroot");
      System.exit(1);
    }

    XfilesScript.scriptSetup();

    final String localRootPath = XfilesCommon.stripTrailingSlash(args[0]);
    final String remoteRootPath = XfilesCommon.stripTrailingSlash(args[1]);

    frame = new JFrame("Xfiles");
    frame.addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e)
      {
	XfilesClient.quitme(0);
      }
    });

    Container cpane = frame.getContentPane();
    clientTree = new JTree();
    compareArea = new JTextArea();
    clientArea = new JTextArea();
    serverArea = new JTextArea();
    statusArea = new JTextArea();
    JPanel    buttonArea = new JPanel();
    /*JButton*/   startButton = new JButton("start");
    JButton   stopButton = new JButton("stop");
    JButton   compareButton = new JButton("diff");
    JButton   clientCopyButton = new JButton("<copy");
    JButton   serverCopyButton = new JButton("copy>");
    JButton   continueButton = new JButton("skip");
    JButton   clientDeleteButton = new JButton("<delete");
    JButton   serverDeleteButton = new JButton("delete>");

    Font      buttonFont = new Font("SanSerif", Font.BOLD, 12);
    startButton.setFont(buttonFont);
    stopButton.setFont(buttonFont);
    compareButton.setFont(buttonFont);
    clientCopyButton.setFont(buttonFont);
    serverCopyButton.setFont(buttonFont);
    continueButton.setFont(buttonFont);
    clientDeleteButton.setFont(buttonFont);
    serverDeleteButton.setFont(buttonFont);


    clientTree.setBorder(new TitledBorder("client tree"));	// TODO: plain
    compareArea.setBorder(new TitledBorder("diff"));	// TODO: plain
    clientArea.setBorder(new TitledBorder("client"));
    serverArea.setBorder(new TitledBorder("server"));
    statusArea.setBorder(new LineBorder(UIManager.getColor("control")));

    cpane.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();

    //c.insets = new Insets(2, 2, 2, 2); no
    c.gridx = 0;
    c.gridy = 0;
    c.gridwidth = 3;
    c.gridheight = 12;
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 0.3;
    c.weighty = 1.0;
    cpane.add(new JScrollPane(clientTree), c);

    c.gridx = 3;
    c.gridwidth = 7;
    c.weightx = 0.7;
    cpane.add(new JScrollPane(compareArea), c);

    c.gridx = 0;
    c.gridy = 12;
    c.gridwidth  = 5;
    c.gridheight = 3;
    //c.fill = GridBagConstraints.HORIZONTAL;
    c.weightx = 0.5;
    c.weighty = 0.2;
    //clientArea.setPreferredSize(new Dimension(XSIZE/2, 200));
    // when showing 'this missing file exists at these other locations'
    // we need a scrollpane if there are many such possible files
    // (e.g. a 'Makefile' is missing on one side)
    cpane.add(new JScrollPane(clientArea), c);

    c.gridx = 5;
    cpane.add(new JScrollPane(serverArea), c);

    c.gridx = 0;
    c.gridy = 15;
    c.gridwidth  = 10;
    c.gridheight = 1;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.HORIZONTAL;
    cpane.add(statusArea, c);

    c.gridwidth = 10;
    c.gridheight = 1;
    c.gridy = 16;
    //c.fill = GridBagConstraints.NONE;
    c.weightx = 0.0;
    c.weighty = 0.0;
    buttonArea.add(startButton);
    buttonArea.add(clientDeleteButton);
    buttonArea.add(clientCopyButton);
    buttonArea.add(continueButton);
    buttonArea.add(serverCopyButton);
    buttonArea.add(serverDeleteButton);
    buttonArea.add(compareButton);
    buttonArea.add(stopButton);
    cpane.add(buttonArea, c);

    startButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	Thread t = new Thread() {
	  public void run() {
	    startButton.setEnabled(false);
	    CMD = CMDnone;
	    XfilesClient.startClient(localRootPath, remoteRootPath);
	    startButton.setEnabled(true);
	  }
	};
	t.start();
      }
    });

    stopButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	CMD = CMDstop;
      }
    });

    compareButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	CMD = CMDcompare;
      }
    });

    clientCopyButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	CMD = CMDcopytoclient;
      }
    });

    serverCopyButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	CMD = CMDcopytoserver;
      }
    });

    continueButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	CMD = CMDskip;
      }
    });

    clientDeleteButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	System.err.println("clientDelete");
	CMD = CMDclientdelete;
      }
    });

    serverDeleteButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	System.err.println("serverDelete");
	CMD = CMDserverdelete;
      }
    });


    clientArea.setText("client");
    serverArea.setText("server");
    statusArea.setText("");

    // check that the client path is valid.
    // server path will be checked after the remote server is contacted.
    {
      File localRootfile = new File(localRootPath);
      if (!localRootfile.exists()) {
	System.err.println("client " + localRootPath + " does not exist.");
	System.exit(1);
      }
      if (XfilesCommon.isLink(localRootfile)) {
	System.err.println("The client side path has a link in it.");
	System.err.println("Use the path that the link refers to instead.");
	System.exit(1);
      }
    }

    try {
      // formerly had
      // root = new Xfiletree(null, localRootPath, false, false);
      // setModel(new DefaultTreeModel(root)
      // root.createChildren(true,true).
      // Probably faster to create the whole tree, then pass it to jtree:
      //root = new Xfiletree(null, localRootPath, true, false);

      root = new Xfiletree(null, localRootPath, false, false);

    } catch(IOException ex) { System.err.println(ex); }
    clientTree.setModel(new DefaultTreeModel(root));

    //TreeSelectionModel sm = clientTree.getSelectionModel();
    //sm.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

    //setupForEditing();

    frame.pack();
    frame.setSize(XSIZE, YSIZE);
    frame.setVisible(true);

    setStatus("graphing file tree");
    // NOTE all changes to the JTree are done in this (the gui) thread
    root.createChildren(true, true);

    //clientTree.expandRow(0);
    selectTreeFile(XfilesGui.root);
    frame.repaint();
    setStatus("");
	
  } //main
}
