// xfiles - synchronize and cross-validate directory trees across a network
// Copyright (C) 1999  j.p.lewis  
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Primary author contact info: www.idiom.com/~zilla  zilla@computer.org

import java.io.*;
import java.util.*;
import XfilesClient.*;

final class XfilesCommon
{
  static boolean gDebug	= false;

  public static BufferedInputStream readFile(String path)
    throws IOException
  {
    return
      new BufferedInputStream
        (new FileInputStream(path));
  }

  //----------------

  public static int readbuf(BufferedInputStream fin, byte[] buf)
    throws IOException
  {
    int len = fin.read(buf, 0, buf.length);
    if (len < 0) fin.close();
    return len;
  }

  //----------------

  // avoid allocation and requent gc. make this local when better gc is common.
  private static byte[] checksumBuf = new byte[8192];

  public final static XfilesClient.checksum checksum(String path)
    throws IOException
  {
    XfilesClient.checksum cs = new XfilesClient.checksum();
 
    BufferedInputStream fin =
      new BufferedInputStream(new FileInputStream(path));
    int len = fin.read(checksumBuf, 0, checksumBuf.length);
    while( len > 0) {

      final int max = Integer.MAX_VALUE;
      for( int i=0; i < len; i++ ) {
	byte c = checksumBuf[i];
	int ci = (int)c;
	if (ci < 0) ci += 128;
	cs.sum = (cs.sum + ci) % max;
	cs.csum[ci] = (cs.csum[ci] + 1) % max;
      }

      if (gDebug) {
	System.err.print("checksum read... " + cs.sum + " :   ");
	for( int i=64; i < 64+5; i++ )
	  System.err.print(cs.csum[i] + " ");
	System.err.println("");
      }

      len = fin.read(checksumBuf, 0, checksumBuf.length);
    }
    fin.close();
    cs.isValid = true;

    return cs;
  } //checksum

  //----------------

  public static void mkDir(String path)
    throws IOException
  {
    File file = new File(path);
    file.mkdirs();
  }

  //----------------

  public static BufferedOutputStream writeFile(String path)
    throws IOException
  {
    return
      new BufferedOutputStream
        (new FileOutputStream(path));
  }

  //----------------

  public static void delete1(String path)
    throws IOException
  {
    System.out.println("delete1 " + path);
    File file = new File(path);
    file.delete();
  }

  //----------------
  
  static void deleteDir(Xfiletree f, Hashtable htab)
    throws IOException
  {
    System.out.println("deleteDir " + f.file.getPath());
    int nchildren = f.children.length;
    for( int i=0; i < nchildren; i++ ) {
      Xfiletree c = f.children[i];
      if (c.isDir) 
	deleteDir(c, htab);
      else
	XfilesCommon.delete1(c.file.getPath());
      htab.remove(c);
    }
    f.children = null;
    f.file.delete();
  } //deleteDir

  //----------------

  public static void delete(String path, Hashtable htab)
    throws IOException, XFException
  {
    Xfiletree f = (Xfiletree)htab.get(path);
    if (f == null) throw new XFExnosuchfile(path);

    if (f.isDir) {
      deleteDir(f, htab);
      // remove f from f.parents children!
      Xfiletree parent = f.parent;
      // XfilesClient notifies the JTree.
      if (parent != null)  parent.deleteChild(f);
      htab.remove(path);
    }
    else {
      XfilesCommon.delete1(path);
      htab.remove(path);
    }
  } //delete

  //----------------

  /**
   * Java link detection:
   * <p>
   * For a link that actually points to something (either a file or
   * a directory), the absolute path is the path through the link,
   * whereas the canonical path is the path the link references.
   * <p>
   * Dangling links appear as files of size zero - 
   * no way to distinguish dangling links from non-existent files
   * other than by consulting the parent directory.
   */
  public static boolean isLink(File file)
  {
    try {
      if (!file.exists())
	return true;
      else

      {
	String cnnpath = file.getCanonicalPath();
	String abspath = file.getAbsolutePath();
	//System.out.println(abspath+" <-> "+cnnpath);
	return !abspath.equals(cnnpath);
      }
    }
    catch (java.io.FileNotFoundException ex) {
      return true;	      // a dangling link.
    }
    catch (IOException ex) { /*ignore other errors*/ }

    return false;
  } //isLink

  //----------------

  public static void assert(boolean b)
  {
    if (!b) {
      Error e = new Error("assert failed");
      // e.printStackTrace(); unnecessary
      throw e;
    }
  }

  //----------------

  public static String stripRoot(String root, int rootLen, String path)
  {
    if (path.startsWith(root)) {
      if (false) {
	System.out.println("startswith");
	System.out.println(root);
	System.out.println(path);
      }
      String stripped = path.substring(rootLen);
      return stripped;
    }
    throw new Error("striproot failed!");
  }

  //----------------

  static String stripExt(String filename)
  {
    int dot = filename.indexOf('.');
    if (dot > 0) filename = filename.substring(0, dot);
    return filename;
  }

  //----------------

  public static String stripTrailingSlash(String path)
  {
    //System.out.print("STRIP " + path + " -> ");
    if (path.endsWith(File.separator))
      path = path.substring(0, path.length() - 1);
    //System.out.println(path);
    return path;
  }

  //----------------

  /**
   * Return true if e.g. path is /xx/yy/foo.txt.gz and filename2 is foo.txt.
   * Proposal, strip off the name part of the path, then see if either
   * filename is a substring of the other.  This does not work
   * because a file 'c' will match any other file with a 'c' in it.
   * Alternate: strip off all extensions, then match.
   */
  static boolean nameIsSimilar(String path, String filename2)
  {
    //System.out.println(":isSimilar "+ path +" "+ filename2);
    int lastslash = path.lastIndexOf("/");
    if (lastslash == -1) return false;
    try {
      String filename1 = path.substring(lastslash+1);
      int dot = filename1.indexOf('.');
      if (dot > 0) filename1 = filename1.substring(0, dot);

      boolean rval = filename1.equals(filename2);

      /***
      boolean rval =  ((filename1.indexOf(filename2) != -1) ||
		       (filename2.indexOf(filename1) != -1));
      ***/
      //System.out.println(">isSimilar "+ filename1 +" "+ filename2 +" "+rval);
      return rval;
    }
    catch(Exception ex) { return false; }
  } //nameIsSimilar

  //----------------

  static String htabLookup(Hashtable htab, String fileName)
  {
    fileName = stripExt(fileName);
    StringBuffer ss = null;
    Enumeration e = htab.keys();
    while( e.hasMoreElements() ) {
      String s = (String)e.nextElement();
      //System.out.println("lookup checking " + s);
      if (XfilesCommon.nameIsSimilar(s, fileName)) {
	if (ss == null) ss = new StringBuffer();
	ss.append(s);
	Xfiletree f = (Xfiletree)htab.get(s);
	if (f != null) {
	  if (f.isDir)
	    ss.append(" (directory)");
	  else 
	    ss.append(" (file)");
	}
	ss.append("\n");
      }
    }
    if (ss != null)
      return ss.toString();
    else
      return null;
  } //htabLookup
} //XfilesCommon
