// xfiles - synchronize and cross-validate directory trees across a network
// Copyright (C) 1999  j.p.lewis  
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Primary author contact info: www.idiom.com/~zilla  zilla@computer.org

//----------------------------------------------------------------

// Program comments:
// 
// Notation -
// localRoot,remoteRoot are the parts above the common tree.
// lpath, rpath are the local, remote versions of the path.
// Cpath is the part below the root, in common to both client and server.
// For example, if the client tree starts at /usr/bozo/src
// and the server tree starts /u2/programmers/bozo/src,
// the cpath "includes/xx.h"
// lpath =  /usr/bozo/src/includes/xx.h
// rpath =  /u2/programmers/bozo/src/includes/xx.h
// cpath =  includes/xx.h
// 
// Hashtables -
// A client hashtable (indexed by lfile, returning a Xfiletree) 
// is used to quickly lookup whether a server file exists on the client
// (in needsAttention(), htab.get(dfile == cfile + / + remotefile)).
// When it does not, the htab is also used to find similarly named files
// that may be in other directories, using XfilesCommon.htabLookup.
// The client htab is also used in several places to retrieve the
// Xfiletree node given the file path- not a necessary use.
//   A server hashtable (indexed by rfile, returning a Xfiletree)
// is used to find any files on the server that are named similarly
// to a client file (for the case where the remote file does not exist,
// but we want to check whether it has been moved to a different directory).
// The server htab is also used internally by the server to find
// the Xfiletree node corresponding to the file.
//   Using filepaths as keys seems wasteful, but for example 15000 files
// times an average path length of 60 is not even a megabyte.  A cfile
// would be the minimum unique key; the extra string garbage creation 
// needed to use these as keys is considered not worth it.
// 
//   XfilesCommon.htabLookup(htab, filename) iterates over the htab keys
// and returns a concatenated string of the filenames that are similar.
//   XfilesCommon.delete(path, root, htab) removes the entry from htab
// as well as deleting the file.
//
// The client and server may not be running on the same OS.
// The rmi serialization is supposed to translate path separators,
// so we transmit the full path to the remote host.
// The server side then does a getPath() and receives
// the correct native path hopefully.
// (If we were to transmit the cpath and let the server
// concatenate, it would have to split the cpath and do
// the path separator translation ourself).
//
// The client file tree is traversed twice:
// once to build up a list of (C)files that need attention
// (these are attached to the userObject of the client Xfiletree),
// a second time visiting those nodes with userObjects
// and asking the user to take action.
//


//----------------------------------------------------------------

import java.io.*;
import java.util.*;
import java.rmi.*;
import java.rmi.server.*;
import javax.swing.tree.*;
import java.rmi.registry.LocateRegistry;


public class XfilesClient {
  // client and interface here

  static Hashtable htab;

  static boolean showAll = false;	// show only changed files/all files
  final static String	TMPFILE		= "/usr/tmp/XfilesTMP";
  final static boolean	gVerbose 	= true;
  final static boolean	gDebug 		= false;

  //----------------------------------------------------------------

  public interface XfilesInterface extends Remote
  {
    public void setRootPath(File path)
      throws RemoteException, XFException;

    public boolean fileExists(File path)
      throws RemoteException;

    public void readTree(File path)
      throws RemoteException, XFException;

    public String lookup(String fileName)
      throws RemoteException, XFException;

    public boolean isDir(File path)
      throws RemoteException, XFException;

    public boolean isLink(File path)
      throws RemoteException, XFException;

    public String[] readDir(File path)
      throws RemoteException, XFException;

    public checksum checksumFile(File path)
      throws RemoteException, XFException;

    public void readFile(File path)
      throws RemoteException, XFException;

    public bytebuf readbuf(int wantlen)
      throws RemoteException, XFException;

    public long getLength(File path)
      throws RemoteException, XFException;

    public long lastModified(File path)
      throws RemoteException, XFException;

    public void mkDir(File path)
      throws RemoteException, XFException;

    public void writeFile(File path)
      throws RemoteException, XFException;

    public void writebuf(byte[] buf, int count)
      throws RemoteException, XFException;

    public void writeClose()
      throws RemoteException, XFException;

    public void delete(File path)
      throws RemoteException, XFException;
  }

  static XfilesInterface remote;

  static class bytebuf implements java.io.Serializable
  {
    byte[] buf;
    long   len;
    bytebuf(int len)
    {
      this.len = len;
      this.buf = new byte[len];
    }
  }

  public static class checksum implements java.io.Serializable
  {
    int		sum;
    int[]	csum;
    boolean 	isValid;
    String	errorMsg;

    checksum() {
      csum = new int[256];
      clear();
    }

    void clear() {
      isValid = false;
      sum = 0;
      for( int i=0; i < 256; i++ ) csum[i] = 0;
      errorMsg = null;
    }
  } //class checksum
  
  //----------------------------------------------------------------

  public static class XFException extends Exception {
    public XFException(String msg) { super(msg); }
  }

  public static class XFExnoremote extends XFException {
    public XFExnoremote(String msg) { super(msg); }
  }

  public static class XFExServerRootIsBad extends XFException {
    public XFExServerRootIsBad(String msg) { super(msg); }
  }

  public static class XFExServerRootIsLink extends XFException {
    public XFExServerRootIsLink(String msg) { super(msg); }
  }

  // thrown by common code that does not know if it is called
  // by client or server.
  public static class XFExnosuchfile extends XFException {
    public XFExnosuchfile(String msg) { super(msg); }
  }

  public static class XFEstop extends XFException {
    public XFEstop(String msg) { super(msg); }
  }

/****************
  static class XFExdifferent extends XFException {
    public XFExdifferent(String msg) { super(msg); }
  }
****************/

  //----------------------------------------------------------------

  static final class Cfile
  {
    //File	file;
    String	cpath;
    String	lpath;
    String	rpath;
    boolean	lexists;
    boolean	rexists;
    boolean	lisdir;
    boolean	risdir;
    boolean	lislink;
    boolean	rislink;
    long	ldate;
    long	rdate;
    long 	llength;
    long	rlength;

    String	message;
    int		condition;
    // conditions:
    final static int	ERROR		= 900;	// i/o error (protection?)
    final static int 	DIFFERENT	= 901;	
    final static int 	NOLOCAL		= 902;
    final static int 	NOREMOTE	= 903;
    final static int 	LINK		= 904;
    final static int 	RNOTDIR		= 905;
    final static int 	LNOTDIR		= 906;

    // pass in a File rather than an Xfiletree:
    // the rest of Xfiletree is not needed,
    // but we need to have a File regardless when doing file.mkdirs()
    // to create a new local directory.
    Cfile(File file, String cpath)
    {
      if (cpath == null) {
	System.err.println("Cpath null!");
	cpath = "";
      }

      //this.file = file;
      this.cpath = cpath;
      lpath = localRoot  + cpath;
      rpath = remoteRoot + cpath;

      // Cfile is sometimes created to represent a remote file
      // with no local equivalent, thus file may be null
      if (file != null) {
	lexists = true;
	ldate = file.lastModified();
	llength = file.length();
	lisdir = file.isDirectory();
      }
      else {
	lexists = false;
	ldate = llength = -1;
      }
      // traverse ignores links so we are not a link
      lislink = false;

      rdate = -1;
      rlength = -1;
      rexists = false;
      try {
	rdate = remote.lastModified(remotePath(cpath));
	rlength = remote.getLength(remotePath(cpath));
	risdir = remote.isDir(remotePath(cpath));
	rislink = remote.isLink(remotePath(cpath));
	rexists = true;
      }
      // ignore exceptions in constructor
      catch(XFException ex) 	{ }
      catch(RemoteException ex) {
	System.err.println("cfile constructor");
	if (lpath != null) System.err.println("lpath"+lpath);
	if (cpath != null) System.err.println("cpath"+cpath);
	XfilesGui.showError(ex);
      }
      catch(Exception ex) { System.err.println(ex); ex.printStackTrace(); }

      message = null;
      condition = 0;
    } //constructor

    public String toString()
    {
      return cpath;
    }
  } //Cfile

  //----------------------------------------------------------------

  final static File remotePath(String cpath)
  {
    return new File(remoteRoot + cpath);
  }

  final static File remotePath(Cfile cfile)
  {
    return new File(cfile.rpath);
  }

  //----------------------------------------------------------------

  final static class hashCB implements Xfiletree.traverseCB
  {
    final public boolean cb(Xfiletree f)
    {
      String lpath = f.file.getPath();
      String cpath = XfilesCommon.stripRoot(localRoot, localRootLen, lpath);
      htab.put(lpath, f);
      f.userObject = null;	// clear out previous userobject as well
      return true;
    }
  } //hashCB

  //----------------------------------------------------------------

  final static boolean comparebuf(String cpath,
				  long rlen, byte[] rbuf,
				  long llen, byte[] lbuf)
  {
    //System.err.println("read " + rlen +" "+ llen);
    if (rlen >= 0 && (rlen != llen)) {
      //System.err.println("dsdifferent1");
      return false;
    }

    // the actual comparison.  some faster way to do this?
    for( int i=0; i < rlen; i++ ) {
      if (rbuf[i] != lbuf[i]) {
	//System.err.println("dsdifferent2 " + lbuf[i] +" "+ rbuf[i]);
	return false;
      }
    }
    return true;
  } //comparebuf

  //----------------

  static boolean compareFull(Cfile cfile)
    throws RemoteException, IOException, Exception
  {
    if (gVerbose) System.out.println("compare " + cfile.lpath);
    remote.readFile(remotePath(cfile));
    BufferedInputStream lfin = XfilesCommon.readFile(cfile.lpath);
    XfilesCommon.assert(lfin != null);
    final int bufsiz = 4096;
    byte[] lbuf = new byte[bufsiz];
    long llen=0, rtotallen=0;
    do {
      bytebuf bbuf = remote.readbuf(bufsiz);
      llen = XfilesCommon.readbuf(lfin, lbuf);
      if (!comparebuf(cfile.lpath, bbuf.len, bbuf.buf, llen, lbuf))
	return false;
      if (bbuf.len > 0) rtotallen += bbuf.len;
    } while( llen >= 0 );

    if (rtotallen != cfile.rlength) {
      cfile.message = "remote bytes read do not match size "
	+ cfile.cpath + " " + cfile.rlength + " " + rtotallen;
      System.err.println(cfile.message);
      return false;
    }

    return true;
  } //compareFull

  //----------------------------------------------------------------

  // has to be static for the thread to access it
  static checksum csLocal = new checksum();

  static boolean compareChecksum(final Cfile cfile)
    throws RemoteException, IOException, Exception
  {
    if (gVerbose) System.out.println("compareChecksum " + cfile.lpath);

    checksum csRemote;
    csLocal.clear();	// unnecessary?
    boolean singlethread = false;

    if (singlethread) {
      csRemote = remote.checksumFile(remotePath(cfile));
      csLocal = XfilesCommon.checksum(cfile.lpath);
    }
    else {
      Thread t = new Thread() {
	public void run() {
	  try {
	    csLocal = XfilesCommon.checksum(cfile.lpath);
	  }
	  catch(Exception ex) {
	    System.err.println("checksum threadexception: " + ex.getMessage());
	    csLocal.errorMsg = ex.getMessage();
	    csLocal.isValid = false;
	  }
	}
      };

      t.start();
      Thread.yield();
      Thread.currentThread().yield();	// which is correct?
      csRemote = remote.checksumFile(remotePath(cfile));
      t.join();	// wait for thread to die
    }

    // Thread questions:
    // Thread.yield() or Thread.currentThread().yield() ?
    // t.wait() works - must be a method in Object()

    // this should be a method in the checksum class.
    // putting it here because not sure if a passing an
    // instance over RMI passes just the data, or the methods too-
    // want to keep the RMI traffic small.
    //
    boolean same = true;
    if (!csRemote.isValid || !csLocal.isValid) same = false;
    if (csLocal.sum != csRemote.sum) same = false;
    for( int i=0; i < 256; i++ ) {
      if (csLocal.csum[i] != csRemote.csum[i]) {
	same = false;
	break;
      }
    }

    if (gDebug)
      System.out.println("checksum returning "
			 + csLocal.isValid + " " + csRemote.isValid + " ->  "
			 + same);
    return same;
  } //compareChecksum

  //----------------------------------------------------------------

    static void dir2Client(Cfile cfile)
    {
      XfilesCommon.assert(cfile.risdir);
      File local = new File(cfile.lpath);
      local.mkdirs();
      try {
	String[] remotefiles = remote.readDir(remotePath(cfile));
	int nremote = remotefiles.length;
	for( int i=0; i < nremote; i++ ) {
	  String dfile = cfile.cpath + File.separator + remotefiles[i];
	  Cfile cdfile = new Cfile(null, dfile);
	  if (cdfile.risdir)
	    dir2Client(cdfile);
	  else {
	    if (!cdfile.rislink)
	      file2Client(cdfile);
	  }
	}
      }
      catch(Exception ex) { XfilesGui.showCondition(ex.getMessage()); }
    } //dir2Client

    //----------------

    private static void _file2client(String cpath, String lpath)
    {
      try {
	XfilesScript.preCopyScript(lpath);
	remote.readFile(remotePath(cpath));
	System.out.println("writing client file " + lpath);
	BufferedOutputStream out = XfilesCommon.writeFile(lpath);
	final int bufsiz = 4096;
	bytebuf bbuf;
	do {
	  bbuf = remote.readbuf(bufsiz);
	  if (bbuf.len > 0)
	    out.write(bbuf.buf, 0, (int)bbuf.len);
	} while(bbuf.len > -1);
	out.close();
	XfilesScript.postCopyScript(lpath);
      }
      catch(Exception ex) {
	XfilesGui.showAlert(EwriteFailed("client", ex.getMessage()));
      }
    } //file2client


    static void file2Client(Cfile cfile)
    {
      System.out.println("copy to client " + cfile);
      if (cfile.risdir) {
	dir2Client(cfile);
      }
      else {
	_file2client(cfile.cpath, cfile.lpath);
      }
    } //file2Client

    //----------------------------------------------------------------

    final static class copier implements Xfiletree.traverseCB
    {
      public boolean cb(Xfiletree f)
      {
	String lpath = f.file.getPath();
	String cpath = XfilesCommon.stripRoot(localRoot, localRootLen, lpath);
	Cfile cfile = new Cfile(f.file, cpath);
	logEntry("recursively copying " + cpath +
			   " to server " + cfile.rpath, true);
	if (f.isDir) {
	  try {
	    remote.mkDir(remotePath(cfile));
	  }
	  catch(Exception ex) { XfilesGui.showError(ex); return false; }
	}
	else
	  file2Server(cfile);
	return true;
      }
    }

    static void file2Server(Cfile cfile)
    {
      System.out.println("copy to server " + cfile);
      try {
	if (cfile.lisdir) {
	  // All we need to do is make the remote directory,
	  // the individual files will be copied over in our traverse.
	  // Note this code would not be entered if there was a
	  // pre-existing directory.
	  if (gVerbose) System.out.println("remote.mkdir " + cfile);
	  //remote.mkDir(cfile.cpath);

	  Xfiletree f = (Xfiletree)htab.get(cfile.lpath);
	  if (f != null) {
	    f.traverse(new copier());
	  }
	}
	else {
	  remote.writeFile(remotePath(cfile));
	  System.out.println("writing server file " + cfile.rpath);
	  BufferedInputStream in = XfilesCommon.readFile(cfile.lpath);
	  final int bufsiz = 4096;
	  byte[] buf = new byte[bufsiz];
	  int count;
	  while( (count=XfilesCommon.readbuf(in, buf)) > -1 ) {
	    remote.writebuf(buf, count);
	  }
	  remote.writeClose();
	}
      }
      catch(IOException ex) { throw new Error(ex.toString()); }
      catch(Exception ex) {
	XfilesGui.showAlert(EwriteFailed("server", ex.getMessage()));
      }
    } //file2Server

    //----------------------------------------------------------------

    static void delete(Cfile cfile)
    {
      try {
	XfilesCommon.delete(cfile.lpath, htab);
	Xfiletree f = (Xfiletree)htab.get(cfile.lpath);
	if (f != null) XfilesGui.deleteTreeFile(f);
      }
      catch(XFException ex) { XfilesGui.showAlert(ex.getMessage()); }
      catch(IOException ex) { XfilesGui.showAlert(ex.getMessage()); }
    } //delete

    //----------------------------------------------------------------

    static void showCompare(Cfile cfile)
    {
      if (gVerbose) System.out.println("compare " + cfile);
      if (cfile.lisdir || cfile.risdir) {
	//TODO: optionpane
	XfilesGui.setStatus("cannot display directory comparison!");
      }
      else {
	_file2client(cfile.cpath, TMPFILE);
	String cmd = "diff " + cfile.lpath + " " + TMPFILE;
	Process p = null;
	int status = 0;
	try {
	  p = Runtime.getRuntime().exec(cmd);
	  BufferedInputStream fin =
	    new BufferedInputStream(p.getInputStream());
	  byte[] buf = new byte[4096];
	  int n;
	  XfilesGui.clearCompareText();
	  while( (n = fin.read(buf, 0, buf.length)) > 0 ) {
	    String s = new String(buf, 0, n);
	    XfilesGui.addCompareText(s);
	  }
	  fin.close();
	  status = p.waitFor();
	}
	catch(Exception ex) {
	  System.err.println(ex.getMessage());
	  ex.printStackTrace();
	  XfilesGui.showAlert(ex.getMessage());
	}
      } //else
    } //showCompare

    //----------------------------------------------------------------

    private static void showLocalInfo(Cfile cfile)
    {
      showLocalInfo(cfile, false);
    }

    private static void showLocalInfo(Cfile cfile, boolean highlight)
    {
      XfilesGui.setClientHighlight(highlight);
      XfilesGui.setClientText(cfile.cpath);
      if (cfile.lexists) {
	if (cfile.lisdir)
	  XfilesGui.addClientText("\n" + "directory");
	else {
	  if (cfile.llength > -1)
	    XfilesGui.addClientText("\n" + cfile.llength + " bytes");
	}
	if (cfile.ldate > -1)
	  XfilesGui.addClientText("\n" + new Date(cfile.ldate).toString());
      }
      else
	XfilesGui.addClientText("\n" + "does not exist.");
    } //showLocalInfo


    private static void showRemoteInfo(Cfile cfile)
    {
      showRemoteInfo(cfile, false);
    }

    private static void showRemoteInfo(Cfile cfile, boolean highlight)
    {
      XfilesGui.setServerHighlight(highlight);
      XfilesGui.setServerText(cfile.cpath);
      if (cfile.rexists) {
	if (cfile.risdir) 
	  XfilesGui.addServerText("\n" + "directory");
	else {
	  if (cfile.rlength > -1)
	    XfilesGui.addServerText("\n" + cfile.rlength + " bytes");
	}
	if (cfile.rdate > -1)
	  XfilesGui.addServerText("\n" + new Date(cfile.rdate).toString());
      }
      else 
	XfilesGui.addServerText("\n" + "does not exist.");
    } //showRemoteInfo

    //----------------

    static void waitForPress()
    {
      XfilesGui.CMD = XfilesGui.CMDnone;

      while( XfilesGui.CMD == XfilesGui.CMDnone ) {
	//System.out.println(".");
	Thread tc = Thread.currentThread();
	tc.yield();
	try { tc.sleep(50); }
	catch(InterruptedException ex) { XfilesGui.showError(ex);}
      }
    } //waitForPress

  //----------------------------------------------------------------

  /**
   * (As documentation, when the a remote file does not exist on the client)
   * Look in the client hashtable to find any directories where
   * files with this same name exist.
   */
  static String localLookup(Cfile cfile)
  {
    XfilesCommon.assert(cfile.lpath != null);
    File f = new File(cfile.lpath);
    //String fileName = "/" + f.getName();
    String fileName = f.getName();
    System.out.println("local.lookup " + fileName);
    return XfilesCommon.htabLookup(htab, fileName);
  } //localLookup

  //----------------------------------------------------------------

  static PrintWriter logFile;

  static void logEntry(String msg, boolean newline)
  {
    if (logFile == null) {
      try {
	logFile = new PrintWriter(new FileWriter("XFILES.LOG"),true);
      }
      catch(IOException e) { System.err.println(e); }
    }
    if (newline) {
      System.out.println(msg);
      logFile.println(msg);
    }
    else {
      System.out.print(msg);	System.out.print("\t");
      logFile.print(msg);	logFile.print("\t");
    }
  } //logEntry

  //----------------------------------------------------------------

  static boolean handle(Cfile cfile)
    throws XFEstop
  {
    {
      Xfiletree f = (Xfiletree)htab.get(cfile.lpath);
      if (f != null) {
	XfilesGui.selectTreeFile(f);
	Thread.yield();
      }
    }

    boolean recurse = true;
    boolean forceSkip = false;

    logEntry(cfile.cpath, false);

    switch(cfile.condition) {

    case Cfile.ERROR:
    case Cfile.LINK:
    case Cfile.LNOTDIR:
    case Cfile.RNOTDIR:
      // one or both files are links, 
      // or one is file, other is directory,
      // cannot handle this
      XfilesGui.setStatus(cfile.message);
      logEntry(cfile.message, false);
      recurse = false;
      forceSkip = true;
      break;


    case Cfile.NOLOCAL:
      XfilesGui.setStatus(cfile.message);
      showRemoteInfo(cfile);
      logEntry(cfile.message, false);
      {
	String s = localLookup(cfile);
	if (s != null) {
	  XfilesGui.setClientHighlight(false);
	  XfilesGui.setClientText("FILES WITH SIMILAR NAMES EXIST AT:\n" + s);
	}
      }
      break;

    case Cfile.NOREMOTE:
      XfilesGui.setStatus(cfile.message);
      showLocalInfo(cfile);
      logEntry(cfile.message, false);
      {
	// Test to see if a directory has been renamed.
	// Better would be to look inside the directory for a sample file
	// and find remote directories that contain that file.
	//
	File f = new File(cfile.lpath);
	String name = f.getName();
	try {
	  System.out.println("noremote name: " + name);
	  String s = remote.lookup(name);
	  if (s != null) {
	    XfilesGui.setServerHighlight(false);
	    XfilesGui.setServerText("FILES WITH SIMILAR NAMES EXIST AT:\n"+ s);
	  }
	}
	catch (Exception ex) { XfilesGui.showError(ex); }
      }
      break;

    case Cfile.DIFFERENT:
      if (cfile.ldate > -1 && cfile.rdate > -1) {
	if (cfile.ldate > cfile.rdate)
	  cfile.message += ", client is newer";
	else
	  cfile.message += ", server is newer";
      }
      showLocalInfo(cfile, cfile.ldate > cfile.rdate);
      showRemoteInfo(cfile, cfile.rdate > cfile.ldate);
      logEntry(cfile.message, false);
      XfilesGui.setStatus(cfile.message);
      break;

    default:
      throw new Error("BAD CONDITION: " + cfile.condition);

    }

    do {

      // think i can first set CMD to none,
      // then call waitforpress,
      // then skip the STATEexecute state and test
      // and rather switch on all possible commands.
      // for the skip command, return a false value for cb to return!
      waitForPress();

      if (forceSkip) XfilesGui.CMD = XfilesGui.CMDskip;

      if (XfilesGui.CMD == XfilesGui.CMDstop ) throw new XFEstop("");

      switch( XfilesGui.CMD ) {

      case XfilesGui.CMDskip:
	recurse = false;
	XfilesGui.CMD = XfilesGui.CMDdone;
	logEntry("skip", true);
	break;

      case XfilesGui.CMDcompare:
	showCompare(cfile);
	XfilesGui.CMD = XfilesGui.CMDnone;		// loop here again
	break;

      case XfilesGui.CMDcopytoserver:
	if (cfile.condition == Cfile.NOLOCAL) {
	  XfilesGui.showAlert("you are copying in the wrong direction");
	  XfilesGui.CMD = XfilesGui.CMDnone;		// loop here again
	}
	else {
	  XfilesGui.setStatus("copy->");
	  file2Server(cfile);
	  XfilesGui.CMD = XfilesGui.CMDdone;
	  logEntry("copy ->", true);
	  recurse = false;
	}
	break;

      case XfilesGui.CMDcopytoclient:
	if (cfile.condition == Cfile.NOREMOTE) {
	  XfilesGui.showAlert("you are copying in the wrong direction");
	  XfilesGui.CMD = XfilesGui.CMDnone;		// loop here again
	}
	else {
	  XfilesGui.setStatus("<-copy");
	  file2Client(cfile);
	  XfilesGui.CMD = XfilesGui.CMDdone;
	  logEntry("<- copy", true);
	}
	break;

      case XfilesGui.CMDclientdelete:
	if (cfile.condition == Cfile.NOLOCAL) {
	  XfilesGui.showAlert("you are deleting on the wrong side");
	  XfilesGui.CMD = XfilesGui.CMDnone;		// loop here again
	}
	else {
	  XfilesGui.setStatus("<-delete");
	  delete(cfile);
	  XfilesGui.CMD = XfilesGui.CMDdone;
	  logEntry("<- delete", true);
	  recurse = false;
	}
	break;

      case XfilesGui.CMDserverdelete:
	if (cfile.condition == Cfile.NOREMOTE) {
	  XfilesGui.showAlert("you are deleting on the wrong side");
	  XfilesGui.CMD = XfilesGui.CMDnone;		// loop here again
	}
	else {
	  try {
	    XfilesGui.setStatus("delete ->");
	    remote.delete(remotePath(cfile));
	    logEntry("delete ->", true);
	  }
	  catch(XFException dex)  { XfilesGui.showAlert(dex.getMessage()); }
	  catch(RemoteException rex)  { XfilesGui.showError(rex); }
	  XfilesGui.CMD = XfilesGui.CMDdone;
	}
	break;

      default:
	XfilesCommon.assert(new Integer(0).toString().equals("bad CMD"));
	break;

      } //switch
    } while( XfilesGui.CMD != XfilesGui.CMDdone );

    XfilesGui.setStatus("");
    XfilesGui.setClientText("");
    XfilesGui.setServerText("");
    XfilesGui.clearCompareText();
    XfilesGui.frame.repaint();
    Thread.yield();

    return recurse;
  } //handle

  //----------------------------------------------------------------

  /** returns one of
   * null 	(file is same on client,server),
   * Cfile	(cfile.message,condition say what to do)
   * Vector	(a vector of Cfiles)
   */
  static final Object needsAttention(Cfile cfile)
  {
    Object rval = null;

    if (!XfilesScript.pathFilter(cfile)) return rval;

    if (!cfile.rexists) {
      cfile.message = "server file does not exist";
      cfile.condition = Cfile.NOREMOTE;
      rval = cfile;
    }

    else if (cfile.rislink) {
      // but local is not link (visitor callback is not called on links).
      cfile.message =
	"server is link, client is not\ncannot handle this situation";
      cfile.condition = Cfile.LINK;
      rval = cfile;
    }

    else if (cfile.lisdir && !cfile.risdir) {
      cfile.message = 
	"client is directory, server is file\ncannot handle this situation";
      cfile.condition = Cfile.RNOTDIR;
      rval = cfile;
    }

    else if (!cfile.lisdir && cfile.risdir) {
      cfile.message =
	"server is directory, client is file\ncannot handle this situation";
      cfile.condition = Cfile.LNOTDIR;
      rval = cfile;
    }

    else if (cfile.lisdir) {
      XfilesCommon.assert(cfile.risdir);
      // compare children, report any in remote that do not exist locally.
      // others will be caught in the file-by-file examination
      Vector missingLocal = null;

      try {
	String[] remotefiles = remote.readDir(remotePath(cfile));
	int nremote = (remotefiles != null) ? remotefiles.length : 0;

	if (nremote != 0) {
	  // hashtable avoids n^2 check here 
	  for( int i=0; i < nremote; i++ ) {
	    String dfile = cfile.lpath + File.separator + remotefiles[i];
	    System.out.print("hashtable lookup " + dfile);
	    if (htab.get(dfile) == null) {
	      System.out.println(" -> no.");
	      if (true) {			// make this configurable
		if (missingLocal == null)  missingLocal = new Vector();
		String cdpath = XfilesCommon.stripRoot(localRoot,
						       localRootLen,
						       dfile);
		Cfile cdfile = new Cfile(null, cdpath);
		cdfile.message = "client file does not exist.";
		cdfile.condition = Cfile.NOLOCAL;
		if (XfilesScript.pathFilter(cdfile))
		  missingLocal.addElement((Object)cdfile);
	      }
	    }
	    else
	      System.out.println(" -> exists.");
	  } //for
	} //nremote!=0
      } //try

      catch(Exception ex) {
	System.err.println(ex);
	ex.printStackTrace();
      }

      if (missingLocal != null) rval = missingLocal;
    } //cfile.lisdir

    else {
      XfilesCommon.assert(!cfile.lisdir && !cfile.risdir);
      try {
	if (!compareChecksum(cfile)) {
	  cfile.message = "files are different";
	  cfile.condition = Cfile.DIFFERENT;
	  rval = cfile;
	}
      }
      catch(Exception ex) {
	System.err.println("compare generated eXception " + ex.getMessage());
	cfile.message = ex.getMessage();
	cfile.condition = Cfile.ERROR;
	rval = cfile;
      }
    }

    if (rval != null &&
	rval instanceof Cfile &&
	((Cfile)rval).condition == 0)
    {
      throw new Error("BAD cfile condition " + ((Cfile)rval).toString());
    }

    return rval;
  } //needsAttention

  //----------------------------------------------------------------

  static final class visitor implements Xfiletree.traverseCB
  {
    // only operate on files below the selected startNode
    Xfiletree startNode = null;
    boolean active;	// we are now past startnode
    boolean scanning;	// scanning or acting
    int selectcount	= 1;	// update gui tree only every nth file

    boolean cbScan(Xfiletree f)
    {
      if (f.isDir) {
	XfilesGui.selectTreeFile(f);
	// encourage the gui to repaint 
	Thread.yield();
      }

      String lpath = f.file.getPath();
      String cpath = XfilesCommon.stripRoot(localRoot, localRootLen, lpath);
      Cfile cfile = new Cfile(f.file, cpath);

      if (gDebug) {
	System.out.println("CB common " + cfile.cpath);
	System.out.println("local " + cfile.lpath);
	System.out.println("remote " + cfile.rpath);
      }

      Object rval = needsAttention(cfile);
      if (rval != null)
	f.userObject = rval;

      return true;
    } //cbScan

    //----------------

    boolean cbAct(Xfiletree f)
      throws XFEstop
    {
      if (f.isDir || (++selectcount % 10 == 0) || f.userObject != null) {
	XfilesGui.selectTreeFile(f);
	Thread.yield();	      // encourage the gui to repaint
      }

      if (f.userObject == null) return true;

      String parentPath = f.file.getParent();
      if (parentPath == null) parentPath = "";
      XfilesGui.setStatus(parentPath + "\n" + f.file.getName());

      if (f.userObject instanceof Vector) {
	Vector v = (Vector)f.userObject;
	int len = v.size();
	// Return true regardless here:
	// these are files that are in this directory on the server only.
	// Returning false means do not traverse children of this directory.
	// TODO: it would be nice to have a way of saying
	// "skip the rest of this directory"
	for( int i=0; i < len; i++ ) {
	  handle((Cfile)(v.elementAt(i)));
	}
	return true;
      }
      else
	return handle((Cfile)f.userObject);
    } //cbAct

    //----------------

    public boolean cb(Xfiletree f)
      throws XFEstop
    {
      if (XfilesGui.CMD == XfilesGui.CMDstop) throw(new XFEstop(""));

      if (!active) {
	if (f == startNode)
	  active = true;
	else
	  return true;
      }

      if (scanning)
	return cbScan(f);
      else
	return cbAct(f);

    } //cb

  } //visitor

  //----------------------------------------------------------------

  static String remoteRoot;
  static String localRoot;
  static int localRootLen;

  static String EserverRootIsBad = 
    "The server root path does not exist\n" +
    "(on Unix -- may be a dangling link)";

  static String EserverRootIsLink = 
    "The server root path contains a link\n" +
    "(use the path it references instead)";

  static String EwriteFailed(String clientOrServer, String path)
  {
    String msg =
      clientOrServer + " file write failed for\n" + path;
    msg += "\n(write permissions / disk space problem?)";
    return msg;
  } //EwriteFailed

  static void doSync(XfilesInterface remote,
		     String localroot,
		     String remoteroot)
  {
    System.out.println("doSync clientroot=" + localroot);
    System.out.println("doSync serverroot=" + remoteroot);

    XfilesGui.setStatus("");
    try {

      remoteRoot = remoteroot;
      localRoot = localroot;
      localRootLen = localRoot.length();
      remote.setRootPath(new File(remoteRoot));

      visitor v = new visitor();

      // selection indicates where to start scanning
      TreePath p = XfilesGui.clientTree.getSelectionPath();
      Object o = p.getLastPathComponent();
      if (o == null)
	v.startNode = XfilesGui.root;
      else
	v.startNode = (Xfiletree)p.getLastPathComponent();;
      System.out.println("starting at " + v.startNode);

      // it would be good to rescan the client side each time
      // (in case the user has made changes outside this program).
      // it does not update the jtree correctly however.
      //v.startNode.createChildren(true, true);

      XfilesGui.selectTreeFile(v.startNode);
      //XfilesGui.clientTree.treeDidChange();
      String lpath = v.startNode.file.getPath();
      String cpath = XfilesCommon.stripRoot(localRoot, localRootLen, lpath);
      XfilesCommon.assert(cpath != null);

      // cfile constructor calls remote.lastModified(), etc.,
      // which require that the remote tree be scanned.
      // Thus must call readTree() before constructing any cfile.
      //
      if (!remote.fileExists(remotePath(cpath))) {
	XfilesGui.showAlert(remoteRoot+cpath + " does not exist.");
	return;
      }

      XfilesGui.setStatus("reading remote tree");
      remote.readTree(remotePath(cpath));

      /****
      Cfile cfile = new Cfile(v.startNode.file, cpath);
      System.out.println("readtree common " + cfile.cpath);
      System.out.println("local " + cfile.lpath);
      System.out.println("remote " + cfile.rpath);
      ****/

      // rebuild the htab every time, to get rid of old paths
      htab = new Hashtable();
      XfilesGui.root.traverse(new hashCB());

      // this is it
      v.active = false;
      v.scanning = true;
      XfilesGui.setStatus("scanning for differences");
      long t1 = System.currentTimeMillis();
      try {
	v.startNode.traverse(v);
      } catch(XFEstop ex) { System.out.println("stopped"); }
      long t2 = System.currentTimeMillis();

      System.out.println("\n\nscanning took " + (t2-t1)/1000.0 + " seconds.");
      System.out.println("free memory = " + Runtime.getRuntime().freeMemory());
      System.out.println("\n");

      v.active = false;
      v.scanning = false;
      XfilesGui.setStatus("");
      try {
	v.startNode.traverse(v);
      } catch(XFEstop ex) { System.out.println("stopped"); }
    }
    catch(XFExServerRootIsBad ex) {
      XfilesGui.showAlert(EserverRootIsBad + "\n" + ex.getMessage());
      quitme(1);
    }
    catch(XFExServerRootIsLink ex) {
      XfilesGui.showAlert(EserverRootIsLink + "\n" + ex.getMessage());
      quitme(1);
    }
    catch(Exception ex) {
      System.err.println("doSync exception");
      XfilesGui.showError(ex);
    }
    XfilesGui.setStatus("done");
  } //doSync

  //----------------------------------------------------------------

  // catch all remaining exceptions
  public static void startClient(String arg0, String arg1)
  {
    try {
      String url = System.getProperty("XFILES", "rmi:///XS");
      remote = (XfilesInterface)Naming.lookup(url);
      doSync(remote, arg0, arg1);
    }
    catch (java.rmi.UnknownHostException ex)
    {
      System.err.println("Failing due to unknownhost, trying again");
      System.err.println(ex.getMessage());
    }
    //
    catch (java.rmi.NotBoundException ex)
    {
      System.err.println(ex);
    }
    //
    catch (java.rmi.ConnectIOException ex)
    {
      System.err.println("Failing with connectIOException, trying again");
      System.err.println(ex.getMessage());
    }
    // Catch and display RMI exceptions
    catch (RemoteException ex)
    {
      System.err.println(ex);
      ex.printStackTrace();
    }
    // Other exceptions are probably user syntax errors, so show usage.
    catch (Exception ex) { 
      System.err.println(ex);
      ex.printStackTrace();
      System.err.println("Usage: java [-DRQNAME=<url>] XfilesClient " + 
			 "<cmd> [<args>]");
      System.err.println("<cmd> \"help\" to get full usage");
      quitme(1);
    } //catch
    catch (Throwable ex) {
      System.err.println("uncaught exception: " + ex);
      ex.printStackTrace();
      quitme(1);
    }
  } //startClient

  //----------------------------------------------------------------

  /**
   * quit, telling rmi distributed gc that we are gone.
   * This is called by the gui.
   */
  static void quitme(int code)
  {
    File f = new File(TMPFILE);
    f.delete();
    remote = null;
    System.gc();
    System.runFinalization();
    System.exit(code);
  }

  //----------------------------------------------------------------

  /** main here is no longer called
   */
  public static void main(String[] args)
  {
    try {
        LocateRegistry.createRegistry(9753);
    }
    catch (RemoteException e) {
        System.out.println("can't create rmi-registry!");
        System.exit(-1);
    }

    System.out.println("#args " + args.length);
    if (args.length != 2) {
      System.err.println("Usage:");
      System.err.println("java -DXFILES=rmi://hostname:9753/XF XfilesGui clientroot serverroot");
      System.exit(1);
    }

    final String localRootPath = XfilesCommon.stripTrailingSlash(args[0]);
    final String remoteRootPath = XfilesCommon.stripTrailingSlash(args[1]);

    startClient(localRootPath, remoteRootPath);
  }

}

