/* 
stattool.c

read paths from stdin and stats them.

Depending on mode, prints various stuff to stdout.

Options:

Filter options:

-c <time>:
  Process only file names that have a ctime newer than <time>
  (in seconds since epoch).


Output options:

-s: sizes mode
  Print total size of files and total number of files.

-n: name mode
  Print paths.

-l:lapback mode
  Print file info in lapback -S format.

For internal use by tbackup.

AUTHOR
  Koen Holtman, koen@blade.stack.urc.tue.nl   (kh1994)

*/


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <getopt.h>

int smode=0;
int cmode=0;
time_t newertime=0;
int nmode=0;
int lmode=0;

void usage();
void doit();
char filetype(int mode);

void main(int argc, char ** argv)
{
 int opt;
 unsigned long t;

 while((opt=getopt(argc,argv,"snlc:"))!=-1)
   {
       switch(opt)
       {
       case 's' : smode=1; break;
       case 'n' : nmode=1; break;
       case 'l' : lmode=1; break;

       case 'c' : cmode=1;  if(sscanf(optarg,"%lu",&t)!=1) usage();
	          newertime=t;
                  break;
       default  : usage();
       }
   }

  doit();
}

void usage()
{
 fprintf(stderr,"usage: stattool [-sn] [-c ctime]\n");
 exit(0);
}

void doit()
{ 
  long tot,nr;
  struct stat fstatus;
  char name[1024];

  tot=nr=0;

  while(fgets(name,1024,stdin)!=NULL)
    {
     if(name[0]!='\0') name[strlen(name)-1]='\0';

     if(lstat(name,&fstatus)) { perror(name); continue; }

    /* handle filter options */
 
    if(cmode) if(! (fstatus.st_ctime>newertime) ) continue;

    /* handle output options */

     /* for smode */
     nr++;
     if(S_ISREG(fstatus.st_mode)) tot+=fstatus.st_size;

     if(nmode) printf("%s\n",name);

     if(lmode) 
     {
        printf("%s %d %d %d %lu %lu %lu %c n %u-%lu\n",
	       name,
               (int)fstatus.st_uid,
               (int)fstatus.st_gid, 
	       (int)fstatus.st_mode,
               (unsigned long)fstatus.st_size, 
               (unsigned long)fstatus.st_mtime,
               (unsigned long)fstatus.st_ctime, 
	       filetype(fstatus.st_mode),
               (unsigned int)fstatus.st_dev, 
               (unsigned long)fstatus.st_ino
              );
     }
    }

  if(smode) printf("%ld %ld\n",tot,nr);

}


/*
  filetype() emulates the following perl code from lapback:

    if( -l "$dir/$file") { $type = "l"; }
    elsif( -f "$dir/$file") { $type = "f"; }
    elsif( -d "$dir/$file") { $type = "d"; }
    elsif( -c "$dir/$file") { $type = "c"; }
    elsif( -b "$dir/$file") { $type = "b"; }
    elsif( -S "$dir/$file") { $type = "s"; }
    elsif( -p "$dir/$file") { $type = "p"; }
    else {
      $type = "u";
      &warning("unknown type for file $prefix$file");
    }
*/

char filetype(int mode)
{
    if(S_ISLNK(mode)) return 'l';
    if(S_ISREG(mode)) return 'f';
    if(S_ISDIR(mode)) return 'd';
    if(S_ISCHR(mode)) return 'c';
    if(S_ISBLK(mode)) return 'b';
    if(S_ISSOCK(mode)) return 's';
    if(S_ISFIFO(mode)) return 'p';
    return 'u';
}
