/*

NAME
  salvage - read data from partially damaged disk.

SYNOPSYS
  salvage [ -b`begin' ] -c`count' devicename >data 2>successmap
  (for internal use by tbackup)

DESCRIPTION
  Reads `count' bytes from devicename, optionally starting at byte
  offset `begin'. 
  If a read error occurs, a block of `?'  characters is output instead
  of the real data.
  The successmap contains one character for each consecutive sector pair
  (each consecutive 1024 bytes) read.  It contains a `.' for a successful
  read, an `X' if there was a read error.

  Aborts with exit code 1 if the user presses `a' or `A'.

BUGS
  If -b is not a multiple of 1024 a garbage succesmap is produced.


AUTHOR
  Koen Holtman, koen@blade.stack.urc.tue.nl   (kh1994)

**/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <fcntl.h>

/* logical floppy block size */
#define BLOCK 1024

 void usage();
 void doit(); 
 
 void handler(int i); 
 void setstuff();
 void resetstuff();


 char *pnam;
 char *fnam=NULL;

 long begin=0;
 long count=-1; 

int main(int argc, char *argv[])
{ 
 int i;

 pnam=argv[0];

 for(i=1; i<argc; i++)
   { if(argv[i][0]!='-')
       {  if(fnam!=NULL) usage();
          fnam=argv[i];
       }
     else
       {
        switch(argv[i][1])
	  {
            case 'b' :
                  if(sscanf(&argv[i][2],"%ld",&begin)!=1) usage();
                  if(begin<0) usage();
                  break;

            case 'c' : 
                  if(count!=-1) usage();
                  if(sscanf(&argv[i][2],"%ld",&count)!=1) usage();
                  if(count<1) usage();
                  break;

            default: usage();
	  }
       }
   }

 if(count==-1) usage();
 if(fnam==NULL) usage();

 setstuff();

 doit();

 resetstuff();
 return 0;
}

void handler(int i) 
{ 
  resetstuff();
  exit(1);
}

struct termios tty;
int fflags;

void setstuff()
{
 signal(SIGINT,handler);

 /* Ugh. It took me far too long to find this. */
 tcgetattr(0, &tty);
 tty.c_lflag&=~ICANON;
 tcsetattr(0,TCSANOW, &tty);
 
 fflags=fcntl(0,F_GETFL);
 fflags|=O_NONBLOCK;
 fcntl(0,F_SETFL,(long)fflags);


} 

void resetstuff()
{
 tty.c_lflag|=ICANON;
 tcsetattr(0,TCSANOW, &tty);

 fflags&=~O_NONBLOCK;
 fcntl(0,F_SETFL,(long)fflags);
}


int apressed()
{
 char c; int r;

 while(1)
   {
     r=read(0,&c,1);
     if(r!=1) return 0;
     if((c=='a') || (c=='A')) return 1;
   }
}

void checkabort()
{
 if(apressed())
   {
     resetstuff();
     exit(1);
   }
}

void doit()
{
 int d,firsttime; long processed,chunk,got,put,totalput,totalgot,err,shift;
 char buf[BLOCK];

 d=open(fnam,O_RDONLY);
 if(d<0)
  { fprintf(stderr,"%s: error opening %s!\n",pnam,fnam);
    exit(1);
  } 

 lseek(d,(off_t)begin,0);
 firsttime=1; 

 /* read in BLOCK size blocks */
 processed=0;
 while(processed<count)
   {
     totalgot=0;
     err=0;

     if(count-processed<BLOCK)
       chunk=count-processed;
     else
       chunk=BLOCK;

     if(firsttime)
     {
	 /* set chunk so that next reads begin on a byte position that is
	    a multiple of BLOCK */
	 
	 if(begin%BLOCK!=0)
	 {
	     shift=BLOCK-(begin%BLOCK);
	     if(chunk>shift) chunk=shift;
	 }
	 firsttime=0;
     }

     /* read chunk bytes*/
     while(totalgot<chunk)
       { 
	 got=read(d,buf+totalgot,chunk-totalgot);
         if(got<0)
	   {
	     err=1;
             /* fill buffer with dummy data */
             while(totalgot<chunk) buf[totalgot++]='?';
             /* go to next block */
             lseek(d,(off_t)begin+processed+chunk,0);
             break;
	   }
         totalgot+=got;
       }

     fprintf(stderr,"%c",err ? 'X' : '.');

     checkabort();

     /* write chunk bytes*/
     totalput=0;
     while(totalput<chunk)
       {
         put=write(1,buf+totalput,chunk-totalput);
         if(put<0)
            { fprintf(stderr,"%s: write error on stderr!\n",pnam);
              exit(1);
            } 
         totalput+=put;
       }

     processed+=chunk;
   }

 close(d);

}

void usage()
{
 fprintf(stderr,
  "usage: %s [ -b`begin' ] -c`count' devicename >data 2>successmap\n"
        ,pnam);
 exit(1);
}

