/*

NAME
  mycp - copy with user abort

SYNOPSYS
  mycp fromfile tofile
  (for internal use by tbackup)

DESCRIPTION
  Copies fromfile to tofile.
  Aborts with exit code 1 of a write error occurs.
  Aborts with exit code 1 if the user presses `i' or `I'.

  v2.0 -- for post-1994 linux kernels which support the 
          floppy error reporting ioctls.

AUTHOR
  Koen Holtman, koen@win.tue.nl   (kh1997)
*/

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <linux/fd.h>
#include <sys/ioctl.h>

 void usage();
 void doit(); 
 
 void handler(int i); 
 void setstuff();
 void resetstuff();


 char *pnam;
 char *infile;
 char *outfile;

 int child=0;

 int f=0,t=0; /* from and to files */


   
int main(int argc, char *argv[])
{ 
 pnam=argv[0];
 if(argc!=3) usage();

 infile=argv[1];
 outfile=argv[2];

 setstuff();

 doit();

 resetstuff();
 return 0;
}

void handler(int i) 
{ 
  fprintf(stderr,"Signal caught. Aborting...\n");
  resetstuff();
  exit(1);
}

struct termios tty;
int fflags;

void setstuff()
{
 signal(SIGINT,handler);

 /* Ugh. It took me far too long to find this. */
 tcgetattr(0, &tty);
 tty.c_lflag&=~ICANON;
 tcsetattr(0,TCSANOW, &tty);
 
 fflags=fcntl(0,F_GETFL);
 fflags|=O_NONBLOCK;
 fcntl(0,F_SETFL,(long)fflags);


} 

void resetstuff()
{
 tty.c_lflag|=ICANON;
 tcsetattr(0,TCSANOW, &tty);

 fflags&=~O_NONBLOCK;
 fcntl(0,F_SETFL,(long)fflags);

 if(child) waitpid(child, NULL, 0);
}


int apressed()
{
 char c; int r;

 while(1)
   {
     r=read(0,&c,1);
     if(r!=1) return 0;
     if((c=='i') || (c=='I')) return 1;
   }
}

void fdflush()
{
  if(t==0) return;
  if (ioctl(t,FDFLUSH,NULL) < 0) perror("mycp");
}

void checkabort()
{
 if(apressed())
   {
     fprintf(stderr,"\nInterrupting write.....\n");
     fflush(stderr);
     /* throw away unsynced writes. This helps a bit to reduce 
	time to exit */
     fdflush();
     
     resetstuff(); /* this also waits for the children */
     fdflush(); /* just to make sure */
     exit(1);
   }
}

void checkwerror(int t) 
{
    struct floppy_write_errors fwe;

    fwe.write_errors=1;
    if(ioctl(t,FDWERRORGET,&fwe) < 0) perror("mycp, FDWERRORGET");
    if(fwe.write_errors!=0)  
    {	
	fprintf(stderr,"%s: write error on floppy %s sector %lu\n", \
		pnam, outfile, fwe.first_error_sector);
	resetstuff();
	exit(1);
	

    }
    
}


void doit()
{
 int count,wcount,w,ccount; long total;
 char buf[4096];

 total=ccount=0;

 f=open(infile,O_RDONLY);
 if(f<0) 
   {
     fprintf(stderr,"%s: error opening %s\n",pnam,infile);
     resetstuff();
     exit(1);
   }

 /* O_SYNC mode will ensure that all data is written after the close */
 t=open(outfile, O_WRONLY | O_CREAT | O_TRUNC | O_SYNC, 0644);
 if(t<0) 
   {
     fprintf(stderr,"%s: error opening %s\n",pnam,outfile);
     resetstuff();
     exit(1);
   }

 /* clear write error and badness information */
 if (ioctl(t,FDWERRORCLR,NULL) < 0) perror("mycp, FDWERRORCLR");

 /* perform the copy. */

 while(1)
 {
     count=read(f,buf,4096);
     if(count<0)
     {
	 fprintf(stderr,"%s: read error on ",pnam);
	 perror(infile);
	 resetstuff();
	 exit(1);
     }
     if(count==0) break;
     
     checkabort();
     
     wcount=0;
     while(wcount<count)
     {
	 w=write(t,buf+wcount,count-wcount);
         if(w<=0)
	 {
	     fprintf(stderr,"%s: write error on %s!\n",pnam,outfile);
	     resetstuff();
	     exit(1);
	 }
         wcount+=w;
     }
 
     total+=count;

     if(ccount++>=16) 
     {
	 /* check for any write errors */
	 checkwerror(t);
	 ccount=0;
     }
 }
 close(f);
     
 /* flush for luck, though close should flush all too */
 if (ioctl(t,FDFLUSH,NULL) < 0) perror("mycp, FDFLUSH");

 checkwerror(t);

 close(t);

 /* check the final write error report */
 t=open(outfile, O_RDONLY);
 checkwerror(t);
 close(t);

}



void usage()
{
 fprintf(stderr,
  "usage: %s <file> <floppy>\n"
        ,pnam);
 exit(1);
}





