/*

NAME
  flowpause - pause dataflow.

SYNOPSYS
  flowpause [-k`kb' -m`mb' -f`maxf' -s -g`grain' -w`file' -h] [file or dirname]

DESCRIPTION
  usage: flowpause [options] [filename or directoryname]

  Works as a filter. Data flows through unchanged. The dataflow is paused
  if the filesystem is nearly full and/or the named directory contains too
  much files. 

  Options:
    -k`kb' -m`mb' : pause dataflow if less than `kb' or `mb' free space left
                    on the filesystem containing file or dirname
    -f`maxf'      : pause dataflow if more than `maxf' files in dirname
    -s            : silent mode, do not print pause/continue messages
    -g`grain'     : when short on space, check device and/or directory when
                    `grain' kb has passed through. (default is the minumum of
                    50 and `kb'/4.)
    -w`file'      : print `file' if free space limit reached
    -h            : print help
  
  
  If no -k or -m is given, flowpause does not check the space on the
  filesystem.
  If no -f is given, the number of files in the directory is not checked.
  If no -k, -m or -f option is given, -m1 is assumed.


EXAMPLES
  Unpack a large archive on a nearly-full filesystem:
    cat /dev/largearchive | flowpause -k100 /usr/src | tar xvf -

  Make an archive in 1.4mb parts. In another shell, these parts are manually
  written out by mtools or ftp and then deleted.
    tar zcvf - /usr/src | flowpause /tmp/bup -m2 -f4 "-wwarning"| \
    split -b1400k - /tmp/bup/arch

  Pause a script if the root device is nearly full:
    echo | flowpause / -k20
  

AUTHOR
  Koen Holtman, koen@blade.stack.urc.tue.nl   (kh1993)

**/



#include <stdio.h>
#include <sys/vfs.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/dir.h>

#define BUFFERSIZE 4*1024

 int fdes;
 struct statfs disk;

 long minkb=-1;     
 long maxfiles=-1;
 long grain=-1;

 int verbose=-1;

 char defaultnam[]=".";
 char *fnam=defaultnam;

 unsigned long long currentfreekb=0;
 /* currentfreekb is a pessimistic estimate of the free space on the device */
 unsigned long long part;
 /* part = #bytes after which free space is checked. */

 void usage();
 void help();
 void doit();
 void waitforroom();
 int nfiles();
 int permeable();

 char *pnam;
 char *reason;
 char *warnfile=NULL;

int main(int argc, char *argv[])
{ 
 int i;

 pnam=argv[0];

 for(i=1; i<argc; i++)
   { if(argv[i][0]!='-')
       {  if(fnam!=defaultnam) usage();
          fnam=argv[i];
       }
     else
       {
        switch(argv[i][1])
	  { case 's' :
                  if(strlen(argv[i])!=2) usage();
        	  verbose=0;
                  break;

            case 'k' :
                  if(minkb!=-1) usage();
                  if(sscanf(&argv[i][2],"%ld",&minkb)!=1) usage();
                  if(minkb<0) usage();
                  break;

            case 'm' : 
                  if(minkb!=-1) usage();
                  if(sscanf(&argv[i][2],"%ld",&minkb)!=1) usage();
                  minkb*=1024;
                  if(minkb<0) usage();
                  break;

            case 'f' : 
                  if(maxfiles!=-1) usage();
                  if(sscanf(&argv[i][2],"%ld",&maxfiles)!=1) usage();
                  if(maxfiles<0) usage();
                  break;

            case 'g' : 
                  if(grain!=-1) usage();
                  if(sscanf(&argv[i][2],"%ld",&grain)!=1) usage();
                  if(grain<1) usage();
                  break;

            case 'w' : 
                  warnfile=&argv[i][2];
                  break;

	    case 'h' :
                  if(strlen(argv[i])!=2) usage();
        	  help();

            default: usage();
	  }
       }
   }

 if((minkb==-1) && (maxfiles==-1)) minkb=1024;

 fdes=open(fnam,O_RDONLY);
 if(fdes<0)
   { fprintf(stderr,"%s: can't open file or directory %s.\n",pnam,fnam);
     exit(1);
   }

 /*
 fprintf(stderr,"%ld files, %ld space, %ld grain\n",maxfiles,minkb,grain);
 */
 
 if(grain!=-1)
   {  part=grain*1024;
   }
 else
   {
      part=50*1024;
      if((minkb!=-1)&&(part>minkb*(1024/4))) part=minkb*(1024/4);
   }

 doit();

 return 0;
}

void usage()
{
 fprintf(stderr,
  "usage: %s [-k`kb' -m`mb' -f`maxf' -s -g`grain' -w`file' -h] [file or dirname]\n"
        ,pnam);
 fprintf(stderr,"  %s -h prints help.\n",pnam);
 exit(1);
}

void help()
{
 fprintf(stderr,"flowpause V0.8\n");
 fprintf(stderr,
  "usage: %s [options] [filename or directoryname]\n"
        ,pnam);
 fprintf(stderr,
" Works as a filter. Data flows through unchanged.\n"
" filename or directoryname is on the filesystem to watch. (default: `.')\n"
"  -k`kb' -m`mb' : pause dataflow if less than `kb' or `mb' free space left\n"
"                  on the filesystem containing file or dirname\n"
"  -f`maxf'      : pause dataflow if more than `maxf' files in dirname\n"
"  -s            : silent mode, do not print pause/continue messages\n"
"  -g`grain'     : when short on space, check device and/or directory when\n"
"                  `grain' kb has passed through. (default is the minumum of\n"
"                  50 and `kb'/4.)\n"
"  -w`file'      : print `file' if free space limit reached\n"
"  -h            : print help\n"
"    If no -k, -m or -f option is given, -m1 is assumed.\n");

 exit(1);
}


unsigned long long freespacekb()
{
 unsigned long long kb;

   if(fstatfs(fdes,&disk)!=0) 
      { fprintf(stderr,"%s: can't stat filesystem!\n",pnam);
        exit(1);
      }

  kb=(unsigned long long)disk.f_bsize* \
     (unsigned long long)disk.f_bfree/ \
     (unsigned long long)(1024);
   
  return kb;
}

int nfiles()
{
 DIR *dir;
 int n;

 dir=opendir(fnam);
 if(dir==NULL)
     { fprintf(stderr,"%s: %s is not a directory.\n",pnam,fnam);
       exit(1);
     }
 
 n=0;
 while(readdir(dir)!=NULL) n++; 

 closedir(dir);

 return n-2;
}


int permeable()
{
/* fprintf(stderr,"perm, currentfreekb=%ld\n",currentfreekb); */

 if(minkb!=-1) 
   { 
/* asking free space is an expensive operation as it syncs the disk,
   so we try to limit it's occurence. */
      if(currentfreekb < minkb+part/256 )
           currentfreekb=freespacekb();

      if(currentfreekb<minkb)
        { reason="free space limit on device reached";
          return 0;
        }
   }
 if(maxfiles!=-1) 
   if(nfiles()>maxfiles)
     { reason="maximum file count in directory reached";
       return 0;
     }

 return -1;
}


void doit()
{
 char buffer[BUFFERSIZE];
 off_t b;
 long bcount,chunk;
 /* chunk = max read/write buffer size */

 chunk=BUFFERSIZE;
 if(chunk>part/4) chunk=part/4;

 /*
 fprintf(stderr,"part:%ld, chunk:%ld\n",part,chunk);
 */

 while(1)
   {  
      if(!permeable()) waitforroom();
      
      bcount=0;
      while(bcount<part)
	{
          b=read(0,buffer,chunk);
          if(b==0) return;
          if(b<0)
            { fprintf(stderr,"%s: error reading stdin!\n",pnam);
                 exit(1);
            } 
   
          bcount+=b;

          if(write(1,buffer,b)!=b)
            { fprintf(stderr,"%s: error writing to stdout!\n",pnam);
                 exit(1);
            } 
	}
        /* substract kb passed through * 4 from free space estimate */
        currentfreekb-=bcount/(1024/4);      
    }

}


void waitforroom()
{
 FILE *f; 
 char c;
 
 if(verbose)
   fprintf(stderr,"%s: %s: Pausing flow -- %s.\n",
     pnam,fnam,reason);

 if((reason[0]=='f') && warnfile!=NULL)
   if((f=fopen(warnfile,"r"))!=NULL)
   {  
    while((c=fgetc(f))>=0) fprintf(stderr,"%c",c);
    fclose(f);
   }

 do
 {
  sleep(4);

 }while(!permeable());

 if(verbose)
   fprintf(stderr,"%s: %s: Continuing flow.\n",pnam,fnam);
}

