/*

NAME
  ecor -- blexta error correction engine

  For internal use by tbackup.
  
  Implements blexta (block level exor trace assembly)
  error correcting codes for floppies.

NOTE
  The trace layout tries to minimize the chance that more than one
  error occurs in a single trace.  The layout could be improved upon if
  1) more statistical information on the occurrence of floppy errors is
  available or 2) more sophisticated scattering techniques are used.

  For best results -x must 1) be larger than the number of sectors
  per track on the floppy and 2) be coprime with this number, so that
  `vertical' error traces are scattered.

        sector 0   ...     sector n
  track 0                X
                    .    X
                    .    X
                    .
  track n 

  XXX is a vertical error trace.  Example: with 18 sec/track -x29 is a good
  number.

AUTHOR
  Koen Holtman, koen@stack.urc.tue.nl   (kh1994)
  Koen Holtman, koen@win.tue.nl         (kh1998)

  v 0.99 -- 1998 --  changed fopen, fseek to open, lseek because newer
                     C library fseek(3) cannot be mixed with read(2)
                     (ack!).


*/



#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define BLOCK 1024

 void doit();
 void prephase();
 void calctraces();

 char *pnam;

int tflag=0; 
int vflag=0;
int mflag=0;
int hflag=0;
int dflag=0;
int rflag=0;
int Dflag=0;
int Eflag=0;
int oneflag=0;

int xsize=0;
int ysize;
int makeecc=0;

char *datanam=NULL;
char *ecornam=NULL;
char *mapnam=NULL;


int dataf, ecorf;
FILE *mapf;


int avalen=-1;
int datalen,ecorlen;

char *map;


void usage()
{
 fprintf(stderr,
  "usage: %s -x`x' -a`avail' [-tvmdrDE1] [-h] datafile eccfile [mapfile]\n"
        ,pnam);
 if(!hflag) exit(1);

 fprintf(stderr,"blexta error correction engine V0.99.\n");
 fprintf(stderr,"For internal use by tbackup.\n");
 fprintf(stderr," -a is the available space for datafile+eccfile.\n");
 fprintf(stderr," -x is a scattering parameter that also influences the ecc file size\n");
 fprintf(stderr," If no mapfile given, makes error correcting codes for datafile in eccfile.\n");
 fprintf(stderr," If mapfile given, corrects errors in datafile using the eccfile.\n");
 fprintf(stderr," Mapfile maps the read errors in datafile and eccfile.\n");
 fprintf(stderr," A `.' is a good 1024 byte block, a `X' is a bad block.\n");
 fprintf(stderr," -t prints ecc traces\n -v is verbose mode\n -m prints mapfiles\n");
 fprintf(stderr," -d is dry run mode (do not change files)\n");
 fprintf(stderr," -r prints number of remaining bad data blocks on stdout\n");
 fprintf(stderr," -D prints size of datafile calculated from -x and -a and exits\n");
 fprintf(stderr," -E prints size of eccfile calculated from -x and -a and exits\n");
 fprintf(stderr," -1 force no preprocess phase when no mapfile given\n");
 fprintf(stderr," -h prints help.\n");
  exit(0);
}

void fater(char *m)
{
    perror(pnam);
    fprintf(stderr,"%s:fatal error: '%s'\n",pnam,m);
    exit(10);
}

int main(int argc, char *argv[])
{ 
 int opt,c,mapi;

 pnam=argv[0];

 while((opt=getopt(argc,argv,"x:a:tvmdrDEh1"))!=-1)
   {
       switch(opt)
       {
       case 'x' : if(sscanf(optarg,"%d",&xsize)!=1) usage(); break;
       case 'a' : if(sscanf(optarg,"%d",&avalen)!=1) usage(); break;
       case 't' : tflag=1; break;
       case 'v' : vflag=1; break;
       case 'm' : mflag=1; break;
       case 'd' : dflag=1; break;
       case 'r' : rflag=1; break;
       case 'D' : Dflag=1; break;
       case 'E' : Eflag=1; break;
       case '1' : oneflag=1; break;
       case 'h' : hflag=1; usage();
       default  : usage();
       }
   }

 if(xsize<=0) usage();
 if(avalen<=0) usage();

 ecorlen=2*xsize+5+5+4;
 datalen=avalen-ecorlen;

 ysize=(datalen+xsize-1)/xsize;

 if(Dflag) { printf("%d\n",datalen); exit(0); }
 if(Eflag) { printf("%d\n",ecorlen); exit(0); }

 calctraces();

 if(tflag) exit(0);

 if( ! ((optind+2==argc) || (optind+3==argc)) ) usage();
 if(optind+2==argc) makeecc=1;

 datanam=argv[optind];
 if(!dflag)
     if((dataf=open(datanam,O_RDWR))<0) fater(datanam);

 ecornam=argv[optind+1];
 if(!dflag)
     if((ecorf=open(ecornam,makeecc ? (O_RDWR|O_CREAT|O_TRUNC) : (O_RDWR),0666))<0) fater(ecornam);

 map=(char *)malloc(avalen);
 if(map==NULL) fater("malloc");

 if (!makeecc) 
 {
     mapnam=argv[optind+2];
     if((mapf=fopen(mapnam,"r"))==NULL) fater(mapnam);
     mapi=0;
     while(mapi<avalen)
     {
	 c=fgetc(mapf);
	 if(c<0) { fprintf(stderr,"Warning:incomplete map\n"); c='.'; }

	 if((c=='.')||(c=='X')) map[mapi++]=c;
     }
     oneflag=1;
 }
 else
 {
     for(mapi=0; mapi<avalen; mapi++)
	 if(mapi<datalen) map[mapi]='.'; else map[mapi]='X';
 }

 if(mflag) 
 { 
     write(2,map,avalen); fprintf(stderr,"\n"); 
 }

 if(!oneflag) prephase();
 
 doit();

 if(mflag) 
 { 
     write(2,map,avalen); fprintf(stderr,"\n"); 
 }

 return 0;
}


struct tracerec{
 struct tracerec *next;
 int len;
 int numpre;
 int *data;
};

struct tracerec *traces=NULL,*lasttrace=NULL;

int curtlen=0; 
int *curtbuf=NULL;

void addtraceelem(int e)
{
    int blen;

    blen=avalen; /* could be shorter */

    if(curtbuf==NULL) 
    {
	curtbuf=(int *)malloc(blen*sizeof(int));
	if(curtbuf==NULL) fater("malloc");
    }
    
    if(curtlen>=blen) fater("trace buffer too short");
    curtbuf[curtlen++]=e;
}

void addtrace()
{
    int *temp;
    struct tracerec *temp2;
    
    temp=(int *)malloc(curtlen*sizeof(int));
    if(temp==NULL) fater("malloc");
    temp2=(struct tracerec *)malloc(sizeof(struct tracerec));
    if(temp2==NULL) fater("malloc");
    
    temp2->next=NULL;
    temp2->len=curtlen;
    temp2->numpre=0;
    temp2->data=temp;
    bcopy(curtbuf,temp,curtlen*sizeof(int));

    if(traces==NULL)
    {
	traces=temp2;
	lasttrace=temp2;
    }
    else
    {
	lasttrace->next=temp2;
	lasttrace=temp2;
    }

    curtlen=0;
}

void tdump()
{
    struct tracerec *t;
    int i;
    
    printf("%d %d\n",xsize,ysize);

    for(t=traces; t!=NULL; t=t->next)
    {
	for(i=0; i<t->len; i++) printf("%d ",t->data[i]);
	printf("\n");
    }

}



void calctraces()
{
 int last,l,t,x,y,p;
 
 last=datalen;

#if 1
 /* vertical traces */ 
 for(x=0; x<xsize; x++)
 {
     for(y=0; y<ysize; y++) 
     {
	 p=x+y*xsize;
	 if(p<datalen) addtraceelem(p);
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

#if 1
 /* diagonal / traces */ 
 for(x=0; x<xsize; x++)
 {
     for(y=0; y<ysize; y++)
     {
	 p=(x+(ysize-y))%xsize+y*xsize;
	 if(p<datalen) addtraceelem(p);
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

#if 1
 /* diagonal \ traces (5x) */ 

 for(t=0; t<5; t++)
 {
     for(x=0; x<xsize; x++)
     {
	 for(y=0; y<ysize; y++)
	 {
	     p=(x+y)%xsize+y*xsize;
	     if(x%5==t)
		 if(p<datalen) addtraceelem(p);
	 }
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

#if 1
 /* horizontal traces (5x) */ 

 for(t=0; t<5; t++)
 {
     for(y=0; y<ysize; y++)
     {
	 for(x=0; x<xsize; x++)
	 {
	     p=x+y*xsize;
	     if(y%5==t)
		 if(p<datalen) addtraceelem(p);
	 }
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

 l=last;

#if 1
 /* internal ecc ecc, mod 2 */

 for(t=0; t<2; t++)
 {
     for(x=datalen; x<l; x++)
     {
	 if(x%2==t) addtraceelem(x);
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

#if 1
 /* internal ecc ecc, mod 16 */
 for(t=0; t<2; t++)
 {
     for(x=datalen; x<l; x++)
     {
	 if((x&8)==t*8) addtraceelem(x);
     }
     addtraceelem(last++);
     addtrace();
 }
#endif

 if(vflag) fprintf(stderr,"Dimensions: datalen %d, ecorlen %d (%.2f%%) avalen %d\n",
		   datalen,ecorlen,(ecorlen*100.0)/avalen,avalen);

 if(last!=avalen) fprintf(stderr,"Internal logic error!!!\n");

 if(tflag) tdump();
}

int good,bad,goodd,badd;
void calcgoodbad()
{
    int i;
    good=bad=0;
    for(i=0; i<avalen; i++)
	if(map[i]=='X') bad++; else good++;

    goodd=badd=0;
    for(i=0; i<datalen; i++)
	if(map[i]=='X') badd++; else goodd++;

    if(vflag)
    {
	fprintf(stderr,"Data blocks: good=%d bad=%d",goodd,badd);
	fprintf(stderr,"  Data+ECC blocks: good=%d bad=%d\n",good,bad);
    }
}

int tryrepair(struct tracerec *t);

void doit()
{
    int pass,passresult,totalresult;

    struct tracerec *t;
      
    pass=totalresult=0; 
    do{
	if(vflag) fprintf(stderr,"Pass %d: ",pass++);
	calcgoodbad();
	if(bad==0) break;
	
	passresult=0;
	for(t=traces; t!=NULL; t=t->next)
	{
	    passresult+=tryrepair(t);
	}  
	totalresult+=passresult;
	if(vflag) fprintf(stderr,"\n");

    }while(passresult>0);

    if(vflag) fprintf(stderr,"Final result: %d repairs made, "
		      "%d bad data blocks remaining.\n",totalresult,badd);

    if(rflag) printf("%d\n",badd);

}



unsigned char buf[BLOCK],xorbuf[BLOCK];

void clearxorbuf()
{
 int i;

 for(i=0; i<BLOCK; i++) xorbuf[i]=0;
}

void doxor()
{
 int i;

 for(i=0; i<BLOCK; i++) xorbuf[i]=xorbuf[i]^buf[i];
}

void doxoron(char *b)
{
 int i;

 for(i=0; i<BLOCK; i++) b[i]=b[i]^buf[i];
}

void readblock(int b)
{
    int f;
    int got,totalgot;

    if(b<datalen)
    {
	f=dataf;
	lseek(f,(long)BLOCK*(long)b,SEEK_SET);
    }
    else
    {
	f=ecorf;
	lseek(f,(long)BLOCK*(long)(b-datalen),SEEK_SET); 
    }

    totalgot=0;

    while(totalgot<BLOCK)
    { 
	got=read(f,buf+totalgot,BLOCK-totalgot);
	if(got<=0) 
        { 
          /* some diagnostics in case I get more bug reports */
	  fprintf(stderr,"b=%d datalen=%d totalgot=%d\n",b,datalen,totalgot);
          fater("read");
        }
	totalgot+=got;
    }
}


void writexorblock(int b)
{
    int f;
    int got,totalgot;

    if(b<datalen)
    {
	f=dataf;
	lseek(f,(long)BLOCK*(long)b,SEEK_SET);
	if(makeecc) fater("Internal error");
    }
    else
    {
	f=ecorf;
	lseek(f,(long)BLOCK*(long)(b-datalen),SEEK_SET); 
    }

    totalgot=0;
    while(totalgot<BLOCK)
    { 
	got=write(f,xorbuf+totalgot,BLOCK-totalgot);
	if(got<=0) fater("write");
	totalgot+=got;
    }
    
}

void writeblockfrom(int b,char *bf)
{
    bcopy(bf,xorbuf,BLOCK);
    writexorblock(b);
}


int tryrepair(struct tracerec *t)
{
    int i;
    int faults,faultpos;

    faults=0; faultpos=-1;
    for(i=0; i<t->len; i++) 
    {
	if(map[t->data[i]]=='X') { faults++; faultpos=t->data[i]; }
    }

    if(faults!=1) return(0);

    if(vflag) fprintf(stderr," %d",faultpos);

    map[faultpos]='.';

    if(dflag) return(1); /* dry run */

    clearxorbuf();
    for(i=0; i<t->len; i++) 
    {
	if(t->data[i]!=faultpos) 
	{
	    readblock(t->data[i]);
	    doxor();
	}
    }
    writexorblock(faultpos);
  
    return(1);
}


/* For efficiency when building an ecor file, we have this
   preprocessing phase.  This is much more efficient if linux does not
   have the buffer space to hold the entire datafile in disk cache.

   Here is how it works:

   An ison table is build:

     ison[b] lists all (4) traces the data block b is on

   Then, the data file is read sequentially, by looking at ison the
   appropriate xors are made in the eccblock in RAM.

   All ecc blocks in ram that have exactly the required xors (trace
   length-1) applied to be the correct ecc value are written, and
   have their map entries set to `.'. 
   
   If no bugs occurred, the ecc doit() then only needs to make
   the internal ecc ecc blocks.
*/

/* max number of traces in an isonrec */
#define MTRACE 4

struct isonrec {
 int len;
 struct tracerec *t[MTRACE];
};


void prephase()
{
    int b,i,ecn,el;
    char *eccblock;
    struct tracerec *t;
    struct isonrec *ison;

    if(vflag) fprintf(stderr,"Preprocess phase\n");
    
    /* make ecc data in ram */
    eccblock=malloc(ecorlen*BLOCK);
    if(eccblock==NULL) 
    {
	fprintf(stderr,"ecor: low on virtual memory, using plan B......\n");
	return;
    }	 

    ison=malloc(datalen*sizeof(struct isonrec));
    if(ison==NULL) 
    {
	free(eccblock);
	fprintf(stderr,"ecor: low on virtual memory, using plan B......\n");
	return;
    }	 
    
    /* calculate ison data */
    for(t=traces; t!=NULL; t=t->next)
    {
	for(i=0; i<t->len; i++)
	{
	    el=t->data[i];
	    if( ! (el<datalen) ) continue;
	    if(ison[el].len>=MTRACE) continue;

	    ison[el].t[ison[el].len++]=t;
	}
    }

    /* read datafile and xor */
    for(b=0; b<datalen; b++)
    {
	readblock(b);

	
	for(i=0; i<ison[b].len; i++)
	{
	    t=ison[b].t[i];
	    ecn=t->data[ (t->len)-1 ];
	    doxoron(&eccblock[(ecn-datalen)*BLOCK]);
	    t->numpre++;
	}  

    }

    /* detect and write complete ecc blocks */
    for(t=traces; t!=NULL; t=t->next)
    {	  
	if(t->numpre==(t->len-1))
	{
	    ecn=( t->data[ (t->len)-1 ] );
 	    map[ecn]='.';
	    writeblockfrom(ecn,&eccblock[(ecn-datalen)*BLOCK]);
	}
    }  
    

}

