/* fifo.c */

#include "taper.h"


/* functions used by restore/verify */

PUBLIC void fifo_send_file_with_info (FILE *fdfifo, const struct file_info *fi,
	const char *file_name, const char *temp_name)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_FILE_INFO, fdfifo);
	fputsz (file_name, fdfifo);
	fputsz (temp_name, fdfifo);
	fwrite(fi, sizeof(*fi), 1, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_end (FILE *fdfifo)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_END, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_error (FILE *fdfifo)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_ERROR, fdfifo);
	fflush (fdfifo);
}


/* return value:
 *   1: fatal error, or end
 *   0:
 *	file_name = compressed_name = NULL -> error
 *	file_name != NULL, compressed_name != NULL -> everything ok
 */
PUBLIC int fifo_receive_file_with_info (FILE *fdfifo, struct file_info *fi,
	char **file_name, char **temp_name)
{
	char magic[100];
	static char fn[MAX_FNAME], tn[MAX_FNAME];

	int c;

	if (fgetsz (magic, 100, fdfifo) == NULL)
		goto fifo_error;
	if (strcmp (magic, FIFO_MAGIC) != 0)
		goto fifo_error;

	if ((c = getc (fdfifo)) == EOF)
		goto fifo_error;
	switch (c) {
	  case FIFO_FILE_INFO:
		if (fgetsz (fn, sizeof (fn), fdfifo) == NULL)
			goto fifo_error;
		if (fgetsz (tn, sizeof (tn), fdfifo) == NULL)
			goto fifo_error;
		if (fread (fi, sizeof(*fi), 1, fdfifo) != 1)
			goto fifo_error;
		*file_name = fn;
		*temp_name = tn;
		return 0;

	  case FIFO_END:
		return 1;		/* end */

	  case FIFO_ERROR:
		*file_name = *temp_name = NULL;
		return 0;		/* continue */

	  default:
		goto fifo_error;
	}

fifo_error:
	write_log ("ERROR: invalid FIFO data");
	return 1;			/* end */
}


/* functions used by backup */

PUBLIC void fifo_send_file (FILE *fdfifo, const char *file_name,
	const char *compressed_name)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_FILE, fdfifo);
	fputsz (file_name, fdfifo);
	fputsz (compressed_name, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_error_with_file (FILE *fdfifo, const char *file_name)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_ERROR_WITH_FILE, fdfifo);
	fputsz (file_name, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_disk_full (FILE *fdfifo)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_DISK_FULL, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_child_ok (FILE *fdfifo)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_CHILD_OK, fdfifo);
	fflush (fdfifo);
}


PUBLIC void fifo_send_no_child (FILE *fdfifo)
{
	fputsz (FIFO_MAGIC, fdfifo);
	putc (FIFO_NO_CHILD, fdfifo);
	fflush (fdfifo);
}


/* return value:
 *   1: fatal error, or no child
 *   0: child ok
 */
PUBLIC int fifo_receive_child_status (FILE *fdfifo)
{
	char magic[100];
	int c;

	if (fgetsz (magic, 100, fdfifo) == NULL)
		goto fifo_error;
	if (strcmp (magic, FIFO_MAGIC) != 0)
		goto fifo_error;

	if ((c = getc (fdfifo)) == EOF)
		goto fifo_error;
	switch (c) {
	  case FIFO_CHILD_OK:
		return 0;

	  case FIFO_NO_CHILD:
		return 1;

	  default:
		goto fifo_error;
	}

fifo_error:
	write_log ("ERROR: invalid FIFO data");
	return 1;
}


char fifo_disk_full_c;		/* to make a unique pointer */

/* return value:
 *   1: fatal error
 *   0:
 *	file_name = compressed_name = NULL -> EOF
 *	file_name = fifo_disk_full, compressed_name = NULL -> disk_full
 *	file_name != NULL, file_name != fifo_disk_full,
 *				compressed_name = NULL -> error
 *	file_name != NULL, compressed_name != NULL -> everything ok
 */
PUBLIC int fifo_receive_file (FILE *fdfifo, char **file_name,
	char **compressed_name)
{
	char magic[100];
	static char fn[MAX_FNAME], cn[MAX_FNAME];

	int c;

	if (fgetsz (magic, 100, fdfifo) == NULL) {
		if (ferror (fdfifo))
			return 1;
		*file_name = *compressed_name = NULL;
		return 0;
	}
	if (strcmp (magic, FIFO_MAGIC) != 0)
		goto fifo_error;

	if ((c = getc (fdfifo)) == EOF)
		goto fifo_error;
	switch (c) {
	  case FIFO_FILE:
		if (fgetsz (fn, sizeof (fn), fdfifo) == NULL)
			goto fifo_error;
		if (fgetsz (cn, sizeof (cn), fdfifo) == NULL)
			goto fifo_error;
		*file_name = fn;
		*compressed_name = cn;
		return 0;

	  case FIFO_ERROR_WITH_FILE:
		if (fgetsz (fn, sizeof (fn), fdfifo) == NULL)
			goto fifo_error;
		*file_name = fn;
		*compressed_name = NULL;
		return 0;

	  case FIFO_DISK_FULL:
		*file_name = fifo_disk_full;
		*compressed_name = NULL;
		return 0;

	  default:
		goto fifo_error;
	}

fifo_error:
	write_log ("ERROR: invalid FIFO data");
	return 1;
}
