/* $Id: span.c,v 1.6 1998/03/23 22:57:35 fraserm Exp $
   $Log: span.c,v $
   Revision 1.6  1998/03/23 22:57:35  fraserm
   changed status messages
   added getopt support for -q quiet and -l volume label options
   more minor bugfixes

   Revision 1.5  1998/03/22 20:36:43  fraserm
   moved blocks_this_volume out of struct and into a global -
   fixed incorrect blocks-this-volume bug

   Revision 1.4  1998/03/22 19:29:56  fraserm
   split into mult. files; added filesystem support

   Revision 1.3  1998/03/07 01:33:57  fraserm
   cosmetic bug

   Revision 1.2  1998/03/07 01:13:45  fraserm
   zero-out id-block before writing

   Revision 1.1  1998/03/06 02:29:04  fraserm
   Initial revision

   Revision 1.1  1998/03/06 02:26:21  fraserm
   Initial revision

*/

#include "span.h"

/* span.h comments this stuff */
int quiet = 0;

int bufsize = DEFAULT_BLOCKSIZE;
char *buffer = NULL;

char progname[MAXLINESIZE];
char label[NAMESIZE] = NO_LABEL;

int volume_number = 0;
long bytes_this_volume;

long first_all, first_volume, last, last_written, expected;

struct device in, out;

void usage ()
{
  printf ("%s: usage:\n\t%s [-q] [-l label] device\nor\n\t%s [-q] [-l label] filesystem filename\n",
	  progname, progname, progname);
}

int main (argc, argv)
  int argc;
  char *argv[];
{
  int actual_read, actual_written;
  int read_so_far, written_so_far, end_loop, out_eof, more;
  struct sigaction sigtmr;
  struct device *blockspec, *standard;
  long want_to_read;
  int opt;

  /* copy program name */
  strncpy (progname, argv[0], MAXLINESIZE-1);
  if (strcmp (progname, OUTNAME) == 0) { /* we write */
    blockspec = &out;
    standard = &in;
    in.fd = 0; /* standard input */
  }
  else { /* we read */
    blockspec = &in;
    standard = &out;
    out.fd = 1; /* standard output */
  }
  strncpy (standard->name, 
	   (standard == &in) ? "stdin" : "stdout",
	   NAMESIZE-1);
  /* check options */
  while ((opt = getopt (argc, argv, "l:q")) != EOF) {
    switch (opt) {
    case 'l': /* volume name */
      strncpy (label, optarg, NAMESIZE - 1);
      break;
    case 'q': /* quiet mode */
      quiet = 1;
      break;
    default: /* huh? */
      usage ();
      die ("error on command line", 0, ERR_COMMAND);
      break;
    }
  }
  if (optind+1 > argc) {
    usage ();
    die ("no block device or filesystem&filename", 0, ERR_COMMAND);
  }
  strncpy (blockspec->name, argv[optind], NAMESIZE-1);
  if (optind+2 <= argc) { /* hmm... two arguments, we have a mounted
			     filesystem! */
    strncpy (blockspec->fname, argv[optind+1], NAMESIZE-1);
  }
  else {
    *(blockspec->fname) = '\0';
  }
  in.bytes_so_far = out.bytes_so_far = 0; /* zero byte counts */
  expected = 0;
  do {
    if (buffer != NULL) {
      free (buffer);
    }
    next_volume (blockspec);
    identify (&in);
    identify (&out);
    if ((*(blockspec->fname) == '\0' && blockspec->id != N_BLOCKDEVICE)
	|| (*(blockspec->fname) != '\0' && blockspec->id != N_FILE)) {
      fprintf (stderr, "%s: %s is not a valid file\n",
	       progname, blockspec->name);
      die ("invalid output file", 0, ERR_NOTBLOCK);
    }
    out.size = SIZE_UNKNOWN;
    if (blockspec == &out) { /* we are writing to the block device, so
				alloc the buffer memory here; if the
				block device is being read from, then
				read_id_block() sets the buffer size based
				on what the id-block indicates the
				block size is */
      bufsize = blockspec->blocksize;
      if ((buffer = (char *) malloc (bufsize)) == NULL) {
	die ("failed to allocate buffer space", errno, ERR_ALLOC);
      }
    }
  } while (blockspec == &in
	   && (expected = read_id_block (blockspec, &more)) == -1);
  ++volume_number;
  /* arm signal handler */
  sigtmr.sa_handler = timertick;
  sigtmr.sa_mask = 0;
  sigtmr.sa_flags = SA_RESTART;
  sigaction (SIGALRM, &sigtmr, NULL);
  end_loop = out_eof = 0;
  /* tot_read = tot_written = last_written = 0; */
  if (blockspec == &out) { /* we write a dummy id block */
    write_id_block (blockspec, NO_SEEK, 0); /* just to reserve space, we'll
					       come back and change it
					       later */
  }
  while (!end_loop) {
    if (expected != 0 && (expected - bytes_this_volume) < bufsize) {
      want_to_read = expected - bytes_this_volume;
      if (want_to_read == 0 && !more) {
	end_loop = 1;
      }
    }
    else {
      want_to_read = bufsize;
    }
    read_so_far = 0;
    while (!end_loop && (read_so_far < want_to_read || want_to_read == 0)) {
      actual_read = read_robust (in.fd, buffer + read_so_far,
				 want_to_read - read_so_far);
      if (actual_read == 0) { /* means eof */
	if (bytes_this_volume < expected) { /* hmmm... EOF before we
					       expect, probably a bug in
					       the writing code */
	  fprintf (stderr, "\n%s: warning: expected %ld bytes, end of volume after %ld\n",
		   progname, expected, bytes_this_volume);
	}
	if (blockspec == &in) { /* then we change volumes */
	  do {
	    next_volume (blockspec);
	  } while ((expected = read_id_block (blockspec, &more)) == -1);
	  want_to_read = bufsize;
	  ++volume_number;
	}
	else {
	  end_loop = 1;
	  close_file (&in);
	}
      }
      else if (actual_read == -1) { /* error, or perhaps no data available */
	die ("unrecoverable read error", errno, ERR_CANTREAD);
      }
      else {
        read_so_far += actual_read;
      }
    }
    in.bytes_so_far += read_so_far;
    written_so_far = 0;
    while (written_so_far < read_so_far) {
      actual_written = write_robust (out.fd, buffer, read_so_far);
      if (actual_written <= 0) {
        print_status (0);
	if (blockspec == &out) {
	  if (errno == ENOSPC) {
	    write_id_block (blockspec, YES_SEEK, 1);
	    next_volume (blockspec);
	    write_id_block (blockspec, NO_SEEK, 0);
	    ++volume_number;
	  }
	  else {
	    die ("unrecoverable error while writing blockdevice",
		 errno, ERR_WBLOCK);
	  }
	}
	else {
	  die ("unrecoverable error while writing output", errno, ERR_WOUT);
	}
      }
      else {
        written_so_far += actual_written;
      }
    }
    out.bytes_so_far += written_so_far;
    bytes_this_volume += written_so_far; /* really should use both in. and
					    out., but we only use one,
					    so get over it */
    if (expected != 0 && written_so_far >= expected) { /* set eof if we have
							  read all that we
							  expected, even though
							  we haven't actually
							  reached end of input
							  volume */
      end_loop = 1;
    }
  }
  if (blockspec == &out) {
    write_id_block (blockspec, YES_SEEK, 0);
  }
  print_status (0);
  if (close_file (blockspec) == -1) {
    die ("close of file returned error", errno, ERR_CLOSE);
  }
  print_status (1);
  fprintf (stderr, "%s: %s %ld bytes %s %d volume%s\n",
	   progname,
	   (blockspec == &in) ? "read" : "wrote", out.bytes_so_far,
	   (blockspec == &in) ? "from" : "over",
	   volume_number, (volume_number == 1) ? "" : "s");
  exit (0);
}
