/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  utils.c                                                      */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <strings.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <ntape.h>

extern void print_utils_popup(void);

char searchstring[ARCHIVENAMELENGTH];

#define NO_UTILS_CELLS 4
static char *utils_cell[] = {" Add archives to tape",
			     " Comment focused archive",
			     " Search archive by substring",
			     " Show complete archive list",
			     " Show complete tape list"};
static int chosen_cell = 0;
static WINDOW *utils_popup;
			       

/*************************************************************************/
/* Name:     utils_chosen                                                */
/* Purpose:  Opens the Utilities popup window                            */
/*************************************************************************/
void utils_chosen()
{
    utils_popup = newwin(NO_UTILS_CELLS+3,29,1,12);
    leaveok(utils_popup,TRUE);
    keypad(utils_popup,TRUE);
    print_utils_popup();
    popup_active = 1;
}


/*************************************************************************/
/* Name:     utils_chosen_cell                                           */
/* Purpose:  Executes a menu entry                                       */
/*************************************************************************/
void utils_chosen_cell()
{
    switch(chosen_cell)
    {
    case 0:
	if (tape->id)
	    FileDlg(NTA_ADD);
	else
	{
	    MessageBox("You must have a selected tape!",1,OKCANCEL,0);
	    destroy_utils_popup();
	}
	break;
    case 1:
	if (tape->id)
	    comment_focussed_archive();
	else
	    MessageBox("You must have a selected tape!",1,OKCANCEL,0);
       	destroy_utils_popup();
	break;
    case 2:
	EntryField("Enter search substring:",ARCHIVENAMELENGTH,
		   searchstring,0,searchstring,WITH_SPECIAL_CHARS);
	touchwin(listbox);
	wrefresh(listbox);
	search_list(searchstring,-1);
	destroy_utils_popup();
	break;
    case 3:
	if (tape->id)
	{
	    /* If you just want to look through the complete archive and then
	       wnat to go back to your last active tape this is selected
	       without rewinding it and checking the header if the position
	       matches. Suggested by Ruppo */

	    last_tape_pos = do_tape_command("tell",0);
	    last_tape_id = tape->id;
	    
	    tape->id = 0;
	    init_list();
	}
	destroy_utils_popup();
	break;
    case 4:
	show_all_tapes();
	destroy_utils_popup();
	break;
    default:
	break;
    }
}


/*************************************************************************/
/* Name:     print_utils_popup                                           */
/* Purpose:  prints the contents of the utils-popup                      */
/*************************************************************************/
void print_utils_popup()
{
    int i;
 
    wmove(utils_popup,0,0);
    for (i = 0; i <= NO_UTILS_CELLS; i++)
    {
	if (i == chosen_cell)
	    set_color_pair(utils_popup,INV_MENU);
	else
	    set_color_pair(utils_popup,MENU);
	mvwaddstr(utils_popup,i+1,0,utils_cell[i]);
	winclrtoeol(utils_popup);
    }
    set_color_pair(utils_popup,MENU);
    box(utils_popup,ACS_VLINE,ACS_HLINE);
    touchwin(utils_popup);
    wrefresh(utils_popup);
}


/*************************************************************************/
/* Name:     utils_scroll_down                                           */
/* Purpose:                                                              */
/*************************************************************************/
void utils_scroll_down()
{
    if (chosen_cell == NO_UTILS_CELLS)
	chosen_cell = 0;
    else
	chosen_cell++;
    print_utils_popup();
}


/*************************************************************************/
/* Name:     utils_scroll_up                                             */
/* Purpose:                                                              */
/*************************************************************************/
void utils_scroll_up()
{
    if (chosen_cell == 0)
	chosen_cell = NO_UTILS_CELLS;
    else
	chosen_cell--;
    print_utils_popup();
}


/*************************************************************************/
/* Name:     destroy_utils_popup                                         */
/* Purpose:                                                              */
/*************************************************************************/
void destroy_utils_popup()
{
    delwin(utils_popup);
    popup_active = 0;
    touchwin(tapestatus);
    wrefresh(tapestatus);
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     add_archive_to_tape                                         */
/* Purpose:  Appends a new archive at the end of the tape                */
/*************************************************************************/
void add_archive_to_tape(char *filename,long size,int count)
{
    FILE *fp1,*fp2 = NULL;
    char name[MAXLEN],tarpath[MAXLEN],dummy[MAXLEN];
    WINDOW *archivedlg;
    time_t now;
    struct tm *today;
    int len, c, i, fertig = 0, x = 0, y = 0, active = 0, pf = 0, cat = 0;
    long filesize;
  
    /* Get first entry of the file list for the default archive name, but only if 
       only one element was marked */
    if ((fp1 = fopen(filename,"r")) == NULL)
    {
	fprintf(stderr,"Sorry! Internal error 101! Look in the fucking manual!\n");
	return;
    }
  
    if (fgets(name, MAXLEN-1, fp1) == NULL)
    {
	fclose(fp1);
	return;    /* Empty list */
    }
    name[strlen(name)-1] = '\0';
    fclose(fp1);
    
    if (count > 1)
	strcpy(name,"");

    strncpy(archive->name,name,ARCHIVENAMELENGTH);
#ifdef DEBUG
    fprintf(stderr,"%s %d\n%s %d\n",archive->name,strlen(archive->name),name,strlen(name));
#endif

    archive->size = size;
    archive->tapeno = tape->id;
    strcpy(archive->description,"");
    archive->add_info = add_info;
    
    /* Build a nice dialog box */
    archivedlg = newwin(22,60,2,(COLS-60)/2);
    leaveok(archivedlg,TRUE);
    keypad(archivedlg,TRUE);

    set_color_pair(archivedlg,INFOLABELS);
    winclr(archivedlg);

    set_color_pair(archivedlg,INFOFRAME);
    wmove(archivedlg,0,0);
    winclrtoeol(archivedlg);
    wmove(archivedlg,1,0);
    winclrtoeol(archivedlg);
    wmove(archivedlg,2,0);
    winclrtoeol(archivedlg);
    mvwaddstr(archivedlg,1,0,"                Archive Information");
    wmove(archivedlg,19,0);
    winclrtoeol(archivedlg);
    wmove(archivedlg,20,0);
    winclrtoeol(archivedlg);
    wmove(archivedlg,21,0);
    winclrtoeol(archivedlg);

    while (!fertig)
    {
	/* Setup buttons and titles */
	if (active == 0)
	    set_color_pair(archivedlg,INV_INFOLABELS);
	else
	    set_color_pair(archivedlg,INFOLABELS);
	mvwaddstr(archivedlg,5,1,"Archive name:");

	if (active == 1)
	    set_color_pair(archivedlg,INV_INFOLABELS);
	else
	    set_color_pair(archivedlg,INFOLABELS);
	mvwaddstr(archivedlg,7,1,"Platform:    ");

	if (active == 2)
	    set_color_pair(archivedlg,INV_INFOLABELS);
	else
            set_color_pair(archivedlg,INFOLABELS);
	mvwaddstr(archivedlg,9,1,"Category:    ");
	
	if (active == 3)
	    set_color_pair(archivedlg,INV_INFOLABELS);
	else
	    set_color_pair(archivedlg,INFOLABELS);
	mvwaddstr(archivedlg,11,1,"Notes:       ");
      
	if (active == 4)
	    set_color_pair(archivedlg,INV_INFOLABELS);
	else
	    set_color_pair(archivedlg,INFOLABELS);
	mvwaddstr(archivedlg,17,1,"Save contents:");

	if (active == 5)
	    set_color_pair(archivedlg,INV_INFOFRAME);
	else
	    set_color_pair(archivedlg,INFOFRAME);
	mvwaddstr(archivedlg,20,5,"   OK   ");
      
	if (active == 6)
	    set_color_pair(archivedlg,INV_INFOFRAME);
	else
	    set_color_pair(archivedlg,INFOFRAME);
	mvwaddstr(archivedlg,20,49," Cancel ");
      
	/* Setup Window-Text */
	set_color_pair(archivedlg,INFOFIELDS);

	mvwaddstr(archivedlg,5,16,"                                        ");
	mvwaddstr(archivedlg,5,16,archive->name);
	if ((active == 0) && (strlen(archive->name) < ARCHIVENAMELENGTH-1))
	    waddch(archivedlg,'_');

	mvwaddstr(archivedlg,7,16,"                    ");
	mvwaddstr(archivedlg,7,16,platform[pf]);

	mvwaddstr(archivedlg,9,16,"                    ");
	mvwaddstr(archivedlg,9,16,category[cat]);

	mvwaddstr(archivedlg,17,16,archive->add_info & CONTENTS_SAVED ? "Yes" : "No ");

	for (i = 11; i < 16; i++)
	    mvwaddstr(archivedlg,i,16,"                                        ");
	wmove(archivedlg,11,16);
	for (i = 0; archive->description[i] != '\0'; i++)
	{
	    if (archive->description[i] == '@')
	    {
		getyx(archivedlg,y,x);
		wmove(archivedlg,y+1,16);
	    }
	    else
		waddch(archivedlg,archive->description[i]);
	}
	getyx(archivedlg,y,x);

	if (active == 3)
	{
	    if (x < 56)
		waddch(archivedlg,'_');
	    else
		wmove(archivedlg,y,x+1);
	}

	touchwin(archivedlg);
	wrefresh(archivedlg);

	c = getch();
	if (c == TAB_KEY || c == KEY_RIGHT)
	{
	    if (active == 6)
		active = 0;
	    else
		active++;
	}
	else if (c == KEY_LEFT)
	{
	    if (active == 0)
		active = 6;
	    else
		active--;
	}
	else if (c == KEY_UP)
	{
	    if (active == 1)
	    {
		pf--;
		if (pf < 0)
		    pf = no_platforms - 1;
	    }
	    else if (active == 2)
	    {
		cat--;
		if (cat < 0)
		    cat = no_categories - 1;
	    }
	    else if (active == 4)
		archive->add_info = 1 - archive->add_info;
	}
	else if (c == KEY_DOWN)
	{
	    if (active == 1)
	    {
		pf++;
		if (pf == no_platforms)
		    pf = 0;
	    }
	    else if (active == 2)
	    {
		cat++;
		if (cat == no_categories)
		    cat = 0;
	    }
	    else if (active == 4)
		archive->add_info = 1 - archive->add_info;
	}
	else if (c == CR_KEY)
	{
	    if (active == 3)
	    {
		getyx(archivedlg,y,x);
		if (y == 15)
		    continue;
		c = '@';
		goto cr;
	    }
	    else if (active == 5)
	    {
		strcpy(archive->pf,platform[pf]);
		strcpy(archive->category,category[cat]);
		fertig = 1;
	    }
	    else if (active == 6)
	    {
		strcpy(archive->name,"");
		fertig = 1;
	    }
	  
	}
	else if (c == BS_KEY1 || c == BS_KEY2 || c == KEY_BACKSPACE)
	{
	    if (active == 0)
	    {
		len = strlen(archive->name);
		if (len > 0)
		    archive->name[--len] = '\0';
	    }
	    if (active == 3)
	    {
		len = strlen(archive->description);
		if (len > 0)
		    archive->description[--len] = '\0';
	    }
	}
	else if (strchr(VALIDCHARSTR,c))
	{
	    if (active == 0)
	    {
		len = strlen(archive->name);
		if (len < ARCHIVENAMELENGTH-1)
		{
		    archive->name[len] = c;
		    archive->name[len+1] = '\0';
		}
	    }
	    else if (active == 3)
	    {
	cr:	      len = strlen(archive->description);
		getyx(archivedlg,y,x); /* Is still the right position */
		if (x < 57 || c == '@')
		{
		    archive->description[len] = c;
		    archive->description[len+1] = '\0';
		}
	    }
	}
    }
    if (!strcmp(archive->name,""))
	goto end;

    if (watch_tape)
    {
	create_watch_win(" Add Archive Log ");
	print_watch_win("Checking if tape is online...");
    }
    /* Just look if the tape is online and write protected */
    if (offline())
    {
	print_watch_win("\nTape offline! That's bad!");
        destroy_watch_win();
	print_footer("Sorry! Tape is offline! No archive added!");
	goto end;
    }
    print_watch_win("Ok.\nChecking if tape is writeable...");
    if (wr_prot())
    {
	print_watch_win("\nTape write protected! That's bad!");
        destroy_watch_win();
	print_footer("Sorry your tape is write protected! No archive added!");
	goto end;
    }
    
    /* Get today's date */
    time(&now);
    today = localtime(&now);
    sprintf(archive->date,"%d %d 19%d",today->tm_mday,
	    today->tm_mon+1,today->tm_year);

    /* Erase whitespaces at the end and do toupper */
    for (i = strlen(archive->name)-1; 
	 archive->name[i] == '\t' || archive->name[i] == ' '; i--)
	archive->name[i] = '\0';
    for (i = 0; i < strlen(archive->name); i++)
	archive->name[i] = toupper(archive->name[i]);

    /* Go to end of tape. The end of tape position is now stored in the
       tape struct. This should be very fast. */
    if (!watch_tape)
	print_footer("Looking for logical end of tape...");
    if (do_tape_command("tell",0) < tape->eot)
    {
	sprintf(dummy,"Ok.\nFast move to block %ld ...",tape->eot);
	print_watch_win(dummy);
	if (do_tape_command("seek",tape->eot) < 0)
	{
	    print_watch_win("Error!");
	    destroy_watch_win();
	    print_footer("There may be differences between database and tape!");
	    goto end;
	}
    }
    print_watch_win("Ok.\nMove to the end of tape data...");
    if (do_tape_command("eod",0) < 0)
    {
	print_watch_win("Error!");
	destroy_watch_win();
	print_footer("Error on accessing tape!");
	goto end;
    }
    print_watch_win("Ok.\nGet the start position for the new archive...");
    if ((archive->position = do_tape_command("tell",0)) < 0)
    {
	print_watch_win("Error!");
	destroy_watch_win();
	print_footer("Error on accessing tape!");
	goto end;
    }
    sprintf(dummy,"Block %ld. Ok.\n",archive->position);
    print_watch_win(dummy);
    
#ifdef DEBUG
    fprintf(stderr,"Name: %s\tSize: %ld\tPosition: %ld\nPlatform: %s\tTape-ID: %ld\nDate: %s\n%s\n",
	    archive->name,archive->size,archive->position,archive->pf,
	    archive->tapeno,archive->date,archive->description);
#endif

    /* Now write the archive on tape */
    get_pwd(tarpath);
    sprintf(name,"cd %s ; tar -cpvvV \"%s\" -T %s", 
	    tarpath, archive->name,filename);
    sprintf(dummy,"Now writing archive to tape executing '%s' ...\n",name);
    print_watch_win(dummy);
    
#ifdef DEBUG
    fprintf(stderr,"Executing '%s'...\n",name);
#endif

    if (!watch_tape)
	print_footer("Writing archive to tape...");
    if ((fp1 = popen(name,"r")) == NULL)
    {
	print_footer("Error on writing archive to tape!");
	goto end;
    }
    if (archive->add_info & CONTENTS_SAVED)
    {
	if ((fp2 = fopen(contents_db,"a")) == NULL)
	{
	    print_footer("Sorry! Could not save contents!");
	    archive->add_info &= ~CONTENTS_SAVED;
	}
	fprintf(fp2,"%ld %ld %s\n",archive->tapeno,archive->position,archive->name);
    }
    while (fgets(name, MAXLEN-1, fp1) != NULL)
    {
	print_watch_win(name);
	if (archive->add_info & CONTENTS_SAVED)
	{
	    if (name[0] != TAR_VOLUME_ID)
	    {
		sscanf(name,"%s %s %ld %s %s %s %s %s",dummy,dummy,&filesize,
		       dummy,dummy,dummy,dummy,tarpath);
		fprintf(fp2,"%ld %s\n",filesize,tarpath);
	    }
	}
    }
    
    if (archive->add_info & CONTENTS_SAVED)
    {
	fprintf(fp2,"\n");
	fclose(fp2);
    }
    pclose(fp1);

    tape->no_archives++;

    print_watch_win("Adding new archive to database...");
    
    backup_database(BACKUP_ARCHIVE);
    if ((fp1 = fopen(archive_db,"a")) == NULL)
    {
	print_watch_win("Error.");
	destroy_watch_win();
	print_footer("Error opening database!");
	goto end;
    }
    fprintf(fp1,"%s\n%ld\n%ld\n%ld\n%u\n%s\n%s\n%s\n%s\n",archive->name,
	    archive->tapeno,archive->size,archive->position,
	    archive->add_info,archive->pf,archive->category,
	    archive->date,archive->description);
    fclose(fp1);

    print_watch_win("Ok.\nUpdating end-of-data field in tape database...");
    
    if (change_eot() == 1)
	print_watch_win("Ok.");
    else
	print_watch_win("Error.");
    
    destroy_watch_win();
    
    init_list();
    search_list(archive->name,archive->position);

    sprintf(name,"Archive %s successfully added to tape!",archive->name);
    print_footer(name);
end: 
    unlink(filename);
    delwin(archivedlg);
}


/*************************************************************************/
/* Name:     extract_archive                                             */
/* Purpose:  Extracts the whole archive                                  */
/*************************************************************************/
void extract_archive()
{
    char cmd[MAXLEN];
    FILE *fp;
    
#ifdef DEBUG
    get_pwd(cmd);
    fprintf(stderr,"Extract in %s\n",cmd);
#endif

    touchwin(listbox);
    wrefresh(listbox);
    touchwin(infobox);
    wrefresh(infobox);
    if (watch_tape)
    {
	create_watch_win(" Extract Archive Log ");
	print_watch_win("Checking if tape is online...");
    }
    if (offline())
    {
	print_watch_win("\nTape offline! That's bad!");
	destroy_watch_win();
	print_footer("Sorry! Tape is offline! Nothing extracted!");
	return;
    }

    sprintf(cmd,"Ok.\nSearching archive at block %ld...",archive->position);
    if (watch_tape)
	print_watch_win(cmd);
    else
	print_footer("Searching archive...");
    if (do_tape_command("seek",archive->position) < 0)
    {
	print_watch_win("Not found");
	destroy_watch_win();
	print_footer("Cannot find archive! Nothing extracted!");
	return;
    }

    if (watch_tape)
	print_watch_win("Ok.\nExtracting archive using 'tar xv'...\n");
    else
	print_footer("Extracting archive...");

    if ((fp = popen("tar xv 2>&1","r")) == NULL)
    {
        print_watch_win("Tape read error!\n");
	destroy_watch_win();
	print_footer("Error on extracting archive!");
	return;
    }
    
    while (fgets(cmd, MAXLEN-1, fp) != NULL)
	print_watch_win(cmd);
    
    pclose(fp);
    
    destroy_watch_win();
    sprintf(cmd,"Archive %s successfully extracted!",archive->name);
    print_footer(cmd);
}


/*************************************************************************/
/* Name:     extract_selected_archive                                    */
/* Purpose:  Extracts a selected part of archive                         */
/*************************************************************************/
void extract_selected_archive(int mode)
{
    FILE *fp,*fp2 = NULL;
    char cmd[MAXLEN],name[MAXLEN],dummy[MAXLEN];
    long size,tapeid,pos;
    int emptyline,found;
    int save_contents_now = 0;
            
#ifdef DEBUG
    get_pwd(cmd);
    fprintf(stderr,"Extract selected in %s\n",cmd);
#endif
    if (archive->add_info & CONTENTS_SAVED)
    {
	print_footer("Getting archive contents from file. This may take a while...");
	
	if ((fp = fopen(contents_db,"r")) == NULL)
	{
	    print_footer("Can't read contents file! Reading from tape instead!");
	    /* Should I unmark the "contents-saved-flag" in this case? I don't! */
	    /*
	    change_info(CHANGE_ADDINFO_NOTSAVED,archive->name,archive->tapeno,
			archive->position,"","","");
	    */
	    goto tape;
	}

	emptyline = 1;
	found = 0;
	while (fgets(cmd, MAXLEN-1, fp) != NULL)
	{
	    cmd[strlen(cmd) - 1] = '\0';
	    if (emptyline)
	    {
#ifdef DEBUG
		fprintf(stderr,"Analyzing the line with the archive header...\n");
#endif
		tapeid = atol(strtok(cmd," "));
		pos = atol(strtok(NULL," "));
#ifdef DEBUG
		fprintf(stderr,"Position: %ld\tTape ID: %ld\n",pos,tapeid);
#endif
		if (tapeid == archive->tapeno && pos == archive->position)
		    found = 1;
		emptyline = 0;
		continue;
	    }
	    if (strlen(cmd) == 0)
		if (found)
		    break;
		else
		    emptyline = 1;
	    else
		emptyline = 0;
	    if (found)
	    {
		sscanf(cmd,"%ld %s",&size,name);
		insert_tar_list(name,size);
	    }
	}
	fclose(fp);

	/* If the archive is not found in the contents database then we
	   should better unmark the contents-saved-flag */
	if (!found)
	{
	    change_info(CHANGE_ADDINFO_NOTSAVED,archive->name,archive->tapeno,
                        archive->position,"","","");
	    print_footer(
		"Hmm... Nothing found in the contents file although the flag is marked...");
	    goto tape_n_save;
	}
    }
    else
    {
tape_n_save:
	if (mode == JUST_VIEW)
	    goto tarlistbox_end;
	
	if (MessageBox("Save the contents of archive ?",2,YESNO,1))
	    save_contents_now = 1;
	touchwin(listbox);
	wrefresh(listbox);
	touchwin(infobox);
	wrefresh(infobox);
tape:
	if (mode == JUST_VIEW)
	    goto tarlistbox_end;
	
	if (watch_tape)
	{
	    create_watch_win(" Archive Contents Log ");
	    print_watch_win("Checking if tape is online...");
	}
	if (offline())
	{
	    print_watch_win("\nTape offline! That's bad!");
	    destroy_watch_win();
	    print_footer("Sorry! Tape is offline! Cannot read tape contents!");
	    return;
	}

	sprintf(cmd,"Ok.\nSearching archive at block %ld...",archive->position);
	if (watch_tape)
	    print_watch_win(cmd);
	else
	    print_footer("Searching archive...");
	if (do_tape_command("seek",archive->position) < 0)
	{
	    print_watch_win("Not found");
	    destroy_watch_win();
	    print_footer("Can't find archive! Nothing extracted!");
	    return;
	}

	if (watch_tape)
	    print_watch_win("Ok.\nReading archive contents using 'tar tv'...\n");
	else
	    print_footer("Read contents...");
		
	if ((fp = popen("tar tv","r")) == NULL)
	{
	    print_watch_win("Tape read error!\n");
	    destroy_watch_win();
	    print_footer("Can't get archive contents!");
	    return;
	}

	if (save_contents_now)
	{
	    if ((fp2 = fopen(contents_db,"a")) == NULL)
	    {
		print_footer("Sorry! Could not save contents!");
		save_contents_now = 0;
	    }
	    fprintf(fp2,"%ld %ld %s\n",archive->tapeno,archive->position,archive->name);
	}
		
	while (fgets(cmd, MAXLEN-1, fp) != NULL)
	{
	    print_watch_win(cmd);
	    if (cmd[0] != TAR_VOLUME_ID) /* Volume */
	    {
		sscanf(cmd,"%s %s %ld %s %s %s %s %s",dummy,dummy,&size,
		       dummy,dummy,dummy,dummy,name);
		insert_tar_list(name,size);
		if (save_contents_now)
		    fprintf(fp2,"%ld %s\n",size,name);
	    }
	}
	pclose(fp);
	if (save_contents_now)
	{
	    fprintf(fp2,"\n");
	    fclose(fp2);
            change_info(CHANGE_ADDINFO_SAVED,archive->name,archive->tapeno,
                        archive->position,"","","");
	}
    }
    destroy_watch_win();
    handle_tar_listbox(mode);
tarlistbox_end:
    return;
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
