/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  tapelist.c                                                   */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <stdlib.h>
#include <time.h>
#include <strings.h>
#include <unistd.h>
#include <ntape.h>

#define WIDTH 51
#define HEIGHT 14

void draw_tape_list(void);
void highlight_tape_list_element(int,int);
int remove_tape_from_db(long,char *);

/* Tape list */
static tape_list_el *tape_head;
static WINDOW *tapelist;
static int clear,no_items,tape_first, tape_focus, active;

/*************************************************************************/
/* Name:     init_tape_list                                              */
/* Purpose:  Init the list of tapes and read tape database file.         */
/*************************************************************************/
void init_tape_list()
{
    FILE *fp;
    char line_id[MAXLEN],line_name[MAXLEN],line_eot[MAXLEN];
    tape_list_el *element,*laufelement;

    if ((fp = fopen(tape_db,"r")) == NULL)
    {
	print_footer("Error opening tape database!");
	nta_exit(-1);
    }

    if (!check_database_version(TAPE_DATABASE,fp,NULL))
	nta_exit(-1);
    
    no_items = 0;
    while (fgets(line_id, MAXLEN-1, fp) != NULL)
    {
	line_id[strlen(line_id) - 1] = '\0';
	if (fgets(line_name, MAXLEN-1, fp) == NULL)
        {
	    print_footer("Error in tape database!");
	    nta_exit(-1);
        }
	line_name[strlen(line_name) - 1] = '\0';
        if (fgets(line_eot, MAXLEN-1, fp) == NULL)
        {
            print_footer("Error in tape database!");
            nta_exit(-1);
        }
        line_eot[strlen(line_eot) - 1] = '\0';
      
	if ((element = (tape_list_el *)malloc(sizeof(tape_list_el))) == NULL)
	{
	    print_footer("Not enough memory. Sorry...");
	    nta_exit(-1);
	}
      
	strcpy(element->name,line_name);
	element->id = atol(line_id);
	element->eot = atol(line_eot);
	element->next = NULL;
	no_items++;
	
	if (tape_head == NULL)
	    tape_head = element;
	else
	{
	    laufelement = tape_head;
	    while (laufelement->next != NULL)
		laufelement = laufelement->next;
	    laufelement->next = element;
	}
    }
    fclose(fp);

#ifdef DEBUG
    laufelement = tape_head;
    while (laufelement != NULL)
    {
	fprintf(stderr,"init_tape_list: %ld %s %ld\n",laufelement->id,
		laufelement->name,laufelement->eot);
	laufelement = laufelement->next;
    }
#endif
}


/*************************************************************************/
/* Name:     append_current_tape                                         */
/* Purpose:  appends the current item if you've just created it          */
/*************************************************************************/
void append_current_tape()
{
    tape_list_el *element,*le;
  
    if ((element = (tape_list_el *)malloc(sizeof(tape_list_el))) == NULL)
    {
	print_footer("Not enough memory! Sorry...");
	nta_exit(-1);
    }
  
    strcpy(element->name,tape->name);
    element->id = tape->id;
    element->eot = tape->eot;
    element->next = NULL;
  
    if (tape_head == NULL)
	tape_head = element;
    else
    {
	le = tape_head;
	while (le->next != NULL)
	    le = le->next;
	le->next = element;
    }
    no_items++;
}


/*************************************************************************/
/* Name:     get_tapename_from_id                                        */
/* Purpose:  name says it all                                            */
/*************************************************************************/
char *get_tapename_from_id(int id,long *eot)
{
    tape_list_el * tapele;
    char *tapename = UNKNOWN;

    tapele = tape_head;
    while (tapele != NULL)
    {
	if (tapele->id == id)
	{
	    tapename = tapele->name;
	    if (eot != NULL)
		*eot = tapele->eot;
	    break;
	}
	tapele = tapele->next;
    }
    return(tapename);
}


/*************************************************************************/
/* Name:     show_all_tapes                                              */
/* Purpose:  Shows all tapes stored in database with remove option       */
/*************************************************************************/
void show_all_tapes()
{
    tape_list_el *p,*q;
    int one_marked = 0,i,c,fertig = 0;

    if (tape_head == NULL)
    {
	print_footer("No registered tapes available. You have to create one at first.");
	return;
    }
    
    tapelist = newwin(HEIGHT,WIDTH,5,(COLS-WIDTH)/2);
    leaveok(tapelist,TRUE);
    keypad(tapelist,TRUE);
    
    tape_first = tape_focus = 0;
    active = 0;
    for (p = tape_head; p != NULL; p = p->next)
	p->marked = 0;
    
    clear = 1;
    draw_tape_list();
    while(!fertig)
    {
	c = getch();
	switch(c)
	{
	case KEY_DOWN:
	    if (active == 0)
	    {
	key_down: 
		clear = 1;
		scroll_down(&tape_focus,&tape_first,HEIGHT - 6,
			    no_items,&highlight_tape_list_element,
			    &draw_tape_list);
	    }
	    break;
	case KEY_UP:
	    if (active == 0)
	    {
		clear = 1;
		scroll_up(&tape_focus,&tape_first,HEIGHT - 6,
			  no_items,&highlight_tape_list_element,
			  &draw_tape_list);
	    }
	    break;
	case KEY_NPAGE:
	    if (active == 0)
	    {
		clear = 1;
		page_down(&tape_focus,&tape_first,HEIGHT - 6,
			  no_items,&highlight_tape_list_element,
			  &draw_tape_list);
	    }
	    break;
	case KEY_PPAGE:
	    if (active == 0)
	    {
		clear = 1;
		page_up(&tape_focus,&tape_first,HEIGHT - 6,
			no_items,&highlight_tape_list_element,
			&draw_tape_list);
	    }
	    break;
	case KEY_A1:
	case KEY_HOME:
	    if (active == 0)
	    {
		clear = 1;
		top_pos(&tape_focus,&tape_first,HEIGHT - 6,
			no_items,&highlight_tape_list_element,
			&draw_tape_list);
	    }
	    break;
	case KEY_C1:
	case KEY_END:
	    if (active == 0)
	    {
		clear = 1;
		bottom_pos(&tape_focus,&tape_first,HEIGHT - 6,
			   no_items,&highlight_tape_list_element,
			   &draw_tape_list);
	    }
	    break;
	case TAB_KEY:
	case KEY_RIGHT:
	    if (active == 2)
		active = 0;
	    else
		active++;
	    clear = 1;
	    draw_tape_list();
	    break;
	case KEY_LEFT:
	    if (active == 0)
		active = 2;
	    else
		active--;
	    clear = 1;
	    draw_tape_list();
	    break;
	case SPACE_KEY:
	    if (active == 0)
	    {
		for (p = tape_head, i = 0; p != NULL && i != tape_focus; i++)
		    p = p->next;
		p->marked = 1 - p->marked;
		highlight_tape_list_element(-1,tape_focus);
		/*clear = 0;
		show_tar_list();*/
		goto key_down;
	    }
	case CR_KEY:
	    if (active == 2)
		fertig = 1;
	    if (active == 1)
	    {
		one_marked = 0;
		p = tape_head;
		while (p != NULL)
		{
		    q = p->next; /* p could be freed in remove_tape_from_db */
		    if (p->marked)
		    {
			one_marked = 1;
			fertig = 1;
			if (remove_tape_from_db(p->id,p->name))
			    one_marked = 2;
			touchwin(listbox);
			wrefresh(listbox);
			touchwin(tapelist);
			wrefresh(tapelist);
		    }
		    p = q;
		}
		if (one_marked == 0)
		    print_footer("No tape marked! Nothing needs to be done!");
		if (one_marked == 2)
		    init_list();
	    }
	    break;
	}
    }
    delwin(tapelist);
}


/*************************************************************************/
/* Name:     draw_tape_list                                              */
/* Purpose:  Draws the tape list                                         */
/*************************************************************************/
void draw_tape_list()
{
    int i;
    tape_list_el *p;
    char line[MAXLEN];
    struct tm *date;
    
    if (clear)
    {
	set_color_pair(tapelist,TAPELIST);
	winclr(tapelist);
    }
    
    set_color_pair(tapelist,TAPELISTHEADER);
    wmove(tapelist,0,0);
    winclrtoeol(tapelist);
    mvwaddstr(tapelist,1,0,"              List of available tapes");
    winclrtoeol(tapelist);
    wmove(tapelist,2,0);
    winclrtoeol(tapelist);
    
    p = tape_head;
    for (i = 0; i < tape_first; i++)
	p = p->next;
    
    for (i = 3; i < HEIGHT - 3; i++)
    {
	if (p->marked)
	    set_color_pair(tapelist,TAPELISTSEL);
	else
	    set_color_pair(tapelist,TAPELIST);
	mvwaddstr(tapelist,i,1,p->name);
	date = localtime(&(p->id));
	sprintf(line,"%02d.%02d.%02d %02d:%02d:%02d  %07ld",date->tm_mday,
		date->tm_mon+1,date->tm_year,date->tm_hour,
		date->tm_min,date->tm_sec,p->eot);
	mvwaddstr(tapelist,i,24,line);
	p = p->next;
	if (p == NULL)
	    break;
    }
    
    set_color_pair(tapelist,TAPELISTHEADER);
    wmove(tapelist,HEIGHT - 3,0);
    winclrtoeol(tapelist);
    wmove(tapelist,HEIGHT - 2,0);
    winclrtoeol(tapelist);
    wmove(tapelist,HEIGHT - 1,0);
    winclrtoeol(tapelist);
    
    if (active == 1)
	set_color_pair(tapelist,INV_TAPELISTBTN);
    else
	set_color_pair(tapelist,TAPELISTBTN);
    mvwaddstr(tapelist,HEIGHT - 2,4," Remove ");
    if (active == 2)
	set_color_pair(tapelist,INV_TAPELISTBTN);
    else
	set_color_pair(tapelist,TAPELISTBTN);
    mvwaddstr(tapelist,HEIGHT - 2,39," Cancel ");
    
    if (no_items)
	highlight_tape_list_element(-1,tape_focus);
    
    touchwin(tapelist);
    wrefresh(tapelist);
}

/*************************************************************************/
/* Name:     hilight_tape_list_element                                   */
/* Purpose:  Inverts the new and old element.                            */
/*************************************************************************/
void highlight_tape_list_element(int old_element,int new_element)
{
    char line[MAXLEN];
    int i;
    tape_list_el *p;
    
    if (active == 0)
    {
	if (new_element >= 0)
	{
	    for (p = tape_head, i = 0; p != NULL && i != new_element; i++)
                p = p->next;
	    if (p->marked)
		set_color_pair(tapelist,INV_TAPELISTSEL);
	    else
		set_color_pair(tapelist,INV_TAPELIST);
	    
	    for (i = 0; i < WIDTH; i++)
		line[i] = mvwinch(tapelist,new_element - tape_first + 3,i);
	    line[i] = '\0';
	    mvwaddstr(tapelist,new_element - tape_first + 3,0,line);
	}
	if (old_element >= 0)
	{
	    for (p = tape_head, i = 0; p != NULL && i != old_element; i++)
		p = p->next;
	    if (p->marked)
		set_color_pair(tapelist,TAPELISTSEL);
	    else
		set_color_pair(tapelist,TAPELIST);
	    for (i = 0; i < WIDTH; i++)
		line[i] = mvwinch(tapelist,old_element - tape_first + 3,i);
	    line[i] = '\0';
	    mvwaddstr(tapelist,old_element - tape_first + 3,0,line);
	}   
	touchwin(tapelist);
	wrefresh(tapelist);
    }
}


/*************************************************************************/
/* Name:     remove_tape_from_db                                         */
/* Purpose:  Removes tape and all corresponding things from databases.   */
/*************************************************************************/
int remove_tape_from_db(long id,char *name)
{
    char msg[MAXLEN],destname[MAXLEN],line[MAXLEN],msg2[MAXLEN];
    FILE *source,*target;
    int found,emptyline;
    long tapeid;
    tape_list_el *p,*q;
    
    sprintf(msg,"Remove tape '%s' from databases??",
	    name);
    if (!MessageBox(msg,2,YESNO,1))
	return(0);
    sprintf(msg,"Erasing all data connected with tape '%s' ...",name);
    print_footer(msg);

    /*** Delete all from contents database ***/
    if ((source = fopen(contents_db,"r")) == NULL)
    {
	print_footer("Can't open contents database. Never mind!");
	goto archive_db;
    }
    sprintf(destname,"%s~",contents_db);
    if ((target = fopen(destname,"w")) == NULL)
    {
	print_footer("Can't write temporary contents database. Never mind!");
	fclose(source);
	goto archive_db;
    }
    emptyline = 1;
    found = 0;
    while (fgets(msg, MAXLEN-1, source) != NULL)
    {
	msg[strlen(msg) - 1] = '\0';
	if (emptyline)
	{
	    tapeid = atol(msg);
	    emptyline = 0;
	    if (tapeid == id)
	    {
		found = 1;
		continue;
	    }
	    goto loop1;
	}
	if (strlen(msg) == 0) /* leere Zeile */
	{
	    emptyline = 1;
	    if (found)
	    {
		found = 0;
		continue;
	    }
	}
	if (found)
	    continue;
loop1:	fprintf(target,"%s\n",msg);
    }
    fclose(target);
    fclose(source);

    unlink(contents_db);
    rename(destname,contents_db);
    
archive_db:
    /*** Delete all from archive database ***/
    sprintf(destname,"%s~",archive_db);
    rename(archive_db,destname);
    if ((source = fopen(destname,"r")) == NULL)
    {
	print_footer("Can't open archive database. That's really bad!");
	return(0);
    }
    if ((target = fopen(archive_db,"w")) == NULL)
    {
	print_footer("Can't write temporary archive database. That's really bad!");
	fclose(source);
	return(0);
    }
    if (!check_database_version(ARCHIVE_DATABASE,source,target))
    {
	print_footer("Wrong database version! That's really bad!");
	fclose(target);
	fclose(source);
	return(0);
    }
    
    while (fgets(line, MAXLEN-1, source) != NULL)   /* name */
    {
	if (fgets(msg, MAXLEN-1, source) == NULL)   /* tape id */
	    goto error;
	if (atoi(msg) == id)
	    found = 1;
	else
	    found = 0;
	if (!found)
	    fprintf(target,"%s%s",line,msg);
	if (fgets(line, MAXLEN-1, source) == NULL)  /* size */
	    goto error;
	if (fgets(msg, MAXLEN-1, source) == NULL)   /* position */
	    goto error;
	if (!found)
	    fprintf(target,"%s%s",line,msg);
        if (fgets(line, MAXLEN-1, source) == NULL)  /* additional info */
            goto error;
        if (fgets(msg, MAXLEN-1, source) == NULL)   /* platform */
            goto error;
	if (!found)
	    fprintf(target,"%s%s",line,msg);
        if (fgets(line, MAXLEN-1, source) == NULL)  /* category */
            goto error;
        if (fgets(msg, MAXLEN-1, source) == NULL)   /* date */
            goto error;
	if (!found)
	    fprintf(target,"%s%s",line,msg);
        if (fgets(line, MAXLEN-1, source) == NULL)  /* description */
            goto error;
	if (!found)
	    fprintf(target,"%s",line);
    }
    fclose(target);
    fclose(source);

    /*** Delete all from tape database ***/
    sprintf(destname,"%s~",tape_db);
    rename(tape_db,destname);
    if ((source = fopen(destname,"r")) == NULL)
    {
	print_footer("Can't open tape database. That's really bad!");
	return(0);
    }
    if ((target = fopen(tape_db,"w")) == NULL)
    {
	print_footer("Can't write temporary tape database. That's really bad!");
	fclose(source);
	return(0);
    }
    if (!check_database_version(TAPE_DATABASE,source,target))
    {
        print_footer("Wrong database version! That's really bad!");
        fclose(target);
        fclose(source);
        return(0);
    }
 
    while (fgets(line, MAXLEN-1, source) != NULL)   /* tape id */
    {
	if (atol(line) == id)
	    found = 1;
	else
	    found = 0;
	if (fgets(msg, MAXLEN-1, source) == NULL)   /* tape name */
	    goto error;
	if (fgets(msg2, MAXLEN-1, source) == NULL)   /* end of tape */
	    goto error;
	if (!found)
	    fprintf(target,"%s%s%s",line,msg,msg2);
    }
    fclose(target);
    fclose(source);

    /* Delete entry from intern list */
    for (p = tape_head, q = NULL; p != NULL && p->id != id; q = p, p = p->next)
	;
    
    if (p != NULL)
    {
	if (q == NULL)
	{
	    tape_head = p->next;
	    free(p);
	}
	else
	{
	    q->next = p->next;
	    free(p);
	}
	no_items--;
	if (tape->id == id)
	    tape->id = 0;
	last_tape_pos = 0;
    }
    return(1);
    
error:
    print_footer("Serious error in database. You have to repair this manually!");
    fclose(target);
    fclose(source);
    return(0);
}


/*************************************************************************/
/* Name:     change_eot                                                  */
/* Purpose:  updates the end-of-tape field after an archive is appended  */
/*************************************************************************/
int change_eot()
{
    int found;
    char srcname[MAXLEN],line[MAXLEN];
    FILE *source,*target;
    tape_list_el *p;

    tape->eot = do_tape_command("tell",0);
    if (tape->eot < 0)
        tape->eot = 0;
    sprintf(srcname,"%s~",tape_db);
    rename(tape_db,srcname);
    if ((source = fopen(srcname,"r")) == NULL)
    {
        print_footer("Can't open tape database. That's really bad!");
        return(0);
    }
    if ((target = fopen(tape_db,"w")) == NULL)
    {
        print_footer("Can't write temporary tape database. That's really bad!");
        fclose(source);
        return(0);
    }
    if (!check_database_version(TAPE_DATABASE,source,target))
    {
        print_footer("Wrong database version! That's really bad!");
        fclose(target);
        fclose(source);
        return(0);
    }
    while (fgets(line, MAXLEN-1, source) != NULL)  /* tape id */
    {
        if (atol(line) == tape->id)
            found = 1;
        else
            found = 0;
        fprintf(target,"%s",line);
        if (fgets(line, MAXLEN-1, source) == NULL)     /* tape name */
            goto error;
        fprintf(target,"%s",line);
        if (fgets(line, MAXLEN-1, source) == NULL)     /* eot */
            goto error;
        if (found)
            fprintf(target,"%ld\n",tape->eot);
        else
            fprintf(target,"%s",line);
    }
    fclose(target);
    fclose(source);

    /* Update intern tape list */
    for (p = tape_head; p != NULL && p->id != tape->id; p = p->next)
        ;
    if (p != NULL)
        p->eot = tape->eot;

    return(1);

error:
    print_footer("Serious error in database. You have to repair this manually!");
    fclose(target);
    fclose(source);
    return(0);
}


/*************************************************************************/
/* Copyright (C) 1995 Matthias Hanisch, Wuerzburg                        */
/*************************************************************************/
