/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  ntape.h                                                      */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/
#ifndef NTAPE_H
#define NTAPE_H

/* #include <ncurses.h> */
#include <curses.h>

/* Compare the end of the first archive database line with this string
   to make sure you have the right database to the right ntape */
#define ARCHIVE_VERSION         "0.15"
#define TAPE_VERSION            "0.18"
#define ARCHIVE_DATABASE        1
#define TAPE_DATABASE           2

/* the id for tar volumes */
#define TAR_VOLUME_ID           'V'

/* some special key codes */
#define SPACE_KEY               32
#define ESCAPE_KEY              27
#define CR_KEY                  10
#define TAB_KEY                 9
#define BS_KEY1                 8
#define BS_KEY2                 0177

/* the following strings should prevent you from using illegal chars */
/* feel free to extend this string. '@' is forbidden. */
#define VALIDCHARSTR            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_0123456789 .,&':-/+\\()="
#define SPECIALCHARSTR          "*?"

#define UNKNOWN                 "Unknown tape"
#define MINCOL                  80
#define MINLINES                24
#define MAXLEN                  1024

#define NTAPEHEADER             "XTAPE\n"  /* for historic reasons. peter, do you remember? :) */
#define NTAPERC                 ".ntaperc"
#define LONGLENGTH              20
#define TAPEDEVICELEN		20
#define TAPENAMELENGTH		21
#define ARCHIVENAMELENGTH	41
#define DATELENGTH		16
#define NO_PLATFORMS		20
#define PFLENGTH		21
#define NO_CATEGORIES		20
#define CATEGORYLENGTH		21
#define DESCRIPTIONLENGTH	210

#define NTA_ADD                 1
#define NTA_EXTRACT             2
#define NTA_EXTRACT_SELECTED    3

#define CONTENTS_SAVED          1
#define DELETED                 2

#define CHANGE_NAME_DESC_PF_CAT 1
#define CHANGE_ADDINFO          10    /* Important! All CHANGE_ADDINFO defines
					 must be higher than this one !!!!! */
#define CHANGE_ADDINFO_COMMENT  11    /* Archive ist "commented" */
#define CHANGE_ADDINFO_SAVED    12    /* Archive contents are saved */
#define CHANGE_ADDINFO_NOTSAVED 13    /* Archive contents are not saved */

#define WITHOUT_SPECIAL_CHARS   0
#define WITH_SPECIAL_CHARS      1

#define OKCANCEL                0
#define YESNO                   1

#define BACKUP_TAPE             0
#define BACKUP_ARCHIVE          1
#define BACKUP_ARCHIVE_RVS      2

#define SILENT                  0
#define LOUD                    1

#define JUST_VIEW               0
#define EXTRACT_PLEEZE          1

#define MENU             1
#define INV_MENU         2
#define STATUSLINE       3
#define ARCHIVELIST      4
#define INV_ARCHIVELIST  5
#define FOOTER           6
#define MSGBOX           7
#define INV_MSGBOX       8
#define INPUTDIALOG      9
#define INV_INPUTDIALOG 10
#define ENTRYFIELD      11
#define TAPELOGWINDOW   12
#define INFOFRAME       13
#define INV_INFOFRAME   14
#define INFOLABELS      15
#define INV_INFOLABELS  16
#define INFOFIELDS      17
#define OPTSFRAME       18
#define INV_OPTSFRAME   19
#define OPTSLABELS      20
#define INV_OPTSLABELS  21
#define OPTSFIELDS      22
#define FDLGHEADER      23
#define FDLGLIST        24
#define INV_FDLGLIST    25
#define FDLGLISTSEL     26
#define FDLGPATH        27
#define FDLGBTN         28
#define INV_FDLGBTN     29
#define CONTHEADER      30
#define CONTLIST        31
#define INV_CONTLIST    32
#define CONTLISTSEL     33
#define CONTBTN         34
#define INV_CONTBTN     35
#define TAPESTATUSWIN   36
#define TAPESTATUSBTN   37
#define TAPELISTHEADER  38
#define TAPELIST        39
#define INV_TAPELIST    40
#define TAPELISTSEL     41
#define TAPELISTBTN     42
#define INV_TAPELISTBTN 43
#define INV_TAPELISTSEL 44
#define INV_FDLGLISTSEL 45
#define INV_CONTLISTSEL 46

/* Typedefinitions */
typedef struct _file_entry
{
    char name[ARCHIVENAMELENGTH];
    long tapeno;
    long size;
    long position;
    unsigned int add_info;
    char date[DATELENGTH];
    char pf[PFLENGTH];
    char category[CATEGORYLENGTH];
    char description[DESCRIPTIONLENGTH];
}
file_entry;

typedef struct _list_el
{
    char name[ARCHIVENAMELENGTH];
    long tapeno;
    long size;
    long position;
    struct _list_el *prev;
    struct _list_el *next;
}
list_el;

typedef struct _tar_list_el
{
    char *name;
    long size;
    int marked;
    struct _tar_list_el *prev;
    struct _tar_list_el *next;
}
tar_list_el;

typedef struct _tape_entry
{
    long id;
    char name[TAPENAMELENGTH];
    long no_archives;
    long eot;
}
tape_entry;

typedef struct _tape_list_el
{
    long id;
    int marked;
    char name[TAPENAMELENGTH];
    long eot;
    struct _tape_list_el *next;
}
tape_list_el;

/* Global variables */
extern const char *VERSION;
extern int sort_type,sort_ascdesc,popup_active,watch_tape,
           no_platforms,no_categories;
extern unsigned int add_info;
extern tape_entry *tape;
extern file_entry *archive;
extern char tape_db[], archive_db[], contents_db[], tapedevice[];
extern WINDOW *std, *ntape, *foot, *listbox, *tapestatus, *menu, *infobox;
extern char platform[NO_PLATFORMS][PFLENGTH];
extern char category[NO_CATEGORIES][CATEGORYLENGTH];
extern char searchstring[ARCHIVENAMELENGTH];
extern long last_tape_pos, last_tape_id;

/* Global functions */

/* init.c */
extern void nta_init(void);

/* initscr.c */
extern void nta_initscr(void);

/* exit.c */
extern void nta_exit(int);

/* list.c */
extern void init_list(void);
extern void top_pos_list(void);
extern void bottom_pos_list(void);
extern void scroll_down_list(void);
extern void scroll_up_list(void);
extern void page_down_list(void);
extern void page_up_list(void);
extern void info_box(void);
extern void search_list(char *,int);
extern void comment_focussed_archive(void);

/* tarlist.c */
extern void insert_tar_list(char *,long);
extern void handle_tar_listbox(int);

/* tapelist.c */
extern void init_tape_list(void);
extern void append_current_tape(void);
extern char *get_tapename_from_id(int,long *);
extern void show_all_tapes(void);
extern int change_eot(void);
extern int remove_tape_from_db(long,char *);

/* footer.c */
extern void print_footer(char *);

/* statusline.c */
extern void print_statusline(int,int);

/* menu.c */
extern void print_menu(void);
extern void change_menu(int);
extern void choose_menu(void);
extern void choose_cell(void);
extern void scroll_down_popup(void);
extern void scroll_up_popup(void);
extern void destroy_popup(void);

/* tape.c */
extern void tape_chosen(void);
extern void tape_chosen_cell(void);
extern void tape_scroll_down(void);
extern void tape_scroll_up(void);
extern void destroy_tape_popup(void);

/* utils.c */
extern void utils_chosen(void);
extern void utils_chosen_cell(void);
extern void utils_scroll_down(void);
extern void utils_scroll_up(void);
extern void destroy_utils_popup(void);
extern void add_archive_to_tape(char *,long,int);
extern void extract_archive(void);
extern void extract_selected_archive(int);

/* options.c */
extern void options_chosen(void);
extern void options_chosen_cell(void);
extern void options_scroll_down(void);
extern void options_scroll_up(void);
extern void destroy_options_popup(void);

/* msgbox.c */
extern int MessageBox(char *,unsigned int,unsigned int,unsigned int);

/* entryfld.c */
extern void EntryField(char *,int,char *,int,char *,int);

/* mt.c */
extern long int do_tape_command(char *,long int);
extern int offline(void);
extern int wr_prot(void);
extern void status_window(void);

/* tools.c */
extern int backup_database(int);
extern int read_header(int);
extern void get_pwd(char *);
extern int get_dir_size(char *);
extern int change_info(int,char *,long,long,char *,char *,char *);
extern int check_database_version(int,FILE *,FILE *);
extern int get_archive_entry(FILE *,file_entry *);
extern int strmatch(char *,char *);

/* filedlg.c */
extern void FileDlg(int);

/* scroll.c */
extern void scroll_down(int *,int *,int,int,void (int,int),void (void));
extern void scroll_up(int *,int *,int,int,void (int,int),void (void));
extern void page_down(int *,int *,int,int,void (int,int),void (void));
extern void page_up(int *,int *,int,int,void (int,int),void (void));
extern void top_pos(int *,int *,int,int,void (int,int),void (void));
extern void bottom_pos(int *,int *,int,int,void (int,int),void (void));

/* win.c */
extern void winclr(WINDOW *);
extern void winclrtoeol(WINDOW *);

/* watch.c */
extern void create_watch_win(char *);
extern void print_watch_win(char *);
extern void destroy_watch_win(void);

/* color.c */
extern void set_color_pair(WINDOW *,int);
extern void config_color_pair(char *,char *);

/* version.c */
extern void print_version(void);

#endif /* !NTAPE_H */
/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
