/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  filedlg.c                                                    */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <sys/stat.h>
#include <dirent.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <ntape.h>

#define REGULAR    1
#define DIRECTORY  2
#define SYMLINK    3
#define CHRDEV     4
#define BLKDEV     5
#define FIFO       6
#define SOCKET     7
#define SYMLINKDIR 8

typedef struct _dir_el
{
    char name[41];
    int type;
    long size;
    char marked;
    struct _dir_el *prev;
    struct _dir_el *next;
}
dir_el;

static WINDOW *filedlg;
static char path[MAXLEN];
static dir_el *dirhead;
static dir_el *dirtail; 
static int first_item,focus_item,no_items,active;
static int clear,modus;

void draw_box(void);
int setup_list(void);
void hilit_element(int,int);

/*************************************************************************/
/* Name:     filedlg                                                     */
/* Purpose:  Initializes file dialog box                                 */
/*************************************************************************/
void FileDlg(int mode)
{
    int i,c,count,fertig = 0;
    long size;
    dir_el *s;
    FILE *fp;
    char filename[MAXLEN];
  
    active = 2;
    modus = mode;

    filedlg = newwin(LINES-7,61,4,(COLS-61)/2);
    leaveok(filedlg,TRUE);
    keypad(menu,TRUE);

    dirhead = NULL;
    get_pwd(path);
    setup_list();
    clear = 1;
    draw_box();
    hilit_element(-1,0);
 
    while (!fertig)
    {
	touchwin(filedlg);
	wrefresh(filedlg);
	c = getch();
	switch(c)
	{
	case KEY_DOWN:
	    if (active == 2)
	    {
	key_down:
		clear = 1;
		scroll_down(&focus_item,&first_item,LINES - 14,
			    no_items,&hilit_element,&draw_box);
	    }
	    break;
	case KEY_UP:
	    if (active == 2)
	    {
		clear = 1;
		scroll_up(&focus_item,&first_item,LINES - 14,
			  no_items,&hilit_element,&draw_box);
	    }
	    break;
	case KEY_NPAGE:
	    if (active == 2)
	    {
		clear = 1;
		page_down(&focus_item,&first_item,LINES - 14,
			  no_items,&hilit_element,&draw_box);
	    }
	    break;
	case KEY_PPAGE:
	    if (active == 2)
	    {
		clear = 1;
		page_up(&focus_item,&first_item,LINES - 14,
			no_items,&hilit_element,&draw_box);
	    }
	    break;
	case KEY_A1:
	case KEY_HOME:
	    if (active == 2)
	    {
		clear = 1;
		top_pos(&focus_item,&first_item,LINES - 14,
			no_items,&hilit_element,&draw_box);
	    }
	    break;
	case KEY_C1:
	case KEY_END:
	    if (active == 2)
	    {
		clear = 1;
		bottom_pos(&focus_item,&first_item,LINES - 14,
			   no_items,&hilit_element,&draw_box);
	    }
	    break;
	case TAB_KEY:
	case KEY_RIGHT:
	    if (active == 2)
	    {
		active = 0;
		hilit_element(focus_item,-1);
	    }
	    else if (active == 1)
	    {
		active++;
		hilit_element(-1,focus_item);
	    }
	    else
	    {
		active++;
		hilit_element(focus_item,-1);
	    }
	    break;
	case KEY_LEFT:
	    if (active == 2)
	    {
		active = 1;
		hilit_element(focus_item,-1);
	    }
	    else if (active == 1)
	    {
		active = 0;
		hilit_element(focus_item,-1);
	    }
	    else
	    {
		active = 2;
		hilit_element(-1,focus_item);
	    }
	    break;
	case CR_KEY:
	    if (active == 2)
	    {
		for (s = dirhead, i = 0; s != NULL && i != focus_item; i++)
		    s = s->next;
		if (s->type == DIRECTORY || s->type == SYMLINKDIR)
		{
		    if (chdir(s->name) < 0)
		    {
			print_footer("Can't stat dir. Changing to HOME instead!");
			if (chdir(getenv("HOME")) < 0)
			{
			    print_footer("Can't stat dir. Changing to / instead!");
			    chdir("/");
			}
		    }
		    get_pwd(path);
		    setup_list();
		    clear = 1;
		    draw_box();
		    hilit_element(-1,0);
		}
	    }
	    else if (active == 1)
		fertig = 1;
	    else
	    {
		if (modus == NTA_ADD)
		{
		    tmpnam(filename);
#ifdef DEBUG
		    fprintf(stderr,"%s\n",filename);
#endif
		    if ((fp = fopen(filename,"w")) == NULL)
		    {
			print_footer("Error creating archive! Giving up...");
			break;
		    }
		    size = count = 0;
		    for (s = dirhead; s != NULL; s = s->next)
		    {
			if (s->marked)
			{
			    fprintf(fp,"%s\n",s->name);
			    if (s->type == DIRECTORY)
			    {
				print_footer("Calculating archive size! This can take a while...");
				size += get_dir_size(s->name);
			    }
			    size += s->size;
			    count++;
			}
		    }
		    fclose(fp);
		    add_archive_to_tape(filename,size,count);
		}
		else if (modus == NTA_EXTRACT)
		    extract_archive();
		else if (modus == NTA_EXTRACT_SELECTED)
		    extract_selected_archive(EXTRACT_PLEEZE);
		fertig = 1;
	    }
	    break;
	case SPACE_KEY:
	    if (active == 2 && modus == NTA_ADD)
	    {
		for (s = dirhead, i = 0; s != NULL && i != focus_item; i++)
		    s = s->next;
		if (strcmp(s->name,".") && strcmp(s->name,".."))
		{
		    s->marked = 1 - s->marked;
		    clear = 0;
		    draw_box();
		}
		hilit_element(-1,focus_item);
		goto key_down;
	    }
	    break;
	default:
	    break;
	}
    }
    delwin(filedlg);
    if (modus == NTA_ADD)
	destroy_utils_popup();
    else
    {
	touchwin(listbox);
	wrefresh(listbox);
    }
}


/*************************************************************************/
/* Name:     hilit_element                                               */
/* Purpose:  Highlight current element                                   */
/*************************************************************************/
void hilit_element(int old,int new)
{
    int i;
    char line[MAXLEN];
    dir_el *s;

    if (new >= 0)
    {
	for (s = dirhead, i = 0; s != NULL && i != new; i++)
            s = s->next;
	if (s->marked)
	    set_color_pair(filedlg,INV_FDLGLISTSEL);
	else
	    set_color_pair(filedlg,INV_FDLGLIST);

	for (i = 0; i < 60; i++)
	{
	    line[i] = mvwinch(filedlg,new - first_item + 3,i);
	}
	line[i] = '\0';
	mvwaddstr(filedlg,new - first_item + 3,0,line);
    }
    if (old >= 0)
    {
	for (s = dirhead, i = 0; s != NULL && i != old; i++)
	    s = s->next;
	if (s->marked)
	    set_color_pair(filedlg,FDLGLISTSEL);
	else
	    set_color_pair(filedlg,FDLGLIST);
	for (i = 0; i < 60; i++)
	{
	    line[i] = mvwinch(filedlg,old - first_item + 3,i);
	}
	line[i] = '\0';
	mvwaddstr(filedlg,old - first_item + 3,0,line);
    }
    for (i = 0; i < 2; i++)
    {
	if (i == active)
	    set_color_pair(filedlg,INV_FDLGBTN);
	else
	    set_color_pair(filedlg,FDLGBTN);
	if (i == 0)
	    mvwaddstr(filedlg,LINES-9,5,"   OK   ");
	else
	    mvwaddstr(filedlg,LINES-9,60-13," Cancel ");
    }
}


/*************************************************************************/
/* Name:     draw_box                                                    */
/* Purpose:  Draw dialog box                                             */
/*************************************************************************/
void draw_box()
{
    int i;
    char line[MAXLEN];
    dir_el *s;

    if (clear)
	werase(filedlg);
    set_color_pair(filedlg,FDLGHEADER);

    wmove(filedlg,0,0);
    winclrtoeol(filedlg);
    if (modus == NTA_ADD)
	mvwaddstr(filedlg,1,0,"         Choose the files you want to be archived!");
    else
	mvwaddstr(filedlg,1,0,"             Choose the destination directory!");
    winclrtoeol(filedlg);
    wmove(filedlg,2,0);
    winclrtoeol(filedlg);
  
    set_color_pair(filedlg,FDLGLIST);
    for (i = 0; i < LINES - 10; i++)
    {
	wmove(filedlg,i+3,0);
	winclrtoeol(filedlg);
    }
  
    s = dirhead;
    for (i = 0; i < first_item; i++)
	s = s->next;
    for (i = 0; i < LINES - 14; i++)
    {
	if (!s->marked)
	    set_color_pair(filedlg,FDLGLIST);
	else
	    set_color_pair(filedlg,FDLGLISTSEL);
	sprintf(line,"  %-40s      ",s->name);
	if (s->type == DIRECTORY)
	    strcat(line,"DIRECTORY");
	else if (s->type == SYMLINK)
            strcat(line,"SYMLINK");
	else if (s->type == SYMLINKDIR)
	    strcat(line,"SYMLINK/DIR");
	else if (s->type == CHRDEV)
            strcat(line,"CHARDEV");
	else if (s->type == BLKDEV)
            strcat(line,"BLOCKDEV");
	else if (s->type == FIFO)
            strcat(line,"PIPE");
	else if (s->type == SOCKET)
            strcat(line,"SOCKET");
	else
	    sprintf(line,"  %-40s      %ld",s->name,s->size);
	mvwaddstr(filedlg,i+3,0,line);
	s = s->next;
	if (s == NULL)
	    break;
    }
    set_color_pair(filedlg,FDLGPATH);
    sprintf(line,"Path: %s",path);
    mvwaddstr(filedlg,LINES-11,0,line);
    winclrtoeol(filedlg);
    
    touchwin(filedlg);
    wrefresh(filedlg);
}


/*************************************************************************/
/* Name:     setup_list                                                  */
/* Purpose:  Reads directory in a linked list                            */
/*************************************************************************/
int setup_list()
{
    char filename[MAXLEN];
    struct stat pstat;
    DIR * dirp;
    struct dirent *entry;
    dir_el *s,*t;
  
    /* Free directory_list */
    s = dirhead;
    while (s != NULL)
    {
	t = s->next;
	free(s);
	s = t;
    }

    /* We have to insert "." and ".." manually if we are not allowed to 
       read the directory's contents. We must assume that '.' and '..' are
       the directories that we expect. You can do dirty things with link(2) */

    if ((dirhead = (dir_el *)malloc(sizeof(dir_el))) == 0)
	return(-1);
    strcpy(dirhead->name,".");
    dirhead->type = DIRECTORY;
    dirhead->size = 0;          /* It's not actually zero. But who cares? */
    dirhead->marked = 0;
    if ((dirtail = (dir_el *)malloc(sizeof(dir_el))) == 0)
	return(-1);
    strcpy(dirtail->name,"..");
    dirtail->type = DIRECTORY;
    dirtail->size = 0;          /* Same as above */
    dirtail->marked = 0;
    dirtail->next = NULL;
    dirtail->prev = dirhead;
    dirhead->prev = NULL;
    dirhead->next = dirtail;
    
    no_items = 2;
    
    /* Fill list with file data */
    first_item = focus_item = 0;
    if ((dirp = opendir(path)) == NULL)
	return(-1);

    while ((entry = readdir(dirp)) != NULL)
    {
	if (!strcmp(entry->d_name,".") || !strcmp(entry->d_name,".."))
	    continue;

	sprintf(filename,"%s/%s",path,entry->d_name);
	if (lstat(filename,&pstat) < 0)
	    continue;
	if ((modus == NTA_EXTRACT || modus == NTA_EXTRACT_SELECTED)
	    && !S_ISDIR(pstat.st_mode))
	    continue;
	if ((s = (dir_el *)malloc(sizeof(dir_el))) == 0)
	    return(-1);
	strncpy(s->name,entry->d_name,40);
	s->size = pstat.st_size;
	s->marked = 0;
	if (S_ISDIR(pstat.st_mode))
	    s->type = DIRECTORY;
	else if (S_ISLNK(pstat.st_mode))
	{
	    s->type = SYMLINK;
	    if (stat(filename,&pstat) < 0)
		continue;
	    if (S_ISDIR(pstat.st_mode))
		s->type = SYMLINKDIR;
	}
	else if (S_ISCHR(pstat.st_mode))
	    s->type = CHRDEV;
	else if (S_ISBLK(pstat.st_mode))
	    s->type = BLKDEV;
	else if (S_ISFIFO(pstat.st_mode))
	    s->type = FIFO;
	else if (S_ISSOCK(pstat.st_mode))
	    s->type = SOCKET;
	else
	    s->type = REGULAR;
	
	t = dirhead;
	while (t != NULL)
	{
	    if (strcmp(t->name,s->name) > 0)
		break;
	    t = t->next;
	}
      
	if (t == NULL)
	{
	    if (dirhead == NULL)
	    {
		s->next = s->prev = NULL;
		dirhead = dirtail = s;
	    }
	    else
	    {
		dirtail->next = s;
		s->prev = dirtail;
		s->next = NULL;
		dirtail = s;
	    }
	}
	else
	{
	    if (t != dirhead)
		t->prev->next = s;
	    else
		dirhead = s;
	    s->prev = t->prev;
	    t->prev = s;
	    s->next = t;
	}
	no_items++;
    }

    closedir(dirp);
    return(1);
}

/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
