/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  color.c                                                      */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <strings.h>
#include <ntape.h>

#define HL(col) ((col)+8)

/* default color pairs */
static unsigned char colors[][2] =
{
    {0,0},                          /* The pair 0 oesn't work */
    {COLOR_WHITE,COLOR_BLUE},       /* The menu at the top */
    {COLOR_BLUE,COLOR_WHITE},

    {COLOR_MAGENTA,COLOR_CYAN},     /* The tape status line */

    {COLOR_WHITE,COLOR_MAGENTA},    /* The archive list */
    {COLOR_BLACK,COLOR_WHITE},

    {COLOR_RED,COLOR_BLUE},         /* The footer */

    {COLOR_BLACK,COLOR_RED},        /* A Messagebox */
    {COLOR_RED,COLOR_BLACK},        /* Its active button */

    {COLOR_BLACK,COLOR_RED},        /* A Textentrydialog */
    {COLOR_RED,COLOR_BLACK},        /* Its active button */
    {COLOR_RED,COLOR_BLACK},        /* Its text entry field */

    {COLOR_WHITE,COLOR_BLUE},       /* The tape operations logging window */

    {COLOR_MAGENTA,COLOR_BLUE},     /* The archive information border */
    {COLOR_BLUE,COLOR_MAGENTA},     /* The active button */
    {COLOR_WHITE,COLOR_MAGENTA},    /* The labels inside archive info */
    {COLOR_MAGENTA,COLOR_WHITE},    /* The active label */
    {COLOR_MAGENTA,COLOR_BLUE},     /* The data fields */

    {COLOR_BLUE,COLOR_CYAN},        /* The options border */
    {COLOR_BLUE,COLOR_WHITE},       /* The active button */
    {COLOR_WHITE,COLOR_BLUE},       /* The tape operations logging window */
    {COLOR_BLUE,COLOR_WHITE},       /* The active label */
    {COLOR_WHITE,COLOR_MAGENTA},    /* The options data fields */

    {COLOR_BLUE,COLOR_CYAN},        /* Top of file selection box */
    {COLOR_WHITE,COLOR_BLUE},       /* File list */
    {COLOR_BLACK,COLOR_WHITE},      /* The highlighted list item */
    {COLOR_CYAN,COLOR_BLUE},        /* Selected items */
    {COLOR_BLUE,COLOR_CYAN},        /* Current path */
    {COLOR_WHITE,COLOR_BLUE},       /* OK and Cancel */
    {COLOR_BLUE,COLOR_CYAN},        /* The active one */

    {COLOR_WHITE,COLOR_BLUE},       /* Top of archive contents box */
    {COLOR_BLUE,COLOR_CYAN},        /* Contents list */
    {COLOR_BLACK,COLOR_WHITE},      /* The highlighted list item */
    {COLOR_MAGENTA,COLOR_CYAN},     /* Selected items */
    {COLOR_WHITE,COLOR_BLUE},       /* OK and Cancel */
    {COLOR_BLUE,COLOR_WHITE},       /* The active one */

    {COLOR_WHITE,COLOR_BLUE},       /* The tape status window */
    {COLOR_BLUE,COLOR_WHITE},       /* The Cancel button */
    
    {COLOR_WHITE,COLOR_BLUE},       /* Top of tape list box */
    {COLOR_BLUE,COLOR_CYAN},        /* The tape list */
    {COLOR_BLACK,COLOR_WHITE},      /* The highlighted list item */
    {COLOR_MAGENTA,COLOR_CYAN},     /* Selected items */
    {COLOR_WHITE,COLOR_BLUE},       /* OK and Cancel */
    {COLOR_BLUE,COLOR_WHITE},       /* The active one */
    {COLOR_BLACK,COLOR_WHITE},      /* The highlighted selected list item */

    {COLOR_BLACK,COLOR_WHITE},      /* The highlighted selected list item in the file dialog box */
    {COLOR_BLACK,COLOR_WHITE}       /* The highlighted selected list item in the archive contents box */
};


/*************************************************************************/
/* Name:     config_color_pair                                           */
/* Purpose:  sets the fore/background color pair as specified in ntaperc */
/*************************************************************************/
void config_color_pair(char *which_pair,char *color_pair)
{
    unsigned char area,fg,bg,hl = 0;
    char *col;
    const char *color[] = 
    {
	"BLACK",
	"RED",
	"GREEN",
	"YELLOW",
	"BLUE",
	"MAGENTA",
	"CYAN",
	"WHITE",
	NULL
    };
    const char *what[] =
    {
	NULL,
	"COLMENU",
	"COLINVMENU",
	"COLSTATUSLINE",
	"COLARCHIVELIST",
	"COLINVARCHIVELIST",
	"COLFOOTER",
	"COLMESSAGEBOX",
	"COLINVMESSAGEBOX",
	"COLINPUTDIALOG",
	"COLINVINPUTDIALOG",
	"COLENTRYFIELD",
	"COLTAPEOPLOG",
	"COLINFOFRAME",
	"COLINVINFOFRAME",
	"COLINFOLABELS",
	"COLINVINFOLABELS",
	"COLINFOFIELDS",
	"COLOPTSFRAME",
	"COLINVOPTSFRAME",
	"COLOPTSLABELS",
	"COLINVOPTSLABELS",
	"COLOPTSFIELDS",
	"COLFDLGHEADER",
	"COLFDLGLIST",
	"COLINVFDLGLIST",
	"COLFDLGLISTSEL",
	"COLFDLGPATH",
	"COLFDLGBTN",
	"COLINVFDLGBTN",
	"COLCONTHEADER",
	"COLCONTLIST",
	"COLINVCONTLIST",
	"COLCONTLISTSEL",
	"COLCONTBTN",
	"COLINVCONTBTN",
	"COLTAPESTATUSWIN",
	"COLTAPESTATUSBTN",
	"COLTAPELISTHEADER",
	"COLTAPELIST",
	"COLINVTAPELIST",
	"COLTAPELISTSEL",
	"COLTAPELISTBTN",
	"COLINVTAPELISTBTN",
	"COLINVTAPELISTSEL",
	"COLINVFDLGLISTSEL",
	"COLINVCONTLISTSEL",
	NULL
    };

    for (area = 1; what[area] != NULL; area++)
	if (!strcmp(what[area],which_pair))
	    break;
    if (what[area] == NULL)
    {
	fprintf(stderr,"Warning! Unrecognized color configuration '%s'...\n",
		which_pair);
	return;
    }
    
    if ((col = strtok(color_pair,"/")) == NULL)
	return;
    if (!strncmp(col,"LIGHT",5))
    {
	hl = 1;
	col = &col[5];
    }
    for (fg = 0; color[fg] != NULL; fg++)
	if (!strcmp(color[fg],col))
	    break;
    if (color[fg] == NULL)
	return;
    if (hl)
	fg |= 8;
        
    if ((col = strtok(NULL,"/")) == NULL)
	return;
    for (bg = 0; color[bg] != NULL; bg++)
	if (!strcmp(color[bg],col))
	    break;
    if (color[bg] == NULL)
	return;

    colors[area][0] = fg;
    colors[area][1] = bg;
#ifdef DEBUG
    fprintf(stderr,"config_color_pair: Fill area %d with Colors %d/%d...\n",
	    area,fg,bg);
#endif
}


/*************************************************************************/
/* Name:     set_color_pair                                              */
/* Purpose:  sets the fore/background color pair                         */
/*************************************************************************/
void set_color_pair(WINDOW *win,int where)
{
    wattrset(win,A_NORMAL);
    
    init_pair(where,colors[where][0] & 7,colors[where][1] & 7);
    if (colors[where][0] & 8)
	wattron(win,COLOR_PAIR(where) | A_BOLD);
    else
	wattron(win,COLOR_PAIR(where) | A_NORMAL);
}

/*************************************************************************/
/* Copyright (C) 1995 Matthias Hanisch, Wuerzburg                        */
/*************************************************************************/
