/********************************************************
 * mtstatus -- Report the status of a HP tape drive	*
 *							*
 * Usage:						*
 *	mtstatus [-a] [-c][-e][-l][-z][-f] [-r] device	*
 *							*
 * 		-a -- Print all reports			*
 * 		-c -- Report capacity			*
 * 		-e -- Error report			*
 * 		-l -- Tape log				*
 * 		-z -- Compression report		*
 * 	 	-f -- Compression mode			*
 * 		-r -- Report raw data			*
 *							*
 * $Header: $						*
 ********************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <string.h>
#include <unistd.h>

#ifdef solaris
#include <sys/scsi/generic/mode.h>
#include <sys/scsi/generic/commands.h>
#include <sys/scsi/impl/types.h>
#include <sys/scsi/impl/uscsi.h>
#include <sys/systeminfo.h>
#endif /* solaris */
#ifdef linux
#include <scsi/scsi.h>
#include <scsi_ioctl.h>
#endif /* linux */

const int HP_LOG_SENSE = 0x4D;	/* Read Log Sense */

static int raw = 0;		/* Output raw data */
static int fd;			/* File descriptor of the tape */

enum page_code {
    P_READ_ERROR  = 0x02,		/* Read error counter */
    P_WRITE_ERROR = 0x03,		/* Write error counter */
    P_TAPE_LOG    = 0x30,		/* Tape log page */
    P_CAPACITY    = 0x31,		/* Capacity command */
    P_COMPRESS    = 0x39		/* Compression */
};

/* The information about each page we're getting */
struct page_info {
   enum page_code code;		/* Page code number */
   char *name;			/* The name of the page */
};

static struct page_info page_list[] = {
	{0,		"Supported pages"},
	{P_READ_ERROR,	"Write Error Counters"},
	{P_WRITE_ERROR,	"Read error Counters"},
	{P_TAPE_LOG,	"Tape log page"},
	{P_CAPACITY,	"Tape capacity log"},
	{0x34,		"Write error counters"},
	{0x35,		"Read error counters"},
	{0x37,		"Drive counters"},
	{0x38,		"Mechanism Counters"},
	{P_COMPRESS,	"Data Compressions log"},
	{-1,		NULL}
};

struct log_info {
    unsigned char page_code;		/* Page code from the log */
    unsigned char reserved;		/* Reserved byte */
    unsigned short int size_code;
};
union scsi_log_struct {
    struct log_info info;	/* Information for a log */
    unsigned char raw[500];	/* Raw SCSI data */
};
/* The log data from the tape drive */
static union scsi_log_struct log;

/********************************************************
 * raw_page -- Print the page in the raw		*
 ********************************************************/
static void raw_page(void)
{
    int i;	/* Index for our dump */
    printf("----------------------------------------------------\n");

    for (i = 0; i < log.info.size_code+4; ++i) {
	printf("%2d: 0x%02x\n", i, log.raw[i]);
    }
    printf("\n\n\n");
}
/********************************************************
 * raw_mode -- Print the mode sense in the raw		*
 ********************************************************/
static void raw_mode(void)
{
    int i;	/* Index for our dump */
    printf("----------------------------------------------------\n");

    for (i = 0; i < log.raw[0]; ++i) {
	printf("%2d: 0x%02x\n", i, log.raw[i]);
    }
    printf("\n\n\n");
}

/* Turn variable into a signed character */
#define SC(what) ((signed char)(what))

/* Turn variable into a unsigned long */
#define UL(value, index) \
     ((unsigned long)((value)[index]))

/* Turn variable into a unsigned int */
#define U(value, index) \
     ((unsigned int)((value)[index]))

/* Turn variable into a unsigned int */
#define L(value, index) \
     ((unsigned int)(SC((value)[index])))

/* Turn two bytes into a a short int */
#define TWO_BYTES(value) (signed short int) (\
    ((U(value,0) << 8) | U(value, 1))	    )

/* Turn three bytes into a a short int */
#define THREE_BYTES(value) \
    ((L( value, 0) << 16L) | \
     (UL(value, 1) << 8L)  | \
      UL(value, 2))

/* Turn four bytes into a a short int */
#define FOUR_BYTES(value) \
   ((UL(value, 0) << 24L) | \
    (UL(value, 1) << 16L) | \
    (UL(value, 2) << 8L)  | \
     UL(value, 3))
/********************************************************
 * print_param2 -- print a two byte parameter		*
 *							*
 * Returns						*
 *	Pointer to first character after the data	*
 ********************************************************/
static u_char *print_param2(
    const char name[],		/* Name of the parameter */
    u_char *cur_ptr		/* Pointer to the data */
) {
    printf("%s Parameter Code %d\n", name, TWO_BYTES(cur_ptr));
    cur_ptr += 2;

    printf("%s Flag 0x%02x\n", name, *cur_ptr);
    cur_ptr += 1;

    printf("%s Length %d\n", name, *cur_ptr);
    if (*cur_ptr != 2) printf("*** Length incorrect\n");
    cur_ptr += 1;

    printf("%s value %d\n", name, TWO_BYTES(cur_ptr));
    cur_ptr += 2;

    printf("\n");
    return (cur_ptr);
}
/********************************************************
 * print_param3 -- print a three byte parameter		*
 *							*
 * Returns						*
 *	Pointer to first character after the data	*
 ********************************************************/
static u_char *print_param3(
    const char name[],		/* Name of the parameter */
    u_char *cur_ptr		/* Pointer to the data */
) {
    printf("%s Parameter Code %d\n", name, TWO_BYTES(cur_ptr));
    cur_ptr += 2;

    printf("%s Flag 0x%02x\n", name, *cur_ptr);
    cur_ptr += 1;

    printf("%s Length %d\n", name, *cur_ptr);
    if (*cur_ptr != 3) printf("*** Length incorrect\n");
    cur_ptr += 1;

    printf("%s value %ld\n", name, THREE_BYTES(cur_ptr));
    cur_ptr += 3;

    printf("\n");
    return (cur_ptr);
}
/********************************************************
 * print_param4 -- print a four byte parameter		*
 *							*
 * Returns						*
 *	Pointer to first character after the data	*
 ********************************************************/
static u_char *print_param4(
    const char name[],		/* Name of the parameter */
    u_char *cur_ptr		/* Pointer to the data */
) {
    printf("%s Parameter Code %d\n", name, TWO_BYTES(cur_ptr));
    cur_ptr += 2;

    printf("%s Flag 0x%02x\n", name, *cur_ptr);
    cur_ptr += 1;

    printf("%s Length %d\n", name, *cur_ptr);
    if (*cur_ptr != 4) printf("*** Length incorrect\n");
    cur_ptr += 1;

    printf("%s value %ld\n", name, FOUR_BYTES(cur_ptr));
    cur_ptr += 4;

    printf("\n");
    return (cur_ptr);
}

/********************************************************
 * print_page_head -- print the page information	*
 *							*
 * Returns						*
 *	Pointer to first character after the data	*
 ********************************************************/
static u_char *print_page_head(
    char *cur_ptr	/* Pointer to the current character */
)
{
    printf("Page code 0x%02x\n", *cur_ptr);
    cur_ptr += 1;

    printf("Reserved1 %d\n", *cur_ptr);
    cur_ptr += 1;

    printf("Length %d\n", TWO_BYTES(cur_ptr));
    cur_ptr += 2;
    printf("\n");
    return (cur_ptr);
}
/********************************************************
 * print_read_write -- Print the read/write pages	*
 ********************************************************/
static void print_read_write(
    const char what[]	/* What we're dumping */
) {
    u_char *cur_ptr;	/* Pointer to current character in the data */

    printf("%s Error Counter Pages\n", what);
    cur_ptr = &log.raw[0];

    cur_ptr = print_page_head(cur_ptr);

    cur_ptr = print_param2("Soft Errors", cur_ptr);
    cur_ptr = print_param4(
		 "Total Times Correction Algorithm Processed", cur_ptr);
    cur_ptr = print_param4("Total Groups Processed", cur_ptr);
    cur_ptr = print_param2("Hard Errors ", cur_ptr);
    printf("\n");
}

/********************************************************
 * print_tape_param -- Print a tape log page parameter	*
 ********************************************************/
static u_char *print_tape_param(
    const char name[],	/* Name of the parameter */
    u_char *cur_ptr	/* Pointer to current character in the data */
) {
    char full_name[100];	/* Full name of the page */

    sprintf(full_name, "%s Groups Written", name);
    cur_ptr = print_param3(full_name, cur_ptr);

    sprintf(full_name, "%s Raw Retries", name);
    cur_ptr = print_param2(full_name, cur_ptr);

    sprintf(full_name, "%s Read", name);
    cur_ptr = print_param3(full_name, cur_ptr);

    sprintf(full_name, "%s ECC Retries", name);
    cur_ptr = print_param2(full_name, cur_ptr);

    return (cur_ptr);
}

/********************************************************
 * print_tape_log -- print the tape log page		*
 ********************************************************/
static void print_tape_log(void)
{
    u_char *cur_ptr;	/* Pointer to current character in the data */

    cur_ptr = &log.raw[0];

    printf("Tape Log Page\n");
    cur_ptr = print_page_head(cur_ptr);
    cur_ptr = print_tape_param("Current", cur_ptr);
    cur_ptr = print_tape_param("Previous", cur_ptr);
    cur_ptr = print_param4("Total Written", cur_ptr);
    cur_ptr = print_param3("Total Raw Retries", cur_ptr);
    cur_ptr = print_param4("Total Read", cur_ptr);
    cur_ptr = print_param3("Total Ecc", cur_ptr);
}
/********************************************************
 * print_tape_capacity -- Print the capacity page	*
 ********************************************************/
static void print_tape_capacity(void)
{
    u_char *cur_ptr;	/* Pointer to the current data */

    cur_ptr = &log.raw[0];

    printf("Tape Capacity Log Page\n");
    cur_ptr = print_page_head(cur_ptr);
    printf("\n");
    cur_ptr = print_param4("Free partition 0", cur_ptr);
    cur_ptr = print_param4("Free partition 1", cur_ptr);
    cur_ptr = print_param4("Capacity partition 0", cur_ptr);
    cur_ptr = print_param4("Capacity partition 1", cur_ptr);
}
/********************************************************
 * print_mechanism -- Print mechanism page		*
 ********************************************************/
static void print_mechanism(void)
{
    u_char *cur_ptr;	/* Pointer to the current data */

    cur_ptr = &log.raw[0];

    printf("Tape Mechanism Counters Page\n");
    cur_ptr = print_page_head(cur_ptr);

    cur_ptr = print_param2("Faults 12v", cur_ptr);
    cur_ptr = print_param2("Mode Sensor Fault", cur_ptr);
    cur_ptr = print_param2("Tension Too Low", cur_ptr);
    cur_ptr = print_param2("Bad Diameter", cur_ptr);
    cur_ptr = print_param2("Capstan Stalled", cur_ptr);
    cur_ptr = print_param2("DPG Calibration Failed", cur_ptr);
    cur_ptr = print_param2("Drum Stalled", cur_ptr);
    cur_ptr = print_param2("Drum has Lost Lock", cur_ptr);
    cur_ptr = print_param2("Drum PG Lost", cur_ptr);
    cur_ptr = print_param2("###", cur_ptr);
}
/********************************************************
 * print_compression -- Print compression page		*
 ********************************************************/
static void print_compression(void)
{
    u_char *cur_ptr;	/* Pointer to the current data */

    cur_ptr = &log.raw[0];

    printf("Tape Compression Transfer Page\n");
    cur_ptr = print_page_head(cur_ptr);

    cur_ptr = print_param4("Number of entries written", cur_ptr);
    cur_ptr = print_param4("Number of entries read", cur_ptr);
    cur_ptr = print_param4("Number of records written", cur_ptr);
    cur_ptr = print_param4("Number of records read", cur_ptr);
    cur_ptr = print_param4("Kilobytes to data compression  ", cur_ptr);
    cur_ptr = print_param4("Kilobytes from data compression", cur_ptr);
    cur_ptr = print_param4("Kilobytes to tape  ", cur_ptr);
    cur_ptr = print_param4("Kilobytes from tape", cur_ptr);
    cur_ptr = print_param4("Logical entity size", cur_ptr);
    cur_ptr = print_param4("Physical entity size", cur_ptr);
    cur_ptr = print_param4("Uncompressed entities", cur_ptr);
}


/********************************************************
 * print_page -- print the current log page		*
 ********************************************************/
static void print_page(void)
{
    switch (log.raw[0]) {
	case 0x02:
	    if (raw) raw_page();
	    print_read_write("Read");
	    break;
	case 0x03:
	    if (raw) raw_page();
	    print_read_write("Write");
	    break;
	case 0x30:
	    if (raw) raw_page();
	    print_tape_log();
	    break;
	case P_CAPACITY:
	    if (raw) raw_page();
	    print_tape_capacity();
	    break;
	case 0x38:
	    if (raw) raw_page();
	    print_mechanism();
	    break;
	case 0x39:
	    if (raw) raw_page();
	    print_compression();
	    break;
	default:
	    raw_page();
	    break;
    }
}
/********************************************************
 * mode_sense -- Do a mode sense to get a page of data	*
 ********************************************************/
static void mode_sense(
    const int page	/* Page index */
)
{
#ifdef linux
    static struct mode_cmd {
	unsigned int inlen;		 /* Length of data going in */
	unsigned int outlen;		 /* Length of data going out */
	unsigned char cmd[6];		 /* The command to execute */
	unsigned char data[sizeof(log)]; /* The data buffer */
    } scsi_cmd;

    scsi_cmd.cmd[0] = MODE_SENSE;
    scsi_cmd.cmd[1] = 0;
    scsi_cmd.cmd[2] = page;

    scsi_cmd.cmd[3] = 0;
    scsi_cmd.cmd[4] = 255;
    scsi_cmd.cmd[5] = 0;

    scsi_cmd.inlen = 0;
    scsi_cmd.outlen = sizeof(log);

    if (ioctl (fd, SCSI_IOCTL_SEND_COMMAND, &scsi_cmd) < 0) { 
	int error = errno;
	perror("SCSI command failed: ");
	printf("Error code: %d\n", error);
	exit (1);
    }
    /* Copy the the data to the log variable */
    /* (I know it's a hack) */
    memcpy(&log, scsi_cmd.cmd, sizeof(log));

#endif /* linux */

#ifdef solaris
    struct uscsi_cmd cmd;	/* The scsi command */
    union scsi_cdb cdb;	/* The command/data block */


    memset(&cmd, '\0', sizeof(cmd));
    memset(&cdb, '\0', sizeof(cdb));

    cdb.scc_cmd = SCMD_MODE_SENSE;
    cdb.scc_lun = 0;	
    cdb.g0_count0 = 255;
    cdb.g0_addr1  = page;

    cmd.uscsi_cdb=(caddr_t)&cdb;
    cmd.uscsi_cdblen=6;
    cmd.uscsi_bufaddr=(caddr_t)&log;
    cmd.uscsi_buflen=sizeof(log);
    cmd.uscsi_flags=USCSI_DIAGNOSE|USCSI_SILENT|USCSI_ISOLATE|USCSI_READ;

    if (ioctl (fd, USCSICMD, &cmd) < 0) { 
	int error = errno;
	perror("SCSI command failed: ");
	printf("Error code: %d\n", error);
	exit (1);
    }
#endif /* solaris */
}
/********************************************************
 * get_page -- Get a page of the LOG_SENSE data		*
 *							*
 * Data is returned in the global structure log		*
 ********************************************************/
static void get_page(
    const int page	/* Page index */
)
{
#ifdef linux
    static struct log_cmd {
	unsigned int inlen;		 /* Length of data going in */
	unsigned int outlen;		 /* Length of data going out */
	unsigned char cmd[10];		 /* The command to execute */
	unsigned char data[sizeof(log)]; /* The data buffer */
    } scsi_cmd;

    scsi_cmd.cmd[0] = HP_LOG_SENSE;
    scsi_cmd.cmd[1] = 0;
    scsi_cmd.cmd[2] = page;

    scsi_cmd.cmd[3] = 0;
    scsi_cmd.cmd[4] = 0;
    scsi_cmd.cmd[5] = 0;
    scsi_cmd.cmd[6] = 0;

    scsi_cmd.cmd[7] = sizeof(log)/256;
    scsi_cmd.cmd[8] = sizeof(log)%256;
    scsi_cmd.cmd[9] = 0;

    scsi_cmd.inlen = 0;
    scsi_cmd.outlen = sizeof(log);

    if (ioctl (fd, SCSI_IOCTL_SEND_COMMAND, &scsi_cmd) < 0) { 
	int error = errno;
	perror("SCSI command failed: ");
	printf("Error code: %d\n", error);
	exit (1);
    }

    /* Copy the the data to the log variable */
    /* (I know it's a hack) */
    memcpy(&log, scsi_cmd.cmd, sizeof(log));

#endif /* linux */
   
#ifdef solaris
    struct uscsi_cmd cmd;	/* The scsi command */
    union scsi_cdb cdb;	/* The command/data block */

    memset(&cmd, '\0', sizeof(cmd));
    memset(&cdb, '\0', sizeof(cdb));

    cdb.scc_cmd = HP_LOG_SENSE;
    cdb.scc_lun = 0;	
    cdb.g1_count1 = sizeof(log) / 256;
    cdb.g1_count0 = sizeof(log) % 256;
    cdb.g1_addr3  = page;

    cmd.uscsi_cdb=(caddr_t)&cdb;
    cmd.uscsi_cdblen=10;
    cmd.uscsi_bufaddr=(caddr_t)&log;
    cmd.uscsi_buflen=sizeof(log);
    cmd.uscsi_flags=USCSI_DIAGNOSE|USCSI_SILENT|USCSI_ISOLATE|USCSI_READ;

    if (ioctl (fd, USCSICMD, &cmd) < 0) { 
	int error = errno;
	perror("SCSI command failed: ");
	printf("Error code: %d\n", error);
	exit (1);
    }
#endif /* solaris */
}
/********************************************************
 * report_all -- report all data pages.			*
 ********************************************************/
static void report_all(void)
{
    int page;	/* Current page number */

    for (page = 0; page_list[page].code != -1; ++page) {
	printf("Page: 0x%02x %s\n", 
		page_list[page].code, page_list[page].name);

	get_page(page_list[page].code);
	print_page();
    }
}
/********************************************************
 * print_one_free -- print the free space of a single	*
 *		partition.				*
 ********************************************************/
static void print_one_free(
    const char what[],		/* The name of the partition */
    const int index_left,	/* The index of the #k left */
    const int index_total	/* The index of the #k total */
)
{
    int space_used;		/* #k used */
    int space_left;		/* #k left */
    int space_total;		/* #k total */
    float percent_used;		/* Total space used */

    space_left  = FOUR_BYTES(&log.raw[index_left]);
    space_total = FOUR_BYTES(&log.raw[index_total]);
    space_used  = space_total - space_left;

    if (space_total == 0)
    	percent_used = 0.0;
    else
	percent_used = ((float)(space_used)) / ((float)(space_total)) * 100.0;

    printf("%s Used %2.1f%% Bytes used %d of %d\n",
    	what, percent_used, 
	space_used, space_total);
}

/********************************************************
 * report_capacity -- print a capacity report		*
 ********************************************************/
static void report_capacity(void)
{

    printf("Capacity ----------------------------------------\n");
    get_page(P_CAPACITY);
    if (raw) raw_page();
    print_one_free("Partition 0",  8, 24);
    print_one_free("Partition 1", 16, 32);
}
/********************************************************
 * report_errors -- print a error counter report	*
 *							*
 * Page format						*
 * 	0 -- page code					*
 *	1 -- reserved					*
 *	2,3 -- page length (1c)				*
 *							*
 *	4,5 -- 0,1 param code (3)			*
 *	6 -- 2 flags					*
 *	7 -- 3 param length (2)				*
 *	8,9 -- 4,5 value				*
 *							*
 *	10,11 -- 0,1 param code (4)			*
 *	12 -- 2 flags					*
 *	13 -- 3 param length (4)			*
 *	14,15,16,17 -- 4,5,6,7 value			*
 *							*
 *	18,19 -- 0,1 param code (5)			*
 *	20 -- 2 flags					*
 *	21 -- 3 param length (4)			*
 *	22,23,24,25 -- 4,5,6,7 value			*
 *							*
 *	26,27 -- 0,1 param code (6)			*
 *	28 -- 2 flags					*
 *	29 -- 3 param length (2)			*
 *	30,31 -- 4,5 value				*
 ********************************************************/
static void report_error(
    const char what[],	/* The name of the counter */
    const int page	/* The page number for the report */
)
{
    int total_corrected;	/* Total errors corrected (param code=3) */
    int total_corrections;	/* Total times cor. alt. proc (pc = 4) */
    int total_groups;		/* Total groups processed (pc=5) */
    int total_hard;		/* Total hard errors (pc=6) */

    get_page(page);
    if (raw) raw_page();

    total_corrected   =  TWO_BYTES(&log.raw[8]);
    total_corrections = FOUR_BYTES(&log.raw[14]);
    total_groups      = FOUR_BYTES(&log.raw[22]);
    total_hard        =  TWO_BYTES(&log.raw[30]);

    printf("%s errors: Soft %3d Hard %3d Fixes %3d Groups %5d\n",
    	what, 
	total_corrected, total_hard, 
	total_corrections, total_groups);
}
/* ### */
struct counters {
    long int groups_written;
    long int raw_retries;
    long int groups_read;
    long int ecc_retries;
};

/********************************************************
 * n_bytes -- get 2, 3, 4 byte value.			*
 ********************************************************/
static long int n_bytes(
    unsigned char value[],	/* The byte version of the value */
    int size			/* Number of bytes in the parameter */
)
{
    switch (size) {
        case 2:
	    return (TWO_BYTES(value));
	case 3:
	    return (THREE_BYTES(value));
	case 4:
	    return (FOUR_BYTES(value));
	default:
	    fprintf(stderr, "Internal error in n_bytes (size = %d)\n",
	    	size);
	    exit (8);
    }
}

/********************************************************
 * grab_counter -- Get the counters for a table log	*
 ********************************************************/
static void grab_counters(
    struct counters *value_ptr,	/* Pointer to the counters (returned) */
    const int offset,		/* Where the blocks start */
    const int v1,		/* Parameter 1 size */
    const int v2,		/* Parameter 2 size */
    const int v3,		/* Parameter 3 size */
    const int v4		/* Parameter 4 size */
)
{
    value_ptr->groups_written  = n_bytes(&log.raw[4*1 + offset],          v1);
    value_ptr->raw_retries     = n_bytes(&log.raw[4*2 + offset+v1],       v2);
    value_ptr->groups_read     = n_bytes(&log.raw[4*3 + offset+v1+v2],    v3);
    value_ptr->ecc_retries     = n_bytes(&log.raw[4*4 + offset+v1+v2+v3], v4);
}
/********************************************************
 * report_tape_log -- print a tape log report		*
 ********************************************************/
static void report_tape_log(void)
{
    struct counters current;
    struct counters previous;
    struct counters total;
    int load_count;

    printf("Tape Log ----------------------------------------\n");
    get_page(P_TAPE_LOG);
    if (raw) raw_page();

    grab_counters(&current,  16*0+ 4,       3, 2, 3, 2);
    grab_counters(&previous, 16*1+ 4+10,    3, 2, 3, 2);
    grab_counters(&total,    16*2+ 4+10+10, 4, 3, 4, 3);
    load_count = TWO_BYTES(&log.raw[16*3+4+10+10+14+4]);

    printf("Current Groups %ld Retrys %ld Read %ld Retrys %ld\n",
	current.groups_written,
	current.raw_retries,
	current.groups_read,
	current.ecc_retries);
    printf("Previous Groups %ld Retrys %ld Read %ld Retrys %ld\n",
	previous.groups_written,
	previous.raw_retries,
	previous.groups_read,
	previous.ecc_retries);
    printf("Total Groups %ld Retrys %ld Read %ld Retrys %ld\n",
	total.groups_written,
	total.raw_retries,
	total.groups_read,
	total.ecc_retries);
    printf("Number of loads %d\n", load_count);
}
/********************************************************
 * report_compress -- report compression		*
 ********************************************************/
static void report_compress(void)
{
    printf("Compression ----------------------------------------\n");
    get_page(P_COMPRESS);
    if (raw) raw_page();

    printf("Entries written %ld read %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*0 + 4]),
    	FOUR_BYTES(&log.raw[4+ 8*1 + 4]));

    printf("Records written %ld read %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*2 + 4]),
    	FOUR_BYTES(&log.raw[4+ 8*3 + 4]));

    printf("KB to compression %ld from %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*4 + 4]),
    	FOUR_BYTES(&log.raw[4+ 8*5 + 4]));

    printf("KB to tape %ld from %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*6 + 4]),
    	FOUR_BYTES(&log.raw[4+ 8*7 + 4]));

    printf("Logical entity size %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*8 + 4]));

    printf("Physical entity size %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*9 + 4]));

    printf("Uncompressed entities %ld\n",
    	FOUR_BYTES(&log.raw[4+ 8*10 + 4]));
}
/********************************************************
 * report_compress_mode -- report compression mode	*
 ********************************************************/
static void report_compress_mode(void)
{
    int buffer_mode;	/* The buffering mode */

    printf("Compression mode ----------------------------------------\n");

    mode_sense(0xF);
    if (raw) raw_mode();

    buffer_mode = (log.raw[2] & 0x70) >> 4;

    printf("Buffer mode %d\n", buffer_mode);
    if ((log.raw[2] & 0x80) != 0)
        printf("Write protected\n");
    else
        printf("Write enabled\n");

    printf("Density code 0x%02x\n", log.raw[4+0]);

    if ((log.raw[4+8+2] & 0x80) != 0)
        printf("Compression is enabled\n");
    else
        printf("Compression is disabled\n");

    if ((log.raw[4+8+2] & 0x40) != 0)
        printf("Compression is supported\n");
    else
        printf("Compression not supported\n");

    if ((log.raw[4+8+3] & 0x80) != 0)
        printf("Decompression enabled\n");
    else
        printf("Decompression not enabled\n");

    printf("Compression algorithm   0x%02lx\n", FOUR_BYTES(&log.raw[4+8+4]));
    printf("Decompression algorithm 0x%02lx\n", FOUR_BYTES(&log.raw[4+8+8]));
}
/********************************************************
 * usage -- tell the user what to do			*
 ********************************************************/
static void usage(void)
{
    printf("Usage: mtstatus [-a] [-c][-e][-l][-z][-f] [-r] device\n");
    printf("Options:\n");
    printf("	-a -- Report all reports\n");
    printf("	-c -- Report capacity\n");
    printf("	-e -- Error report\n");
    printf("	-l -- Tape log\n");
    printf("	-r -- Report raw data\n");
    printf("	-z -- Compression report\n");
    printf(" 	-f -- Compression mode\n");
    exit (8);
}
int main (int argc, char *argv[])
{ 
    /* Report types */
    /* These are go into a bitmapped variable */
    /* and are used to decide which reports to print */

#define R_ALL 	 	-1		/* Report everything */
#define R_CAPACITY 	(1 << 0)	/* Report capacity */
#define R_ERROR 	(1 << 1)	/* Report error log */
#define R_TAPE_LOG	(1 << 2)	/* Report tape log */
#define R_COMPRESS	(1 << 3)	/* Report compression */
#define R_COMP_MODE	(1 << 4)	/* Report compression mode */

#define R_FIRST 	R_CAPACITY	/* First report */
#define R_LAST 	 	R_COMP_MODE	/* Second report */

    int report = 0;		/* Reports to print */

    while ((argc > 1) && (argv[1][0] == '-')) {
	switch (argv[1][1]) {
	    case 'e':
		report |= R_ERROR;
		break;
	    case 'c':
		report |= R_CAPACITY;
		break;
	    case 'l':
	        report |= R_TAPE_LOG;
		break;
	    case 'z':
	        report |= R_COMPRESS;
		break;
	    case 'f':
	        report |= R_COMP_MODE;
		break;
	    case 'a':
		report = R_ALL;
		break;
	    case 'r':
		raw = 1;
		break;
	    default:
		printf("Bad option %s\n", argv[1]);
		usage();
	}
	--argc;
	++argv;
    }
    /* If no reports selected, then use the defaults */
    if (report == 0)
        report = R_CAPACITY | R_COMP_MODE;

    if (argc != 2) {
        usage();
    }

    if ((fd = open(argv[1], O_RDONLY | O_NONBLOCK)) < 0) { 
	if ((fd = open(argv[1], O_RDONLY | O_NONBLOCK)) < 0) { 
	    printf ("Failed to open device %s\n", argv[1]);
	    exit (1);
	}
    }

    if (report == R_ALL)
	report_all();
    else {
	int cur_report;

        for (cur_report = R_FIRST; 
	     cur_report != (R_LAST << 1); 
	     cur_report <<= 1) {

	    if ((cur_report & report) == 0)
	        continue;

	    switch (cur_report) {
		case (int)R_CAPACITY:
		    report_capacity();
		    break;
		case R_ERROR:
		    report_error("Read ", 2);
		    report_error("Write", 3);
		    break;
		case R_TAPE_LOG:
		    report_tape_log();
		    break;
		case R_COMPRESS:
		    report_compress();
		    break;
		case R_COMP_MODE:
		    report_compress_mode();
		    break;
		default:
		    fprintf(stderr, "Internal error: bad report %d\n", 
		    		cur_report);
		    exit (8);
	    }
	}
    }
    return(0);
}
