/******************************************************************************
 * Copyright (C)1996-98 Klaus Alexander Seistrup @ Magnetic Ink, Dianalund, DK
 *
 * Author : 1996 Klaus Alexander Seistrup
 * Created: November 12, 1996
 * $Header: /usr/local/src/Master/vcstime/vcstime.c,v 1.12 1997/04/26 08:41:01 kas Exp $
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITH-
 * OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 ******************************************************************************/

#include <sys/resource.h>
#include <sys/time.h>
#include <getopt.h>
#include <locale.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <time.h>

#ifndef HAVE_SYSCONF
# ifndef RLIMIT_NOFILE
#  include <posix1_lim.h>
# endif
#endif

#define PGMNAME		"vcstime"
#define PGMVER		"0"
#define PGMREV		"2"
#define PGMPL		"2"
#define PGMDATE		"26.IV.1997"

#define DO_DETACH	0x0001

#define END_LONG_OPTS	{NULL,0,NULL,0}

#define MakeID(name,ver,rev,pl,date) \
"@(#)" name " " ver "." rev "." pl " (" date ")\n"

#define USAGE \
"Usage: %s [OPTION]\n"\
"  options are:\n"\
"    -s, --sticky .... don't detach from parent process\n"\
"    -c, --copying ... print copying policy\n"\
"    -h, --help ...... this brief help text\n"\
"    -V, --version ... output version information\n"

#define MINIGPL	\
"Copyright (C) 1996-98 Klaus Alexander Seistrup @ Magnetic Ink, DENMARK.\n"\
"\n"\
"This program is free software; you can redistribute it and/or modify it\n"\
"under the terms of the GNU General Public License as published by the Free\n"\
"Software Foundation; either version 2 of the License, or (at your option)\n"\
"any later version.\n"\
"\n"\
"This program is distributed in the hope that it will be useful, but WITH-\n"\
"OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or\n"\
"FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for\n"\
"more details.\n"\
"\n"\
"You should have received a copy of the GNU General Public License along\n"\
"with this program; if not, write to the Free Software Foundation, Inc.,\n"\
"675 Mass Ave, Cambridge, MA 02139, USA.\n"

typedef enum
{
  FALSE, TRUE
}
bool;

static int  fdprintf __P ((int fd, const char *fmt, ...));
static void detach __P ((void));
static void fatal __P ((const char *)) __attribute__ ((noreturn));
static void log_signal __P ((int)) __attribute__ ((const));
static void sigterm_handler __P ((int));
static void sigwinch_handler __P ((int));
static void vcstime __P ((void)) __attribute__ ((noreturn));

static const char PgmID[] = MakeID (PGMNAME, PGMVER, PGMREV, PGMPL, PGMDATE);
static const char RCSid[] = "$Id: vcstime.c,v 1.12 1997/04/26 08:41:01 kas Exp $";
static const char VCS_NAME[] = "/dev/vcs0";

int XPos;

static void
fatal (const char *msg)
{
  syslog (LOG_USER | LOG_ERR, "%s: %m\n", msg);
  exit (EXIT_FAILURE);
}

#ifndef HAVE_BASENAME
/**/
static char *
basename (char *pathname)
{
  char *filename = strrchr (pathname, '/');

  if (filename)
    if (*(filename+1))
      return filename+1;

  return pathname;
}
#endif

/**/
static void
detach (void)
{
#ifdef HAVE_SYSCONF
  long nfds;
#else
# ifdef RLIMIT_NOFILE
  struct rlimit rl;
# endif
  int nfds;
#endif
  int err, fd;

  switch ((err = fork ()))
    {
    case 0:
      break;
    case -1:
      fatal ("fork() didn't");
      /* break; */
    default:
      exit (0);			/* Parent exits here... */
      /* break; */
    }
#ifdef HAVE_SYSCONF
  nfds = sysconf (_SC_OPEN_MAX);
#else
# ifdef RLIMIT_NOFILE
  if (getrlimit (RLIMIT_NOFILE, &rl) < 0)
    fatal ("getrlimit() failed");

  nfds = rl.rlim_max;
# else
  nfds = _POSIX_OPEN_MAX;
# endif
#endif
  /* Close all file descriptors */
  for (fd = nfds - 1; fd >= 0; --fd)
    (void) close (fd);

  (void) chdir ("/");		/* Make sure we aren't on a mounted fs */
  (void) setsid ();		/* Run child in a new session */
  errno = 0;			/* Probably got set to EBADF from close() */
  syslog (LOG_USER | LOG_INFO, PgmID + 4);
}

static void
log_signal (int sig)
{
  if (sig)
    {
      syslog (LOG_USER | LOG_INFO, "%s.\n", strsignal (sig));
      (void) signal (sig, SIG_DFL);
    }
}

/**/
static void
sigterm_handler (int sig)
{
  int *xpos = &XPos;

  log_signal (sig);

  if (*xpos > 0)
    {
      int fda = open (VCS_NAME, O_WRONLY);

      if (fda >= 0)
	{
	  long lbuf[2];

	  lbuf[0] = lbuf[1] = 0x20202020;   /* Oh, well... *;) */

	  (void) lseek (fda, *xpos, SEEK_SET);
	  (void) write (fda, (void *) lbuf, 7);
	  (void) close (fda);
	}
    }

  if (sig)
    exit (EXIT_SUCCESS);
}

/**/
static void
sigwinch_handler (int sig)
{
  static const char VCSA_NAME[] = "/dev/vcsa0";
  int fda;

  log_signal (sig);

  if ((fda = open (VCSA_NAME, O_RDONLY)) >= 0)
    {
      char rc[2];

      if (read (fda, (void *) rc, 2) == 2)
	{
	  sigterm_handler (0);

	  XPos = (int) (rc[1] > 7 ? rc[1] : 80) - 7;
	  (void) signal (SIGWINCH, sigwinch_handler);
	}
      else
	fatal (VCSA_NAME);

      (void) close (fda);
    }
  else
    fatal (VCSA_NAME);
}

/**/
static void
vcstime (void)
{
  int fd = open (VCS_NAME, O_WRONLY);

  if (fd >= 0)
    {
      int *xpos = &XPos;
      struct timeval *now = (struct timeval *) alloca (sizeof (*now));
      char *buf = (char *) alloca (8);

      buf[0] = buf[6] = 0x20;

      sigwinch_handler (0);
      (void) signal (SIGTERM, sigterm_handler);

      for (;;)
	{
	  long   tid;
	  struct tm *tp;

	  (void) gettimeofday (now, NULL);
	  tp = localtime ((time_t *) &(now->tv_sec));

	  tid = tp->tm_hour;
	  buf[1] = (char) (tid / 10) + '0';
	  buf[2] = (char) (tid % 10) + '0';

	  buf[3] = (char) (now->tv_usec < 500000 ? ':' : ' ');

	  tid = tp->tm_min;
	  buf[4] = (char) (tid / 10) + '0';
	  buf[5] = (char) (tid % 10) + '0';

	  (void) lseek (fd, *xpos, SEEK_SET);
	  (void) write (fd, (void *) buf, 7);

	  now->tv_sec  = 0L;
	  now->tv_usec = (250000L - now->tv_usec % 250000L) * 1000L;

	  (void) nanosleep ((struct timespec *) now, NULL);
	}
    }
  fatal (VCS_NAME);
}

/**/
static int
fdprintf(int fd, const char *fmt, ...)
{
  int cnt;
  va_list args;
  char *buf = (char *) alloca (1024);

  va_start (args, fmt);
  cnt = vsprintf (buf, fmt, args);
  va_end (args);

  return write (fd, buf, cnt);
}

/**/
int
main (int argc, char *const *argv)
{
  static const struct option long_opts[] =
    {
	{"sticky",         0, NULL, 's'},
	{"copying-policy", 0, NULL, 'c'},
	{"help",           0, NULL, 'h'},
	{"version",        0, NULL, 'V'},
        END_LONG_OPTS
    };
  char *program_name = (char *) basename (argv[0]);
  bool sticky = FALSE;
  int opt, idx = 0;

  (void) setlocale (LC_ALL, "");
  openlog (program_name, LOG_CONS | LOG_PID, LOG_USER);

  while ((opt = getopt_long (argc, argv, "schV?", long_opts, &idx)) != EOF)
    {
      switch (opt)
	{
	case 's':
	  sticky = TRUE;
	  break;
	case 'c':
	  (void) fdprintf (STDOUT_FILENO, MINIGPL);
	  exit (EXIT_SUCCESS);
	  /* break; */
	case 'h':
	  (void) fdprintf (STDOUT_FILENO, USAGE, program_name);
	  exit (EXIT_SUCCESS);
	  /* break; */
	case 'V':
	  (void) fdprintf (STDOUT_FILENO, PgmID + 4);
	  exit (EXIT_SUCCESS);
	  /* break; */
	default:
	  (void) fdprintf (STDERR_FILENO, "Try `%s --help' for more information.\n", program_name);
	  exit (EXIT_FAILURE);
	  /* break; */
	}
    }

  if (!sticky)
    detach ();

  vcstime ();

  /* NOTREACHED */
  return EXIT_FAILURE;
}

/* EOF */
