#include "kcrontab.moc"
#include <qregexp.h>


// -- class KCrontab implementation

KCrontab::KCrontab(const char *)
{
  setCaption(kapp->getCaption());

  config = kapp->getConfig();

  setupToolBar();
  setupStatusBar();
  setupMenuBar();
  setupAccelerators();
  setupData();
 
  /* ********************************************************************** */
 
  frMain = new QFrame(this, "frMain");
  frMain->setMinimumSize(580, 350);

  /* ********************************************************************** */

  khead = new KHeader(frMain, "khead", 8, KHeader::Resizable);
  khead->setMinimumSize(560, 20);

  khead->setHeaderLabel(0, "*", AlignLeft | AlignVCenter);
  khead->setHeaderLabel(1, i18n("Hour"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(2, i18n("Minute"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(3, i18n("Day"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(4, i18n("Month"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(5, i18n("Weekday"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(6, i18n("User"), AlignLeft | AlignVCenter);
  khead->setHeaderLabel(7, i18n("Command"), AlignLeft | AlignVCenter);

  crontable = new CronTable(frMain, "crontable");
  crontable->setMinimumSize(564, 60);
  crontable->setFrameStyle(51);
  crontable->setLineWidth(2);
  crontable->setAutoDelete(TRUE);
  crontable->setUser(user);

  connect(crontable, SIGNAL(doubleClick()),
          this, SLOT(slotEdit()));
  connect(khead, SIGNAL(sizeChanged(int, int)),
          crontable, SLOT(setColumnWidth(int, int)));
  connect(crontable, SIGNAL(hSliderMoved(int)),
          khead, SLOT(setOrigin(int)));

  config->setGroup(kapp->appName());  
  QString s;
  int     n;
  for (int i = 0; i <= khead->numHeaders() - 1; i++)
  {
    s.sprintf("Header%d", i);
    n = config->readNumEntry(s);
    if (!n)
      if (i == 0)
        khead->setHeaderSize(i, 15);
      else if (i < khead->numHeaders() - 1)
        khead->setHeaderSize(i, 60);
      else
        khead->setHeaderSize(i, 213);
    else
      khead->setHeaderSize(i, n);
  }

  /* ********************************************************************** */

  mainLayout = new QGridLayout(frMain, 2, 1, 4);
  mainLayout->addWidget(khead, 0, 0, AlignCenter);
  mainLayout->setRowStretch(0, 0);
  mainLayout->addWidget(crontable, 1, 0, AlignCenter);
  mainLayout->setRowStretch(1, 1);
  mainLayout->activate();

  /* ********************************************************************** */

  setView(frMain, FALSE);
  setMinimumSize(600, 300);
  resize(600, 300);

  config->setGroup (kapp->appName());
  int width  = config->readNumEntry("Width");
  int height = config->readNumEntry("Height");

  if (width < minimumSize().width())
    width = minimumSize().width();
  if (height < minimumSize().height())
    height = minimumSize().height();

  resize(width, height);

  changed = FALSE;
  editVariables = FALSE;
  slotRead();

  crontable->setFocus();
}

KCrontab::~KCrontab()
{
  delete userList;

  delete statusbar;
  delete toolbar;
  delete file;
  delete options;
  delete help;
  delete menubar;
}

void KCrontab::closeEvent(QCloseEvent *e)
{
  (void) e;
  slotExit();
}

 /* *********************************************************************** */

void KCrontab::setupMenuBar()
{
  file    = new QPopupMenu;
  edit    = new QPopupMenu;
  options = new QPopupMenu;
  help    = kapp->getHelpMenu(FALSE, "kcrontab 0.1.7\n\n" 
	                      "(c) by Michael Bialas <michael@quinto.ruhr.de>\n"
                              "and David Klemm <klemm@abacus.at>");
	
  file->insertItem(i18n("&New crontab"),
                   this, SLOT(slotNew()));
  file->insertItem(i18n("&Read crontab"),
                   this, SLOT(slotRead()));
  if (getuid() == 0)
    file->insertItem(i18n("Read &system crontab"),
                     this, SLOT(slotReadSystem()));
  file->insertItem(i18n("&Write crontab"),
                   this, SLOT(slotWrite()));
  file->insertSeparator();
  file->insertItem(i18n("E&xit"),
                   this, SLOT(slotExit()));

  edit->insertItem(i18n("&Add Entry"),
                   this, SLOT(slotAdd()));
  edit->insertItem(i18n("&Edit Entry"),
                   this, SLOT(slotEdit()));
  edit->insertItem(i18n("&Remove Entry"),
                   this, SLOT(slotRemove()));
  edit->insertSeparator();
  edit->insertItem(i18n("Edit &Variables"),
                   this, SLOT(slotVariable()));

  config->setGroup (kapp->appName());
  int n;
  
  n = config->readNumEntry ("Hide_Toolbar");
  if (!n) 
    hide_toolbar = FALSE;	
  else hide_toolbar = n;

  if (hide_toolbar)
  {
    toolID = options->insertItem(i18n("Show &Tool Bar"),
                                 this, SLOT(slotToggleToolBar()));
    enableToolBar(KToolBar::Hide, toolbarID);
  }
  else
  {
    toolID = options->insertItem(i18n("Hide &Tool Bar"),
                                 this, SLOT(slotToggleToolBar()));
    enableToolBar(KToolBar::Show, toolbarID);
  }
			       
  n = config->readNumEntry ("Hide_Statusbar");
  if (!n) 
    hide_statusbar = FALSE;		
  else
    hide_statusbar = n;

  if (hide_statusbar)
  {
    statID = options->insertItem(i18n("Show &Status Bar"),
                                 this, SLOT(slotToggleStatusBar()));
    enableStatusBar(KStatusBar::Hide);
  }
  else
  {
    statID = options->insertItem(i18n("Hide &Status Bar"),
                                 this, SLOT(slotToggleStatusBar()));
    enableStatusBar(KStatusBar::Show);
  }
	
  options->insertSeparator();
  options->insertItem(i18n("Sa&ve settings"),
                      this, SLOT(slotSaveSettings()));

  menubar = new KMenuBar(this, "menubar");
  menubar->insertItem(i18n("&File"), file); 
  menubar->insertItem(i18n("&Edit"), edit);
  menubar->insertItem(i18n("&Options"), options);
  menubar->insertSeparator();
  menubar->insertItem(i18n("&Help"), help);

  setMenu(menubar);
}

void KCrontab::setupToolBar()
{
  KIconLoader *loader = kapp->getIconLoader();
  QPixmap     pixmap;

  toolbar = new KToolBar(this, "toolbar");

  pixmap = loader->loadIcon("filenew.xpm");
  toolbar->insertButton(pixmap, 0,
                        SIGNAL(clicked()), this, SLOT(slotNew()),
                        TRUE, i18n("New crontab"));

  pixmap = loader->loadIcon("fileopen.xpm");
  toolbar->insertButton(pixmap, 1,
                        SIGNAL(clicked()), this, SLOT(slotRead()),
                        TRUE, i18n("Read crontab"));

  pixmap = loader->loadIcon("filefloppy.xpm");
  writeID = toolbar->insertButton(pixmap, 2,
                                  SIGNAL(clicked()), this, SLOT(slotWrite()),
                                  TRUE, i18n("Write crontab"));
 
  toolbar->insertSeparator();

  pixmap = loader->loadIcon("editpaste.xpm");
  toolbar->insertButton( pixmap, 3,
                        SIGNAL(clicked()), this, SLOT(slotAdd()), 
                        TRUE, i18n("Add Entry"));
			  
  pixmap = loader->loadIcon("openbook.xpm");
  toolbar->insertButton( pixmap, 4,
                        SIGNAL(clicked()), this, SLOT(slotEdit()),
                        TRUE, i18n("Edit Entry"));
			  
  pixmap = loader->loadIcon("editcut.xpm");
  toolbar->insertButton(pixmap, 5,
                        SIGNAL(clicked()), this, SLOT(slotRemove()),
                        TRUE, i18n("Remove Entry"));

  toolbar->insertSeparator();

  pixmap = loader->loadIcon("help.xpm");
  toolbar->insertButton(pixmap, 6,
                        SIGNAL(clicked()), this, SLOT(slotHelp()),
                        TRUE, i18n("Help"));

  toolbar->setBarPos(KToolBar::Top);
  toolbarID = addToolBar(toolbar);
}

void KCrontab::setupStatusBar()
{

  config->setGroup(kapp->appName());
  int n;

  n = config->readNumEntry("Hide_Statusbar");
  if (!n) 
    hide_statusbar = FALSE;
  else
    hide_statusbar = n;
  
  statusbar = new KStatusBar(this, "statusbar");
  statusbar->insertItem("", 0);

  setStatusBar(statusbar);
}

void KCrontab::setupAccelerators()
{
  KStdAccel stdaccel(kapp->getConfig());
  accel = new QAccel(this);

  accel->connectItem(accel->insertItem(stdaccel.quit()),
                     kapp, SLOT(quit()));
  accel->connectItem(accel->insertItem(stdaccel.openNew()),
                     this, SLOT(slotNew()));
  accel->connectItem(accel->insertItem(stdaccel.open()),
                     this, SLOT(slotRead()));
  accel->connectItem(accel->insertItem(stdaccel.save()),
                     this, SLOT(slotWrite()));
}

void KCrontab::setupData()
{
  struct passwd *pw;
  
  pw = getpwuid(getuid());
  user = pw->pw_name;

  userList = new QStrList();
  userList->clear();

  // Read in user-list
  QFile file("/etc/passwd");
  if (file.open(IO_ReadOnly))
  {
    QTextStream stream(&file);
    QString     line;

    while (!stream.eof())
    {
      line = stream.readLine();
      userList->append(strtok(line.data(), ":"));
    }
    file.close();
  }
}

// -- Slots

void KCrontab::slotNew()
{
  switch (QMessageBox::information(this, kapp->appName(), 
          i18n("All entries will be deleted.\nAre you sure?"),
          i18n("OK"), i18n("Cancel"), 0, 1))
  {
    case 0: changed = FALSE;
            crontable->clear();
            crontable->repaint();
            variables.clear();
            statusbar->changeItem(i18n("New crontab..."), 0);
            break;

    case 1:
            break;
  }
}

void KCrontab::slotRead()
{
  readCrontab(FALSE);
}

void KCrontab::slotReadSystem()
{
  readCrontab(TRUE);
}

void KCrontab::slotWrite()
{
  CronRow *row;

  if (editVariables)
  {
    statusbar->changeItem(i18n("Write cancelled, because the edit " \
                               "variables dialog is open."), 0);
    return;
  }

  for (int i = 0; i < crontable->count() - 1; i++)
  {
    row = (CronRow *) crontable->getRow(i);
    if (row->isEdited())
    {
      statusbar->changeItem(i18n("Write cancelled, because an edit entry " \
                                 "dialog is open."), 0);
      return;
    }
  }

  changed = FALSE;

  if (syscron)
    tmpFilePath = "/etc/crontab";
  else
    tmpFilePath.sprintf("/tmp/kcrontab.%d", getpid());

  QFile out(tmpFilePath.data());
  if (out.open(IO_WriteOnly))
  {
    QTextStream stream(&out);

    for (int i = 0; i < (int) variables.count(); i++)
      stream << variables.at(i) << '\n';

    for (int i = 0; i < crontable->count(); i++)
      stream << crontable->line(i).data() << '\n';

    out.close();
  }

  if (syscron)
  {
    statusbar->changeItem(i18n("Crontab written..."), 0);
    return;
  }

  QString command;
  command.sprintf("crontab %s", tmpFilePath.data());
  int ret = system(command.data());

  if (ret != 0)
    QMessageBox::critical(this, kapp->appName(), 
                          i18n("Could not start \"crontab\"."));
  else
   statusbar->changeItem(i18n("Crontab written..."), 0);
  
  unlink(tmpFilePath.data());
}

void KCrontab::slotExit()
{
  if (changed)
  {
    switch (QMessageBox::information(this, kapp->appName(),
            i18n("The crontab has been modified.\n" \
                 "Would you like to save?"),
            i18n("Yes"), i18n("No"), i18n("Cancel"), 0, 2))
    {
      case 0: // Save changes
              slotWrite();
              break;

      case 1: // Don't save, exit
              break;

      case 2: // Cancelled
              return;
              break;
    }
  }

  kapp->quit();
}

void KCrontab::slotAdd()
{
  statusbar->changeItem(i18n("Adding an entry..."), 0);

  edEntry = new EditEntry(syscron);
  edEntry->setUser(userList, user);
  connect(edEntry, SIGNAL(newEntry(CronRow *, bool)),
          this, SLOT (slotNewEntry(CronRow *, bool)));
  edEntry->show();
}

void KCrontab::slotEdit()
{
  int index = crontable->currentItem();

  if (index == -1)
  {
    slotAdd();
    return;
  }

  CronRow *row = (CronRow *) crontable->getRow(index);

  if (!row->isEdited())
  {
    statusbar->changeItem(i18n("Editing an entry..."), 0);

    edEntry = new EditEntry(syscron, row);
    edEntry->setUser(userList, user);
    connect(edEntry, SIGNAL(newEntry(CronRow *, bool)),
            this, SLOT (slotNewEntry(CronRow *, bool)));
    edEntry->show();
  }
  else
    statusbar->changeItem(i18n("Editing in progress."), 0);
}

void KCrontab::slotRemove()
{
  int index = crontable->currentItem();

  if (index == -1)
  {
    statusbar->changeItem(i18n("Select an entry first."), 0);
    return;
  }

  CronRow *row = (CronRow *) crontable->getRow(index);
  if (row->isEdited())
  {
    statusbar->changeItem(i18n("Current entry is edited, " \
                               "so it could not be deleted."), 0);
    return;
  }

  changed = TRUE;
  crontable->deleteRow(index);
}

void KCrontab::slotVariable()
{
  if (!editVariables)
  {
    statusbar->changeItem(i18n("Editing variables..."), 0);

    editVariables = TRUE;

    EditVariable *edVariable = new EditVariable(0, "edVariable", &variables);
    connect(edVariable, SIGNAL(onClose(bool)), SLOT(slotVariableClose(bool)));
    edVariable->show();
  }
}

void KCrontab::slotToggleToolBar()
{
  if (hide_toolbar)
  {
    hide_toolbar = FALSE;
    enableToolBar(KToolBar::Show, toolbarID);
    options->changeItem(i18n("Hide &Tool Bar"), toolID);
  }
  else
  {
    hide_toolbar = TRUE;
    enableToolBar(KToolBar::Hide, toolbarID);
    options->changeItem(i18n("Show &Tool Bar"), toolID);
  }
}

void KCrontab::slotToggleStatusBar()
{
  if (hide_statusbar)
  {
    hide_statusbar = FALSE;
    enableStatusBar(KStatusBar::Show);
    options->changeItem(i18n("Hide &Status Bar"), statID);
  }
  else
  {
    hide_statusbar = TRUE;
    enableStatusBar(KStatusBar::Hide);
    options->changeItem(i18n("Show &Status Bar"), statID);
  }
}

void KCrontab::slotSaveSettings ()
{
  config->setGroup(kapp->appName());
  config->writeEntry("Width", width());
  config->writeEntry("Height", height());
  config->writeEntry("Hide_Toolbar", hide_toolbar);
  config->writeEntry("Hide_Statusbar", hide_statusbar);

  QString s;
  for (int i = 0; i <= khead->numHeaders() - 1; i++)
  {
      s.sprintf("Header%d", i);
      config->writeEntry(s, khead->getHeaderSize(i));
  }
  config->sync();
}

void KCrontab::slotHelp()
{
  kapp->invokeHTMLHelp("", "");
}

void KCrontab::slotNewEntry(CronRow *row, bool editMode)
{
  changed = TRUE;

  if (editMode)
    crontable->repaint();
  else
  {
    crontable->addItem(row);
    crontable->setCurrentItem(crontable->count() - 1);
  }
}

void KCrontab::slotVariableClose(bool _changed)
{
  changed = _changed;
  editVariables = FALSE;
}

void KCrontab::readCrontab(bool _syscron)
{
  QString fname;
  QString comment;
  CronRow *row;
  int     ret, i;

  if (editVariables)
  {
    statusbar->changeItem(i18n("Read cancelled, because the edit " \
                               "variables dialog is open."), 0);
    return;
  }

  for (i = 0; i < crontable->count() - 1; i++)
  {
    row = (CronRow *) crontable->getRow(i);
    if (row->isEdited())
    {
      statusbar->changeItem(i18n("Read cancelled, because an edit entry " \
                                 "dialog is open."), 0);
      return;
    }
  }

  if (changed)
    if (QMessageBox::information(this, kapp->appName(), 
        i18n("Reading the crontab will destroy your changes.\n"
             "Are you sure?"), i18n("OK"), i18n("Cancel"), 0, 1))
    {
      statusbar->changeItem(i18n("Read cancelled."), 0);
      return;
    } 

  changed = FALSE;
  syscron = _syscron;
  crontable->setCrontype(syscron);

  crontable->setUpdatesEnabled(FALSE);
  crontable->clear();
  variables.clear();

  if (!syscron)
  {
    QString command;
    fname.sprintf("/tmp/kcrontab.%03d", getpid());
    command.sprintf("crontab -l > %s 2> /tmp/kcrontab.err", fname.data());
    ret = system(command.data());
    setCaption("kcrontab - " + user);
  }
  else
  {
    fname = "/etc/crontab";
    ret   = 0;
    setCaption("kcrontab - /etc/crontab");
  }

  if (ret == 0)
  {  
    fin.setName(fname.data());
    if (fin.open(IO_ReadOnly))
    {
      QString     line;
      QString     name, value;
      QTextStream stream(&fin);
      QRegExp     reWhiteSpace("^\\s*$", TRUE, FALSE);
      QRegExp     reVars("^\\s*[A-Z0-9_]+\\s*=\\s*.+$", FALSE, FALSE);
      QRegExp     reComment("^\\s*#.*$", TRUE, FALSE);
                              
      while (!stream.eof())
      {
	line = stream.readLine();

	if (line.find(reWhiteSpace) != -1)
          comment = "";
	else if (line.find(reComment) != -1)
	{
          if (line.left(2) == "#\\")
	  {
	    crontable->addItem(line, comment);
	    comment = "";
	  }
	  else
	  {
            comment = line.right(line.length() - 1);
	    comment = comment.stripWhiteSpace();
	  }
	}
	else if (line.find(reVars) != -1)
	{
          comment = "";

	  int eq  = line.find('=');
	  name    = line.left(eq);
	  name    = name.stripWhiteSpace();
	  value   = line.remove(0, eq + 1);
	  value   = value.stripWhiteSpace();

	  variables.append(name + '=' + value);
	}
	else
	{
	  crontable->addItem(line, comment);
	  comment = "";
	}
      }
      fin.close();
    }
  }
  else if (ret != 256)
    statusbar->changeItem(i18n("Error while reading crontab."), 0);

  if (!syscron)
  {
    unlink(fname.data());
    unlink("/tmp/kcrontab.err");
  }

  crontable->setUpdatesEnabled(TRUE);
  crontable->repaint();

  if (crontable->count() > 0)
  {
    statusbar->changeItem(i18n("Crontab read."), 0);
    crontable->setCurrentItem(0);
  }
  else
    statusbar->changeItem(i18n("Crontab is empty."), 0);
}
