/*
 *   main.c handlings for the DCF serial clock daemon.
 *
 *   Copyright 1997        quacks@paula.owl.de (Joerg Krause)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "dcf77.h"

static void	InitSignalHandlers(void);
static void	PutPID(void);
static int	CheckPID(void);
static int	CheckCmdline(int pid);

char *myname;			/* our name from the command line */
char *port     = "/dev/cua0";	/* Default port for the clk */
int  cmos      = FALSE; 	/* TRUE == Update CMOS Clk (requires RTC Kernel) */
int  lastday    = 0;		/* Day of the last time update */
int  loglevel   = LEV_ERR_ONLY;	/* notification level */
int  rtc        = -1;		/* File descriptor for /dev/rtc */
int  fd         = -1;		/* File descriptor for /dev/cuaX **/
int  kill_other = FALSE;	/* try to kill already running dcf77d ? */
int  pidfile    = -1;


char *version  = "dcf77d 1.01";

int makedaemon(void)
{
  pid_t pid;

  if ( ( pid = fork()) < 0)
    return(-1);
  else
  {
    if (pid != 0)
      exit(0); /* Quitting parent process */
  }
  /* Child is running */
  
  setsid();

  chdir("/");

  umask(0);

  return(0);
}


/*
 * Pick up a numeric argument.  It must be nonnegative and in the given
 * range (except that a vmax of 0 means unlimited).
 */
static long numarg(char *meaning, long vmin, long vmax)
{
  char *p;
  long val;
  
  val = strtol(optarg, &p, 10);
  if (*p)
  {
    fprintf(stderr, "illegal %s -- %s", meaning, optarg);
    exit(1);
  }
  if (val < vmin || (vmax && val > vmax))
  {
    fprintf(stderr, "%s must be between %ld and %ld", meaning, vmin, vmax);
    exit(1);
  }
  return (val);
}

static void usage(char *progname)
{
  printf("%s (c) 1997 by Joerg Krause\n\n",version);
  printf("usage: %s [-k] [-h] [-c] [-p device] [-l loglevel]\n",progname);
}

int main(int argc, char *argv[])
{
  char *name;
  int ch;

  /* check if we're really invoked by root
  ** or a users with root-privilegs
  */
  if (getuid() != 0)
  {
    fprintf(stderr, "dcf77d: Sorry, must be root to execute this.\n");
    exit(1);
  }

  /* get our name */
  if ((name = (char*)strrchr(argv[0], '/')) != NULL)
  {
    name++;
    myname = strdup(name);
  }
  else
  {
    myname = strdup(argv[0]);
  }
  
  /* Scan thru the commandline */
  while ((ch = getopt(argc,argv, "khcl:p:")) != -1)
  {
    switch (ch)
    {
      case 'h':
	/* print help */
	usage(argv[0]);
	exit(0);
	break;
      case 'p':
	/* tty or cua device to use */
	port = optarg;
	break;
      case 'c':
	/* Set C-MOS time aswell */
	cmos = TRUE;
	break;
      case 'l':
	/* loglevel */
	loglevel = numarg("Log level: 0=Quiet, 1=Errors, 2=All,",0L,2L);
	break;
      case 'k':
	/* kill already running daemons */
	kill_other = TRUE;
	break;	
      default:
	usage(argv[0]);
	exit(1);
	break;
    }
  }

  /* check for running daemon */
  if (CheckPID() != FALSE)
  {
    fprintf(stderr, "dcf77d: Already running.\n");
    exit(1);
  }

  /* install sig-handler */
  InitSignalHandlers();

  /* Create a daemon */
  if (makedaemon() != 0)
  {
    fprintf(stderr,"Failed to create daemon process");
    exit(1);
  }

  /* put PID file */
  PutPID();
  
  /* Start logging */
  openlog("dcf77d", 0, LOG_DAEMON);

  startclock();

  exit(0);
}


/*******************
** InitSignalHandler() -- 
** setup signal handlers.
*/

static void InitSignalHandlers()
{
  /* install signal handler */

  if (signal(SIGINT, SIG_IGN) != SIG_IGN)
  {
    signal(SIGINT, Done);
  }
  
  if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
  {
    /* maybe restart on HUP ?? */
    signal(SIGHUP, Done);
  }

  if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
  {
    signal(SIGQUIT, Done);
  }
  
  signal(SIGTERM, Done);

  return;
}

/******************/


/*******************
** Done --  
** sighandler and cleanup.
*/

void Done(int sig)
{

  /* cleanup and exit quiet */
  if (pidfile != -1)
  {
    close(pidfile);
    unlink(DCF_PID);
  }
  
  if (fd != -1)
    close(fd);

  /* paranoia */
  if ((cmos) && (rtc != -1))
    close(rtc);

  exit(0);
}

/******************/


/*******************
** PutPID -- 
** save PID to DCF_PID file.
*/

static void PutPID()
{

  char pidbuffer[MAX_PID_LEN];
  int pidlen;

  snprintf(pidbuffer, MAX_PID_LEN, "%d", (int)getpid());
  pidlen = strlen(pidbuffer);
  
  if ((pidfile = open(DCF_PID,
		      (O_RDWR|O_CREAT|O_TRUNC),
		      (S_IRUSR|S_IWUSR))) == -1)
  {
    fprintf(stderr, "dcf77d: Couldn't open %s.\n", DCF_PID);
    exit(1);
  }

  if (write(pidfile, (void *)pidbuffer, pidlen) != pidlen)
  {
    close(pidfile);
    pidfile = -1;
    unlink(DCF_PID);
    fprintf(stderr, "dcf77d: Couldn't write PID to %s.\n", DCF_PID);
    exit(1);
  }

  return;
}

/******************/


/*******************
** CheckPID -- 
** check for another instance already running.
**
** Be sure that we never write to a pidfile
** with insecure perms in PutPID().
**
** returns FALSE if other process doesn't exist,
** TRUE otherwise.
**
** exits whenever an error occurs or the pidfile
** has wrong format.
**
** note that sig-handlers should be installed
** after this is called.
*/

static int CheckPID()
{
  struct stat o_pid_stat;
  int o_pid_fd = -1;
  char o_pid_string[MAX_PID_LEN];
  int o_pid_len, o_pid;

  errno = 0;
  if (stat(DCF_PID, &o_pid_stat) == -1)
  {
    /* if file does not exist,
    ** simply return FALSE indicating
    ** that all is clear and we're ready to go.
    */
    if (errno == ENOENT)
      return(FALSE);
    else
    {
      fprintf(stderr, "dcf77d: Can't stat %s.\n", DCF_PID);
      exit(1);
    }
  }

  /* before accessing the pid-file do
  ** some paranoia checks.
  ** (must be a regular file, belong to root, has rw for
  **  root set and is not executable for anyone and has not
  **  S_ISUID or S_ISGID bit set)
  */
  if ((o_pid_stat.st_uid == 0) &&
      (S_ISREG(o_pid_stat.st_mode)) &&
      (o_pid_stat.st_mode & (S_IRUSR|S_IWUSR)) &&
      (!(o_pid_stat.st_mode &(S_ISUID|S_ISGID|S_IXUSR|S_IXGRP|S_IXOTH))))
  {
    /* o.k., access */
    if ((o_pid_fd = open(DCF_PID, O_RDONLY)) == -1)
    {
      fprintf(stderr, "dcf77d: Error while accessing %s.\n",DCF_PID);
      exit(1);
    }
      
    if ((o_pid_len = read(o_pid_fd,o_pid_string,(MAX_PID_LEN-1))) == -1)
    {
      close(o_pid_fd);
      fprintf(stderr, "dcf77d: Error while reading %s.\n",DCF_PID);
      exit(1);
    }      

    close(o_pid_fd);
        
    /* check if read buffer contains a valid process id */
    if (o_pid_len > 0)
    {
      char *endptr;
      
      /* force last char to be \0-termination*/
      o_pid_string[o_pid_len + 1] = '\0';

      /* convert */
      o_pid = strtoul(o_pid_string, &endptr, 10);

      if (((void *)endptr != (void *)(o_pid_string + o_pid_len)) ||
	  (o_pid == ULONG_MAX) ||
	  (o_pid < 2))
      {
	fprintf(stderr, "dcf77d: Wrong data in %s.\n", DCF_PID);
	exit(1);
      }

      /* check if there is a running process with this pid */
      errno = 0;
      if (kill(o_pid, 0) == -1)
      {
	if (errno == ESRCH)
	{
	  /* no such process, maybe an old file,
	  ** try to delete it.
	  */
	  unlink(DCF_PID);
	  /* ... and go */
	  return(FALSE);
	}
      }
      
      /* there is a running process,
      ** check its /proc entry
      */
      if (CheckCmdline(o_pid) == TRUE)
      {
	/* process with same name already running */
	if (kill_other == TRUE)
	{
	  int i;
	  /* try to kill it */
	  kill(o_pid, SIGTERM);
	  for (i = 0; i < 5; i++)
	  {
	    /* wait */
	    sleep(1);
	    errno = 0;
	    if (kill(o_pid, 0) == -1)
	    {
	      /* has it gone ? */ 
	      if (errno == ESRCH)
	      {
		/* file should have gone too,
		** but unlink it anyway */
		unlink(DCF_PID);
		return(FALSE);
	      }
	    }
	  }
	  /* process has not terminated */
	  fprintf(stderr, "dcf77d: Unable to kill process %d.\n", o_pid); 
	  return(TRUE);
	}

	/* process should not be killed */
	return(TRUE);
      }
      else
      {
	/* process with same name not found,
	** pidfile should be invalid
	*/
	unlink(DCF_PID);
	return(FALSE);
      }
    }
  }
  
  /* wrong perms or file-length 0 */
  fprintf(stderr, "dcf77d: Sorry Guy, wrong permissions for %s.\n", DCF_PID);
  exit(1);
}

/******************/


/*******************
** CheckCmdline --  
** check for occurence of myname in /proc/pid/cmdline
** entry for process with 'pid'
*/

static int CheckCmdline(int pid)
{
  char proc_cmd[PROC_CMDLINE_LEN];
  int proc_cmd_fd;
  char *cmdline;
  int cmd_len;
  char *ptr;
  int mynamelen;
  int i;
  int found;

  /* check if this process was invoked
  ** with the same name as we are
  ** (is this secure ?)
  */

  found = FALSE;
  snprintf(proc_cmd, PROC_CMDLINE_LEN, PROC_CMDLINE_FILE, pid);

  if ((cmdline = (char *)calloc((PATH_MAX + 1), sizeof(char))) == NULL)
  {
    fprintf(stderr, "dcf77d: Couldn't allocate memory.\n");
    exit(1);
  }
  
  if ((proc_cmd_fd = open(proc_cmd, O_RDONLY)) == -1)
  {
    free(cmdline);
    fprintf(stderr, "dcf77d: Error while accessing %s.\n", proc_cmd);
    exit(1);
  }
  
  if ((cmd_len = read(proc_cmd_fd, cmdline, PATH_MAX)) == -1)
  {
    free(cmdline);
    close(proc_cmd_fd);
    fprintf(stderr, "dcf77d: Error while reading from %s.\n", proc_cmd);
    exit(1);
  }

  close(proc_cmd_fd);
  
  if (cmd_len > 0)
  {
    /* find myname in cmdline */
    ptr = cmdline;
    mynamelen = strlen(myname);
    i = 0;
    do  
    {
      if (*ptr == '/')
      {
	ptr++;
	i = 0;
	while ((i < mynamelen) && (*ptr == myname[i]))
	{
	  ptr++;
	  i++;
	}
	
	/* found ? */
	if (i == mynamelen)
	{
	  found = TRUE;
	  break;
	}
      }
    } while (ptr++ != (cmdline + cmd_len - 1));
    
    /* maybe no abs. path */
    if (found == FALSE)
    {
      ptr = cmdline;
      i = 0;
      while ((i < mynamelen) && (*ptr == myname[i]))
      {
	ptr++;
	i++;
      }		  
    
      if (i == mynamelen)
      {
	/* yepp, found ! */
	found = TRUE;
      }
    }
  }

  free(cmdline);
  
  /* found = TRUE if myname was found,
  ** FALSE otherwise.
  */
  return(found);
}

/******************/





