/* The code should compile with either ANSI C or K&R compilers. */

/*
 *      Copyright (c) 1993 by California Institute of Technology.
 *      Written by William Deich.  Not derived from licensed software.

 *    You may distribute under the terms of either the GNU General Public
 *    License or the Artistic License, as specified in the README file.
 */


#ifndef SCO
    /* Don't define _XOPEN_SOURCE for SCO: although it uses this for compliance
     * reasons, this define causes some problems with password file negotiation.
     */
#define  _XOPEN_SOURCE
#endif

#define  _ALL_SOURCE	/* AIX needs this to define NSIG */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>

#ifdef __clix__
#include <sys/signal.h>
#ifdef _NSIG
#define NSIG _NSIG
#endif
#else
#include <signal.h>
#endif

#include <sys/wait.h>
#include <sys/param.h>

#include <sys/stat.h>
#ifndef S_IWGRP
#define S_IWGRP 0000020
#endif
#ifndef S_IWOTH
#define S_IWOTH 0000002
#endif

#include <pwd.h>
#include <grp.h>
#include <unistd.h>

#ifdef USE_GETHOSTBYNAME
#include <sys/types.h>
#include <netdb.h>
#endif

#ifdef USE_SYSLOG
#include <syslog.h>
#endif

#ifdef __STDC__
#include <stdlib.h>
#else
#include <malloc.h>
#endif

#ifdef SUNOS5
#include <shadow.h>
#include <sys/time.h>
#include <sys/systeminfo.h>
extern int sysinfo();
#define gethostname(buf, lbuf) (sysinfo(SI_HOSTNAME, (buf), (lbuf)))
#undef NSIG
#define NSIG _sys_nsig
#else
extern int gethostname();
#include <time.h>
#endif

#ifdef SCO
#include <limits.h>
#include <sys/types.h>
#include <sys/signal.h>
#ifdef SCO_SHADOW_PWD
#include <shadow.h>
#endif
#include <malloc.h>
#endif

#ifdef _POSIX_PATH_MAX
#undef MAXPATHLEN
#define MAXPATHLEN _POSIX_PATH_MAX
#else
#ifndef MAXPATHLEN
#define MAXPATHLEN 255
#endif
#endif

#include "version.h"

/*
 * Super allows users to execute other programs, particularly
 * scripts, as root (or another user/group), without unduly
 * compromising security.
 * 
 * Use:
 * 
 *     $0 [-d] [-h | -? | -H | -V] commandname [args...]
 * 
 * Options:
 *	-d -- debug mode: show what would be done, but don't actually
 *		execute a command.
 *	-h, -? -- help: print list of allowed commands, then exit.
 *	-H -- a long-winded variant of -h.
 *	-V -- print version information, then exit.
 * 
 * The super.tab file names each command that super will execute, and
 * says who can use it. It contains lines like:
 * 
 *  commandname    FullPathName   ok-user-pats | options
 * 
 * The `ok-user-pats' are (by default) ed-style patterns, with the addition of
 * csh-style brace expansion.  The actual regular-expression code is
 * Ozan Yigit's implementation of regex; see regex.c for details.
 * (Alternatively, an option can be set so that shell-style patterns are
 * used; see global option "patterns=xxx", below.)

 * The commandname is normally used as just a plain string, but it is actually
 * a pattern that is matched against the user's input command (again, brace
 * expansion is allowed).
 
 * If the commandname is matched, the FullPathName normally specifies the
 * actual command to execute.  In the special case that FullPathName contains
 * an asterisk, the asterisk is replaced by the user's "cmd" string, and
 * this forms the actual command to exec.  This wildcard form allows you to
 * give away access to MANY commands for _TRUSTED_ users -- be very careful
 * about what you give away here.

 * Each ok-user-pattern is in the format
 *	user[:group][@host]   |    :group[@host]   |   @host

 * By default, the "group" field can match either the group name or
 * the user's gid -- this allows the use of group numbers in the
 * /etc/passwd file that aren't in the /etc/group file.  See the
 * #define's of MATCH_DECIMAL_UID and MATCH_DECIMAL_GID for details.

 * To make it easy to exclude some users/groups/hosts, any of these
 * ok-user-patterns can be prefixed with `!' to mean that if matched,
 * the user may NOT execute the command.  Regular patterns and
 * negated patterns are read left-to-right, and the last matched
 * pattern ``wins''.  Thus if the valid user list is
 *		j.*  !joe
 * then the first entry allows any username beginning with `j' to execute
 * the command, but the second entry disallows user `joe'.
 * If entered in the reverse order,
 *		!joe  j.*
 * then the second entry, `j.*', allows all users beginning `j', and
 * therefore the first entry has no effect.

 *  The options can be mixed in with user patterns.  Each option is of the
 *  form   key=value  (no whitespace around the equal sign is allowed).
 *  The valid keys are:
 *  o	info=xxx	This line is printed when help is given.

 *  o	uid=xxx		By default, FullPathName is executed with effective
 *			uid=root, but no changes to the gid or real uid.
 *			This option sets the real and effective user id to xxx.
 *			N.B.  Uid's and gid's are first tried as names, then
 *			as numbers.

 *  o  gid=xxx		By default, the group id remains unchanged when
 *			FullPathName is executed.  This option sets the real
 *			and effective group id to xxx.

 *  o  u+g=zzz		The real and effective uid are set to zzz and
 *			the real and effective gid is set to zzz's login gid.
 *			(u+g conflicts with uid=xxx or gid=yyy, and can't be
 *			used w/ them).

 *  o  env=var[,...]	Super() discards most environment variables and keeps
 *			only a restricted set (see below).  This option adds
 *			the variables var,...  to the kept set.  Use at your
 *			own risk.

 *  o  fd=m[,...]	Super() closes most file descriptors and normally keeps
 *			open just fd=0,1,2 before running the FullPathName.
 *			This option adds the descriptors m,...  to the open set.

 *  o   password=y|n	If y, the user's password is required to execute
 *			the command.  Default taken from global setting.
 *			** This can be entered as either local or global option.
 *			** The local password option, if present, overrides
 *			** the global setting.

 *  o  timeout=mmm	Passwords are good for mmm minutes; then they
 *			have to be re-entered to execute another password
 *			command.  <=0 requires password every time.
 *			Default taken from global setting.
 *			** This can be entered as either local or global option.
 *			** The local timeout option, if present, overrides
 *			** the global setting.

 *  o	nargs=[mmm-]nnn	In addition to any command-line options supplied
 *			with the command string (e.g. "/Full/Path/Name arg1 "),
 *			the user must supply from mmm to nnn args (if mmm and
 *			nnn are given), else exactly nnn args (if nnn only
 *			is given), else any number of args.

 *  o	owner=xxx	The owner of the FullPathName must be xxx, or it
 *			won't be executed.  N.B.  xxx is first tried as a
 *			username, then as a uid.
 *			** This can be entered as either local or global option.
 *			** The local option, if present, overrides
 *			** the global setting.

 * Global options.
 * The above options are set on a per-command basis.  The following options
 * are global.  To set them, the FIRST non-comment line of the super.tab file
 * should read

 *	 	/	/	[globaloptions]

 * (i.e., commandname="/", FullPathName="/", then options).

 * If these options appear mixed in with regular control lines, the
 * response of super(1) is undefined.

 *  o  logfile=xxx	Enable logging information.  Each invocation of super()
			(aside from ones for help) generates an entry in
			file xxx.

 *  o  loguid=xxx	The logfile is opened for writing using uid=xxx
 *			(xxx can be either a username or numeric uid).
 *			This option is useful
 *			when a network of hosts are sharing a cross-mounted
 *			logfile.  Note that networks are typically configured
 *			to not allow root on one host to have root access to
 *			files on another host, so this option allows you to
 *			have the file created/opened under another uid that
 *			does have cross-host access, such as the uid of a
 *			system manager.
 *			Default: loguid=0.

 *  o  syslog=y|n	If y, logfile messages are logged using syslog().
 *			(This is done independently of the setting of
 *			logfile=xxx).

 *  o  mail="mail-command"	Notices of super failures are piped to
 *			the shell command mail-command.  For instance,
 *			mail="mail -s Super joeblow" will cause error
 *			messages to be mailed to user joeblow (on some
 *			systems you may need to substitute "mailx" for "mail").
 *			Note: the mail-command is passed as an argument
 *			to popen(3), and the message piped in.

 *  o	owner=xxx	The owner of the FullPathName must be xxx, or it
 *			won't be executed.  N.B.  xxx is first tried as a
 *			username, then as a uid.
 *			** This can be entered as either local or global option.
 *			** The local option, if present, overrides
 *			** the global setting.

 *  o   password=y|n	If y, the user's password is required to execute
 *			the command.  Default=n.
 *			** This can be entered as either local or global option.
 *			** The local password option, if present, overrides
 *			** the global setting.

 *  o  timeout=mmm	Passwords are good for mmm minutes; then they
 *			have to be re-entered to execute another password
 *			command.  <=0 requires password every time.
 *			Default = 5 min.
 *			** This can be entered as either local or global option.
 *			** The local timeout option, if present, overrides
 *			** the global setting.

 *  o  renewtime=y|n	If y, each successfully-executed password-requiring
 *			command causes the timestamp to be changed to the
 *			currrent time, allowing another timeout minutes of
 *			use before the password need be re-entered.  def=n.

 *  o  timestampbyhost=y|n	If y, timestamp files are unique for each host
 *			-- they won't be shared by the same username when
 *			one timestamp directory is cross-mounted.  If n, the
 *			cross-mounted timestamp directories use the same
 *			timestamp file for all accounts with the same username.

 *  o  timestampuid=xxx	If given, timestamp files are created using uid=xxx
 *			(xxx can be a username or uid).  The default is to
 *			use uid=root.  The motivation is the same as for
 *			the loguid option: to allow a cross-mounted directory
 *			to be used for timestamp files, even though root may
 *			be mapped to nobody for NFS accesses.

 *  o  patterns=xxx	Select the pattern-matching type.  xxx must be one of:

 *		    "shell" -- patterns are as follows (approximately
 *				Bourne-shell style), with the addition
 *				of csh-style brace expansion:
 *				\x	force x to be a plain character;
 *				?	matches any single character;
 *				*	matches any sequence of 0 or more chars;
 *				[x..y]	matches any single character in the set;
 *				[^x..y]	matches any single char NOT in the set;
 *				^xxx	inverts the sense of the match --
 *					the string is `matched' if it doesn't
 *					match pattern xxx.  Note that this is
 *					different from the negation !pat.  For
 *					example, the pattern in `!joe' is just
 *					`joe', and thereby matches only user
 *					joe (who is then disallowed because the
 *					pattern is preceded with `!').
 *					In contrast, the pattern in
 *					`^joe' is all four characters ``^joe'',
 *					which matches any string other than joe.
 *					Example 1:
 *						j*  ^joe
 *					matches all names beginning with
 *					j (`j*'), _and_ all other names except
 *					for joe.
 *					Example 2:
 *						j*  !joe
 *					matches all names beginning with
 *					j (`j*'), and then disallows joe
 *					(`!joe').


 *		    "regex" -- patterns are ed-style, with the addition
 *				of csh-style brace expansion.

 *			DEFAULT: regex.

 *  o  user/group/host	Ordinary user/group/host patterns can be put in the
 *			global options list.  The list is of the form
 *			  [ before_pat before_pat ... <> ] after_pat after_pat
 *			When matching user/group/host patterns for each
 *			command, all user/group/host patterns up to `<>'
 *			are processed _before_ all of the patterns on the
 *			per-command line; and all the rest of the patterns
 *			are processed _after_ all of the patterns on the
 *			per-command line.

 * If a user is allowed to execute a given <commandname>, the <fullpathname>
 * is exec'd, with <commandname> as argv[0].
 *
 * If a symlink is made from, say, mycmd to super, then executing
 *	mycmd args...
 * or
 *	/usr/local/bin/superlinks/mycmd args...
 * is equivalent to executing
 *	super mycmd args...

 * Notes:
 *	1) when interpreting $argv[0] as the cmd, the leading directory parts
 *		are stripped off.
 *	2) Links mustn't be named "super" -- the super program won't recognize
 *		it's a link.
 * 
 * For security, the environment variables are discarded, save for TERM, LINES
 * COLUMNS, and the env=... list.  If TERM contains any characters other than
 * [a-z][A-Z][0-9]_+.:/-, it is discarded.  If LINES or COLUMNS contains any
 * characters other than [0-9], they are discarded.  To these are added
 * reasonable values for IFS, PATH, USER and HOME (USER and HOME are set
 * to the username and login directory, respectively, of the real uid when
 * the command is run).  LOGNAME is set to the same as USER.  ORIG_USER,
 * ORIG_LOGNAME, and ORIG_HOME are set to the username and login directory
 * of the person invoking super().  SUPERCMD is set to the <commandname>.
 * All descriptors excepting 0,1,2 and the fd=... set are closed.
 * Signals are all reset to have default handling.

 * If the super.tab file isn't owned by root, or if the super.tab file
 * is group- or world-writeable, won't run setuid-root.
 * (If caller is root, won't run at all.  Otherwise, euid reverts to real uid).

 */


static char *SEP = " \t\v\n";	/* How to split fields on input lines */
static char *QM	= "\"'";	/* Quotemarks in fields */
static char *CM	= "#";		/* Comments in input file */
static char *SAFE_IFS = "IFS= \t\n";

#ifndef SAFE_PATH
#define SAFE_PATH "PATH=/bin:/usr/bin:/usr/ucb"
#endif
static char *SafePath = SAFE_PATH ;

/* The name under this program assumes it is installed.  If argv[0] isn't
 * [/.../]ONETRUENAME, we assume we're running via symlink.
 */
#ifndef ONETRUENAME
#define ONETRUENAME "super"
#endif

#ifndef SUPERFILE
#define SUPERFILE "/usr/local/lib/super.tab"
#endif

#ifndef TIMESTAMP_DIR
#define TIMESTAMP_DIR "/usr/local/lib/superstamps"
#endif

#ifndef MAXFD
int getdtablesize( /* void */ );
#define MAXFD (getdtablesize()-1)
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif

/* If defined, then user patterns are allowed to match the uid as well as
 * the actual username.  We DISABLE this by default because users are
 * almost always identified by username.
 */
/* #define MATCH_DECIMAL_UID */

/* If defined, then group patterns are allowed to match the
 * gid as well as the group name, if any.  We ENABLE this by default
 * because it's not unusual for users to be put into unnamed groups
 * in the password file.
 */
#define MATCH_DECIMAL_GID


/* maximum number of tries at entering the password */
#define MAXTRY 3

/* Global option settings that aren't stored elsewhere */
struct {
    char owner[32];
} globalopt = { "" };

/* Password information */
struct PassInfo {
    int required;	/* -1 means not in use */
    int timeout;	/* Time before password must be re-entered */
    int renewtime;	/* update the timestamp file with each use of cmd? */
    int perhost;	/* create timestamp files separately for each host? */
    char user[1024];	/* value of timestampuid=xxx, if entered */
};


/* This struct is filled in a bit at a time until it completely
 * describes the caller, the program to invoke, any options
 * set in the control line, etc.
 */
struct superinfo {
    char caller[128];			/* who's invoking prog */
    char hostname[MAXHOSTNAMELEN];	/* whence came the user */
    char fqdn[MAXHOSTNAMELEN+2];	/* fully-qualified hostname */
#ifdef __STDC__
    uid_t orig_uid;	/* caller's uid */
    gid_t orig_gid;	/* caller's gid */
#else
    int orig_uid;
    int orig_gid;
#endif

    char *info;		/* value of info=xxx option */

    char *user;		/* value of uid=xxx option */
    char *group;	/* value of gid=xxx option */
    char *u_g;		/* value of u+g=xxx option */
#ifdef __STDC__
    uid_t new_uid;	/* new uid, computed from uid=xxx or u+g=xxx */
    gid_t new_gid;	/* new gid, computed from gid=xxx or u+g=xxx */
#else
    int new_uid;
    int new_gid;
#endif
    char owner[32];	/* value of owner=xxx option */
    char *env;		/* value of env=var[,...] option */
    char *fdlist;	/* value of fd=nnn[,...] option */
    int usr_args[2];	/* number of user-entered args allowed */

    int *fd;		/* descriptors from fdlist string */
    struct PassInfo passinfo;	/* password requirements on this command */
    char encr[20];	/* encrypted password */
    char salt[4];	/* salt from password */

    int line;		/* For error messages: line in superfile */
    int nl;		/* 	No. lines used for this cmd in superfile */
    char *superfile;	/* 	name of superfile */
} si;

/* an expandable input buffer */
struct Ebuf {
    char *buf;
    int l;
    int nalloc;
};
static struct Ebuf ebuf = { NULL, 0, 0 };

#ifdef USE_NETGROUP
extern int innetgr();
#endif

#ifdef NEED_MEMSET
void *memset(/* void *s, int c, size_t n */ )
#endif
void init_strqtokS(/* void */ );
void logmsg( /* (char *) cmd, (char **) args */ );
int get_encrypted_pw( /* void */ );
int Error( /* int show_perror, int die, char * fmt, ... */);
int taglines( /* int retval */ );
int set_u_g( /* void */ );
int checkenv( /* (char *) name,  (char *) value,  (char *) pat */ );
int ingroup( /* (char *) user, (gid_t) gid, (char *) gp_pat */ );
void opensuperlog( /* void */ );
FILE *open_writer( /* char *user, char *filename */ );
void close_writer( /* void */ );
int check_pass( /* char *cmd */ );
char *makedirname( /* char *prefix, char *hostname, char *path */ );
int makedir( /* char *directories */ );
int get_password( /* char *cmd, char *user,  char *salt,  char *encr */ );
char *str_val( /* char *left, char *str */ );
void printhelp( /* int verbose, char *cmd, char *path */);
int match_word( /* int match, char *wd, int debug */ );
void strtolower( /* char *string */ );
int check_owner( /* char *file */ );

extern int re_exec();
extern char *re_comp();
int shell_compare();
char *shell_compile();

extern int error_stderr;		/* stderr() bool, used by Error() */
extern int error_syslog;		/* syslog() bool, used by Error() */
extern char *error_command;		/* command used by Error() */
extern char *error_user;		/* username printed by Error() */

extern unsigned char *strqS_qm;		/* For fast access by strqtokS */
unsigned char my_qm[256];
extern unsigned char *strqS_cc;		/* For fast access by strqtokS */
unsigned char my_cc[256];


char *prog;				/* this program */

/* Information for logging use */
struct loginfo {
    FILE *fp;			/* logfile pointer */
    char filename[1024];	/* logfile name */
    char user[1024];		/* value of loguid=xxx, if entered */
    int uid;			/* UID under which we open logfile */
} log = { NULL, "", "", 0 };

/* Global password requirements */
struct PassInfo passinfo = {
		0,	/* not required */
		5,	/* 5-minute expiration */
		0,	/* renew timestamp with ea. command use? */
		1,	/* create timestamp files separately for each host? */
		""};	/* value of timestampuid=xxx, if entered */

/* Routines used to compile/match user/group/host patterns */
char *(*pat_compile)() = re_comp;
int (*pat_compare)() = re_exec;
int need_re_anchor = 1;

/* Global user/group/host patterns */
struct List {
    char *pat;
    struct List *next;
};
struct List g_before={NULL,NULL};	/* list of pats before per-cmd pats */
struct List g_after={NULL,NULL};	/* list of pats after per-cmd pats */
struct List *last_g_after=&g_after;
int before_after = 0;			/* !set to !0 when we see <> */

#define NELEM(x) (sizeof(x)/(sizeof(*x)))
#define UNTIL(e) while(!(e))

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
main(argc, argv)
int argc;
char **argv;
{
    int status, debug=0;
    char *s, *cmd, *path, *approve();
    char **buttonup(), **newargs(), **arglist, **envp;
    extern char *error_prog;	/* same as prog, but used by Error() if set */
    void debug_print( /* char **argv, char *path, char **arglist */ );

    s = strrchr(argv[0], '/');
    prog = (s && *(s+1)) ? s+1 : argv[0];
    error_prog = ONETRUENAME;

    if (gethostname(si.hostname, sizeof(si.hostname)) == -1)
	(void) Error(1, 1, "Couldn't get your hostname: ");
    strtolower(si.hostname);
    si.fqdn[0] = '\0';

    /* Decide if we were invoked as "super cmd args...", or 
     * as a link: cmd args...
     */
    if (strcmp(prog, ONETRUENAME) == 0) {
	/* Invoked as super cmd args... */
	if (argv[1] && strcmp(argv[1], "-d") == 0) {
	    debug = 1;
	    argv++;
	}
	argv++;
	cmd = argv[0];
    } else {
	/* Assume it's been invoked via link to super */
	s = strrchr(argv[0], '/');
	cmd = (s && *(s+1)) ? s+1 : argv[0];
    }

    init_strqtokS();

    /* Check for permission to execute, and change uid/gid as necessary */
    if ((path = approve(cmd, debug)) == NULL)
	return 1;
    else if (*path == '\0')
	return 0;

    /* Check that the file to execute has proper ownership */
    if (check_owner(path) != 0)
	return 1;

    /* Get the arglist for the command, and null-terminate cmd if not
     * already done so.
     */
    arglist = newargs(path, argv);

    /* Button up for security, and get a modified environment */
    envp = buttonup(cmd, debug);

    /* Set uid/gid if necessary */
    status = set_u_g();

    if (debug) {
	debug_print(path, arglist);
	return 0;
    }
    if (status == -1)
	return 1;

    /* Check password requirements */
    if (check_pass(cmd) != 0)
	return 1;

    /* Log an informational message at LOG_INFO priority, not at
     * the usual error priority.
     */
    {
	extern int error_priority;
	int old_pri = error_priority;
	error_priority = LOG_INFO;
	logmsg(cmd, arglist);
	error_priority = old_pri;
    }
    /* Close the logfile writer, if any: we are done logging, and going
     * to exec the prog.
     */
    close_writer();

    if (execve(path, arglist, envp) == -1) {
#ifdef INTERPRETER_HACK
	if (errno == ENOEXEC) {
	    /* Open the file, check for "#!interpreter [argument]" */
	    FILE *fp;
	    char *interp, *argument, line1[1024];

	    if ((fp = fopen(path, "r")) &&		/* open the file */
		fgets(line1, sizeof(line1), fp) &&	/* read first line */
		(strlen(line1) < sizeof(line1)-1) &&	/* not too long? */
		(strncmp(line1, "#!", 2) == 0) &&	/* begins "#!"? */
		(interp = strtok(line1+2, " \t\n"))) {	/* has interpreter? */

		argument = strtok(NULL, " \t\n");	/* get opt argument */

		/* Adjust the arglist -- recall it has room for this */
		if (argument) {
		    arglist -= 2;
		    arglist[0] = arglist[2];
		    arglist[1] = argument;
		    arglist[2] = path;
		} else {
		    arglist--;
		    arglist[0] = arglist[1];
		    arglist[1] = path;
		}
		(void) execve(interp, arglist, envp);
	    }
	}
#endif
	/* If here, we failed to exec the prog.  Re-open the logfile we
	 * closed above and write a message.
	 */
	if (log.fp) opensuperlog();
	(void) Error(1, 1, "command `%.500s': couldn't exec `%s': ", cmd, path);
    }
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Print the debug info */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
debug_print(path, arglist)
char *path;
char **arglist;
{
    char **sp;
    (void) fprintf(stderr, "%s -d:\n", prog);
    (void) fprintf(stderr, "\tCommand: `%s'\n", arglist[0]);
    (void) fprintf(stderr, "\tPath: `%s'\n", path);
    for (sp=arglist; *sp; sp++)
	(void) fprintf(stderr, "\tArgv[%d]:  %s\n", sp-arglist, *sp);
    if (si.usr_args[0] < 0)
	(void) fprintf(stderr,
			    "\tAny number of user-entered args allowed.\n");
    else if (si.usr_args[0] == si.usr_args[1] && si.usr_args[0] == 0)
	(void) fprintf(stderr, "\tNo user-entered args are allowed.\n");
    else if (si.usr_args[0] == si.usr_args[1])
	(void) fprintf(stderr,
		    "\t%d user-entered arg%s required.\n",
		    si.usr_args[0],
		    si.usr_args[0] == 1? " is" : "s are");
    else
	(void) fprintf(stderr,
		    "\t%d - %d user-entered args are required.\n",
		    si.usr_args[0], si.usr_args[1]);
    (void) fprintf(stderr, "\tExtra environment variables:\n");
	(void) fprintf(stderr, "\t\t%s\n",
		(si.env == NULL || *si.env == '\0') ? "(none)" : si.env);
    (void) fprintf(stderr,
		    "\tFile descriptors not to be closed:\n\t\t0,1,2");
    if (si.fdlist)
	(void) fprintf(stderr, ",%s", si.fdlist);
    (void) fprintf(stderr, "\n\n\tID's:\treal effective\n");
    (void) fprintf(stderr, "\tuid\t%d\t%d\n", getuid(), geteuid());
    (void) fprintf(stderr, "\tgid\t%d\t%d\n", getgid(), getegid());
    (void) fprintf(stderr, "\n\tPassword required: %s\n",
			    si.passinfo.required ? "yes" : "no");
    if (si.passinfo.required) {
	(void) fprintf(stderr, "\tPassword timeout: %d min\n",
						    si.passinfo.timeout);
	(void) fprintf(stderr,
    "\tUpdate timestamp with each use of a password-requiring command? %s\n",
				    si.passinfo.renewtime ? "Yes" : "No");
    }
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Log a super call -- If "args" isn't a null ptr, it's printed inside
 *		parentheses, with whitespace separating the arguments.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void
logmsg(cmd, args)
char *cmd;
char **args;
{
    char *logbuf, **ap;
    int e;
    int loglen = strlen(cmd) + 4;

    /* Determine buffer length needed to hold all arguments */
    if (args)
	for (ap = args; *ap; ap++)
	    loglen += strlen(*ap) + 1;

    if (!(logbuf = malloc(loglen)))
	(void) Error(0, 2, "failed to malloc space for logging command\n");

    if (args) {
	sprintf(logbuf, "%s (", cmd);
	for (ap = args; *ap; ) {
	    strcat(logbuf, *ap++);
	    strcat(logbuf, " ");
	}
	logbuf[loglen-3] = ')';
	logbuf[loglen-2] = '\n';
	logbuf[loglen-1] = '\0';
    } else {
	strcpy(logbuf, cmd);
    }

    /* Log the message using Error(), but make sure msg doesn't go to stderr */
    e = error_stderr;
    error_stderr = 0;
    Error(0, 0, logbuf);
    error_stderr = e;
} 

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get the arglist for the command, and null-terminate cmd if nec */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char **
newargs(path_plus, argv)
char *path_plus;	/* string = "path [options]".  Null-terminate and put
			 * options into front of arglist. */
char **argv;		/* options to add to arglist */
{
    int nuser, nalloc, nargs, nxtra;
    char **arglist, **ap;
    char *s, *strqtokS();

    /* Count user-entered args. */
    for (ap = argv; *ap; )
	ap++;

    /* Check number of user-entered args is ok. */
    if (si.usr_args[0] >= 0) {
	nargs = ap - argv - 1;
	if (nargs < si.usr_args[0] || nargs > si.usr_args[1]) {
	    if (si.usr_args[0] == si.usr_args[1] && si.usr_args[0] == 0)
		    (void) Error(0, 2,
			"you may not give any arguments to `%.500s'\n",
			argv[0]);
	    else if (si.usr_args[0] == si.usr_args[1])
		(void) Error(0, 2,
		    "You must give %d argument%s to `%.500s'\n",
		    si.usr_args[0], si.usr_args[0] == 1 ? "" : "s", argv[0]);
	    else
		(void) Error(0, 2,
		    "You must give %d - %d arguments to `%.500s'\n",
			si.usr_args[0], si.usr_args[1], argv[0]);
	}
    }

    /* Start off with space for user-entered args + 100 args in super.tab.
     * We'll re-alloc if necessary.
     */
    nuser = (ap - argv) + 3;
    nalloc = nuser + 100;
    arglist = (char **) malloc(sizeof(char *) * nalloc);
    if (!arglist)
	(void) Error(1, 2, 
	    "failed to malloc space for %d ptrs\n", nalloc);

    /* Leave room for two extra args at front, in case we are handling
     * the "#! interpreter [opt]" file for OS's that don't support it.
     */
    arglist += 2;

    /* First copy the command to the new arglist */
    arglist[0] = *argv++;

    /* Then copy the extra args from super.tab to the arglist,
     * re-allocing the arglist as the number of args grows.
     */
    s=strqtokS(path_plus, SEP, QM, "", 0); /* Don't put FullPath into arglist */
    for(nxtra=0, ap=&arglist[1], s=strqtokS(NULL, SEP, NULL, NULL, 0);  s;
				s = strqtokS(NULL, SEP, NULL, NULL, 0)) {
	nxtra++;
	if (nuser + nxtra >= nalloc) {
	    char **newarglist;
	    nalloc *= 2;
	    newarglist = (char **) realloc((void *) arglist, nalloc);
	    if (!newarglist)
		(void) Error(1, 2, 
		    "failed to realloc space for %d ptrs\n", nalloc);
	    ap = newarglist + (ap - arglist);
	    arglist = newarglist;
	}
	*ap++ = s;
    }

    /* Now add the user-supplied args at the end */
    while (*argv)
	*ap++ = *argv++;
    *ap = NULL;

    return arglist;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get a safe environment for execution of the command */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char **
buttonup(cmd, debug)
char *cmd;		/* name of command being started */
int debug;		/* print debug info? */
{
    /* Closes all descriptors save 0,1,2 and the super.tab-specified fd list.
     * Resets all signal-handling to SIG_DFL.
     * Discards all env. variables save for TERM, LINES, COLUMNS, and
     * any variables listed in the super.tab file.
     * Don't allow TERM to have any but [-/:+._a-zA-Z0-9].
     * Don't allow LINES, COLUMNS to have anything but digits.
     * To these are added reasonable values for IFS, PATH, USER, and HOME.
     * USER and HOME refer to the uid under which the command is executed;
     * LOGNAME is set to the same as USER, and SUPERCMD is set to cmd.
     * ORIG_USER, ORIG_LOGNAME, and ORIG_HOME refer to the user who invoked
     * super.
     * Returned:
     *	a pointer to the modified environment list.
     */
    int i, fd, maxfd;
    char *s, *Getenv();
    int getlogdir();
    int checkenv();
    int fd_log;
    static char *env[200];
    static char User[100];		/* USER */
    static char Logname[100];		/* LOGNAME (alias for USER) */
    static char Home[MAXPATHLEN+5];	/* HOME */
    static char OrigUser[100];		/* ORIG_USER */
    static char OrigLogname[100];	/* ORIG_LOGNAME */
    static char OrigHome[MAXPATHLEN+9];	/* ORIG_HOME */
    static char Cmd[1200];		/* SUPERCMD */
    void (*signal())();

    fd_log = log.fp ? fileno(log.fp) : -1; /* don't close logfile yet */
    maxfd = MAXFD;

    for (fd=3; fd <= maxfd; fd++)
	if (si.fd[fd] == 0 && fd != fd_log)
	    (void) close(fd);
    
    for (i=0; i<NSIG; i++)
       (void) signal(i, SIG_DFL);

    s = si.user ? si.user : si.u_g ? si.u_g : si.caller;
    (void) sprintf(OrigUser, "ORIG_USER=%s", si.caller);
    (void) sprintf(User, "USER=%s", s);
    (void) sprintf(OrigLogname, "ORIG_LOGNAME=%s", si.caller);
    (void) sprintf(Logname, "LOGNAME=%s", s);
    (void) sprintf(Cmd, "SUPERCMD=%s", cmd);
    (void) strcpy(Home, "HOME=");
    (void) getlogdir(s, Home+5);
    (void) strcpy(OrigHome, "ORIG_HOME=");
    (void) getlogdir(si.caller, OrigHome+10);
    i = 0;
    env[i] = Getenv("TERM");
    if (env[i] && checkenv(debug,
		"TERM", env[i]+5, "^[-/:+._a-zA-Z0-9]*$") != -1) i++;
    env[i] = Getenv("LINES");
    if (env[i] && checkenv(debug, "LINES", env[i]+6, "^[0-9]*$") != -1) i++;
    env[i] = Getenv("COLUMNS");
    if (env[i] && checkenv(debug, "COLUMNS", env[i]+8, "^[0-9]*$") != -1) i++;
    env[i++] = SAFE_IFS;
    env[i++] = SafePath;
    env[i++] = User;
    env[i++] = Logname;
    env[i++] = Home;
    env[i++] = Cmd;
    env[i++] = OrigUser;
    env[i++] = OrigLogname;
    env[i++] = OrigHome;

    /* Now add the extra environment variables requested in the
     * super.tab file.
     */
    for (s=strtok(si.env, ","); i < NELEM(env)-1 && s; s=strtok(NULL, ",")) {
	env[i] = Getenv(s);
	if (env[i])
	    i++;
    }
    if (i >= NELEM(env)-1)
	(void) Error(0, 1,
	    "Asked to save too many environment variables (max allowed %d).\n",
	    NELEM(env)-1);

    env[i] = (char *) NULL;

    return &env[0];
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get environment variable */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
Getenv(s)
char *s;
{
    /* Like getenv(), but returns ptr to the <name> in "name=xxxx",
     * not just the xxxx.
     */
    char **envp; 
    int l;
    extern char **environ;

    if (!s)
	return (char *) NULL;
    l = strlen(s);
    for (envp=environ; *envp ; envp++)
	if (strncmp(*envp, s, l) == 0  &&  *(*envp+l) == '=')
	    return *envp;
    return (char *) NULL;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check that an environment variable only includes allowed characters */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Returns 0 if pat matched; -1 otherwise.  */
int
checkenv(debug, name, value, pat)
int debug;	/* print debug info? */
char *name;	/* variable name to check (e.g. "TERM") */
char *value;	/* contents of variable (e.g. "vt100") */
char *pat;	/* pattern that value must match */
{

    if (!value)
	return -1;

    if (debug)
	(void) fprintf(stderr,
	    "\tcheckenv args: name=\"%s\"; value=\"%s\"; pat=\"%s\"\n",
	    name, value, pat);
    
    /* Environment variables are always checked with re_comp/re_exec:
     * the patterns are fixed internally, not supplied by the user.
     */
    if (re_comp(pat))
	return Error(0, 0,
	    "checkenv(): couldn't compile pattern `%.500s'.\n", pat);

    if (re_exec(value) != 1)
	return Error(0, 0,
	    "checkenv(): $%.100s (=%.100s) doesn't match pattern %.500s.\n",
			name, value, pat);
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check ownership of the file */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_owner(file)
char *file;
{
    /* Return 0 if file ownership ok (or if no check required); -1 if not ok */
    struct stat st;
    struct passwd *owner_pw;

    if (!file || file == NULL)
	return Error(1, 0, "check_owner(): passed null ptr or empty string\n");

    if (*si.owner == '\0')
	return 0;	/* no checks required */

    if (stat(file, &st) == -1)
	return Error(1, 0, "stat() failed on file `%s': ", file);

    /* Convert owner string to a uid */
    owner_pw = getpwnam(si.owner);
    if (!owner_pw) {
	/* Maybe owner was a uid, not a username */
	char c;
	int numeric, i;
	numeric = (sscanf(si.owner, "%d%c", &i, &c) == 1);
	if (numeric)
	    owner_pw = getpwuid(i);
    }
    if (!owner_pw)
	return Error(0, 0,
	"Can't check owner=%s: no such user or uid as `%s' in password file.\n",
	si.owner);

    if (st.st_uid != owner_pw->pw_uid)
	return Error(0, 0,
	    "Actual owner of `%s' is uid %d, but superfile \
requires owner to be %d (%s).\n", file, st.st_uid,
	    owner_pw->pw_uid, owner_pw->pw_name);
    
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Set user and group according to the specified args */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
set_u_g()
{
    /* Return 0 on success, -1 on failure */
    struct passwd *uid_pw;
    struct group *gp;
    void setgrent(), endgrent();
    int i, numeric=0;

    si.new_uid = si.orig_uid;
    si.new_gid = si.orig_gid;

    /* Check gid=xxx */
    if (si.group) {
	char c;
	numeric = (sscanf(si.group, "%d%c", &i, &c) == 1);
	si.new_gid = i;

	setgrent();
	for (gp = getgrent(); gp; gp = getgrent()) {
	    if ((strcmp(si.group, gp->gr_name) == 0) ||
		(numeric && si.new_gid == gp->gr_gid)) {
		/* Found the gid in the group file */
		si.new_gid = gp->gr_gid;
		break;
	    }
	}
	endgrent();
	if (!gp)
	    return Error(0, 0,
		"Can't set gid: no such group or gid as `%s' in group file.\n",
		si.group);
    }

    /* Check uid=xxx u+g=yyy */
    if (si.user || si.u_g) {
	if (!si.user)
	    si.user = si.u_g;
	uid_pw = getpwnam(si.user);
	if (!uid_pw) {
	    char c;
	    numeric = (sscanf(si.user, "%d%c", &i, &c) == 1);
	    si.new_uid = i;
	    if (numeric)
		uid_pw = getpwuid(si.new_uid);
	}
	if (uid_pw) {
	    si.new_uid = uid_pw->pw_uid;
	    if (si.u_g)
		si.new_gid = uid_pw->pw_gid;
	} else {
	    return Error(0, 0,
	    "Can't set uid: no such user or uid as `%s' in password file.\n",
	    si.user);
	}
    }

    /* Now set uid & gid */
    if (si.group || si.u_g)
	if (setgid(si.new_gid) == -1)
	    return Error(1, 0, "setgid(gid=%d) failed: ", si.new_gid);
    if (si.user || si.u_g)
	if (setuid(si.new_uid) == -1)
	    return Error(1, 0, "setuid(uid=%d) failed: ", si.new_uid);

    if (si.passinfo.required) {
	/* Get caller's encrypted password */
	i = get_encrypted_pw();
	if (i != 0)
	    return i;
    }

    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Puts the encrypted password in si.encr, and the salt in si.salt.
 * Returns 0 on success, -1 on failure to obtain the password.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#ifdef SUNOS5 
int
get_encrypted_pw()
{
    /* Shadow password lookup for Sunos 5.x */
    struct spwd *caller_pw;
    if (!(caller_pw = getspnam(si.caller)))
	return Error(1, 0,
	    "Failed to obtain shadow password entry for user %s: ",
	    si.caller);
    strcpy(si.encr, caller_pw->sp_pwdp);
    strncpy(si.salt, caller_pw->sp_pwdp, 2);
    si.salt[2] = '\0';
    return 0;
}

#else
#ifdef SCO_SHADOW_PWD
int
get_encrypted_pw()
{
    /* Shadow password lookup for SCO 3.2v4 */
    struct spwd *caller_pw;
    if (!(caller_pw = getspnam(si.caller)))
	return Error(1, 0,
	    "Failed to obtain shadow password entry for user %s: ",
	    si.caller);
    strcpy(si.encr, caller_pw->sp_pwdp);
    strncpy(si.salt, caller_pw->sp_pwdp, 2);
    si.salt[2] = '\0';
    return 0;
}


#else
int
get_encrypted_pw()
{
    /* Main part of this routine does standard password lookup.
     * Ifdef'd code handles some shadow passwords.
     */
    struct passwd *caller_pw;

#ifdef _HPUX_SOURCE
    /* See if we can do shadow password lookup for HPUX 9.x */
    static struct stat st;
    if (stat("/.secure/etc/passwd", &st) == 0) {
	/* Shadow password file exists; use it */
	struct s_passwd *caller_pw;
	if (!(caller_pw = getspwnam(si.caller)))
	    return Error(1, 0,
		"Failed to obtain shadow password entry for user %s: ",
		si.caller);
	strcpy(si.encr, caller_pw->pw_passwd);
	strncpy(si.salt, caller_pw->pw_passwd, 2);
	si.salt[2] = '\0';
	return 0;
    }
#endif

    if (!(caller_pw = getpwnam(si.caller)))
	return Error(0, 0, "No password entry for user %s.\n", si.caller);
    strcpy(si.encr, caller_pw->pw_passwd);
    strncpy(si.salt, caller_pw->pw_passwd, 2);
    si.salt[2] = '\0';
    return 0;
}
#endif
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Checks if password is needed, and if so, prompts for same.
 * Returns 0 on success, -1 on error.

 * The timestamp directory faces the same problem as the logfile: if the
 * administrator wants to share an NFS-mounted directory across hosts
 * on which root is translated to nobody for NFS access, we have to be
 * able to create the timestamp file under a special uid.  This is done
 * just as in open_writer(): we fork, setuid(), and do the file
 * manipulation in the child.  This allows us to implement a special uid
 * for the timestamp file, without needing the operating system to
 * offer saved uid's or interprocess file-descriptor passing, etc.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_pass(cmd)
char *cmd;
{
    char file[MAXPATHLEN];
    struct stat st;
    int l, istat, file_exists, timed_out, got_pass;
    int status;
    pid_t child;

    timed_out = 0;

    if (!si.passinfo.required)
	return 0;			/* don't need password */

    /* Create or update the timestamp file even if the lifetime is 0
     * (always ask for password).  We do this because the user may
     * execute _another_ command which has a password expiry > 0,
     * and which will be happy to use the password that was already
     * entered with the 0-lifetime command.
     */
    child = fork();
    if (child == -1) {
	return Error(1, 0, "Failed to create child for timestamp processing: ");

    } else if (child > 0) {
	/* In parent -- wait to see if the child succeeded */
	if (wait(&status) < 0)
	    return Error(1, 0, "wait() on timestamp creation process failed: ");
	else if (status == 0)
	    return 0;
	else
	    return Error(0, 0, "Timestamp creation failed\n");
	
    } else {
	/* In child.  setuid, then make and/or test the directory */
	if (*si.passinfo.user != '\0') {
	    si.user = si.passinfo.user;
	    si.group = NULL;
	    si.u_g = NULL;
	    if (set_u_g() == -1) {
		(void) Error(1, 0,
		"failed to setuid %s before setting timestamp file: ",
		si.user);
		exit(1);
	    }
	}
	/* Make the timestamp directory name */
	if (!makedirname(passinfo.perhost ? TIMESTAMP_DIR : "",
							si.hostname, file))
	    exit(1);

	/* Make the timestamp directory */
	if (!makedir(file))
	    exit(1);

	/* Make the file in the timestamp directory */
	l = strlen(file) + 1 + strlen(si.caller);
	if (l >= MAXPATHLEN)
	    Error(1, 1,
	    "Can't create timestamp file: it would exceed MAXPATHLEN = %d\n",
	    MAXPATHLEN);
	strcat(file, "/");
	strcat(file, si.caller);

	istat = stat(file, &st);
	if (istat != 0 && errno != ENOENT)
	    Error(1, 1, "Failed to stat timestamp file `%s': ", file);

	file_exists = (istat == 0);
	if (file_exists)
	    timed_out = (si.passinfo.timeout < 1) ||
			((time(NULL)-st.st_mtime) > si.passinfo.timeout*60);

	got_pass=0;
	if (!file_exists || timed_out) {
	    got_pass = (get_password(cmd, si.caller, si.salt, si.encr) == 1);
	    if (!got_pass)
		exit(1);
	}

	/* NOTE: A race condition is possible between two super's, with the
	 * worst-case effect of an error message and failure to run the
	 * requested command.
	 */

	/* If file exists, and we haven't (a) gotten the password again, or
	 * (b) supposed to automatically refresh the timestamp, do nothing to
	 * the file except ensure that we own it.

	 * Otherwise create the file (unlink it first if it exists).
	 */
	if (file_exists && !(got_pass || si.passinfo.renewtime)) {
	    if (st.st_uid != geteuid())
		Error(0, 1,
	    "Timestamp file `%s' is owned by uid=%d, but expected owner=%d.\n\
\tN.B. If you recently changed the value of timestampuid=xxx, all existing\n\
\tfiles in the timestamp directory _may_ have the wrong owner; delete them.\n\
\t(No security hole appears when you delete a timestamp file.)\n",
		file, st.st_uid, geteuid());

	} else {
	    if (file_exists) {
		if (unlink(file) != 0)
		    Error(1, 1,
			"Failed to unlink() timestamp file `%s': ", file);
	    }
	    if (open(file, O_WRONLY|O_CREAT|O_TRUNC|O_EXCL, 0200) == -1)
		Error(1, 1,
		    "Failed to open() timestamp file `%s': ", file);
	}
	exit(0);
    }
    /* UNREACHABLE */
    Error(0, 1, "Unreachable code!\n");
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Gets a user's encrypted password. Returns -1 on failure, +1 on success */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
get_password(cmd, user, salt, encr)
char *cmd, *user, *salt, *encr;
{
    /* No such file or password timed out -- get password */
    int ntry, match;
    char msg[500];
    char *crypt(), *getpass();
    char *encrypted = NULL;
    if (strcmp(encr, "") == 0) {
	return Error(0, 0,
    "Command `%.1000s' requires a password, but user `%s' has no password\n",
    cmd, user);
    }
    for (ntry=0, match=0; ntry < MAXTRY && !match; ntry++) {
	if (ntry == 0) {
	    (void) sprintf(msg,
    "Your password is required for super command `%s'...\nPassword: ",
		cmd);
	} else {
	    strcpy(msg, "Password incorrect\nPassword: ");
	}

	encrypted = crypt(getpass( msg ), salt);
	if (encr && encrypted)
	    match = (strcmp(encr, encrypted) == 0);
	else
	    match = 0;
    }
    if (!match)
	return Error(0, 0, "Password incorrect\n");
    return 1;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Copies in to out, prefixing with "^" and suffixing with "$"
 * if these are missing.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
anchor(in, out)
char *in;
char *out;
{
    void re_anchor();
    if (need_re_anchor)
	re_anchor(in, out);
    else
	(void) strcpy(out, in);
}

void
re_anchor(in, out)
char *in;
char *out;
{
    int i;
    i = (*in != '^');
    if (i)
	out[0] = '^';
    (void) strcpy(out+i, in);
    i = strlen(out);
    if (out[i-1] != '$')
	out[i++] = '$';
    out[i] = '\0';
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Grow an expandable buffer */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
grow(cb, nb)
struct Ebuf *cb;
int nb;			/* amount to grow, bytes */
{
    if (cb->buf)
	cb->buf = realloc(cb->buf, cb->nalloc += nb);
    else
	cb->buf = malloc(cb->nalloc += nb);

    return cb->buf;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Grow buffer if less than 1K free */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
checksize(cb)
struct Ebuf *cb;
{
    if (cb->nalloc - cb->l  <  1024)
	return grow(cb, 2048);
    else
	return cb->buf;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check if string s1 ends with string s2 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *ends(s1, s2)
char *s1, *s2;
/* If s1 ends with string s2, a pointer to the ending of s1 is returned;
 * else null
 */
{
    int l1, l2;
    l1 = strlen(s1);
    l2 = strlen(s2);
    if (l1 < l2)
	return NULL;
    else if (strcmp(s1+l1-l2, s2) == 0)
	return s1+l1-l2;
    else
	return NULL;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Do fgets to get one logical line: join lines that are terminated
 * with backslash-newline.  Don't discard backslash or newline (so that
 * we can print the exact text, if desired).
 * The result is stored in "ebuf" and a pointer to the string
 * is returned.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
fillbuffer(fp, nl)
FILE *fp;
int *nl;	/* If non-null, number of lines read in is returned */
{
    char c, *s;
    ebuf.l = 0;		/* clear the extensible buffer */

    /* Collect lines until we have a non-zero buffer (which happens with
     * the first line, of course) and it isn't terminated "\\\n".
     */
    if (nl) *nl = 0;
    UNTIL(ebuf.l && !(s=ends(ebuf.buf, "\\\n"))) {
	
	if (!checksize(&ebuf)) {
	    /* Needed to, but couldn't increase the allocated space */
	    return NULL;
	}
	if (!fgets(ebuf.buf+ebuf.l, ebuf.nalloc - ebuf.l, fp))
	    return NULL;
	c = *(ebuf.buf + ebuf.l) ;
	if (ebuf.l != 0 && !(isspace(c) || c == '#')) {
	    /* Continued lines must be indented unless they are comments. */
	    (void) Error(0, 1,
	    "format error in super.tab file:\n\tcontinued line not indented: \
Problem text:\n%s\n\t\n", ebuf.buf);
	}
	ebuf.l += strlen(ebuf.buf+ebuf.l);
	if (nl) (*nl)++;
    }
    return ebuf.buf;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Try to match a user/group/host pattern against the present user. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
match_word(match, wd, debug, prefix)
int match;	/* Return input value of match on failure; 1 on success */
char *wd;
int debug;
char *prefix;
{
    int i, iwd;
    char *tp, *tok, tokbuf[1000];
    char **wdlist;
    int globbraces();
    int check_ugh();

    /* Do brace globbing on each word */
    /* fprintf(stderr, "globbraces -> ``%s''\n", wd); */
    if ((i=globbraces(wd, &wdlist)) != 0) {
	(void) taglines(Error(0, 0, "Missing `%c'.\n", i));
    } else {
	for (iwd=0; (tok=wdlist[iwd]); iwd++) {
	    strcpy(tokbuf, tok);
#ifdef NO_NEGATION
	    tp = tokbuf;
#else
	    tp = (tokbuf[0] == '!') ? tokbuf+1 : tokbuf ;
#endif
	    if (check_ugh(wd, tp) == 0) {
		match = 1;
		if (debug)
		    (void) fprintf(stderr,
		    "\tMatched %s pat=%s (user=%s gid=%d hostname=%s)\n",
		    prefix, tok, si.caller, si.orig_gid, si.hostname);
#ifndef NO_NEGATION
		if (tp == tokbuf+1) {
		    /* pattern was of form !pat */
		    if (debug)
			(void) fprintf(stderr,
			"\tPattern is negated -- it's prefixed with `!'.\n");
		    match = 0;
		}
#endif
	    } else if (debug) {
		(void) fprintf(stderr,
		    "\tNo match %s pat=%s (user=%s gid=%d hostname=%s)\n",
		    prefix, tok, si.caller, si.orig_gid, si.hostname);
	    }
	}
    }

    return match;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Try to match a string to a pattern. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
match_pattern(match, str, pattern, debug)
int match;	/* Return input value of match on failure; 1 on success */
char *str;
char *pattern;
int debug;
{
    int i, ipat;
    char *tp, *tok, tokbuf[1000];
    char **patlist;
    int globbraces();
    char chkbuf[1024];

    /* Do brace globbing on the pattern */
    /* fprintf(stderr, "globbraces -> ``%s''\n", pattern); */
    if ((i=globbraces(pattern, &patlist)) != 0) {
	(void) taglines(Error(0, 0, "Missing `%c'.\n", i));
    } else {
	for (ipat=0; (tok=patlist[ipat]); ipat++) {
	    strcpy(tokbuf, tok);
	    tp = tokbuf;
	    anchor(tok, chkbuf);			/* Anchor all matches */
	    if ((*pat_compile)(chkbuf) != NULL)
		return taglines(Error(0, 0,
			    "bad command pattern: `%s'.\n", pattern));
		if ((*pat_compare)(str) == 1) {
		    if (debug)
			(void) fprintf(stderr,
			    "\tMatched str=%s to pat=%s\n", str, pattern);
		    return 1;
		} else if (debug) {
		    (void) fprintf(stderr,
			"\tNo match str=%s to pat=%s\n", str, pattern);
	    }
	}
    }
    return match;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check a single user/group/host string
 * Return -1 on failure to match; 0 on success.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_ugh(origtext, token)
char *origtext;		/* original text containing token -- for err messages */
char *token;		/* user/group/host pattern */
{
    char chkbuf[1024];
    char *userpat, *grouppat, *hostpat;
    int match, i;

    if (si.orig_uid == 0)
	return 0;			/* root is always legit */

    /* Split into user:group@host */
    hostpat = strchr(token, '@');
    if (hostpat && !*(hostpat+1))
	return taglines(Error(0, 0, "Missing hostname in `%s'.\n", origtext));
    else if (hostpat)
	*hostpat++ = '\0';

#ifdef USE_NETGROUP
    if (hostpat && *hostpat == '+') {	/* Interpret as +netgroupname */
	if (*(hostpat+1) == '\0')
	    return taglines(Error(0, 0,
		"Missing netgroupname in `%s'.\n", origtext));
	if (!innetgr(hostpat+1, si.hostname, (char *) NULL, (char *) NULL))
	    return -1;
    }
#else
    if (hostpat && *hostpat == '+')	/* Disallow +name */
	return taglines(Error(0, 0, 
	    "hostnames may not begin with `+' since this super()\n\
was compiled without -DUSE_NETGROUP.\n"));
#endif

    /* *** ***** *** *** *** *** *** *** *** *** *** *** *** *** *** ***
     * *** NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE
     * ***
     * *** If compiled without -DUSE_NETGROUP, the following is unreachable.
     * *** In that case, don't worry if your compiler complains about that.
     * ***
     * *** ***** *** *** *** *** *** *** *** *** *** *** *** *** *** ***
     */
    if (hostpat && *hostpat != '+') {	/* Interpret as a hostname pattern */
	strtolower(hostpat);
	anchor(hostpat, chkbuf);	/* Force all matches to be anchored */
	if ((*pat_compile)(chkbuf) != NULL)
	    return taglines(Error(0, 0, "bad host pattern: `%s'.\n", origtext));
#ifndef USE_GETHOSTBYNAME
	if ((*pat_compare)(si.hostname) != 1)
	    return -1;
#else
	if ((*pat_compare)(si.hostname) != 1) {
	    /* Try inspecting the fully-qualified domain name */
	    void namehost();
	    char *dot, *odot;
	    if (si.fqdn[0] == '\0') {
		namehost(si.hostname, si.fqdn, sizeof(si.fqdn)-1);
		strtolower(si.fqdn);
		strcat(si.fqdn, ".");
	    }
	    /* Try matches in following order: a.b.c.d, a.b.c, a.b, a */
	    for (odot=NULL, dot=strrchr(si.fqdn, '.'); dot;
				    dot = strrchr(si.fqdn, '.')) {
		if (odot)
		    *odot = '.';		/* restore previous dot */
		odot = dot;
		*dot = '\0';
		if ((*pat_compare)(si.fqdn) == 1)
		    break;
	    }
	    if (odot)
		*odot = '.';
	    if (!dot)
		return -1;
	}
#endif
    }

    grouppat = strchr(token, ':');
    userpat = token;
    if (*token == '\0' && !hostpat) {
	/* Nothing in pattern?! */
	return taglines(Error(0, 0, "Unacceptable pattern `%s'.\n", origtext));

    } else if (*token == '\0') {
	userpat = grouppat = "^.*$";		/* only hostname given */

    } else if (grouppat && *(grouppat+1)) {	/* pat is "uuu:ggg or ":ggg" */
	if (token == grouppat)
	    userpat = "^.*$";			/* pat is ":ggg" */
	*grouppat++ = '\0';

    } else {					/* pat is "uuu" or "uuu:" */
	if (grouppat)
	    *grouppat = '\0';			/* pat is "uuu:" */
	grouppat = "^.*$";
    }
    anchor(userpat, chkbuf);			/* Anchor all matches */
    if ((*pat_compile)(chkbuf) != NULL)
	return taglines(Error(0, 0, "bad user pattern: `%s'.\n", origtext));
    match = (*pat_compare)(si.caller);
#ifdef MATCH_DECIMAL_UID
    if (match != 1) {
	/* Enabling MATCH_DECIMAL_UID allows the userpat to be
	 * numeric, as an alternative to being interpreted as a
	 * user name: after checking the username, we check if the
	 * user's uid, as a decimal text value, matches the user
	 * pattern userpat.
	 */
	char buf[20];
	(void) sprintf(buf, "%d", si.orig_uid);
	match = (*pat_compare)(buf);
    }
#endif
    if (match != 1)
	return -1;

    anchor(grouppat, chkbuf);
    i = ingroup(si.caller, si.orig_gid, chkbuf);
    if (i == -1)
	return taglines(Error(0, 0, "bad group pattern\n", origtext));
    else if (i != 1)
	return -1;
    
    return 0;				/* Success! */
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Collect global xxx=yyy options and make the global user/group/pat list. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
global_option(word)
char *word;	/* opt=xxx or user/group/host */
{
    /* Return 0 on success, -1 if formatting error. */
    extern int error_syslog;
    char *s;

    if (strcmp("\\", word) == 0) {
	return 0;
    } else if (!(s=strchr(word, '=')) || s == word) {
	/* It's a user/group/host pattern */
	if (strcmp(word, "<>") == 0) {
	    /* End of g_before list.  Since we accumulate into
	     * g_after list, just move that list over to g_before.
	     */
	    if (before_after != 0)
		(void) Error(0, 1,
			"Multiple use of `<>' in global options list.\n");
	    before_after = 1;
	    g_before.next = g_after.next;
	    g_after.next = NULL;
	    last_g_after = &g_after;
	} else {
	    /* It's a user/group/host pattern.  Put it into the g_after list.
	     * If we eventually see "<>", we'll move the list to
	     * g_before (see above).
	     */
	    last_g_after->next = (struct List *) malloc(sizeof(struct List));
	    if (!(last_g_after->next))
		Error(0, 1,
		"Failed to malloc space for global user/group/host list\n");
	    last_g_after = last_g_after->next;
	    last_g_after->next = NULL;
	    if (!(last_g_after->pat = malloc(strlen(word)+1)))
		Error(0, 1,
		"Failed to malloc space for global user/group/host list\n");
	    strcpy(last_g_after->pat, word);
	}
    } else if ((s = str_val("patterns", word))) {
	word = s;
	if (strcmp(word, "shell") == 0) {
	    pat_compile = shell_compile;
	    pat_compare = shell_compare;
	    need_re_anchor = 0;
	} else if (strcmp(word, "regex") == 0) {
	    pat_compile = re_comp;
	    pat_compare = re_exec;
	    need_re_anchor = 1;
	} else {
	    return taglines(Error(0, 1,
		"Invalid pattern type `%s'.  Valid: \"shell\" and \"regex\"\n",
		word));
	}

    } else if ((s = str_val("syslog", word))) {
	if (strcmp(s, "n") == 0) {
	    error_syslog = 0;
	} else {
	    error_syslog = 1;
	    if (strcmp(s, "y") != 0)
		(void) taglines(Error(0, 1,
	"Invalid option value in `%s' -- value must be `y' or `n'; using `y'.\n"
	));
#ifndef USE_SYSLOG
	    Error(0, 1,
    "Can't enable syslog() calls -- not compiled with USE_SYSLOG defined.\n");
#endif
	}

    } else if ((s = str_val("mail", word))) {
	char *p = malloc(strlen(s)+1);
	if (!p)
	    Error(1, 1, "Can't malloc space for option `%s'\n", word);
	strcpy(p, s);
	error_command = p;

    } else if ((s = str_val("logfile", word))) {
	strcpy(log.filename, s);

    } else if ((s = str_val("loguid", word))) {
	strcpy(log.user, s);

    } else if ((s = str_val("timestampuid", word))) {
	strcpy(passinfo.user, s);

    } else if ((s = str_val("timestampbyhost", word))) {
	if (strcmp(s, "n") == 0)
	    passinfo.perhost = 0;
	else if (strcmp(s, "y") == 0)
	    passinfo.perhost = 1;
	else
	    (void) taglines(Error(0, 1,
	"Global option value `timestampbyhost' must have value `y' or `n'\n"));

    } else if ((s = str_val("renewtime", word))) {
	if (strcmp(s, "y") == 0) {
	    passinfo.renewtime = 1;
	} else {
	    passinfo.renewtime = 0;
	    if (strcmp(s, "n") != 0)
		(void) taglines(Error(0, 1,
    "Global option value `renewtime' must have value `y' or `n'; using`n'\n"));
	}

    } else if ((s = str_val("timeout", word))) {
	passinfo.timeout = atoi(s);

    } else if ((s = str_val("password", word))) {
	if (strcmp(s, "n") == 0) {
	    passinfo.required = 0;
	} else {
	    passinfo.required = 1;
	    if (strcmp(s, "y") != 0)
		(void) taglines(Error(0, 1,
	"Invalid option value in `%s' -- value must be `y' or `n'; using `y'.\n"
	));
	}

    } else if ((s = str_val("owner", word))) {
	strncpy(globalopt.owner, s, sizeof(globalopt.owner)-1);
	globalopt.owner[sizeof(globalopt.owner)-1] = '\0';

    } else {
	return taglines(Error(0, 1,
			"Unrecognized global option `%s'.\n", word));
    }

    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Collect local xxx=yyy options. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
local_option(word)
char *word;	/* null -> clear all per-command options; else opt=xxx */
{
    /* Return 0 on success, -1 if formatting error. */
    int i, maxfd;
    char *s;

    if (!word) {
	/* Clear options */
	maxfd = MAXFD;
	si.info = NULL;
	si.user = si.group = si.u_g = NULL;
	si.env = NULL;
	strcpy(si.owner, globalopt.owner);
	si.fdlist = NULL;
	if (!si.fd) {
	    si.fd = (int *) malloc(sizeof(int) * (maxfd + 1));
	    if (!si.fd)
		(void) Error(1, 1,
			"Failed to malloc space for file descriptor list: ");
	}
	for (i=0; i<=maxfd; i++) {
	    si.fd[i] = 0;
	}
	si.usr_args[0] = si.usr_args[1] = -1;
	/* should use struct assignment, but that's not portable to
	 * all k&r compilers.
	 */
	si.passinfo.required = passinfo.required;
	si.passinfo.timeout = passinfo.timeout;
	si.passinfo.renewtime = passinfo.renewtime;
	si.passinfo.perhost = passinfo.perhost;
	strcpy(si.passinfo.user, passinfo.user);

	return 0;
    }
    
    if ((s = str_val("info", word))) {
	si.info = s;

    } else if ((s = str_val("timeout", word))) {
	si.passinfo.timeout = atoi(s);

    } else if ((s = str_val("password", word))) {
	if (strcmp(s, "n") == 0) {
	    si.passinfo.required = 0;
	} else {
	    si.passinfo.required = 1;
	    if (strcmp(s, "y") != 0)
		(void) taglines(Error(0, 1,
	"Invalid option value in `%s' -- value must be `y' or `n'; using `y'.\n"
	));
	}

    } else if ((s = str_val("owner", word))) {
	strncpy(si.owner, s, sizeof(si.owner)-1);
	si.owner[sizeof(si.owner)-1] = '\0';

    } else if ((s = str_val("uid", word))) {
	si.user = s;

    } else if ((s = str_val("gid", word))) {
	si.group = s;

    } else if ((s = str_val("u+g", word))) {
	si.u_g = s;

    } else if ((s = str_val("env", word))) {
	si.env = s;

    } else if ((s = str_val("fd", word))) {
	int n;
	char *w;
	char buf[500];
	maxfd = MAXFD;
	si.fdlist = s;
	strcpy(buf, s);
	for (w=strtok(buf, ","); w; w=strtok(NULL, ",")) {
	    if ((n=atoi(w)) >= 0 && n <= maxfd)
		si.fd[n] = 1;
	    else
		(void) taglines(Error(0, 1,
		"Ridiculous value in file descriptor list: `%s'\n", word));
	}

    } else if ((s = str_val("nargs", word))) {
	int i, m, n;
	i = sscanf(s, "%u-%u", &m, &n);
	switch(i) {
	case 1:     si.usr_args[0] = si.usr_args[1] = m;
		    break;
	case 2:     si.usr_args[0] = m;
		    si.usr_args[1] = n;
		    break;
	default:
	    return taglines(Error(0, 1,
			"nargs must be nargs=nnn or  nargs=mmm-nnn.\n",
			word));
	}

    } else {
	return taglines(Error(0, 1,
			"Unrecognized local option `%s'.\n", word));

    }
    if ((si.user || si.group) && si.u_g)
	return taglines(Error(0, 1,
		"Option u+g=zzz can't be used with uid=xxx or gid=yyy\n"));
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Parse one control line (may contain several text lines). */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*	Return -1 if caller should give up parsing the file;
 *	return 0 if caller should loop to next chunk (either because we
 *		are doing help processing or because of nomatch);
 *	return 1 if success, in which case *path pts to the fullpathname.
 */
int
parseline(givehelp, buf, usrcmd, path, commandfound, debug)
int givehelp;		/* request is to give help, not check valid command */
char *buf;		/* Input chunk.  Gets modified by strqtokS(). */
char *usrcmd;		/* Command user wants to execute -- can be NULL if
			 * givehelp!=0. */
char **path;		/* On success, returned with *path -> fullpathname */
char **commandfound;	/* Returns ptr to command name; NULL if no match */
int debug;
{
    char *command;
    char *wd;
    char *bbuf, *s, *p, *q;
    int n, lpath, match;
    int global_option();
    int local_option();
    extern char *strqS_start, *strqtokS();
    static char FullPath[2*MAXPATHLEN];
    struct List *l;

    bbuf = buf;
    buf += strspn(buf, SEP);	/* skip leading whitespace */
    if (!*buf || *buf == '#')
	return 0;		/* Discard empty lines and pure-comment lines */

    if (bbuf != buf) {
	return taglines(Error(0, 0, "format error in super(1) file: \n\
\tCommandPatterns must begin in column 1.\n"));
    }

    /* Get the command and full path.  */
    strqS_qm = my_qm;
    strqS_cc = my_cc;
    command = strqtokS(buf, SEP, NULL, NULL, 1);
    *path = strqtokS(NULL, SEP, NULL, NULL, 1);
    if (!command || !*path)
	return taglines(Error(0, 0,
    "format error in super.tab file: CmdPattern or FullPathName missing.\n"));

    if (debug)
	fprintf(stderr, "\tParsing line with command = `%s'\n", command);

    if (strcmp("/", command) == 0 && strcmp("/", *path) == 0) {
	/* Process global options and return */

	/* reset global user/group/host patterns pointer, so that more names
	 * form a new list, and are not appended to the old one.
	 */
	last_g_after = &g_after;
	for (strqS_qm = my_qm, strqS_cc = my_cc,
		wd=strqtokS(NULL, SEP, NULL, NULL, 1); wd;
				    wd = strqtokS(NULL, SEP, NULL, NULL, 1)) {
	    if (global_option(wd) == -1) {
		Error(0, 1,
		    "Note: unexpected return from global_option(\"%s\")\n\
\tafter error was found in superfile.\n", wd);
		return -1;
	    }
	}
	if (*log.filename != '\0')
	    opensuperlog();
	return 0;

    } else if (!givehelp && match_pattern(0, usrcmd, command, debug) != 1) {
	/* Skip non-matching commands */
	return 0;
    }

    if (!givehelp) {
	/* Check if we are supposed to generate a modified FullPath */

	/* Watch out for sneaky users */
	if (strcspn(usrcmd, SEP) != strlen(usrcmd) || strchr(usrcmd, '\\'))
	    return Error(0, 0,
		"You entered `%s',\nbut you may NOT include whitespace or backslashes in the cmd!\n",
		usrcmd);

	/* Determine the length of the filename part of the path -- remember
	 * that it may contain whitespace-separated arguments following the
	 * filename.
	 */
	lpath = strcspn(*path, SEP);
	p = strchr(*path, '*');	/* Check for asterisk in filename part */
	n = p ? p - *path : 0;	/* number of characters before asterisk */
	if (p && n < lpath) {
	    /* Wildcard command -- replace asterisk with the usrcmd */

	    /* FullPath is twice MAXPATHLEN, because this allows room for
	     * MAXPATHLEN characters of arguments as well as MAXPATHLEN
	     * characters for the filename itself.
	     */
	    if ((strlen(*path) + strlen(usrcmd)) > sizeof(FullPath))
		return taglines(Error(0, 0,
		"Ridiculously long path would be formed from Cmd=`%.500s%s'\n\
and FullPathName=`%.500s%s'\n", usrcmd, strlen(usrcmd)>500 ? "..." : "",
				*path, strlen(*path)>500 ? "..." : ""));
	    if ((q=strchr(p+1, '*')) && q < (*path + lpath))
		return taglines(Error(0, 0,
"FullPathName can only have 1 asterisk, but the superfile entry\nreads `%s'\n",
		    *path));
	    if (n > 0)
		strncpy(FullPath, *path, n);
	    strcpy(FullPath + n, usrcmd);
	    strcat(FullPath, p+1);
	    *commandfound = usrcmd;
	    *path = FullPath;
	} else {
	    /* Regular command */
	    *commandfound = command;
	}
    } else {
	/* givehelp */
	*commandfound = command;
    }

    /* Replace "\\\n" with " \n" -- the backslash has been left in until
     * here so that we have choice of printing help info with nearly identical
     * text to that in the super.tab file.  Don't get rid of newline as
     * comments only go up to newlines.
     */
    s = strqS_start;
    while ((s = strchr(s, '\\'))) {
	if (*(s+1) == '\n')
	    *s++ = ' ';
	else
	    s++;
    }

    match = 0;
    /* Next process global before-command-line ok-user-patterns */
    for (l = g_before.next; l; l = l->next)
	    match = match_word(match, l->pat, debug, "global");

    /* Now process command-line ok-user-patterns and options */
    for (strqS_qm = my_qm, strqS_cc = my_cc,
	    wd=strqtokS(NULL, SEP, NULL, NULL, 1); wd;
				    wd = strqtokS(NULL, SEP, NULL, NULL, 1)) {

	if ((p=strchr(wd, '=')) && p > wd && *(p-1) != '\\') {
	    /* Is an option */
	    if (local_option(wd) == -1)
		Error(0, 1,
		    "Note: unexpected return from local_option(\"%s\")\n\
\tafter error was found in superfile.\n", wd);
	    continue;

	} else {
	    /* Keep looking for a matching ok-user pattern or a
	     * negating pattern.
	     */
	    match = match_word(match, wd, debug, "per-cmd");

	}
    }

    /* Finally process global after-command-line ok-user-patterns */
    for (l = g_after.next; l; l = l->next)
	    match = match_word(match, l->pat, debug, "global");

    return match;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Look up command "usrcmd" in file "superfile".  Return path to execute
 * if approved, empty string if no action should be taken, NULL ptr
 * if error.  As a ``side effect'': Sets the fields in the superinfo struct.
 * approve() returns:
 *	- NULL ptr if error:
 *		a) username not found;
 *		b) superfile can't be opened for reading;
 *		c) no such command as usrcmd in superfile;
 *		d) user not allowed to execute this command;
 *		e) invalid superfile contents.
 *	- ptr to empty string if all ok, but no program should be executed.
 *	- ptr to path of file to exec, if user allowed to do so.
 *	Any error also generates a message to stderr.

 * New calls to approve() overwrite the buffer containing the returned path.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
approve(usrcmd, debug)
char *usrcmd;		/* command we're to check on.
			 * If command is one of:
			 *	"" | "-h" | "-?" | "-H"
			 *	then a list of super commands is printed
			 *	instead of attempting to execute anything.
			 * The "-h" and "-?" forms give terse help,
			 * while the -H form gives long help.
			 * If command is "-V", then version info is printed
			 *	and nothing is executed.
			 */
int debug;		/* !0 If we are in debug mode */

{
    char *path, *buf;
    int local_option();
    int givehelp, giveversion, verbose;
    char *commandfound;
    struct passwd *usrpw;
    struct stat st;
    FILE *fp;

    si.orig_uid = getuid();
    si.orig_gid = getgid();
    usrpw = getpwuid(si.orig_uid);
    if (!usrpw)
	(void) Error(0, 1, "approve(): Couldn't get your password entry: ");
    (void) strcpy(si.caller, usrpw->pw_name);
    error_user = si.caller;

    if ((fp = fopen(SUPERFILE, "r")) == NULL)
	(void) Error(1, 1, "Couldn't open super.tab file `%s': ", SUPERFILE);

    /* Make sure that, if we are running as root, the super.tab file
     * is owned by root, and is not writable by group/world.
     */
    if (geteuid() == 0) {
	if (stat(SUPERFILE, &st) == -1)
	    (void) Error(1, 1,
			"Couldn't stat super.tab file `%s': ", SUPERFILE);
	if (st.st_uid != 0) {
	    if (getuid() == 0) {
		(void) Error(0, 1, "super.tab file `%s' isn't owned by root,\n\
but we are being run by root.  Bailing out.\n", SUPERFILE);
	    } else {
		(void) Error(0, 0, "super.tab file `%s' isn't owned by root,\n\
but our euid==0.   Reverting to uid=%d.\n", SUPERFILE, getuid());
		setuid(getuid());
	    }
	}
	if (st.st_mode & (S_IWGRP || S_IWOTH)) {
	    if (getuid() == 0) {
		(void) Error(0, 1, "super.tab file `%s' is owned by root,\n\
but is group- or world-writeable.  Bailing out.\n", SUPERFILE);
	    } else {
		(void) Error(0, 0, "super.tab file `%s' is owned by root,\n\
but is group- or world-writeable.  Reverting to uid=%d.\n",
			SUPERFILE, getuid());
		setuid(getuid());
	    }
	}
    }

    /* Do we (1) print version info, (2) give help, or
     * (3) match a command with this user?
     */
    giveversion = (usrcmd && strcmp(usrcmd, "-V") == 0);
    givehelp = (usrcmd == NULL) ||
			strcmp(usrcmd, "-H")==0 ||
			strcmp(usrcmd, "-h")==0 || strcmp(usrcmd, "-?")==0;
    verbose = (usrcmd != NULL && strcmp(usrcmd, "-H") == 0);

    if (givehelp || giveversion)
	(void) printf("%s version %s patchlevel %s\n",
				prog, Version, Patchlevel);
    if (giveversion && !givehelp)
	return "";
    if (givehelp) {
	if (verbose) {
	    (void) printf("Use:\n\t%s command [args]\n\n", prog);
	    (void) printf("Super.tab file: `%s'\n\n", SUPERFILE);
	}
	(void) printf("Commands available to user %s%s:\n\n", si.caller,
		verbose ? " (use option `-h' for a terse listing)" :
		" (use option `-H' for a long-winded listing)");
	if (!verbose) {
	    (void) printf("Command Name    Comments\n");
	    (void) printf("or Pattern              \n");
	    (void) printf("------------    --------\n");
	}
    }

    for (commandfound=NULL, si.line=1;
			(buf=fillbuffer(fp, &si.nl)); si.line += si.nl) {
	if (local_option(NULL) == -1) {
		Error(0, 1, "Note: unexpected return from local_option(NULL)\n\
\tafter error was found in superfile.\n");
	    return NULL;
	}

	switch(parseline(givehelp, buf, usrcmd, &path, &commandfound, debug)) {
	case -1:
	    return NULL;
	    break;

	case 0:
	    break;

	default:
	    /* Have an acceptable line */
	    if (givehelp) {
		printhelp(verbose, commandfound, path);
	    } else {
		return path;
	    }
	}
    }
    if (givehelp)
	return "";
    else if (!commandfound)
	(void) Error(0, 0, "No such super command as `%.500s'.\n", usrcmd);
    else
	(void) Error(0, 0, "%s - Permission denied to user %s\n",
							usrcmd, si.caller);
    return NULL;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Prints help information for a command */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
printhelp(verbose, commandfound, path)
char *verbose, *commandfound, *path;
{
    if (!verbose) {
	(void) printf("%-15s", commandfound);
	if (si.info)
	    (void) printf(" %s\n", si.info);
	else
	    (void) putchar('\n');
	return;
    }

    /* ASSERT verbose!=0 */

    if (si.info)
	(void) printf("# %s:\n", si.info);
    (void) printf("%s %s -> %s\n", prog, commandfound, path);
    if (si.user || si.group || si.u_g || si.fdlist || si.owner || si.env) {
	(void) fputs("\t(Executes with:", stdout);
	if (si.user) (void) printf(" uid=%s", si.user);
	if (si.group) (void) printf(" gid=%s", si.group);
	if (si.u_g) (void) printf(" u+g=%s", si.u_g);
	if (si.env) (void) printf(" env=%s", si.env);
	if (si.fdlist) (void) printf(" fdlist=%s", si.fdlist);
	if (*si.owner) (void) printf(" owner=%s", si.owner);
	(void) fputs(")\n", stdout);
    }
    if (si.usr_args[0] < 0)
	;
    else if (si.usr_args[0] == si.usr_args[1] &&
					    si.usr_args[0] == 0)
	(void) printf("\t(No user-entered args are allowed)\n");
    else if (si.usr_args[0] == si.usr_args[1])
	(void) printf("\t(%d user-entered arg%s required)\n",
	    si.usr_args[0], si.usr_args[0] == 1? " is" : "s are");
    else
	(void) printf("\t(%d - %d user-entered args are required)\n",
	    si.usr_args[0], si.usr_args[1]);

    if (si.passinfo.required) {
	if (si.passinfo.timeout == 0)
	    (void) printf("\t(Password is required on each use)\n");
	else
	    (void) printf("\t(Password is required; good for %d minutes%s)\n",
		si.passinfo.timeout, si.passinfo.renewtime ? 
		    ", extended with each use" : "");
    }
    (void) putchar('\n');
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Determines if user's group matches a group pattern. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
ingroup(user, gid, gp_pat)
char *user;
int gid;
char *gp_pat;	/* pattern to match */
{
    /* Use:
     *	ingroup(user, gid, gp_pat)
     * Returns:
     *	1 if the user is in a group matching the regex pattern gp_pat.
     *	0 if the user isn't in a group matching the pattern.
     *	-1 if pattern failed to compile.

     * SIDE-EFFECT: uses pat_compile/pat_compare!
     *			-- messes up caller's use of same!

     * Examples:
     *	ingroup("joe", joes_gid, "xyz")
     * returns !0 if user joe is in group "xyz".
     *	ingroup("joe", joes_gid, "xy.*")
     * returns !0 if user joe is in any group matching "xy.*".

     */

    struct group *gp;
    char **mem;
    char buf[20];
    void setgrent(), endgrent();

    if ((*pat_compile)(gp_pat) != (char *)0 )
	return -1;

    /* Search group file for groups user is in.  For each group of which
     * the user is a member, test a match to the pattern.
     */
    setgrent();
    for (gp = getgrent(); gp; gp = getgrent()) {
	/* The gr_mem list only shows usernames added in the /etc/group file,
	 * and not any users assigned to the group in the passwd file.
	 * Thus discover group membership by first checking the user's
	 * group from the password file (gp->gr_gid) against this group's
	 * gid, then check to see if this user is in the gp->gr_mem list.
	 */
	if (gid != gp->gr_gid) {
	    for (mem = gp->gr_mem; *mem ; mem++)
		if (strcmp(*mem, user) == 0)
		    break;
	    if (!*mem)
		continue;			/* not in group */
	}
	/* if here, the user is in group gp; now check if group
	 * name gp->gr_name matches group pattern gp_pat.
	 */
	if ((*pat_compare)(gp->gr_name) == 1) {
	    /* successful match -- user is in a group that matches gp_pat */
	    endgrent();
	    return 1;
	}
#ifdef MATCH_DECIMAL_GID
	else {
	    /* Enabling MATCH_DECIMAL_GID allows the gp_pat to be
	     * numeric, as an alternative to being interpreted as a
	     * group name: we check if the group id gp->gr_gid, as a
	     * decimal text value, matches the group pattern gp_pat.
	     */
	    (void) sprintf(buf, "%d", gp->gr_gid);
	    if ((*pat_compare)(buf) == 1){
		/* successful match -- user is in a group that matches gp_pat */
		endgrent();
		return 1;
	    }
	}
#endif
    }

#ifdef MATCH_DECIMAL_GID
    /* We haven't found any group from /etc/group to which we belong that
     * matches the pattern.  It is possible that the user's group id from the
     * password file isn't in the /etc/group file at all, in which case the
     * user's group won't have matched the pattern since we've only checked
     * /etc/group entries so far.  Now check the numeric id from the
     * /etc/passwd file against the pattern.
     */
    (void) sprintf(buf, "%d", gid);
    if ((*pat_compare)(buf) == 1){
	endgrent();
	return 1;
    }
#endif

    endgrent();
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get login directory of a user */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int getlogdir(user, buf)
char *user;
char *buf;
{
    /* Gets the login directory of the named user, and puts it into buf.
     * If user==NULL || *user == '\0', the current user is obtained.
     * Best if buf is MAXPATHLEN long.
     * 0 is returned on success; -1 on error.
     */

    struct passwd *pass;
    char *p;
    char *getlogin();

    buf[0] = '\0';
    if (user != NULL && *user != '\0') {
	/* Name given; use getpwnam */
	pass = getpwnam(user);
    } else if ((p = getlogin()) != NULL) {
	/* No name given; use current login name */
	pass = getpwnam(p);
    } else {
	/* No user given && getlogin() returned NULL; use current uid */
	pass = getpwuid(getuid());
    }

    if (pass == (struct passwd *) NULL)
	return -1;

    (void) strcpy(buf, pass->pw_dir);

    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* For tagging error text in the superfile */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
taglines(retval)
int retval;
{
    if (si.nl <= 1)
	(void) fprintf(stderr, "\tLine %d in file `%s'\n", si.line, SUPERFILE);
    else
	(void) fprintf(stderr, "\tLines %d..%d in file `%s'\n",
			    si.line, si.line + si.nl - 1, SUPERFILE);
    return retval;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Opens the logfile. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
opensuperlog()
{
    extern FILE *error_logfile;	/* to tell Error() where the log is */

    if (log.fp)
	(void) fclose(log.fp);

    if (*log.filename == '\0') {
	(void) Error(0, 0, "opensuperlog(): logfile name is (nil)\n");
	return;
    }

    log.fp = open_writer(log.user, log.filename);

    error_logfile = log.fp;		/* so that Error() writes here too */
    return;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
    /* In order to implement the loguser=xxx option, we (1) create a pipe,
     * (2) fork, in the child setuid to loguid; (3) child opens logfile;
     * (4) child copies from pipe to logfile.  Meanwhile, we return a pointer
     * to a stream to the pipe as the log stream seen by the parent program.
     * This allows us to implement a special uid for the logfile writer,
     * without needing the operating system to offer saved uid's or
     * interprocess file-descriptor passing, etc.
     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
FILE *
open_writer(user, filename)
char *user;
char *filename;
{
    FILE *fp;
    int fildes[2];
    pid_t child;

    fp = NULL;

    if (pipe(fildes) == -1) {
	(void) Error(1, 0, "Failed to created pipe for logfile; no logging: ");
	return NULL;
    }
    child = fork();
    if (child == -1) {
	(void) Error(1, 0, "Failed to create child for logfile; no logging: ");
	return NULL;

    } else if (child > 0) {
	/* In parent -- close read side, and aim logstream at write side */
	(void) close(fildes[0]);
	if (!(fp = fdopen(fildes[1], "w"))) {
	    (void) Error(1, 0,
		"failed to fdopen logfile pipe writer; no logging: ");
	    (void) close(fildes[1]);
	    return NULL;
	}

    } else if (child == 0) {
	/* In child.  Open log file and copy from pipe to log. */
	FILE *input;
	char text[2000];
	(void) close(fildes[1]);
	if (!(input = fdopen(fildes[0], "r"))) {
	    (void) Error(1, 0,
		"failed to fdopen logfile pipe reader; no logging: ");
	    (void) close(fildes[1]);
	    exit(1);
	}
	if (user && *user != '\0') {
	    si.user = user;
	    si.group = NULL;
	    si.u_g = NULL;
	    if (set_u_g() == -1) {
		(void) Error(1, 0,
		"failed to setuid %s before opening logfile; no logging: ",
		log.user);
		exit(1);
	    }
	}
	if (!(fp = fopen(log.filename, "a"))) {
	    if (user && *user != '\0')
		(void) Error(1, 0,
		    "failed to open logfile `%s' using uid `%s': ",
		    log.filename, si.user);
	    else
		(void) Error(1, 0,
			"failed to open logfile `%s': ", log.filename);
	    exit(1);
	}
	while (fgets(text, sizeof(text), input)) {
	    if (fputs(text, fp) == EOF)
		(void) Error(1, 0, "fputs to logfile failed: ");
	}
	(void) fclose(fp);
	exit(0);
    }
    return fp;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Closes the logfile stream, then calls wait(). */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void
close_writer()
{
    if (!log.fp)
	return;

    if (fclose(log.fp) == EOF)
	(void) Error(1, 0, "failed to close log.fp: ");
    (void) wait((int *) NULL);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
    /* Construct a host-unique directory name:
     * xyz.home.caltech.edu -> prefix/xyz.home.caltech.edu/user
     * If hostname is empty string, file is prefix/user.

     * WARNING: the hostname used is that from gethostname().
     * Note that this is not necessarily unique across
     * internet domains, since it is frequently not a
     * fully-qualified domain name.  Therefore you should NOT
     * share the timestamp directory outside the local domain.
     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
makedirname(prefix, hostname, file)
char *prefix, *hostname, *file;
{
    int l;
    char *s, str[MAXPATHLEN];

    l = strlen(prefix) + 1 + strlen(hostname) + 1;
    if (l >= MAXPATHLEN) {
	Error(1, 0,
	"Can't create timestamp directory: it would exceed MAXPATHLEN = %d\n",
	MAXPATHLEN);
	return NULL;
    }

    strcpy(file, prefix);

    if (!*hostname)
	return file;

#ifdef L14
    strncpy(str, hostname, 14);
    str[14] = '\0';	/* in case exactly 14 chars were copied */
#else
    strcpy(str, hostname);
#endif
    for (s = strrchr(str, '.'); s; *s = '\0', s = strrchr(str, '.')) {
	strcat(file, "/");
	strcat(file, s+1);
    }
    strcat(file, "/");
    strcat(file, str);

    return file;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Creates a directory, including any needed directories leading
 * to it.  Returns 1 on success, dies otherwise.
 * WARNING: doesn't check if final component is a directory.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
makedir(dir)
char *dir;	/* path with directories only. */
{
    static struct stat st;
    char *q;
    char path[MAXPATHLEN];

    /* First create directories along way, if necessary */
    strcpy(path, dir);

    for (q=path; q && *q; ) {

	/* skip leading slashes */
	while (*q == '/')
	    q++;

	/* check directory before next slash */
	q = strchr(q, '/');
	if (q)
	    *q = '\0';

	/* Stat directory; if missing, create it */
	if (stat(path, &st) != 0) {
	    if (errno != ENOENT) {
		Error(1, 1, "Failed to stat directory `%s'\n", path);
	    } else {
		if (mkdir(path, 0700) != 0)
		    Error(1, 1, "Failed to create directory `%s'\n", path);
	    }
	}

	/* Restore slash */
	if (q)
	    *q = '/';
    }

    return 1;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Input is string like   "lhs=rhs"
 * If there is no `=', null ptr is returned.
 * If lhs == left, then ptr to rhs is returned; else null pointer.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *
str_val(left, str)
char *left, *str;
{
    char *s = strchr(str, '=');

    if (!s					/* equal sign? */
	|| strlen(left) != (s - str)		/* not same size as `left'? */
	|| strncmp(left, str, s-str) != 0)	/* lhs != left */
	return NULL;
    
    return s+1;
}


/*
 * -----------------------------------------
 * re_comp()/re_exec()-style interface to wildmat.
 * -----------------------------------------
 */
static char *shell_pattern;
char *
shell_compile(s)
char *s;
{
    shell_pattern = s;
    return NULL;
}

int
shell_compare(s)
char *s;
{
    extern int wildmat();
    return wildmat(s, shell_pattern);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
init_strqtokS(void)
{
    unsigned char *p;
    memset(my_qm, '\0', sizeof(my_qm));
    for (p=(unsigned char *) QM; *p; )
	my_qm[*p++] = 1;

    memset(my_cc, '\0', sizeof(my_cc));
    for (p=(unsigned char *) CM; *p; )
	my_cc[*p++] = 1;
}

#ifdef USE_GETHOSTBYNAME
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Returns fully-qualified domain name.
 * Exits with error message if there are problems.
 * This is a modified version of a routine by Martin Shepherd,
 * (mcs@astro.caltech.edu)
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
namehost(hostname, fqdn, l_fqdn)
char *hostname;	/* input: hostname for which fqdn is desired */
char *fqdn;	/* output: returned with fqdn */
int l_fqdn;	/* input: length of fqdn buffer */
{
    char buf[MAXHOSTNAMELEN+2];
    struct hostent *host_entry;   /* Host info returned by gethostbyname() */

    if (strlen(hostname) > sizeof(buf)-1)
	Error(0, 1,
	"hostname `%s' (%d chars) is too long for buffer (%d chars)\n",
	hostname, strlen(hostname), sizeof(buf)-1);

    strcpy(buf, hostname);
    strcat(buf, ".");
    /*
     * Look up further information on the named host.
     */
    host_entry = gethostbyname(buf);
    if (host_entry == NULL)
	Error(0, 1, "host `%s' not recognized by gethostbyname().\n", hostname);

    if (strlen(host_entry->h_name) > l_fqdn)
	Error(0, 1,
	"fqdn `%s' (%d chars) is too long for buffer (%d chars)\n",
	host_entry->h_name, strlen(host_entry->h_name), l_fqdn);

    strcpy(fqdn, host_entry->h_name);
}
#endif

/*
 * Downcase a null-terminated string.
 */
void
strtolower(s)
char *s;
{
    for (; *s; s++)
	*s = tolower(*s);
}


#ifdef NEED_MEMSET
void *
memset(s, c, n)
void *s;
int c;
int n;
{
    register int i;
    register unsigned char *p = (unsigned char *) s;

    for (i=0; i<n; i++)
	*p++ = (unsigned char) c;
}
#endif

