/*
 * Copyright (c) 1997 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.  This program
 * is provided "as is", and I accept no responsibility for security
 * problems either inherent in the program or from its actual use.
 *
 * Program to allow specified users to execute specified commands as root.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>


#ifdef	LINUX_PAM
#include <security/pam_appl.h>
#include <security/pam_misc.h>
#endif


#define	PROGRAM		"su1"
#define	VERSION		"4.2"

#define	PRIVFILE	"/etc/su1.priv"
#define	CACHEFILE	"/etc/su1.cache"
#define	INITLOGFILE	"/etc/su1.log"

#define	INITASK		KEY_ALWAYS

#define	WILDCHARS	"*?[]"
#define	ROOTNAME	"root"
#define	ROOTUID		0
#define	ROOTGID		0
#define	COMMENTCHAR	'#'
#define	UMASK		0022
#define	MAXARGS		10000
#define	MAXDEFINES	1000
#define	MAXALIASES	1000
#define	MAXPATHSIZE	1024
#define	MAXVARIABLES	1000
#define	MAXFD		1024
#define	MAXGROUPS	256
#define	SYSLOGBUFSIZE	256
#define	STDIN		0
#define	STDOUT		1
#define	STDERR		2


/*
 * Maximum ask interval in seconds (one day).
 * This cannot be increased by the privilege file.
 */
#define	MAXINTERVAL	(24 * 60 * 60)


/*
 * Boolean values.
 */
typedef	int		BOOL;

#define	FALSE		((BOOL) 0)
#define	TRUE		((BOOL) 1)


/*
 * Keyword identifiers.
 */
typedef	enum
{
	KEY_NULL, KEY_DEFINE, KEY_ALIAS, KEY_PATHS, KEY_LOGFILE,
	KEY_SYSLOG, KEY_ALL, KEY_NONE, KEY_SUCCESSES, KEY_FAILURES,
	KEY_REFUSALS, KEY_PASSWORD, KEY_VARIABLES, KEY_ASK, KEY_ALWAYS,
	KEY_ONCE, KEY_INTERVAL, KEY_NEVER, KEY_ALLOW, KEY_REFUSE,
	KEY_EXACT, KEY_PREFIX, KEY_ANY
} KEYWORD;



/*
 * Keyword table.
 */
typedef	struct	{
	char *		name;		/* name of keyword */
	KEYWORD		code;		/* keyword id */
} KEYTABLE;


static	KEYTABLE	keytable[] =
{
	{"define",	KEY_DEFINE},
	{"alias",	KEY_ALIAS},
	{"paths",	KEY_PATHS},
	{"logfile",	KEY_LOGFILE},
	{"syslog",	KEY_SYSLOG},
	{"all",		KEY_ALL},
	{"none",	KEY_NONE},
	{"successes",	KEY_SUCCESSES},
	{"failures",	KEY_FAILURES},
	{"refusals",	KEY_REFUSALS},
	{"password",	KEY_PASSWORD},
	{"variables",	KEY_VARIABLES},
	{"ask",		KEY_ASK},
	{"always",	KEY_ALWAYS},
	{"once",	KEY_ONCE},
	{"interval",	KEY_INTERVAL},
	{"never",	KEY_NEVER},
	{"allow",	KEY_ALLOW},
	{"refuse",	KEY_REFUSE},
	{"exact",	KEY_EXACT},
	{"prefix",	KEY_PREFIX},
	{"any",		KEY_ANY},
	{NULL,		KEY_NULL}
};


/*
 * Initial set of trusted program paths.
 * This list is NOT ended with a NULL entry.
 */
static	char *	init_paths[] =
{
	"/bin",
	"/usr/bin",
	"/usr/local/bin",
	"/sbin",
	"/usr/sbin",
	"/usr/local/sbin"
};


/*
 * Initial set of trusted environment variables.  This list does not
 * need to include HOME, SHELL, or PATH since they are set specially.
 * This list is NOT ended with a NULL entry.
 */
static	char *	init_variables[] =
{
	"TERM"
};


/*
 * Cache entry for password authentication.
 * These entries are used to by-pass asking for passwords.
 * WARNING: There could be danger in the use of this feature
 * since we cannot absolutely guarantee that two login sessions
 * for the same user can be distinguished, or that an old login
 * session is terminated.  In addition, we cannot be sure that
 * a change to the privilege file will be noticed.
 */
typedef	struct	{
	uid_t	uid;		/* user id */
	dev_t	dev;		/* stdin device */
	int	inode;		/* stdin inode */
	pid_t	parent_pid;	/* parent process pid */
	pid_t	session_id;	/* session id */
	long	offset;		/* offset in privilege file where used */
	time_t	last_used;	/* time entry was last used */
} CACHE_ENTRY;


/*
 * Header of cache file.
 */
typedef	struct	{
	long	magic;		/* magic number */
	time_t	priv_time;	/* modification time of privilege file */
	long	priv_size;	/* size of privilege file */
	long	entries;	/* number of entries in file */
} CACHE_HEADER;

#define	CACHE_MAGIC	0x76439511


/*
 * Local variables.
 */
static	char *		priv_file = PRIVFILE;
static	FILE *		priv_fp = NULL;
static	long		priv_line;
static	long		priv_offset;
static	long		password_offset;

static	char *		cache_file = CACHEFILE;
static	int		cache_fd = -1;
static	int		cache_slot = -1;
static	CACHE_HEADER	cache_header;

static	KEYWORD		ask_type = INITASK;
static	int		ask_interval;

static	BOOL		syslog_successes;
static	BOOL		syslog_failures;
static	BOOL		syslog_refusals;
static	BOOL		running_our_name;
static	BOOL		clear_flag;
static	BOOL		verbose_flag;
static	BOOL		no_cache_flag;

static	char *		password = NULL;
static	char *		logfile = NULL;
static	char *		myname;
static	char *		mygroups[MAXGROUPS];
static	char *		homevar;
static	char *		shellvar;
static	char *		paths;
static	char *		variables;
static	int		mygroupcount;
static	uid_t		myuid;
static	gid_t		mygids[MAXGROUPS];
static	int		cmdargc;
static	char **		cmdargv;
static	char *		cmdname;
static	char *		defines[MAXDEFINES];
static	char *		aliases[MAXALIASES];


/*
 * The environment variable head.
 */
extern	char **		environ;


/*
 * Local procedures.
 * Use prototypes if available.
 */
#ifdef	__STDC__
#define	PROTO(a)	a
#else
#define	PROTO(a)	()
#endif


static	char *	MakeList PROTO((int, char **));
static	char *	FindDefine PROTO((char *));
static	char *	FindAlias PROTO((char *));
static	char *	MakeString PROTO((char *));
static	KEYWORD	FindKeyword PROTO((char *));
static	BOOL	ReadLine PROTO((char *, int));
static	BOOL	CheckCommand PROTO((int, char **, KEYWORD));
static	BOOL	CheckUserList PROTO((char *));
static	BOOL	UserAndGroupNameMatchesMine PROTO((char *));
static	BOOL	PatternMatch PROTO((char *, char *));
static	int	ScanArgument PROTO((char *));
static	long	ScanNumber PROTO((char *));
static	void	CheckPassword PROTO((void));
static	void	IdentifyMyself PROTO((void));
static	void	WriteLogFile PROTO((void));
static	void	WriteSyslog PROTO((KEYWORD));
static	void	ExecuteCommand PROTO((KEYWORD));
static	void	DisplayCommand PROTO((char **));
static	void	ReadPrivilegeFile PROTO((void));
static	void	ExamineLine PROTO((int, char **));
static	void	BadFile PROTO((char *));
static	void	ParseDefine PROTO((int, char **));
static	void	ParseAlias PROTO((int, char **));
static	void	ParsePaths PROTO((int, char **));
static	void	ParseLogfile PROTO((int, char **));
static	void	ParseSyslog PROTO((int, char **));
static	void	ParsePassword PROTO((int, char **));
static	void	ParseAsk PROTO((int, char **));
static	void	ParseAllow PROTO((int, char **));
static	void	ParseRefuse PROTO((int, char **));
static	void	ParseVariables PROTO((int, char **));
static	void	VerifyNameFormat PROTO((char *));
static	BOOL	InitializeCacheEntry PROTO((CACHE_ENTRY *, long));
static	BOOL	IsPasswordCached PROTO((long, long));
static	void	RememberCacheEntry PROTO((long));
static	void	OpenCacheFile PROTO((void));
static	void	CloseCacheFile PROTO((void));
static	void	ReadCacheHeader PROTO((CACHE_HEADER *));
static	void	ClearMyCacheEntries PROTO((void));
static	void	WriteCacheHeader PROTO((CACHE_HEADER *));
static	void	ReadCacheEntry PROTO((int, CACHE_ENTRY *));
static	void	WriteCacheEntry PROTO((int, CACHE_ENTRY *));
static	void	VerifyDescriptors PROTO((void));
static	void	InitializeEnvironment PROTO((void));
static	char *	AllocateEnvironmentString PROTO((char *, char *));
static	char *	AllocateString PROTO((char *));
static	char *	Allocate PROTO((int));
static	void	CheckLocalPassword PROTO((void));

#ifdef	LINUX_PAM
static	void	CheckPamPassword PROTO((void));
#else
static	void	DefaultPassword PROTO((void));
#endif


int
main(argc, argv)
	int	argc;
	char **	argv;
{
	char *	str;

	VerifyDescriptors();

	/*
	 * Make sure that we are running as root,
	 * otherwise it is pointless to run this program.
	 */
	if (geteuid() != ROOTUID) {
		fprintf(stderr, "Not running as root\n");

		return 1;
	}

	/*
	 * Set our umask to a reasonable value.
	 * This default allows read and execute to everyone,
	 * but writing only to the owner.
	 */
	umask(UMASK);

	/*
	 * Make sure we have a reasonable number of arguments.
	 */
	if (argc <= 0) {
		fprintf(stderr, "No program arguments supplied\n");

		return 1;
	}

	if (argc > MAXARGS - 3) {
		fprintf(stderr, "Too many arguments in command\n");

		return 1;
	}

	/*
	 * Save the last component of our program name away for later
	 * use and then remove it from the argument list.
	 * Note that it may be restored later if the name turns out
	 * to match a command alias.  Remember whether the name is our
	 * default program name so that we can do slightly different
	 * things in that case.
	 */
	cmdname = strrchr(argv[0], '/');

	if (cmdname)
		argv[0] = ++cmdname;
	else
		cmdname = argv[0];

	running_our_name = (strcmp(cmdname, PROGRAM) == 0);

	argc--;
	argv++;

	/*
	 * Find out information about myself.
	 */
	IdentifyMyself();

	/*
	 * If we are running under our default program name then parse
	 * some command line options.  We don't parse options for any other
	 * program names since they might belong to an aliased command.
	 */
	while (running_our_name && (argc > 0) && (**argv == '-'))
	{
		argc--;
		str = *argv++;

		for (str++; *str; str++) {
			switch (*str) {
				case 'c':
					clear_flag = TRUE;
					break;

				case 'v':
					verbose_flag = TRUE;
					break;

				case 'n':
					no_cache_flag = TRUE;
					break;

				default:
					fprintf(stderr, "Unknown option -%c\n",
						*str);

					return 1;
			}
		}
	}

	/*
	 * Look for illegal option combinations.
	 */
	if (clear_flag && no_cache_flag) {
		fprintf(stderr, "Cannot use -c and -n together\n");

		return 1;
	}

	/*
	 * If the clear flag was specified then act on that.
	 * But first make sure that there are no more arguments.
	 */
	if (clear_flag) {
		if (argc > 0) {
			fprintf(stderr, "No arguments allowed with -c\n");

			return 1;
		}

		ClearMyCacheEntries();

		return 0;
	}

	/*
	 * We are doing the normal execute a command case.
	 * Save the remaining command arguments for later use.
	 */
	cmdargc = argc;
	cmdargv = argv;

	/*
	 * If we are running as ourself, and there is no command
	 * supplied, then give a usage.  Otherwise the command may
	 * be an alias which has no arguments, and so we cannot
	 * complain here.
	 */
	if (running_our_name && (argc < 1)) {
		fprintf(stderr, "[%s version %s]\n", PROGRAM, VERSION);
		fprintf(stderr, "usage: su1 [-vn] command\n");
		fprintf(stderr, "   or: su1 -c [-v]\n");

		return 1;
	}

	/*
	 * Set up a few initial values.
	 */
	logfile = MakeString(INITLOGFILE);

	paths = MakeList(sizeof(init_paths) / sizeof(init_paths[0]),
		init_paths);

	variables = MakeList(sizeof(init_variables) / sizeof(init_variables[0]),
		init_variables);

	/*
	 * Open up the cache file in case we need it,
	 */
	OpenCacheFile();

	/*
	 * Scan the privilege file to see whether we can run the command.
	 * If we are successful, then this won't return.
	 */
	ReadPrivilegeFile();

	/*
	 * The command has been refused.
	 * Write a syslog entry if required, complain, and then quit.
	 */
	WriteSyslog(KEY_REFUSALS);
	fprintf(stderr, "Permission denied for command\n");

	return 1;
}


/*
 * Read the privilege file.  If we find that the desired command can
 * be executed, we do that and so never return.  If the command is
 * not found, we will return from here.
 */
static void
ReadPrivilegeFile()
{
	char *		cp;
	int		argc;
	char *		argv[MAXARGS];
	char		buf[1024*10];
	struct	stat	statbuf;

	priv_fp = fopen(priv_file, "r");

	if (priv_fp == NULL) {
		perror(priv_file);
		exit(1);
	}

	if (fstat(fileno(priv_fp), &statbuf)) {
		fprintf(stderr, "Cannot stat privilege file\n");
		exit(1);
	}

	if ((statbuf.st_uid != ROOTUID) || (statbuf.st_mode & 022)) {
		fprintf(stderr, "Privilege file is not secure\n");
		exit(1);
	}

	while (ReadLine(buf, sizeof(buf))) {
		cp = buf;

		while (isblank(*cp))
			cp++;

		if (*cp == COMMENTCHAR)
			continue;

		argc = 0;

		/*
		 * Break up the line into arguments.
		 */
		while (*cp) {
			if (argc >= MAXARGS)
				BadFile("too many arguments");

			argv[argc++] = cp;

			cp += ScanArgument(cp);
		}

		ExamineLine(argc, argv);
	}

	fclose(priv_fp);
}


/*
 * Scan the text contained in the specified buffer to collect together
 * the next command argument.  The argument is usually terminated by a NULL
 * or blank character.  However, single or double quote marks can be used
 * to include blanks within the argument.  Within quote marks, backslash
 * can be used to quote those characters or itself to include them within
 * the argument.  The argument text is shifted upwards to be contiguous
 * from the beginning of the buffer, and is NULL terminated.  Leading and
 * trailing blanks are removed.  The number of characters scanned from the
 * buffer is returned (NOT the length of the returned argument string).
 * Does not return on errors.
 */
static int
ScanArgument(buf)
	char *	buf;
{
	int	ch;
	int	quote;
	char *	incp;
	char *	outcp;

	incp = buf;
	outcp = buf;

	while (isblank(*incp))
		incp++;

	for (;;) {
		/*
		 * Get the next input character, and if it is a NULL or a
		 * blank character then we are done with the argument.
		 */
		ch = *incp++;

		if ((ch == '\0') || isblank(ch))
			break;

		/*
		 * If this is not a quote character then copy the input
		 * character unchanged into the output and continue.
		 */
		if ((ch != '\'') && (ch != '"')) {
			*outcp++ = ch;

			continue;
		}

		/*
		 * Here we are beginning a quoted string.
		 * Remember the quote character and continue reading
		 * characters until we see the same quote character again.
		 * Handle backslash as a single character quote which
		 * can escape anything except a NULL character.
		 */
		quote = ch;

		for (;;) {
			ch = *incp++;

			if (ch == quote)
				break;

			if (ch == '\\')
				ch = *incp++;

			if (ch == '\0')
				BadFile("unterminated quoted string");

			*outcp++ = ch;
		}
	}

	/*
	 * Skip all trailing blanks, null terminate the output string,
	 * and return the number of characters that were scanned.
	 */
	while (isblank(*incp))
		incp++;

	*outcp = '\0';

	return incp - buf;
}


/*
 * Read a line of the privilege file, taking into account continuation
 * characters.  Returns TRUE if a line was read.
 * Also remembers the file offset of the beginning of the line
 * and the current line number.
 */
static BOOL
ReadLine(buf, buflen)
	char *	buf;
	int	buflen;
{
	char *	cp;
	BOOL	gotdata;

	buflen--;
	gotdata = FALSE;

	priv_offset = ftell(priv_fp);

	while (fgets(buf, buflen, priv_fp)) {
		gotdata = TRUE;
		priv_line++;

		cp = buf + strlen(buf);

		if (cp == buf)
			BadFile("missing newline character");

		cp--;

		if (*cp != '\n')
			BadFile("line too long");

		*cp = '\0';

		if (cp == buf)
			return TRUE;

		cp--;

		if (*cp != '\\')
			return TRUE;

		*cp++ = ' ';
		buflen -= (cp - buf);
		buf = cp;
	}

	if (ferror(priv_fp))
		BadFile("read error");

	if (gotdata)
		BadFile("premature end of file");

	return FALSE;
}


/*
 * Examine one line of the privilege file.
 */
static void
ExamineLine(argc, argv)
	int	argc;
	char **	argv;
{
	if (argc-- <= 0)
		return;

	switch (FindKeyword(*argv++)) {
		case KEY_DEFINE:
			ParseDefine(argc, argv);
			break;

		case KEY_ALIAS:
			ParseAlias(argc, argv);
			break;

		case KEY_PATHS:
			ParsePaths(argc, argv);
			break;

		case KEY_LOGFILE:
			ParseLogfile(argc, argv);
			break;

		case KEY_SYSLOG:
			ParseSyslog(argc, argv);
			break;

		case KEY_PASSWORD:
			ParsePassword(argc, argv);
			break;

		case KEY_VARIABLES:
			ParseVariables(argc, argv);
			break;

		case KEY_ASK:
			ParseAsk(argc, argv);
			break;

		case KEY_ALLOW:
			ParseAllow(argc, argv);
			break;

		case KEY_REFUSE:
			ParseRefuse(argc, argv);
			break;

		default:
			BadFile("unknown keyword at beginning of line");
	}
}


/*
 * Parse the "define" directive.
 * This creates a list of user.group names.
 */
static void
ParseDefine(argc, argv)
	int	argc;
	char **	argv;
{
	int	i;
	char *	str;
	char *	cp;

	if (argc <= 0)
		BadFile("missing define name");

	str = MakeList(argc, argv);

	/*
	 * To avoid confusion for the user, disallow wildcards in the name.
	 */
	if (strcspn(str, WILDCHARS) != strlen(str))
		BadFile("wildcards used in define name");

	/*
	 * Verify that the list of user.group names looks reasonable.
	 * Don't make the check for the define name itself.
	 */
	cp = str + strlen(str) + 1;

	while (*cp) {
		VerifyNameFormat(cp);

		cp += strlen(cp) + 1;
	}

	/*
	 * Search the list of defines to see if we are replacing an
	 * old definition, and add the new definition to the list.
	 */
	for (i = 0; i < MAXDEFINES; i++) {
		if (defines[i] == NULL) {
			defines[i] = str;

			return;
		}

		if (strcmp(defines[i], str) == 0) {
			free(defines[i]);
			defines[i] = str;

			return;
		}
	}

	BadFile("too many define strings");
}


/*
 * Parse the "alias" directive.
 */
static void
ParseAlias(argc, argv)
	int	argc;
	char **	argv;
{
	int	i;
	char *	str;

	if (argc <= 0)
		BadFile("missing alias name");

	if ((argc >= 2) && (strcmp(argv[0], argv[1]) == 0))
		BadFile("recursive alias");

	str = MakeList(argc, argv);

	for (i = 0; i < MAXALIASES; i++) {
		if (aliases[i] == NULL) {
			aliases[i] = str;

			/*
			 * If this alias name matches our program name,
			 * then include our program name as part of the
			 * user's command arguments.
			 */
			if (strcmp(str, cmdname) == 0) {
				cmdargc++;
				cmdargv--;
			}

			return;
		}

		if (strcmp(aliases[i], str) == 0) {
			free(aliases[i]);
			aliases[i] = str;

			return;
		}
	}

	BadFile("too many alias strings");
}


/*
 * Parse the "paths" directive.
 */
static void
ParsePaths(argc, argv)
	int	argc;
	char **	argv;
{
	int	ac;
	char **	av;

	if (argc <= 0)
		BadFile("no paths specified");

	ac = argc;
	av = argv;

	while (ac-- > 0) {
		if (**av++ != '/')
			BadFile("path names must be absolute");
	}

	free(paths);
	paths = MakeList(argc, argv);
}


/*
 * Parse the "variables" directive.
 */
static void
ParseVariables(argc, argv)
	int	argc;
	char **	argv;
{
	if (argc > MAXVARIABLES)
		BadFile("too many environment variables");

	free(variables);
	variables = MakeList(argc, argv);
}


/*
 * Parse the "logfile" directive.
 */
static void
ParseLogfile(argc, argv)
	int	argc;
	char **	argv;
{
	if (argc <= 0)
		BadFile("missing log file path");

	if (argc > 1)
		BadFile("only one log file path can be specified");

	if (**argv != '/')
		BadFile("log file path must be absolute");

	free(logfile);
	logfile = MakeString(*argv);
}


/*
 * Parse the "syslog" directive.
 */
static void
ParseSyslog(argc, argv)
	int	argc;
	char **	argv;
{
	/*
	 * Turn off all the syslog flags until they are turned back on.
	 */
	syslog_successes = FALSE;
	syslog_failures = FALSE;
	syslog_refusals = FALSE;

	if (argc <= 0)
		BadFile("missing syslog argument");

	/*
	 * Look for the special keyword to say turn syslog off.
	 */
	if ((argc == 1) && (FindKeyword(*argv) == KEY_NONE))
		return;

	/*
	 * Check all the arguments for our keywords.
	 * Enable those flags which are specified.
	 */
	while (argc-- > 0) {
		switch (FindKeyword(*argv++)) {
			case KEY_ALL:
				syslog_successes = TRUE;
				syslog_failures = TRUE;
				syslog_refusals = TRUE;
				break;

			case KEY_SUCCESSES:
				syslog_successes = TRUE;
				break;

			case KEY_FAILURES:
				syslog_failures = TRUE;
				break;

			case KEY_REFUSALS:
				syslog_refusals = TRUE;
				break;

			case KEY_NONE:
				BadFile("inappropriate use of none for syslog");

			default:
				BadFile("unknown value specified for syslog keyword");
		}
	}
}


/*
 * Parse the "password" directive.
 */
static void
ParsePassword(argc, argv)
	int	argc;
	char **	argv;
{
	password_offset = priv_offset;

	if (password)
		free(password);

	password = NULL;

	if (argc <= 0)
		return;

	if (argc > 1)
		BadFile("only one password can be specified");

	if (strlen(*argv) < 3)
		BadFile("password must be more than two characters");

	password = MakeString(*argv);
}


/*
 * Parse the "ask" directive.
 */
static void
ParseAsk(argc, argv)
	int	argc;
	char **	argv;
{
	if (argc <= 0)
		BadFile("missing value for ask keyword");

	ask_type = FindKeyword(*argv);
	ask_interval = 0;

	switch (ask_type) {
		case KEY_NEVER:
		case KEY_ALWAYS:
		case KEY_ONCE:
			if (argc > 1)
				BadFile("too many arguments for ask keyword");

			ask_interval = MAXINTERVAL;

			return;

		case KEY_INTERVAL:
			if (argc < 2)
				BadFile("missing argument for ask interval");

			if (argc > 2)
				BadFile("too many arguments for ask interval");

			ask_interval = ScanNumber(argv[1]) * 60;

			if ((ask_interval < 0) || (ask_interval > MAXINTERVAL))
				BadFile("illegal ask interval");

			return;

		default:
			BadFile("unknown value specified for ask keyword");
	}
}


/*
 * Parse the "allow" directive.
 * This is where the command is executed if it is successful.
 * In that case, this routine does not return.
 */
static void
ParseAllow(argc, argv)
	int	argc;
	char **	argv;
{
	char *	user;
	KEYWORD	type;

	if (argc < 2)
		BadFile("insufficient arguments for allow keyword");

	user = *argv++;
	type = FindKeyword(*argv++);
	argc -= 2;

	switch (type) {
		case KEY_PREFIX:
		case KEY_EXACT:
			if (!CheckCommand(argc, argv, type))
				return;

			break;

		case KEY_ANY:
			break;

		default:
			BadFile("bad command type specified for allow keyword");
	}

	if (cmdargc <= 0)
		return;

	if (!CheckUserList(user))
		return;

	/*
	 * The user can execute his chosen command.  Ask for and check
	 * the password, write the cache entry if required, write the log
	 * file and syslog if required, and finally execute his command.
	 * We never return from here.
	 */
	fclose(priv_fp);

	CheckPassword();
	CloseCacheFile();
	WriteLogFile();
	WriteSyslog(KEY_SUCCESSES);

	ExecuteCommand(type);

	exit(1);
}


/*
 * Parse the "refuse" directive.
 * It this applies, then this routine does not return.
 */
static void
ParseRefuse(argc, argv)
	int	argc;
	char **	argv;
{
	char *	user;
	KEYWORD	type;

	if (argc < 2)
		BadFile("insufficient arguments for refuse keyword");

	user = *argv++;
	type = FindKeyword(*argv++);
	argc -= 2;

	switch (type) {
		case KEY_PREFIX:
		case KEY_EXACT:
			if (!CheckCommand(argc, argv, type))
				return;

			break;

		case KEY_ANY:
			break;

		default:
			BadFile("bad command type specified for refuse keyword");
	}

	if (cmdargc <= 0)
		return;

	if (!CheckUserList(user))
		return;

	/*
	 * The user has explicitly been refused to do this command.
	 * Write a syslog entry if required and terminate the program
	 * immediately with an error.
	 */
	WriteSyslog(KEY_REFUSALS);
	fprintf(stderr, "Permission denied for command\n");
	exit(1);
}


/*
 * Complain about a bad line in the privilege file and exit.
 */
static void
BadFile(str)
	char *	str;
{
	fprintf(stderr, "Error in \"%s\", line %ld: %s\n",
		priv_file, priv_line, str);

	exit(1);
}


/*
 * Check the specified string to see if it is a keyword.
 * Returns the keyword value if so, or KEY_NULL if it was not found.
 */
static KEYWORD
FindKeyword(str)
	char *	str;
{
	KEYTABLE *	key;

	for (key = keytable; key->name; key++) {
		if (strcmp(str, key->name) == 0)
			return key->code;
	}

	return KEY_NULL;
}


/*
 * Append an array of words together to form a long list of strings.
 * The strings are separated by nulls, and the end of the list is
 * marked by an extra null.
 */
static char *
MakeList(argc, argv)
	int	argc;
	char **	argv;
{
	int	len;
	int	ac;
	char **	av;
	char *	str;
	char *	cp;

	ac = argc;
	av = argv;

	/*
	 * First scan the list to count up the total length of the
	 * string needed, and check for any null names in the list.
	 */
	len = 1;

	while (ac-- > 0) {
		if (**av == '\0')
			BadFile("null list entry");

		len += strlen(*av++) + 1;
	}

	/*
	 * Now allocate the string and append the strings separated
	 * by null characters, with two nulls at the end.
	 */
	str = Allocate(len);

	cp = str;

	while (argc-- > 0) {
		strcpy(cp, *argv++);
		cp += strlen(cp) + 1;
	}

	*cp = '\0';

	return str;
}


/*
 * Make a string into a one-element list.
 * This is just a copy of the string terminated by two null characters.
 */
char *
MakeString(str)
	char *	str;
{
	int	len;
	char *	cp;

	len = strlen(str);

	cp = Allocate(len + 2);

	memcpy(cp, str, len + 1);
	cp[len + 1] = '\0';

	return cp;
}


/*
 * Check some command arguments to see if they match the ones the
 * user is trying to execute.  Returns TRUE if so.
 */
static BOOL
CheckCommand(argc, argv, type)
	int	argc;
	char **	argv;
	KEYWORD	type;
{
	char **	hisargv;

	if (argc <= 0)
		BadFile("missing command for user keyword");

	hisargv = cmdargv;

	if (cmdargc < argc)
		return FALSE;

	if ((type == KEY_EXACT) && (cmdargc != argc))
		return FALSE;

	while (argc-- > 0) {
		if (strcmp(*argv++, *hisargv++))
			return FALSE;
	}

	return TRUE;
}


/*
 * Check our user name against a specified user name or list of user names.
 * Each name can actually be a combination user and group name separated
 * by a single period, such as "dbell.users".  Both the user name and the
 * group name must match.  Wildcard patterns can be used to match the user or
 * group names.  If the name contains no period, then the group name is "*",
 * and so will match any group.  Returns TRUE if our name is in the list.
 */
static BOOL
CheckUserList(name)
	char *	name;
{
	char *	str;

	/*
	 * See if the name represents a list of users.
	 * If so, then replace the name by the list of user names.
	 */
	str = FindDefine(name);

	if (str == NULL)
		str = MakeString(name);

	/*
	 * Check our name against each one of the names in the list.
	 */
	while (*str) {
		if (UserAndGroupNameMatchesMine(str))
			return TRUE;

		str += strlen(str) + 1;
	}

	return FALSE;
}


/*
 * Check the specified user.group name string against our own user name
 * and group names, applying possible wildcards separately to each one.
 * Returns TRUE if both the user name and the group name matches.
 * A missing group name means that all groups match.
 *
 * WARNING:
 *	This routine temporarily modifies the incoming argument so that it
 *	cannot be a constant string.
 */
static BOOL
UserAndGroupNameMatchesMine(wildname)
	char *	wildname;
{
	char *	period;
	BOOL	match;
	int	i;

	VerifyNameFormat(wildname);

	/*
	 * See if the name contains a period.
	 * If not, then all groups match so just check the user name.
	 */
	period = strchr(wildname, '.');

	if (period == NULL)
		return PatternMatch(wildname, myname);

	/*
	 * There is a period, and so a group name needs to be checked too.
	 * Temporarily replace the period by a null character for the user
	 * name check.  If the name doesn't match, then return FALSE.
	 */
	*period = '\0';

	match = PatternMatch(wildname, myname);

	*period = '.';

	if (!match)
		return FALSE;

	/*
	 * The user name matched, so now we need to match the group
	 * name against our list of group names.  If it matches any
	 * one of those, then return TRUE.
	 */
	wildname = period + 1;

	for (i = 0; i < mygroupcount; i++) {
		if (PatternMatch(wildname, mygroups[i]))
			return TRUE;
	}

	/*
	 * The name doesn't match any of our groups, so return FALSE.
	 */
	return FALSE;
}


/*
 * Verify that a combined user and group name entry is of a valid format.
 * This basically means that it contains zero or one periods, and if it
 * does contain a period, then something is present before and after it.
 * (Periods are used to separate user names from group names.)
 * Does not return on an error.
 */
static void
VerifyNameFormat(name)
	char *	name;
{
	char *	cp;

	if ((name == NULL) || (*name == '\0'))
		BadFile("null user.group name");

	cp = strchr(name, '.');

	if (cp == NULL)
		return;

	if ((cp == name) || (cp[1] == '\0'))
		BadFile("null side in user.group name");

	if (strchr(cp + 1, '.') != NULL)
		BadFile("multiple periods in user.group name");
}


/*
 * Search for a definition for the given define name, and return the list.
 * This does not return the initial element of the list (which is the name).
 * If the list name was not found, a NULL pointer is returned.
 */
static char *
FindDefine(name)
	char *	name;
{
	int	i;
	char *	str;

	for (i = 0; i < MAXDEFINES; i++) {
		str = defines[i];

		if ((str == NULL) || strcmp(str, name))
			continue;

		str += strlen(str) + 1;

		return str;
	}

	return NULL;
}


/*
 * Search for a definition for the given alias name, and return the list.
 * This does not return the initial element of the list (which is the name).
 * If the list name was not found, a NULL pointer is returned.
 */
static char *
FindAlias(name)
	char *	name;
{
	int	i;
	char *	str;

	for (i = 0; i < MAXALIASES; i++) {
		str = aliases[i];

		if ((str == NULL) || strcmp(str, name))
			continue;

		str += strlen(str) + 1;

		return str;
	}

	return NULL;
}


/*
 * Get my user id, my user name, my group ids, my group names,
 * and my HOME environment variable.
 * We don't call cuserid, getlogin, or the USER environment variable
 * because none of these are reliably set.  Instead we use the password
 * file.  The disadvantage of this is that we can't distinguiush between
 * two user names having the same user id.
 */
static void
IdentifyMyself()
{
	struct passwd *	pwd;
	struct group *	grp;
	int		i;

	/*
	 * Find out who I really am.
	 */
	myuid = getuid();

	/*
	 * Get my password entry.
	 */
	pwd = getpwuid(myuid);

	if (pwd == NULL) {
		fprintf(stderr, "Cannot find the password entry for uid %d\n",
			myuid);

		exit(1);
	}

	/*
	 * Save my user name.
	 */
	myname = AllocateString(pwd->pw_name);

	/*
	 * Get my primary group id as the first element of a group array,
	 * and then get the supplementary groups if any as further elements.
	 * Remember the total number of groups.
	 */
	mygids[0] = getgid();

	mygroupcount = getgroups(MAXGROUPS - 1, &mygids[1]);

	if (mygroupcount < 0) {
		perror("cannot get list of groups");

		exit(1);
	}

	mygroupcount++;

	/*
	 * Get and save the group names for each of the group ids in the array.
	 */
	for (i = 0; i < mygroupcount; i++) {
		grp = getgrgid(mygids[i]);

		if (grp == NULL) {
			fprintf(stderr,
				"Cannot find the group entry for gid %d\n",
				mygids[i]);

			exit(1);
		}

		mygroups[i] = AllocateString(grp->gr_name);
	}

	/*
	 * Initialize some environment variables that we might need later.
	 */
	homevar = AllocateEnvironmentString("HOME", pwd->pw_dir);
	shellvar = AllocateEnvironmentString("SHELL", pwd->pw_shell);
}


/*
 * Ask for the required password before allowing the command to execute.
 * This password can be one given in the privilege file, or else the real
 * root password.  This checks the cache file if allowable, and writes an
 * entry in the cache file if appropriate.  If a password is required and
 * is wrong, a syslog entry may be written and the routine does not return.
 * This can get the password either ourselves, or by using the PAM routines,
 * depending on how the program is compiled.
 */
static void
CheckPassword()
{
	/*
	 * If a password is never required then we win.
	 */
	if (ask_type == KEY_NEVER)
		return;

	/*
	 * If the password is only required occasionally then check
	 * the cache to see if it has already been given recently.
	 * If so, then revalidate its time for another interval and
	 * we win.
	 */
	if ((ask_type == KEY_ONCE) || (ask_type == KEY_INTERVAL)) {
		if (IsPasswordCached(password_offset, ask_interval)) {
			RememberCacheEntry(password_offset);

			return;
		}
	}

#ifdef	LINUX_PAM
	/*
	 * If there is no target password set from the privilege file,
	 * then ask PAM to ask for and check the real root password.
	 * Otherwise ask the user for the password and validate it
	 * against the password from the privilege file.  These routines
	 * don't return if the password is wrong.
	 */
	if (password == NULL)
		CheckPamPassword();
	else
		CheckLocalPassword();
#else
	/*
	 * If there is no target password set from the privilege file,
	 * then default it to the root password.  Then ask the user for
	 * the password and validate it against the target one.
	 * This doesn't return if the password is wrong.
	 */
	DefaultPassword();
	CheckLocalPassword();
#endif

	/*
	 * The password was right, so we can execute his command.
	 * If we only have to ask within a time interval then cache the
	 * fact that he knew it so that he won't have to type it again soon.
	 */
	if ((ask_type == KEY_ONCE) || (ask_type == KEY_INTERVAL))
		RememberCacheEntry(password_offset);
}


/*
 * Default the password if necessary to be the root password.
 * Doesn't return if the root password cannot be obtained.
 * This routine assumes that shadow passwords are not in use.
 * This routine is not used if PAM authentication is compiled in.
 */
#ifndef	LINUX_PAM

static void
DefaultPassword()
{
	struct passwd *	pwd;

	/*
	 * If a particular password is already set, then use that one.
	 */
	if (password)
		return;

	/*
	 * There was no password set from the privilege file.
	 * So remember the root password as the one to check against.
	 * Get this from the normal /etc/passwd file.
	 */
	pwd = getpwnam(ROOTNAME);

	if (pwd == NULL) {
		fprintf(stderr, "Cannot get root password\n");

		exit(1);
	}

	/*
	 * Save the password.
	 */
	password = AllocateString(pwd->pw_passwd);
}

#endif


/*
 * Ask for a password, encrypt it, and compare it against the target
 * password value.  Doesn't return if the password is invalid.
 */
static void
CheckLocalPassword()
{
	char *	str;
	BOOL	tried;

	/*
	 * If there is no password set, then something is wrong.
	 */
	if (password == NULL) {
		fprintf(stderr, "No password has been set\n");

		exit(1);
	}

	/*
	 * If the target password is empty, then we win.
	 * But unbelievable that any system administrator would do this!
	 */
	if (*password == '\0')
		return;

	/*
	 * The password is required.
	 * Read it from the user, encrypt it, and then compare it.
	 * Remember if the password given was non-null.
	 */
	str = getpass("Password: ");
	tried = (*str != '\0');
	str = crypt(str, password);

	/*
	 * If the password is wrong, then write a syslog entry if required,
	 * sleep for a second (if a non-null password was tried), complain
	 * and then exit.  The sleep is to slow down password guessers.
	 */
	if (strcmp(str, password)) {
		WriteSyslog(KEY_FAILURES);

		if (tried)
			sleep(1);

		fprintf(stderr, "Incorrect password\n");

		exit(1);
	}
}


/*
 * Function to ask the PAM modules to ask for and check the root password.
 * Doesn't return if the authentication failed.
 */
#ifdef	LINUX_PAM

static void CheckPamPassword()
{
	pam_handle_t *	pamh;
	int		status;

	static struct pam_conv conv = {
		misc_conv,
		NULL
	};

	/*
	 * Initialise for calling PAM.
	 */
	status = pam_start(PROGRAM, ROOTNAME, &conv, &pamh);

	if (status != PAM_SUCCESS) {
		fprintf(stderr, "PAM initialization failed: %s\n",
			pam_strerror(status));

		exit(1);
	}

	/*
	 * Ask PAM to get and validate the root password.
	 */
	status = pam_authenticate(pamh, PAM_SILENT);

	/*
	 * If the authentication was not correct, then log that fact,
	 * sleep a second to password guessers, give an error message,
	 * and exit.
	 */
	if (status != PAM_SUCCESS) {
		WriteSyslog(KEY_FAILURES);

		sleep(1);

		fprintf(stderr, "%s\n", pam_strerror(status));

		exit(1);
	}

	/*
	 * The password is valid.  Now check the account management
	 * restrictions defined for root.  For su1, I personally think
	 * this should be defined to always succeed, but I suppose that
	 * the system administrators should still have their say about
	 * whether PAM restrictions are imposed on its use.
	 */
	status = pam_acct_mgmt(pamh, 0);

	if (status != PAM_SUCCESS) {
		WriteSyslog(KEY_REFUSALS);

		fprintf(stderr, "%s\n", pam_strerror(status));

		exit(1);
	}

	/*
	 * Authentication is successful!
	 * Close down the PAM module and return.
	 */
	status = pam_end(pamh, PAM_SUCCESS);

	if (status != PAM_SUCCESS) {
		fprintf(stderr, "PAM release failed: %s\n",
			pam_strerror(status));

		exit(1);
	}
}

#endif


/*
 * Scan a numeric value from a string.
 * Only nonnegative numbers are allowed.
 */
static long
ScanNumber(cp)
	char *	cp;
{
	long	val;

	if (*cp == '\0')
		BadFile("missing number");

	val = 0;

	while ((*cp >= '0') && (*cp <= '9'))
		val = val * 10 + *cp++ - '0';

	if (*cp || (val < 0))
		BadFile("bad number");

	return val;
}


/*
 * Write out the log file.  This is only used for successes.
 * There is a not-too-important race in this routine for multiple users.
 */
static void
WriteLogFile()
{
	FILE *	fp;
	int	ac;
	char **	av;
	time_t	timebuf;

	if (strcmp(logfile, "/dev/null") == 0)
		return;

	fp = fopen(logfile, "a");

	if (fp == NULL) {
		fprintf(stderr, "Cannot open log file\n");
		exit(1);
	}

	time(&timebuf);
	fprintf(fp, "%-8s %16.16s ", myname, ctime(&timebuf));

	ac = cmdargc;
	av = cmdargv;

	while (ac-- > 0)
		fprintf(fp, " %s", *av++);

	fprintf(fp, "\n");
	fflush(fp);

	if (ferror(fp)) {
		fclose(fp);
		fprintf(stderr, "Error writing log file\n");
		exit(1);
	}

	if (fclose(fp)) {
		fprintf(stderr, "Error closing log file\n");
		exit(1);
	}
}


/*
 * Write out the result of a command to syslog.
 * This can be used for successes, failures, or refusals.
 */
static void
WriteSyslog(type)
	KEYWORD	type;
{
	char *	cp;
	char *	tag;
	char **	av;
	int	ac;
	int	level;
	int	used;
	int	len;
	BOOL	wantlog;
	char	buf[SYSLOGBUFSIZE + 8];

	/*
	 * See whether we want to syslog this type of result
	 * and set up the tag string and level for the message.
	 */
	switch (type) {
		case KEY_SUCCESSES:
			wantlog = syslog_successes;
			level = LOG_NOTICE;
			tag = "success";
			break;

		case KEY_FAILURES:
			wantlog = syslog_failures;
			level = LOG_WARNING;
			tag = "failure";
			break;

		case KEY_REFUSALS:
			wantlog = syslog_refusals;
			level = LOG_NOTICE;
			tag = "refusal";
			break;

		default:
			fprintf(stderr, "Unknown syslog type\n");
			exit(1);
	}

	/*
	 * If this type of syslog is not wanted, then return.
	 */
	if (!wantlog)
		return;

	/*
	 * Syslog is wanted, so build up the argument string while
	 * checking for overflows.  If the whole command won't fit
	 * then end it with some dots.
	 */
	used = 0;
	buf[0] = '\0';

	ac = cmdargc;
	av = cmdargv;

	while (ac-- > 0) {
		buf[used++] = ' ';

		len = strlen(*av);

		if (used + len > SYSLOGBUFSIZE) {
			strcpy(&buf[used], "...");
			break;
		}

		strcpy(&buf[used], *av++);
		used += len;
	}

	/*
	 * Go through the command string and replace unprintable
	 * characters with question marks.
	 */
	for (cp = buf; *cp; cp++) {
		if (!isprint(*cp))
			*cp = '?';
	}

	/*
	 * OK, open the syslog facility and format the log entry.
	 * This contains the user name, the tag string, and the command.
	 * The time, date, hostname, and program name are added by syslog.
	 */
	openlog(PROGRAM, LOG_CONS, LOG_AUTHPRIV);
	syslog(level, "%s %s:%s", myname, tag, buf);
	closelog();
}


/*
 * Execute the user's command after setting our group and user ids and
 * initializing our environment.  The type argument specifies whether
 * or not the user can execute any command.
 */
static void
ExecuteCommand(type)
	KEYWORD	type;
{
	int	i;
	int	j;
	int	cmdlen;
	char *	cp;
	char *	args[MAXARGS];
	char	execpath[MAXPATHSIZE];

	/*
	 * Make sure that our environment is good in order to
	 * run other programs safely.
	 */
	InitializeEnvironment();

	/*
	 * Change our real group and user ids to be root so that
	 * we can run privileged programs that check the user id
	 * without them complaining.
	 */
	if (setgid(ROOTGID)) {
		perror("cannot set group id");
		exit(1);
	}

	if (setuid(ROOTUID)) {
		perror("cannot set user id");
		exit(1);
	}

	/*
	 * Build the list of arguments.
	 */
	for (i = 0; i < cmdargc; i++)
		args[i] = cmdargv[i];

	args[i] = NULL;

	/*
	 * If the command type allows any command to be executed,
	 * then see if it contains a slash.  If so, then run it explicitly.
	 * Otherwise we will only run it from the explicitly set PATH.
	 * This helps to avoid accidentally running malicious programs
	 * from places like /usr/games or the current directory.
	 */
	if ((type == KEY_ANY) && (strchr(args[0], '/') != NULL)) {
		DisplayCommand(args);

		execv(args[0], args);

		fprintf(stderr, "Cannot execute \"%s\" using ", args[0]);
		perror("execv");
		exit(1);
	}

	/*
	 * The command is restricted to a limited set of commands run from
	 * a limited number of directories.  See if the command name matches
	 * one of the alias strings.  If so, then replace the command name
	 * with the alias definition.
	 */
	cp = FindAlias(cmdargv[0]);

	if (cp) {
		i = 0;

		while (*cp) {
			args[i++] = cp;
			cp += strlen(cp) + 1;
		}

		for (j = 1; j < cmdargc; j++) {
			if (i >= MAXARGS - 1) {
				fprintf(stderr, "Too many arguments\n");
				exit(1);
			}

			args[i++] = cmdargv[j];
		}

		args[i] = NULL;
	}

	/*
	 * If the command name contains a slash, then make sure that it
	 * is an absolute path, and then try to execute that exact program.
	 * Relative paths are not allowed since the user could be cd'd
	 * anywhere and so trick us.
	 */
	if (strchr(args[0], '/')) {
		if (args[0][0] != '/') {
			fprintf(stderr, "Command is not an absolute pathname\n");
			exit(1);
		}

		DisplayCommand(args);

		execv(args[0], args);

		fprintf(stderr, "Cannot execute \"%s\" using ", args[0]);
		perror("execv");
		exit(1);
	}

	/*
	 * The command name is a generic name, so search down our own
	 * internal path list, trying to execute the program from each path.
	 */
	DisplayCommand(args);

	cmdlen = strlen(args[0]);

	for (cp = paths; *cp; cp += strlen(cp) + 1) {
		if (strlen(cp) + cmdlen > MAXPATHSIZE - 2)
			BadFile("path name of command is too long");

		strcpy(execpath, cp);
		strcat(execpath, "/");
		strcat(execpath, args[0]);

		execv(execpath, args);
	}

	fprintf(stderr, "Failed to exec \"%s\" using ", args[0]);
	perror("execv");
	exit(1);
}


/*
 * Display the command which is about to be executed, if that was enabled.
 * The command line arguments are terminated with a NULL pointer.
 */
static void
DisplayCommand(args)
	char **	args;
{
	if (!verbose_flag)
		return;

	printf("Executing:");

	while (*args)
		printf(" %s", *args++);

	printf("\n");
}


/*
 * Look in the cache file to see if we are allowed to bypass asking
 * for a password.  Returns TRUE if the password is not required.
 * This also looks for and saves the index of an obsolete cache entry
 * which can be reused if we need to write a new one.
 */
static BOOL
IsPasswordCached(offset, interval)
	long	offset;
	long	interval;
{
	int		slot;
	long		elapsed;
	CACHE_ENTRY	entry;
	CACHE_ENTRY	cache_entry;

	if (!InitializeCacheEntry(&cache_entry, offset))
		return FALSE;

	/*
	 * Look for our entry.
	 */
	for (slot = 0; slot < cache_header.entries; slot++)
	{
		ReadCacheEntry(slot, &entry);

		/*
		 * Check this cache entry to see if it can be reclaimed.
		 * This is true if the last used time is too old.
		 * Try to use the earliest slot number in the file.
		 * If we do not yet have a cache entry, then this one
		 * will be used.
		 */
		elapsed = cache_entry.last_used - entry.last_used;

		if (cache_slot < 0)
		{
			if ((elapsed < 0) || (elapsed > MAXINTERVAL))
				cache_slot = slot;
		}

		/*
		 * Check things which must match in order to have a valid
		 * cache entry for us.  If they do not match, then skip
		 * this entry.
		 */
		if ((entry.uid != cache_entry.uid) ||
			(entry.parent_pid != cache_entry.parent_pid) ||
			(entry.session_id != cache_entry.session_id) ||
			(entry.dev != cache_entry.dev) ||
			(entry.inode != cache_entry.inode) ||
			(entry.offset != cache_entry.offset))
		{
			continue;
		}

		/*
		 * We found an old cache entry for us that matches everything
		 * expect perhaps a recent time.  Remember this slot number
		 * for storing our new cache entry into.
		 */
		cache_slot = slot;

		/*
		 * Return TRUE depending on whether this cache entry is new
		 * enough to be used.  If it is too old, then a password is
		 * again required.
		 */
		return ((elapsed >= 0) &&
			(elapsed <= MAXINTERVAL) &&
			(elapsed <= interval));
	}

	return FALSE;
}


/*
 * Add our current cache entry to the cache file for future use.
 * This should be called after the cache file has been scanned
 * in order to set the slot number where we can write our entry.
 * If no free slot number is found, a new one at the end of the
 * file will be allocated.  Doesn't return on an error.
 */
static void
RememberCacheEntry(offset)
	long	offset;
{
	CACHE_ENTRY	cache_entry;

	if (!InitializeCacheEntry(&cache_entry, offset))
		return;

	/*
	 * If we already found a slot number, then write it to that one.
	 */
	if (cache_slot >= 0) {
		WriteCacheEntry(cache_slot, &cache_entry);

		return;
	}

	/*
	 * We need to allocate a slot number.
	 * This will be one past the current number of entries.
	 * Write the entry, and then the new header.
	 */
	cache_slot = cache_header.entries++;

	WriteCacheEntry(cache_slot, &cache_entry);
	WriteCacheHeader(&cache_header);
}


/*
 * Open up the cache file for reading and writing.
 * The file is locked for exclusive access so that multiple updates
 * are safe.  The file is created if necessary.  Doesn't return on errors.
 */
static void
OpenCacheFile()
{
	CACHE_HEADER	header;
	struct	stat	statbuf;

	/*
	 * Open the cache file for reading and writing,
	 * creating it if necessary.
	 */
	cache_fd = open(cache_file, O_RDWR | O_CREAT, 0600);

	if (cache_fd < 0) {
		fprintf(stderr, "Cannot open cache file\n");
		exit(1);
	}

	/*
	 * Lock the file for exclusive use.
	 */
	if (flock(cache_fd, LOCK_EX) < 0) {
		fprintf(stderr, "Cannot lock cache file\n");
		exit(1);
	}

	/*
	 * Find out about the file we have opened and make sure that
	 * it is owned by root and is not writable by anyone else.
	 */
	if (fstat(cache_fd, &statbuf) < 0) {
		fprintf(stderr, "Cannot stat cache file\n");
		exit(1);
	}

	if ((statbuf.st_uid != ROOTUID) || (statbuf.st_mode & 022)) {
		fprintf(stderr, "Cache file is not secure\n");
		exit(1);
	}

	/*
	 * If the cache file is empty, then write an initial header to it,
	 * otherwise read in the old header.
	 */
	if (statbuf.st_size > 0)
		ReadCacheHeader(&header);
	else {
		header.magic = CACHE_MAGIC;
		header.priv_time = 0;
		header.priv_size = 0;
		header.entries = 0;

		WriteCacheHeader(&header);
	}

	/*
	 * Find out the size and last modification time of the privilege
	 * file, and verify that those values match the ones saved in
	 * the cache file header.  If not, then the cache file is obsolete
	 * and so we must reinitialize it.
	 */
	if (stat(priv_file, &statbuf) < 0) {
		fprintf(stderr, "Cannot stat privilege file\n");
		exit(1);
	}

	if ((statbuf.st_size != header.priv_size) ||
		(statbuf.st_mtime != header.priv_time))
	{
		header.priv_time = statbuf.st_mtime;
		header.priv_size = statbuf.st_size;
		header.entries = 0;

		WriteCacheHeader(&header);
	}

	/*
	 * Remember the cache header for later use and clear the slot where
	 * we want to write our new entry to.
	 */
	cache_header = header;
	cache_slot = -1;
}


/*
 * Close the cache file.
 */
static void
CloseCacheFile()
{
	if (close(cache_fd) < 0)
		perror(cache_file);

	cache_fd = -1;
}


/*
 * Read the header from the cache file and verify it.
 * Doesn't return on an error.
 */
static void
ReadCacheHeader(header)
	CACHE_HEADER *	header;
{
	int	cc;

	if (lseek(cache_fd, 0L, SEEK_SET) < 0) {
		perror(cache_file);
		exit(1);
	}

	cc = read(cache_fd, (char *) header, sizeof(CACHE_HEADER));

	if (cc < 0) {
		perror(cache_file);
		exit(1);
	}

	if (cc != sizeof(CACHE_HEADER)) {
		fprintf(stderr, "Short read of cache header\n");
		exit(1);
	}

	if (header->magic != CACHE_MAGIC) {
		fprintf(stderr, "Cache file has an incorrect magic number\n");
		exit(1);
	}

	if ((header->entries < 0) || (header->priv_size < 0)) {
		fprintf(stderr, "Cache file has an invalid header\n");
		exit(1);
	}
}


/*
 * Verify a cache file header and write it to the cache file.
 * Doesn't return on an error.
 */
static void
WriteCacheHeader(header)
	CACHE_HEADER *	header;
{
	int	cc;

	if ((header->magic != CACHE_MAGIC) || (header->entries < 0) ||
		(header->priv_size < 0))
	{
		fprintf(stderr, "Attempt to write bad cache header\n");
		exit(1);
	}

	if (lseek(cache_fd, 0L, SEEK_SET) < 0) {
		perror(cache_file);
		exit(1);
	}

	cc = write(cache_fd, (char *) header, sizeof(CACHE_HEADER));

	if (cc < 0) {
		perror(cache_file);
		exit(1);
	}

	if (cc != sizeof(CACHE_HEADER)) {
		fprintf(stderr, "Short write of cache header\n");
		exit(1);
	}
}


/*
 * Read the cache entry for the specified slot number.
 * Doesn't return on an error.
 */
static void
ReadCacheEntry(slot, entry)
	int		slot;
	CACHE_ENTRY *	entry;
{
	long	pos;
	int	cc;

	if ((slot < 0) || (slot > cache_header.entries)) {
		fprintf(stderr, "Reading bad cache slot number %d\n", slot);
		exit(1);
	}

	pos = sizeof(CACHE_HEADER) + (sizeof(CACHE_ENTRY) * slot);

	if (lseek(cache_fd, pos, SEEK_SET) < 0) {
		perror(cache_file);
		exit(1);
	}

	cc = read(cache_fd, (char *) entry, sizeof(CACHE_ENTRY));

	if (cc < 0) {
		perror(cache_file);
		exit(1);
	}

	if (cc != sizeof(CACHE_ENTRY)) {
		fprintf(stderr, "Short read of cache entry %d\n", slot);
		exit(1);
	}
}


/*
 * Write the cache entry for the specified slot number.
 * Doesn't return on an error.
 */
static void
WriteCacheEntry(slot, entry)
	int		slot;
	CACHE_ENTRY *	entry;
{
	long	pos;
	int	cc;

	if ((slot < 0) || (slot > cache_header.entries)) {
		fprintf(stderr, "Writing bad cache slot number %d\n", slot);
		exit(1);
	}

	pos = sizeof(CACHE_HEADER) + (sizeof(CACHE_ENTRY) * slot);

	if (lseek(cache_fd, pos, SEEK_SET) < 0) {
		perror(cache_file);
		exit(1);
	}

	cc = write(cache_fd, (char *) entry, sizeof(CACHE_ENTRY));

	if (cc < 0) {
		perror(cache_file);
		exit(1);
	}

	if (cc != sizeof(CACHE_ENTRY)) {
		fprintf(stderr, "Short write of cache entry %d\n", slot);
		exit(1);
	}
}


/*
 * Initialize a new cache entry for us if we can.
 * This can fail in various ways in which case we return FALSE
 * to force a password to be used.
 */
static BOOL
InitializeCacheEntry(entry, offset)
	CACHE_ENTRY *	entry;
	long		offset;
{
	int		parent_pid;
	int		session_id;
	int		stdin_dev;
	int		stdin_inode;
	time_t		now;
	char		dummychar;
	struct	stat	statbuf;

	memset((char *) entry, 0, sizeof(CACHE_ENTRY));

	/*
	 * If the cache file is disabled, then stop now.
	 */
	if (no_cache_flag)
		return FALSE;

	/*
	 * Get our parent's pid, which is probably the shell running
	 * the terminal session.  Give up if we have no parent.
	 */
	parent_pid = getppid();

	if ((parent_pid == 0) || (parent_pid == 1))
		return FALSE;

	/*
	 * Get our session id which should uniquely identify our session.
	 */
	session_id = getsid(0);

	if ((session_id == 0) || (session_id == 1))
		return FALSE;

	/*
	 * Find out information about my STDIN file descriptor.
	 * This must be a terminal device, must be owned by us,
	 * and must be readable.
	 */
	if (!isatty(STDIN))
		return FALSE;

	if (fstat(STDIN, &statbuf) < 0)
		return FALSE;

#if	PARANOID
	/*
	 * This check would be nice to have, but it makes su1 fail
	 * if you have su'd to another user.  So disable this check if
	 * you want convenience, at the risk of losing a little security.
	 * (The loss of security only applies to other login sessions for
	 * the same user id.)
	 */
	if (statbuf.st_uid != myuid)
		return FALSE;
#endif

	if (read(STDIN, &dummychar, 0) < 0)
		return FALSE;

	stdin_dev = statbuf.st_rdev;
	stdin_inode = statbuf.st_ino;

	/*
	 * Get the current time of day.
	 */
	time(&now);

	/*
	 * OK, fill in the cache entry with the information.
	 */
	entry->uid = myuid;
	entry->dev = stdin_dev;
	entry->inode = stdin_inode;
	entry->parent_pid = parent_pid;
	entry->session_id = session_id;
	entry->offset = offset;
	entry->last_used = now;

	return TRUE;
}


/*
 * Clear all of the cache entries which are for my user id.
 * This will force the asking of passwords again ahead of when they
 * would normally time out.  Does not return on an error.
 */
static void
ClearMyCacheEntries()
{
	int		slot;
	CACHE_ENTRY	entry;

	OpenCacheFile();

	/*
	 * Look for any of our entries and clear them.
	 */
	for (slot = 0; slot < cache_header.entries; slot++)
	{
		ReadCacheEntry(slot, &entry);

		if (entry.uid != myuid)
			continue;

		memset((char *) &entry, 0, sizeof(CACHE_ENTRY));

		WriteCacheEntry(slot, &entry);
	}

	CloseCacheFile();

	if (verbose_flag)
		fprintf(stderr, "Cache entries cleared for %s.\n", myname);
}


/*
 * Routine to see if a text string is matched by a wildcard pattern.
 * Returns TRUE if the text is matched, or FALSE if it is not matched
 * or if the pattern is invalid.
 *  *		matches zero or more characters
 *  ?		matches a single character
 *  [abc]	matches 'a', 'b' or 'c'
 *  \c		quotes character c
 *  Adapted from code written by Ingo Wilken.
 */
static BOOL
PatternMatch(pattern, text)
	char *	pattern;
	char *	text;
{
	char *	retrypat;
	char *	retrytxt;
	int	ch;
	BOOL	found;

	retrypat = NULL;
	retrytxt = NULL;

	while (*text || *pattern) {
		ch = *pattern++;

		switch (ch) {
			case '*':  
				retrypat = pattern;
				retrytxt = text;
				break;

			case '[':  
				found = FALSE;

				while ((ch = *pattern++) != ']') {
					if (ch == '\\')
						ch = *pattern++;

					if (ch == '\0')
						return FALSE;

					if (*text == ch)
						found = TRUE;
				}

				if (!found) {
					pattern = retrypat;
					text = ++retrytxt;
				}

				/* fall into next case */

			case '?':  
				if (*text++ == '\0')
					return FALSE;

				break;

			case '\\':  
				ch = *pattern++;

				if (ch == '\0')
					return FALSE;

				/* fall into next case */

			default:        
				if (*text == ch) {
					if (*text)
						text++;

					break;
				}

				if (*text) {
					pattern = retrypat;
					text = ++retrytxt;
					break;
				}

				return FALSE;
		}

		if (pattern == NULL)
			return FALSE;
	}

	return TRUE;
}


/*
 * Initialize the environment to be passed to the executed program.
 * Only a selected subset of the environment variables are allowed.
 * This is necessary so that the path of shared libraries cannot be changed
 * by the user to make otherwise innocent programs do the wrong thing.
 * Warning: homevar and shellvar must have been previously set up.
 */
static void
InitializeEnvironment()
{
	int		fd;
	int		sig;
	int		count;
	char *		name;
	char *		value;
	char *		var;
	char *		cp;
	char *		pathvar;
	static	char *	new_environ[MAXVARIABLES + 10];

	/*
	 * Create the default path from the current path list.
	 * This is done by walking though the path list to get its length,
	 * allocating a string large enough for the path and the PATH name,
	 * and then concatenating the parts into the string.
	 */
	cp = paths;

	while (*cp)
		cp += strlen(cp) + 1;

	pathvar = Allocate(cp - paths + 10);

	strcpy(pathvar, "PATH=");

	for (cp = paths; *cp; cp += strlen(cp) + 1) {
		if (cp != paths)
			strcat(pathvar, ":");

		strcat(pathvar, cp);
	}

	/*
	 * Create the list of new environment variables.
	 * This is made up of a few special values, and a list of values
	 * which are allowed to be copied from the existing environment.
	 * This cannot overflow because of the limit on the size of the
	 * variables list, and the fact that the table has extra room
	 * for the few special entries.
	 */
	count = 3;

	for (name = variables; *name; name += strlen(name) + 1)
	{
		/*
		 * See if the next variable name exists in the environment.
		 * If not, then ignore it.
		 */
		value = getenv(name);

		if (value == NULL)
			continue;

		/*
		 * Allocate an environment variable string.
		 */
		var = AllocateEnvironmentString(name, value);

		/*
		 * If the name is one of the specially handled variable names,
		 * then just remember the new value of the variable,
		 * otherwise add the variable to the new environment.
		 */
		if (strcmp(name, "PATH") == 0)
			pathvar = var;
		else if (strcmp(name, "HOME") == 0)
			homevar = var;
		else if (strcmp(name, "SHELL") == 0)
			shellvar = var;
		else
			new_environ[count++] = var;
	}

	/*
	 * Set the final values of the special variables at the front
	 * of the environment table.
	 */
	new_environ[0] = pathvar;
	new_environ[1] = homevar;
	new_environ[2] = shellvar;

	/*
	 * Terminate the list of environment variables.
	 */
	new_environ[count] = NULL;

	/*
	 * Set the environment variable head to our new table.
	 */
	environ = new_environ;

	/*
	 * Close all of the file descriptors that we can
	 * that aren't the standard ones.
	 */
	for (fd = 3; fd < MAXFD; fd++)
		(void) close(fd);

	/*
	 * Reset all of the signal handlers to the defaults.
	 */
	for (sig = 1; sig < 32; sig++)
		(void) signal(sig, SIG_DFL);
}


/*
 * Make sure that the standard file descriptors are opened.
 * If not, then stop right now without doing anything.
 * This check is required so that we can't be tricked into
 * writing error output into opened privilege files.
 */
static void
VerifyDescriptors()
{
	struct	stat	statbuf;

	if ((fstat(STDIN, &statbuf) < 0) || (fstat(STDOUT, &statbuf) < 0)
		|| (fstat(STDERR, &statbuf) < 0))
	{
		_exit(99);
	}
}


/*
 * Allocate an environment variable having the specified name and value.
 * The returned string is of the form "name=value".
 */
static char *
AllocateEnvironmentString(name, value)
	char *	name;
	char *	value;
{
	char *	str;

	/*
	 * Allocate a string large enough for the name, the value,
	 * the separating equals sign, and the terminating NULL.
	 */
	str = Allocate(strlen(name) + strlen(value) + 2);

	strcpy(str, name);
	strcat(str, "=");
	strcat(str, value);

	return str;
}


/*
 * Allocate a new NULL-terminated string having the specified value.
 */
static char *
AllocateString(str)
	char *	str;
{
	char *	newstr;
	int	len;

	len = strlen(str) + 1;

	newstr = Allocate(len);

	memcpy(newstr, str, len);

	return newstr;
}


/*
 * Allocate memory and exit if it cannot be gotten.
 */
static char *
Allocate(size)
	int	size;
{
	char *	cp;

	cp = malloc(size);

	if (cp == NULL) {
		fprintf(stderr, "Cannot allocate %d bytes of memory\n", size);

		exit(1);
	}

	return cp;
}


/* END CODE */
