/*
 * Copyright 1989 - 1994, John F. Haugh II
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John F. Haugh, II
 *      and other contributors.
 * 4. Neither the name of John F. Haugh, II nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JOHN HAUGH AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JOHN HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <utmp.h>

#include <stdio.h>
#include <grp.h>

#include "config.h"
#include "defines.h"
#include "pwd.h"

#ifndef	NGROUPS
#ifdef	NGROUPS_MAX
#define	NGROUPS	NGROUPS_MAX
#endif
#endif

#ifdef	USE_SYSLOG
#include <syslog.h>

#ifndef	LOG_WARN
#define	LOG_WARN	LOG_WARNING
#endif
#endif

#ifndef	lint
static char rcsid[] = "$Id: setup.c,v 1.3 1995/12/17 03:50:36 marekm Exp $";
#endif

#ifndef	SU
extern	struct	utmp	utent;
#endif	/* !SU */

long	strtol ();
#ifdef	HAVE_ULIMIT
long	ulimit ();
#endif

void	addenv ();
extern	char	*getdef_str();
extern	int	getdef_bool();
extern	int	getdef_num();

#ifndef SU
/*
 * is_my_tty -- determine if "tty" is the same as TTY stdin is using
 */

static int
is_my_tty (tty)
char	*tty;
{
	struct	stat	by_name, by_fd;

	if (stat (tty, &by_name) == -1)
		return 0;

	fstat (0, &by_fd);

	if (by_name.st_rdev != by_fd.st_rdev)
		return 0;
	else
		return 1;
}
#endif

/*
 * setup - initialize login environment
 *
 *	setup() performs the following steps -
 *
 *	set the login tty to be owned by the new user ID with TTYPERM modes
 *	change to the user's home directory
 *	set the process nice, ulimit, and umask from the password file entry
 *	set the group ID to the value from the password file entry
 *	set the supplementary group IDs
 *	set the user ID to the value from the password file entry
 *	set the HOME, SHELL, MAIL, PATH, and LOGNAME or USER environmental
 *	variables.
 */

void	setup (info)
struct	passwd	*info;
{
	extern	int	errno;
	char	buf[BUFSIZ];
#ifndef	SU
	char	tty[sizeof utent.ut_line + 8];
	char	*group;		/* TTY group name or number */
	struct	group	*grent;
#endif
	char	*cp;
	char	*maildir;	/* the directory in which the mailbox resides */
	char	*mailfile;	/* the name of the mailbox */
	int	i;
	long	l;

#ifndef	SU

	/*
	 * Determine the name of the TTY device which the user is logging
	 * into.  This device will (optionally) have the group set to a
	 * pre-defined value.
	 */

	if (utent.ut_line[0] != '/')
		(void) strcat (strcpy (tty, "/dev/"), utent.ut_line);
	else
		(void) strcpy (tty, utent.ut_line);

	/*
	 * See if login.defs has some value configured for the port group
	 * ID.  Otherwise, use the user's primary group ID.
	 */

	if (! (group = getdef_str ("TTYGROUP")))
		i = info->pw_gid;
	else if (group[0] >= '0' && group[0] <= '9')
		i = atoi (group);
	else if ((grent = getgrnam (group)))
		i = grent->gr_gid;
	else
		i = info->pw_gid;

	/*
	 * Change the permissions on the TTY to be owned by the user with
	 * the group as determined above.
	 */

	if (! is_my_tty (tty)) {
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "unable to determine TTY name, got %s\n",
				tty);
		closelog ();
#endif
		exit (1);
	}
	
	if (chown (tty, info->pw_uid, i) ||
			chmod (tty, getdef_num("TTYPERM", 0600))) {
		(void) sprintf (buf, "Unable to change tty %s", tty);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "unable to change tty `%s' for user `%s'\n",
			tty, info->pw_name);
		closelog ();
#endif
		perror (buf);
		exit (errno);
	}
#endif

	/*
	 * Change the current working directory to be the home directory
	 * of the user.  It is a fatal error for this process to be unable
	 * to change to that directory.  There is no "default" home
	 * directory.
	 */
	/*
	 * FIXME - we do it as root, this may fail on NFS-mounted home
	 * directories.  --marekm
	 */
	if (chdir (info->pw_dir) == -1) {
		(void) sprintf (buf, "Unable to cd to \"%s\"", info->pw_dir);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "unable to cd to `%s' for user `%s'\n",
			info->pw_dir, info->pw_name);
		closelog ();
#endif
		perror (buf);
		exit (errno);
	}

	/*
	 * See if the GECOS field contains values for NICE, UMASK or ULIMIT.
	 * If this feature is enabled in /etc/login.defs, we make those
	 * values the defaults for this login session.
	 */

	if ( getdef_bool("QUOTAS_ENAB") ) {
		for (cp = info->pw_gecos ; cp != NULL ; cp = strchr (cp, ',')) {
			if (*cp == ',')
				cp++;

			if (strncmp (cp, "pri=", 4) == 0) {
				i = atoi (cp + 4);
				if (i >= -20 && i <= 20)
					(void) nice (i);

				continue;
			}
#if defined(HAVE_ULIMIT) || defined(HAVE_RLIMIT)
			if (strncmp (cp, "ulimit=", 7) == 0) {
				l = strtol (cp + 7, (char **) 0, 10);
#ifndef HAVE_ULIMIT
				{
					struct rlimit rlimit_fsize;

					bzero(&rlimit_fsize, sizeof rlimit_fsize);
					rlimit_fsize.rlim_cur =
					rlimit_fsize.rlim_max = l * 512;
				}
#else
				(void) ulimit (2, l);
#endif
				continue;
			}
#endif
			if (strncmp (cp, "umask=", 6) == 0) {
				i = strtol (cp + 6, (char **) 0, 8) & 0777;
				(void) umask (i);

				continue;
			}
		}
	}

	/*
	 * Set the real group ID to the primary group ID in the password
	 * file.
	 */

	if (setgid (info->pw_gid) == -1) {
		puts ("Bad group id");
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "bad group ID `%d' for user `%s'\n",
			info->pw_gid, info->pw_name);
		closelog ();
#endif
		exit (errno);
	}
#if NGROUPS > 1

	/*
	 * For systems which support multiple concurrent groups, go get
	 * the group set from the /etc/group file.
	 */

	if (initgroups (info->pw_name, info->pw_gid) == -1) {
		puts ("initgroups failure");
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "initgroups failed for user `%s'\n",
			info->pw_name);
		closelog ();
#endif
		exit (errno);
	}
#endif /* NGROUPS > 1 */

	/*
	 * Set the real UID to the UID value in the password file.
	 */

#ifndef	BSD
	if (setuid (info->pw_uid))
#else
	if (setreuid (info->pw_uid, info->pw_uid))
#endif
	{
		puts ("Bad user id");
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "bad user ID `%d' for user `%s'\n",
			info->pw_uid, info->pw_name);
		closelog ();
#endif
		exit (errno);
	}

	/*
	 * Create the HOME environmental variable and export it.
	 */

	(void) strcat (strcpy (buf, "HOME="), info->pw_dir);
	addenv (buf);

	/*
	 * Create the SHELL environmental variable and export it.
	 */

	if (info->pw_shell == (char *) 0 || ! *info->pw_shell)
		info->pw_shell = "/bin/sh";

	(void) strcat (strcpy (buf, "SHELL="), info->pw_shell);
	addenv (buf);

	/*
	 * Create the PATH environmental variable and export it.
	 */

	cp = getdef_str( info->pw_uid == 0 ? "ENV_SUPATH" : "ENV_PATH" );
	addenv( cp != NULL ? cp : "PATH=/bin:/usr/bin" );

	/*
	 * Export the user name.  For BSD derived systems, it's "USER", for
	 * all others it's "LOGNAME".  We set both of them.
	 */

	(void) strcat (strcpy (buf, "USER="), info->pw_name);
	addenv (buf);

	(void) strcat (strcpy (buf, "LOGNAME="), info->pw_name);
	addenv (buf);

	/*
	 * Create the MAIL environmental variable and export it.  login.defs
	 * knows the prefix.
	 */

	if ( (cp=getdef_str("MAIL_DIR")) != NULL ) {
		maildir = cp;
		mailfile = info->pw_name;
	} else if ( (cp=getdef_str("MAIL_FILE")) != NULL) {
		maildir = info->pw_dir;
		mailfile = cp;
	} else {
		maildir = MAIL_SPOOL_DIR;
		mailfile = info->pw_name;
	}
		
	(void) strcat (strcat (strcat (strcpy (buf,
		"MAIL="), maildir), "/"), mailfile);
	addenv (buf);
}
