/*
 * Copyright 1989 - 1994, John F. Haugh II
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John F. Haugh, II
 *      and other contributors.
 * 4. Neither the name of John F. Haugh, II nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JOHN HAUGH AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JOHN HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include "pwd.h"
#include <time.h>

#include "config.h"
#include "defines.h"
#include "lastlog.h"

#ifndef	lint
static char rcsid[] = "$Id: lastlog.c,v 1.2 1995/12/17 05:04:49 marekm Exp $";
#endif

FILE	*lastlogfile;	/* lastlog file stream */
off_t	user;		/* one single user, specified on command line */
int	days;		/* number of days to consider for print command */
time_t	seconds;	/* that number of days in seconds */

int	uflg;		/* set if user is a valid user id */
int	tflg;		/* print is restricted to most recent days */
struct	lastlog	lastlog; /* scratch structure to play with ... */
struct	stat	statbuf; /* fstat buffer for file size */

int freadcode;

extern	int	optind;
extern	char	*optarg;
extern	char	*asctime ();
extern	struct	passwd	*getpwuid ();
extern	struct	passwd	*getpwnam ();
extern	struct	passwd	*getpwent ();
extern	struct	tm	*localtime ();
extern	void	print ();
extern	void	print_one (struct passwd *pwent);

#define	DAY	(24L*3600L)
#define	NOW	(time ((time_t *) 0))

int
main (argc, argv)
int	argc;
char	**argv;
{
	int	c;
	struct	passwd	*pwent;

	if ((lastlogfile = fopen (LASTLOG_FILE,"r")) == (FILE *) 0) {
		perror (LASTLOG_FILE);
		exit (1);
	}
	while ((c = getopt (argc, argv, "u:t:")) != EOF) {
		switch (c) {
			case 'u':
				pwent = getpwnam (optarg);
				if (! pwent) {
					fprintf (stderr, "Unknown User: %s\n", optarg);
					exit (1);
				}
				uflg++;
				user = pwent->pw_uid;
				break;
			case 't':
				days = atoi (optarg);
				seconds = days * DAY;
				tflg++;
				break;
		}
	}
	print ();
	fclose (lastlogfile);
	exit (0);
	/*NOTREACHED*/
}

void
print ()
{
	off_t	offset;
	struct	passwd	*pwent = NULL;

	if (uflg) {
		offset = user * sizeof lastlog;
		fstat (fileno (lastlogfile), &statbuf);
		if (offset >= statbuf.st_size)
			return;

		fseek (lastlogfile, (off_t) user * sizeof lastlog, 0);
		if (fread ((char *) &lastlog, sizeof lastlog, 1, lastlogfile) == 1)
			print_one (pwent);
		else
			perror (LASTLOG_FILE);
	} else {
		setpwent ();
		while ((pwent = getpwent ())) {
			user = pwent->pw_uid;
			fseek (lastlogfile, (off_t) user * sizeof lastlog, 0);
			if (fread ((char *) &lastlog, sizeof lastlog, 1,
					lastlogfile) != 1)
				continue;

			if (tflg && NOW - lastlog.ll_time > seconds)
				continue;

			print_one (pwent);
		}
	}
}

void
print_one (pwent)
struct	passwd	*pwent;
{
	static	int	once;
	char	*cp;
	struct	tm	*tm;

	if (! pwent)
		return;

	if (! once) {
#if defined SVR4 || defined __linux__
		printf ("Username         Port     From             Latest\n");
#else
		printf ("Username                Port     Latest\n");
#endif
		once++;
	}
	tm = localtime (&lastlog.ll_time);
	cp = asctime (tm);
	cp[24] = '\0';

	if(lastlog.ll_time == (time_t) 0)
		cp = "**Never logged in**\0";

#if defined SVR4 || defined __linux__
	printf ("%-16s %-8.8s %-16.16s %s\n", pwent->pw_name,
		lastlog.ll_line, lastlog.ll_host, cp);
#else
	printf ("%-16s\t%-8.8s %s\n", pwent->pw_name,
		lastlog.ll_line, cp);
#endif
	return;
}
