/*
 * Copyright 1991 - 1994, John F. Haugh II
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John F. Haugh, II
 *      and other contributors.
 * 4. Neither the name of John F. Haugh, II nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JOHN HAUGH AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JOHN HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char rcsid[] = "$Id: groupdel.c,v 1.2 1995/12/16 21:51:39 marekm Exp $";
#endif

#include <sys/types.h>
#include <stdio.h>
#include <grp.h>
#include <ctype.h>
#include <fcntl.h>
#include "pwd.h"

#include "config.h"
#include "prototypes.h"
#include "defines.h"

#ifdef	SHADOWGRP
#include "gshadow.h"
#endif

#ifdef	USE_SYSLOG
#include <syslog.h>
#endif

char	group_name[BUFSIZ];
char	*Prog;
int	errors;

#ifdef	NDBM
extern	int	gr_dbm_mode;
extern	int	sg_dbm_mode;
#endif

extern	struct	group	*getgrnam();
extern	int	gr_lock();
extern	int	gr_unlock();
extern	int	gr_open();
extern	int	gr_remove();

#ifdef	SHADOWGRP
extern	int	sgr_lock();
extern	int	sgr_unlock();
extern	int	sgr_open();
extern	int	sgr_remove();
#endif

/*
 * usage - display usage message and exit
 */

void
usage ()
{
	fprintf (stderr, "usage: groupdel group\n");
	exit (2);
}

/*
 * grp_update - update group file entries
 *
 *	grp_update() writes the new records to the group files.
 */

void
grp_update ()
{
#ifdef	NDBM
	struct	group	*ogrp;
#endif

	if (! gr_remove (group_name)) {
		fprintf (stderr, "%s: error removing group entry\n", Prog);
		errors++;
	}
#ifdef	NDBM

	/*
	 * Update the DBM group file
	 */

	if (access (GROUP_PAG_FILE, 0) == 0) {
		if ((ogrp = getgrnam (group_name)) &&
				! gr_dbm_remove (ogrp)) {
			fprintf (stderr, "%s: error removing group dbm entry\n",
				Prog);
			errors++;
		}
	}
	endgrent ();
#endif	/* NDBM */

#ifdef	SHADOWGRP

	/*
	 * Delete the shadow group entries as well.
	 */

	if (! sgr_remove (group_name)) {
		fprintf (stderr, "%s: error removing shadow group entry\n",
			Prog);
		errors++;
	}
#ifdef	NDBM

	/*
	 * Update the DBM shadow group file
	 */

	if (access (SGROUP_PAG_FILE, 0) == 0) {
		if (! sg_dbm_remove (group_name)) {
			fprintf (stderr,
				"%s: error removing shadow group dbm entry\n",
				Prog);
			errors++;
		}
	}
	endsgent ();
#endif	/* NDBM */
#endif	/* SHADOWGRP */
#ifdef	USE_SYSLOG
	syslog (LOG_INFO, "remove group `%s'\n", group_name);
#endif	/* USE_SYSLOG */
	return;
}

/*
 * close_files - close all of the files that were opened
 *
 *	close_files() closes all of the files that were opened for this
 *	new group.  This causes any modified entries to be written out.
 */

void
close_files ()
{
	if (! gr_close ()) {
		fprintf (stderr, "%s: cannot rewrite group file\n", Prog);
		errors++;
	}
	(void) gr_unlock ();
#ifdef	SHADOWGRP
	if (! sgr_close ()) {
		fprintf (stderr, "%s: cannot rewrite shadow group file\n",
			Prog);
		errors++;
	}
	(void) sgr_unlock ();
#endif	/* SHADOWGRP */
	return;
}

/*
 * open_files - lock and open the group files
 *
 *	open_files() opens the two group files.
 */

void
open_files ()
{
	if (! gr_lock ()) {
		fprintf (stderr, "%s: unable to lock group file\n", Prog);
		exit (1);
	}
	if (! gr_open (O_RDWR)) {
		fprintf (stderr, "%s: unable to open group file\n", Prog);
		exit (1);
	}
#ifdef	SHADOWGRP
	if (! sgr_lock ()) {
		fprintf (stderr, "%s: unable to lock shadow group file\n",
			Prog);
		exit (1);
	}
	if (! sgr_open (O_RDWR)) {
		fprintf (stderr, "%s: unable to open shadow group file\n",
			Prog);
		exit (1);
	}
#endif	/* SHADOWGRP */
	return;
}

/*
 * group_busy - check if this is any user's primary group
 *
 *	group_busy verifies that this group is not the primary group
 *	for any user.  You must remove all users before you remove
 *	the group.
 */

void
group_busy (gid)
GID_T	gid;
{
	struct	passwd	*pwd;

	/*
	 * Nice slow linear search.
	 */

	setpwent ();

	while ((pwd = getpwent ()) && pwd->pw_gid != gid)
		;

	endpwent ();

	/*
	 * If pwd isn't NULL, it stopped becaues the gid's matched.
	 */

	if (pwd == (struct passwd *) 0)
		return;

	/*
	 * Can't remove the group.
	 */

	fprintf (stderr, "%s: cannot remove user's primary group.\n", Prog);
	exit (1);
}

/*
 * main - groupdel command
 *
 *	The syntax of the groupdel command is
 *	
 *	groupdel group
 *
 *	The named group will be deleted.
 */

int
main (argc, argv)
int	argc;
char	**argv;
{
	struct	group	*grp;

	/*
	 * Get my name so that I can use it to report errors.
	 */

	Prog = basename(argv[0]);

	if (argc != 2)
		usage ();

	strncpy (group_name, argv[1], BUFSIZ);

#ifdef	USE_SYSLOG
	openlog (Prog, LOG_PID|LOG_CONS|LOG_NOWAIT, LOG_AUTH);
#endif	/* USE_SYSLOG */

	/*
	 * The open routines for the DBM files don't use read-write
	 * as the mode, so we have to clue them in.
	 */

#ifdef	NDBM
	gr_dbm_mode = O_RDWR;
#ifdef	SHADOWGRP
	sg_dbm_mode = O_RDWR;
#endif	/* SHADOWGRP */
#endif	/* NDBM */

	/*
	 * Start with a quick check to see if the group exists.
	 */

	if (! (grp = getgrnam (group_name))) {
		fprintf (stderr, "%s: group %s does not exist\n",
			Prog, group_name);
		exit (9);
	}
#ifdef	USE_NIS

	/*
	 * Make sure this isn't a NIS group
	 */

	if (__isgrNIS ()) {
		char	*nis_domain;
		char	*nis_master;
		int	nis_masterlen;

		fprintf (stderr, "%s: group %s is a NIS group\n",
			Prog, group_name);

		if (! yp_get_default_domain (&nis_domain) &&
				! yp_master (nis_domain, "group.byname",
				&nis_master)) {
			fprintf (stderr, "%s: %s is the NIS master\n",
				Prog, nis_master);
		}
		exit (6);
	}
#endif

	/*
	 * Now check to insure that this isn't the primary group of
	 * anyone.
	 */

	group_busy (grp->gr_gid);

	/*
	 * Do the hard stuff - open the files, delete the group entries,
	 * then close and update the files.
	 */

	open_files ();

	grp_update ();

	close_files ();
	exit (errors == 0 ? 0:1);
	/*NOTREACHED*/
}
