/*
 * Copyright 1990 - 1994, John F. Haugh II
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John F. Haugh, II
 *      and other contributors.
 * 4. Neither the name of John F. Haugh, II nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JOHN HAUGH AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JOHN HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <stdio.h>
#include "pwd.h"
#include <grp.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#if defined(USG) || defined(SUN4) || defined(__linux__)
#include <termio.h>
#ifdef SYS3
# include <sys/ioctl.h>
#endif	/* SYS3 */
#else /* SUN || BSD */
#include <sgtty.h>
#endif /* !SUN && !BSD */
#include "config.h"
#include "prototypes.h"
#include "defines.h"

#ifdef	SHADOWGRP
#include "gshadow.h"
#endif

#ifdef	USE_SYSLOG
#include <syslog.h>

#ifndef	LOG_WARN
#define	LOG_WARN	LOG_WARNING
#endif
#endif

#ifndef	lint
static char rcsid[] = "$Id: gpmain.c,v 1.3 1995/12/17 05:04:49 marekm Exp $";
#endif

char	name[BUFSIZ];
char	pass[BUFSIZ];
char	pass2[BUFSIZ];

struct	group	grent;

char	*Prog;
char	*user;
char	*group;
char	*admins;
char	*members;

int	aflg;
int	Aflg;
int	dflg;
int	Mflg;
int	rflg;
int	Rflg;

#ifndef	RETRIES
#define	RETRIES	3
#endif

extern	char	*l64a ();
extern	char	*crypt ();
extern	char	*pw_encrypt ();
extern	int	errno;
extern	long	a64l ();
extern	void	entry ();
extern	time_t	time ();
extern	char	*xmalloc ();
extern	char	*getpass ();
#ifdef	NDBM
#ifdef	SHADOWGRP
extern	int	sg_dbm_mode;
#endif
extern	int	gr_dbm_mode;
#endif

/*
 * usage - display usage message
 */

void
usage ()
{
	fprintf (stderr, "usage: %s [ -r|R ] group\n", Prog);
	fprintf (stderr, "       %s [ -a user ] group\n", Prog);
	fprintf (stderr, "       %s [ -d user ] group\n", Prog);
#ifdef	SHADOWGRP
	fprintf (stderr, "       %s [ -A user[,user] ][ -M user[,user] group\n",
		Prog);
#else
	fprintf (stderr, "       %s [ -M user[,user] group\n", Prog);
#endif
	exit (1);
}

/*
 * comma_to_list - convert comma-separated list to (char *) array
 */

char **
comma_to_list (comma)
char	*comma;
{
	char	*members;
	char	**array;
	int	i;
	char	*cp, *cp2;

	/*
	 * Make a copy since we are going to be modifying the list
	 */

	members = xstrdup (comma);

	/*
	 * Count the number of commas in the list
	 */

	for (cp = members, i = 0;;i++)
		if ((cp2 = strchr (cp, ',')))
			cp = cp2 + 1;
		else
			break;

	/*
	 * Add 2 - one for the ending NULL, the other for the last item
	 */

	i += 2;

	/*
	 * Allocate the array we're going to store the pointers into.
	 */

	array = (char **) xmalloc (sizeof (char *) * i);

	/*
	 * Now go walk that list all over again, this time building the
	 * array of pointers.
	 */

	for (cp = members, i = 0;;i++) {
		array[i] = cp;
		if ((cp2 = strchr (cp, ','))) {
			*cp2++ = '\0';
			cp = cp2;
		} else {
			array[i + 1] = (char *) 0;
			break;
		}
	}

	/*
	 * Return the new array of pointers
	 */

	return array;
}

/*
 * check_list - check a comma-separated list of user names for validity
 *
 *	check_list scans a comma-separated list of user names and checks
 *	that each listed name exists.
 */

int
check_list (users)
char	*users;
{
	char	*end;
	char	*start;
	char	user[16];
	int	errors = 0;
	int	len;

	for (start = users;start && *start;start = end) {
		if ((end = strchr (start, ','))) {
			if ((len = end - start) > 15)
				len = 15;

			strncpy (user, start, len);
			user[len] = 0;
			end++;
		} else {
			if ((len = strlen (start)) > 15)
				len = 15;

			strncpy (user, start, len);
			user[len] = 0;
		}

		/*
		 * This user must exist.
		 */

		if (! getpwnam (user)) {
			fprintf (stderr, "%s: unknown user %s\n", Prog, user);
			errors++;
		}
	}
	return errors;
}

/*
 * gpasswd - administer the /etc/group file
 *
 *	-a user		add user to the named group
 *	-d user		remove user from the named group
 *	-r		remove password from the named group
 *	-R		restrict access to the named group
 *	-A user,...	make list of users the administrative users
 *	-M user,...	make list of users the group members
 */

int
main (argc, argv)
int	argc;
char	**argv;
{
	extern	int	optind;
	extern	char	*optarg;
	int	flag;
	int	i;
	void	die ();
	char	*cp;
	char	*getlogin ();
	char	*getpass ();
	int	amroot;
	int	retries;
	int	ruid = getuid();
	struct	group	*gr = 0;
	struct	group	*getgrnam ();
	struct	group	*sgetgrent ();
#ifdef	SHADOWGRP
	struct	sgrp	*sg = 0;
	struct	sgrp	sgent;
	struct	sgrp	*getsgnam ();
#endif
	struct	passwd	*pw = 0;
#ifndef	__STDC__
	struct	passwd	*getpwuid ();
	struct	passwd	*getpwnam ();
#endif

	/*
	 * Make a note of whether or not this command was invoked
	 * by root.  This will be used to bypass certain checks
	 * later on.  Also, set the real user ID to match the
	 * effective user ID.  This will prevent the invoker from
	 * issuing signals which would interfer with this command.
	 */

	amroot = getuid () == 0;
#ifdef	NDBM
#ifdef	SHADOWGRP
	sg_dbm_mode = O_RDWR;
#endif
	gr_dbm_mode = O_RDWR;
#endif
	setuid (geteuid ());

	Prog = basename(argv[0]);

#ifdef	USE_SYSLOG
	openlog (Prog, LOG_PID|LOG_CONS|LOG_NOWAIT, LOG_AUTH);
#endif
	setbuf (stdout, (char *) 0);
	setbuf (stderr, (char *) 0);

	while ((flag = getopt (argc, argv, "a:d:grRA:M:")) != EOF) {
		switch (flag) {
			case 'a':	/* add a user */
				aflg++;
				user = optarg;
				break;
			case 'A':
				Aflg++;
				admins = optarg;
				break;
			case 'd':	/* delete a user */
				dflg++;
				user = optarg;
				break;
			case 'g':	/* no-op from normal password */
				break;
			case 'M':
				Mflg++;
				members = optarg;
				break;
			case 'r':	/* remove group password */
				rflg++;
				break;
			case 'R':	/* restrict group password */
				Rflg++;
				break;
			default:
				usage ();
		}
	}

	/*
	 * Make sure exclusive flags are exclusive
	 */

	if (aflg + dflg + rflg + Rflg + (Aflg || Mflg) > 1)
		usage ();

	/*
	 * If the password is being changed, the input and output must
	 * both be a tty.  The typical keyboard signals are caught
	 * so the termio modes can be restored.
	 */

	if (! aflg && ! dflg && ! rflg && ! Rflg && ! Aflg && ! Mflg) {
		if (! isatty (0) || ! isatty (1))
			exit (1);

		die (0);			/* save tty modes */

		signal (SIGHUP, die);
		signal (SIGINT, die);
		signal (SIGQUIT, die);
		signal (SIGTERM, die);
#ifdef	SIGTSTP
		signal (SIGTSTP, die);
#endif
	}

	/*
	 * Determine the name of the user that invoked this command.
	 * This is really hit or miss because there are so many ways
	 * that command can be executed and so many ways to trip up
	 * the routines that report the user name.
	 */

	if ((cp = getlogin ()) && (pw = getpwnam (cp)) && pw->pw_uid == ruid) {
					/* need user name */
		STRFCPY(name, cp);
	} else if ((pw = getpwuid (ruid))) /* get it from password file */
		STRFCPY(name, pw->pw_name);
	else {				/* can't find user name! */
		fprintf (stderr, "Who are you?\n");
		exit (1);
	}
	if (! (pw = getpwnam (name)))
		goto failure;		/* can't get my name ... */
		
	/*
	 * Get the name of the group that is being affected.  The group
	 * entry will be completely replicated so it may be modified
	 * later on.
	 */

	if (! (group = argv[optind]))
		usage ();

	if (! (gr = getgrnam (group))) {
		fprintf (stderr, "unknown group: %s\n", group);
		exit (1);
	}
	grent = *gr;
	grent.gr_name = xstrdup (gr->gr_name);
	grent.gr_passwd = xstrdup (gr->gr_passwd);

	for (i = 0;gr->gr_mem[i];i++)
		;
	grent.gr_mem = (char **) xmalloc ((i + 1) * sizeof (char *));
	for (i = 0;gr->gr_mem[i];i++)
		grent.gr_mem[i] = xstrdup (gr->gr_mem[i]);
	grent.gr_mem[i] = (char *) 0;
#ifdef	SHADOWGRP
	if ((sg = getsgnam (group))) {
		sgent = *sg;
		sgent.sg_name = xstrdup (sg->sg_name);
		sgent.sg_passwd = xstrdup (sg->sg_passwd);

		for (i = 0;sg->sg_mem[i];i++)
			;
		sgent.sg_mem = (char **) xmalloc (sizeof (char *) * (i + 1));
		for (i = 0;sg->sg_mem[i];i++)
			sgent.sg_mem[i] = xstrdup (sg->sg_mem[i]);
		sgent.sg_mem[i] = 0;

		for (i = 0;sg->sg_adm[i];i++)
			;
		sgent.sg_adm = (char **) xmalloc (sizeof (char *) * (i + 1));
		for (i = 0;sg->sg_adm[i];i++)
			sgent.sg_adm[i] = xstrdup (sg->sg_adm[i]);
		sgent.sg_adm[i] = 0;
	} else {
		sgent.sg_name = xstrdup (group);
		sgent.sg_passwd = grent.gr_passwd;
		grent.gr_passwd = "!";

		for (i = 0;grent.gr_mem[i];i++)
			;
		sgent.sg_mem = (char **) xmalloc (sizeof (char *) * (i + 1));
		for (i = 0;grent.gr_mem[i];i++)
			sgent.sg_mem[i] = xstrdup (grent.gr_mem[i]);
		sgent.sg_mem[i] = 0;

		sgent.sg_adm = (char **) xmalloc (sizeof (char *) * 2);
		if (sgent.sg_mem[0]) {
			sgent.sg_adm[0] = xstrdup (sgent.sg_mem[0]);
			sgent.sg_adm[1] = 0;
		} else
			sgent.sg_adm[0] = 0;

		sg = &sgent;
	}
#endif

#ifdef	SHADOWGRP

	/*
	 * The policy here for changing a group is that 1) you must be
	 * root or 2). you must be listed as an administrative member.
	 * Administrative members can do anything to a group that the
	 * root user can.
	 */

	if (! amroot) {
		for (i = 0;sgent.sg_adm[i];i++)
			if (strcmp (sgent.sg_adm[i], name) == 0)
				break;

		if (sgent.sg_adm[i] == (char *) 0)
			goto failure;
	}
#else

	/*
	 * The policy here for changing a group is that 1) you must bes
	 * root or 2) you must be the first listed member of the group.
	 * The first listed member of a group can do anything to that
	 * group that the root user can.  The rationale for this hack is
	 * that the FIRST user is probably the most important user in
	 * this entire group.
	 */

	if (! amroot) {
		if (grent.gr_mem[0] == (char *) 0)
			goto failure;

		if (strcmp (grent.gr_mem[0], name) != 0)
			goto failure;
	}

#endif	/* SHADOWGRP */

	/*
	 * Removing a password is straight forward.  Just set the
	 * password field to a "".
	 */

	if (rflg) {
		grent.gr_passwd = "";
#ifdef	SHADOWGRP
		sgent.sg_passwd = "";
#endif
#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "remove password from group %s\n", group);
#endif
		goto output;
	} else if (Rflg) {

	/*
	 * Same thing for restricting the group.  Set the password
	 * field to "!".
	 */

		grent.gr_passwd = "!";
#ifdef	SHADOWGRP
		sgent.sg_passwd = "!";
#endif
#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "restrict access to group %s\n", group);
#endif
		goto output;
	}

	/*
	 * Adding a member to a member list is pretty straightforward
	 * as well.  Call the appropriate routine and split.
	 */

	if (aflg) {
		if (getpwnam (user) == (struct passwd *) 0) {
			fprintf (stderr, "%s: unknown user %s\n", Prog, user);
			exit (1);
		}
		printf ("Adding user %s to group %s\n", user, group);
		grent.gr_mem = add_list (grent.gr_mem, user);
#ifdef	SHADOWGRP
		sgent.sg_mem = add_list (sgent.sg_mem, user);
#endif
#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "add member %s to group %s\n", user, group);
#endif
		goto output;
	}

	/*
	 * Removing a member from the member list is the same deal
	 * as adding one, except the routine is different.
	 */

	if (dflg) {
		int	removed = 0;

		for (i = 0;grent.gr_mem[i];i++)
			if (strcmp (user, grent.gr_mem[i]) == 0)
				break;

		printf ("Removing user %s from group %s\n", user, group);

		if (grent.gr_mem[i] != (char *) 0) {
			removed = 1;
			grent.gr_mem = del_list (grent.gr_mem, user);
		}
#ifdef	SHADOWGRP
		for (i = 0;sgent.sg_mem[i];i++)
			if (strcmp (user, sgent.sg_mem[i]) == 0)
				break;

		if (sgent.sg_mem[i] != (char *) 0) {
			removed = 1;
			sgent.sg_mem = del_list (sgent.sg_mem, user);
		}
#endif
		if (! removed) {
			fprintf (stderr, "%s: unknown member %s\n", Prog, user);
			exit (1);
		}
#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "remove member %s from group %s\n",
				user, group);
#endif
		goto output;
	}

	/*
	 * Replacing the entire list of members is simple.  Check the list
	 * to make sure everyone is a real user.  Then slap the new list
	 * in place.
	 */

	if (Mflg) {

		/*
		 * Only root can replace the entire list.
		 */

		if (! amroot)
			goto failure;

		/*
		 * Check the list for validity, then put it in.
		 */

		if (check_list (members))
			exit (1);

#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "set members of %s to %s\n", group, members);
#endif
#ifdef	SHADOWGRP
		sgent.sg_mem = comma_to_list (members);
		grent.gr_mem = comma_to_list (members);

		if (! Aflg)
			goto output;
#else
		grent.gr_mem = comma_to_list (members);

		goto output;
#endif
	}

#ifdef	SHADOWGRP

	/*
	 * Replacing the entire list of administators is simple.  Check the
	 * list to make sure everyone is a real user.  Then slap the new
	 * list in place.
	 */

	if (Aflg) {

		/*
		 * Only root can replace the entire list.
		 */

		if (! amroot)
			goto failure;

		/*
		 * Check the list for validity, then put it in.
		 */

		if (check_list (admins))
			exit (1);

#ifdef	USE_SYSLOG
		syslog (LOG_INFO, "set administrators of %s to %s\n",
				group, members);
#endif
		sgent.sg_adm = comma_to_list (admins);

		goto output;
	}
#endif

	/*
	 * A new password is to be entered and it must be encrypted,
	 * etc.  The password will be prompted for twice, and both
	 * entries must be identical.  There is no need to validate
	 * the old password since the invoker is either the group
	 * owner, or root.
	 */

	printf ("Changing the password for group %s\n", group);

	for (retries = 0;retries < RETRIES;retries++) {
		if (! (cp = getpass ("New Password:")))
			exit (1);
		else {
			STRFCPY(pass, cp);
			bzero (cp, strlen (cp));
		}
		if (! (cp = getpass ("Re-enter new password:")))
			exit (1);
		else {
			STRFCPY(pass2, cp);
			bzero (cp, strlen (cp));
		}
		if (strcmp (pass, pass2) == 0)
			break;

		bzero (pass, sizeof pass);
		bzero (pass2, sizeof pass2);

		if (retries + 1 < RETRIES)
			puts ("They don't match; try again");
	}
	bzero (pass2, sizeof pass2);

	if (retries == RETRIES) {
		fprintf (stderr, "%s: Try again later\n", Prog);
		exit (1);
	}
#ifdef	SHADOWGRP
	sgent.sg_passwd = pw_encrypt (pass, (char *) 0);
#else
	grent.gr_passwd = pw_encrypt (pass, (char *) 0);
#endif
	bzero (pass, sizeof pass);
#ifdef	USE_SYSLOG
	syslog (LOG_INFO, "change the password for group %s\n", group);
#endif

	/*
	 * This is the common arrival point to output the new group
	 * file.  The freshly crafted entry is in allocated space.
	 * The group file will be locked and opened for writing.  The
	 * new entry will be output, etc.
	 */

output:
	signal (SIGHUP, SIG_IGN);
	signal (SIGINT, SIG_IGN);
	signal (SIGQUIT, SIG_IGN);
#ifdef	SIGTSTP
	signal (SIGTSTP, SIG_IGN);
#endif

	if (! gr_lock ()) {
		fprintf (stderr, "%s: can't get lock\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "failed to get lock for /etc/group\n");
#endif
		exit (1);
	}
#ifdef	SHADOWGRP
	if (! sgr_lock ()) {
		fprintf (stderr, "%s: can't get shadow lock\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "failed to get lock for /etc/gshadow\n");
#endif
		exit (1);
	}
#endif
	if (! gr_open (O_RDWR)) {
		fprintf (stderr, "%s: can't open file\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot open /etc/group\n");
#endif
		exit (1);
	}
#ifdef	SHADOWGRP
	if (! sgr_open (O_RDWR)) {
		fprintf (stderr, "%s: can't open shadow file\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot open /etc/gshadow\n");
#endif
		exit (1);
	}
#endif
	if (! gr_update (&grent)) {
		fprintf (stderr, "%s: can't update entry\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot update /etc/group\n");
#endif
		exit (1);
	}
#ifdef	SHADOWGRP
	if (! sgr_update (&sgent)) {
		fprintf (stderr, "%s: can't update shadow entry\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot update /etc/gshadow\n");
#endif
		exit (1);
	}
#endif
	if (! gr_close ()) {
		fprintf (stderr, "%s: can't re-write file\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot re-write /etc/group\n");
#endif
		exit (1);
	}
#ifdef	SHADOWGRP
	if (! sgr_close ()) {
		fprintf (stderr, "%s: can't re-write shadow file\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot re-write /etc/gshadow\n");
#endif
		exit (1);
	}
	(void) sgr_unlock ();
#endif
	if (! gr_unlock ()) {
		fprintf (stderr, "%s: can't unlock file\n", Prog);
		exit (1);
	}
#ifdef	NDBM
	if (access (GROUP_PAG_FILE, 0) == 0 && ! gr_dbm_update (&grent)) {
		fprintf (stderr, "%s: can't update DBM files\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot update /etc/group DBM files\n");
#endif
		exit (1);
	}
	endgrent ();
#ifdef	SHADOWGRP
	if (access (SGROUP_PAG_FILE, 0) == 0 && ! sg_dbm_update (&sgent)) {
		fprintf (stderr, "%s: can't update DBM shadow files\n", Prog);
#ifdef	USE_SYSLOG
		syslog (LOG_WARN, "cannot update /etc/gshadow DBM files\n");
#endif
		exit (1);
	}
	endsgent ();
#endif
#endif
	exit (0);
	/*NOTREACHED*/

failure:
	fprintf (stderr, "Permission denied.\n");
	exit (1);
	/*NOTREACHED*/
}

/*
 * die - set or reset termio modes.
 *
 *	die() is called before processing begins.  signal() is then
 *	called with die() as the signal handler.  If signal later
 *	calls die() with a signal number, the terminal modes are
 *	then reset.
 */

void
die (killed)
int	killed;
{
#if defined(BSD) || defined(SUN)
	static	struct	sgtty	sgtty;

	if (killed)
		stty (0, &sgtty);
	else
		gtty (0, &sgtty);
#else
#if defined(SVR4) || defined (SUN4) || defined(__linux__)
	static	struct	termios	sgtty;

	if (killed)
		tcsetattr (0, TCSANOW, &sgtty);
	else
		tcgetattr (0, &sgtty);
#else	/* !SVR4 */
	static	struct	termio	sgtty;

	if (killed)
		ioctl (0, TCSETA, &sgtty);
	else
		ioctl (0, TCGETA, &sgtty);
#endif	/* SVR4 */
#endif	/* BSD || SUN */
	if (killed) {
		putchar ('\n');
		fflush (stdout);
		exit (killed);
	}
}
