/*
 * Copyright 1989 - 1994, John F. Haugh II
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by John F. Haugh, II
 *      and other contributors.
 * 4. Neither the name of John F. Haugh, II nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JOHN HAUGH AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JOHN HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <stdio.h>
#include <errno.h>
#ifndef BSD
#include <sys/wait.h>
#endif
#include "config.h"
#include "defines.h"
#include "pwd.h"
#include "grp.h"
#ifdef	SHADOWPWD
#include "shadow.h"
#endif
#ifdef	SHADOWGRP
#include "gshadow.h"
#endif
#ifdef  AIX
#include <userpw.h>
#include <usersec.h>
#include <userconf.h>
#endif

#ifndef	AGING
#if defined(SHADOWPWD) || defined(AIX)
#define AGING	0
#endif
#else
#if !defined(SHADOWPWD) && !defined(AIX) && !defined(ATT_AGE)
#undef AGING
#endif
#endif

#if defined(SHADOWPWD) || defined(AGING) /*{*/

#ifndef	lint
static char rcsid[] = "$Id: age.c,v 1.2 1995/12/17 03:50:36 marekm Exp $";
#endif


static	char	*EXPIRE_DAY = "Your password will expire in 1 day.\n";
static	char	*EXPIRE_DAYS = "Your password will expire in %d days.\n";
static	char	*PASSWORD_EXPIRED = "Your password has expired.";
static	char	*PASSWORD_INACTIVE = "Your password is inactive.";
static	char	*LOGIN_EXPIRED = "Your login has expired.";
static	char	*CONTACT_SYSADM = "  Contact the system administrator.\n";
static	char	*NEW_PASSWORD = "  Choose a new password.\n";

#define	DAY	(24L*3600L)
#ifdef	ITI_AGING
#define	SCALE	(DAY)
#else
#define	SCALE	(1)
#endif

extern	time_t	time ();
extern	char	*xstrdup();

#ifdef	SHADOWPWD	/*{*/

/*
 * pwd_to_spwd - create entries for new spwd structure
 *
 *	pwd_to_spwd() creates a new (struct spwd) containing the
 *	information in the pointed-to (struct passwd).
 */

static struct spwd *
pwd_to_spwd (pw)
struct	passwd	*pw;
{
	static	struct	spwd	tspwd;
	struct	spwd	*sp = &tspwd;

	/*
	 * Nice, easy parts first.  The name and passwd map directly
	 * from the old password structure to the new one.
	 */

	sp->sp_namp = xstrdup (pw->pw_name);
	sp->sp_pwdp = xstrdup (pw->pw_passwd);
#ifdef	ATT_AGE

	/*
	 * AT&T-style password aging maps the sp_min, sp_max, and
	 * sp_lstchg information from the pw_age field, which appears
	 * after the encrypted password.
	 */

	if (pw->pw_age[0]) {
		sp->sp_max = (c64i (pw->pw_age[0]) * 7) * SCALE;

		if (pw->pw_age[1])
			sp->sp_min = (c64i (pw->pw_age[1]) * 7) * SCALE;
		else
			sp->sp_min = (10000L) * SCALE;

		if (pw->pw_age[1] && pw->pw_age[2])
			sp->sp_lstchg = (a64l (pw->pw_age + 2) * 7) * SCALE;
		else
			sp->sp_lstchg = time ((time_t *) 0) / (DAY/SCALE);
	} else {
		sp->sp_min = 0;
		sp->sp_max = (10000L * SCALE);
		sp->sp_lstchg = time ((time_t *) 0) / (DAY/SCALE);
	}
#else	/* !ATT_AGE */
	/*
	 * BSD does not use the pw_age field and has no aging information
	 * anywheres.  The default values are used to initialize the
	 * fields which are in the missing pw_age field;
	 */

	sp->sp_min = 0;
	sp->sp_max = (10000L * SCALE);
	sp->sp_lstchg = time ((time_t *) 0) / (DAY/SCALE);
#endif	/* ATT_AGE */

	/*
	 * These fields have no corresponding information in the password
	 * file.  They are set to uninitialized values.
	 */

	sp->sp_warn = -1;
	sp->sp_inact = -1;
	sp->sp_expire = -1;
	sp->sp_flag = -1;

	return sp;
}
#endif /*}*/

/*
 * isexpired - determine if account is expired yet
 *
 *	isexpired calculates the expiration date based on the
 *	password expiration criteria.
 */

/*ARGSUSED*/

#ifdef	SHADOWPWD
int
isexpired (pw, sp)
struct	passwd	*pw;
struct	spwd	*sp;
#else
int
isexpired (pw)
struct	passwd	*pw;
#endif
{
	long	clock;
#ifdef	AIX
	int	minage = 0;
	int	maxage = 10000;
	int	curage = 0;
	struct	userpw	*pu;
#endif

	clock = time ((time_t *) 0) / (DAY/SCALE);

#ifdef	SHADOWPWD
	/*
	 * Quick and easy - there is an expired account field
	 * along with an inactive account field.  Do the expired
	 * one first since it is worse.
	 */

	if (sp->sp_expire > 0 && sp->sp_expire < clock)
		return 3;

	if (sp->sp_inact > 0 && sp->sp_lstchg > 0 && sp->sp_max > 0 &&
			sp->sp_inact + sp->sp_lstchg + sp->sp_max < clock)
		return 2;
#endif
#ifdef	AIX	/*{*/
        /*
         * The aging information lives someplace else.  Get it from the
         * login.cfg file
         */

        if (getconfattr (SC_SYS_PASSWD, SC_MINAGE, &minage, SEC_INT))
                minage = -1;

        if (getconfattr (SC_SYS_PASSWD, SC_MAXAGE, &maxage, SEC_INT))
                maxage = -1;

        pu = getuserpw (pw->pw_name);
        curage = (time (0) - pu->upw_lastupdate) / (7*86400L);

	if (maxage != -1 && curage > maxage)
		return 1;
#else	/*} !AIX */

	/*
	 * The last and max fields must be present for an account
	 * to have an expired password.  A maximum of >10000 days
	 * is considered to be infinite.
	 */

#ifdef	SHADOWPWD
	if (sp->sp_lstchg == -1 ||
			sp->sp_max == -1 || sp->sp_max >= (10000L*SCALE))
		return 0;
#endif
#ifdef	ATT_AGE
	if (pw->pw_age[0] == '\0' || pw->pw_age[0] == '/')
		return 0;
#endif

	/*
	 * Calculate today's day and the day on which the password
	 * is going to expire.  If that date has already passed,
	 * the password has expired.
	 */

#ifdef	SHADOWPWD
	if (sp->sp_lstchg + sp->sp_max < clock)
		return 1;
#endif
#ifdef	ATT_AGE
	if (a64l (pw->pw_age + 2) + c64i (pw->pw_age[1]) < clock / 7)
		return 1;
#endif
#endif	/*} AIX */
	return 0;
}

/*
 * expire - force password change if password expired
 *
 *	expire() calls /bin/passwd to change the user's password
 *	if it has expired.
 */

#ifdef	SHADOWPWD
int
expire (pw, sp)
struct	passwd	*pw;
struct	spwd	*sp;
#else
int
expire (pw)
struct	passwd	*pw;
#endif
{
	int	status;
	int	child;
	int	pid;

#ifdef	SHADOWPWD
	if (! sp)
		sp = pwd_to_spwd (pw);
#endif

	/*
	 * See if the user's password has expired, and if so
	 * force them to change their password.
	 */

#ifdef	SHADOWPWD
	switch (status = isexpired (pw, sp))
#else
	switch (status = isexpired (pw))
#endif
	{
		case 0:
			return 0;
		case 1:
			printf (PASSWORD_EXPIRED);
			break;
		case 2:
			printf (PASSWORD_INACTIVE);
			break;
		case 3:
			printf (LOGIN_EXPIRED);
			break;
	}

	/*
	 * Setting the maximum valid period to less than the minimum
	 * valid period means that the minimum period will never
	 * occur while the password is valid, so the user can never
	 * change that password.
	 */

#ifdef	SHADOWPWD
	if (status > 1 || sp->sp_max < sp->sp_min)
#else
	if (status > 1 || c64i (pw->pw_age[0]) < c64i (pw->pw_age[1]))
#endif
	{
		puts (CONTACT_SYSADM);
		exit (1);
	}
	puts (NEW_PASSWORD);
	fflush (stdout);

	/*
	 * Close all the files so that unauthorized access won't
	 * occur.  This needs to be done anyway because those files
	 * might become stale after "passwd" is executed.
	 */

#ifdef	SHADOWPWD
	endspent ();
#endif
	endpwent ();
#ifdef	SHADOWGRP
	endsgent ();
#endif
	endgrent ();

	/*
	 * Execute the /bin/passwd command.  The exit status will be
	 * examined to see what the result is.  If there are any
	 * errors the routine will exit.  This forces the user to
	 * change their password before being able to use the account.
	 */

	if ((pid = fork ()) == 0) {

		/*
		 * Set the UID to be that of the user.  This causes
		 * passwd to work just like it would had they executed
		 * it from the command line while logged in.
		 */

		if (setuid (pw->pw_uid))
			_exit (errno);

		execl (BIN_DIR "passwd", "passwd", pw->pw_name, (char *) 0);
		puts ("Can't execute " BIN_DIR "passwd");
		fflush (stdout);

		_exit (errno);
	} else if (pid == -1) {
		perror ("passwd");
		exit (errno);
	}
	while ((child = wait (&status)) != pid && child != -1)
		;

	if (child == pid && status == 0)
		return 1;

	exit (1);
	/*NOTREACHED*/
}

/*
 * agecheck - see if warning is needed for password expiration
 *
 *	agecheck sees how many days until the user's password is going
 *	to expire and warns the user of the pending password expiration.
 */

#ifdef	SHADOWPWD
void
agecheck (pw, sp)
struct	passwd	*pw;
struct	spwd	*sp;
#else
void
agecheck (pw)
struct	passwd	*pw;
#endif
{
	long	clock = time ((long *) 0) / (DAY/SCALE);
	long	remain;

#ifdef	SHADOWPWD
	if (! sp)
		sp = pwd_to_spwd (pw);

	/*
	 * The last, max, and warn fields must be supported or the
	 * warning period cannot be calculated.
	 */

	if (sp->sp_lstchg == -1 || sp->sp_max == -1 || sp->sp_warn == -1)
		return;
#else
	if (pw->pw_age[0] == '\0')
		return;
#endif

#ifdef	SHADOWPWD
	if ((remain = (sp->sp_lstchg + sp->sp_max) - clock) <= sp->sp_warn)
#else
	if ((remain = (a64l (pw->pw_age + 2) + c64i (pw->pw_age[0])) * 7
			- clock) <= getdef_num ("PASS_WARN_AGE", 7))
#endif
	{
		remain /= SCALE;
		if (remain >= 0) {
			if (remain == 1)
				printf (EXPIRE_DAY);
			else
				printf (EXPIRE_DAYS, remain);
		}
	}
}
#endif /*}*/
