//
// Copyright (C) 1995  Lars Berntzon
//
#include <sadblib.hh>
#include <sadist.hh>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <sys/wait.h>


//////////////////////////////////////////////////////////////////
//		S A D I S T : : S A D I S T
//		---------------------------
// Description:
//	Constructor for SADist
//
// Arguments:
//	options		- Command line options
//	
//////////////////////////////////////////////////////////////////
SADist::SADist(const Options &options) :
    skip(0),
    directory(0),
    distribution(0),
    dataBase(0),
    host(0),
    pStatus(0),
    pOptions(options)
{
    outputFileName[0] = 0;

    //
    // Registrate tcl interpreter.
    //
    interp = Tcl_CreateInterp();

    //
    // Registrate database tcl commands.
    //
    SADB::Tcl_AppInit(interp);

    //
    // Registrate extended TCL commands.
    //
#ifdef USE_TCLX
    TclXCmd_Init(interp);
#endif

    //
    // Registrate sadist commands.
    //
    Tcl_CreateCommand(interp, "error", priv_proc_error, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "sarand", priv_proc_sarand, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "include", priv_proc_include, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "include-cond", priv_proc_include, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "include-bin", priv_proc_include, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "include-bin-cond", priv_proc_include, (ClientData) this, NULL);
    Tcl_CreateCommand(interp, "text", priv_proc_text, (ClientData) this, NULL);

    //
    // Connect TCL and C++ variables.
    //
    Tcl_LinkVar(interp, "SAskip", (char *)&skip, TCL_LINK_BOOLEAN);
    Tcl_LinkVar(interp, "SAmax_processes", (char *)&pOptions.max_processes, TCL_LINK_INT);
    Tcl_LinkVar(interp, "SAdirectory", (char *)&directory, TCL_LINK_STRING);
    Tcl_LinkVar(interp, "SAdistribution", (char *)&distribution, TCL_LINK_STRING);
    Tcl_LinkVar(interp, "SAdatabase", (char *)&dataBase, TCL_LINK_STRING);
    Tcl_LinkVar(interp, "host", (char *)&host, TCL_LINK_STRING);

    //
    // Source the sadist.tcl file, this is so that list_distributions
    // shall work before actually distributed anything.
    //
    pStatus = source_setupfiles();

    return;
}


//////////////////////////////////////////////////////////////////
//		S A D I S T : : D I S T R I B U T E
//		-----------------------------------
// Description:
//	Distribute all selected distributions to hosts.
//
// Arguments:
//	distributions	- List of distribution type.
//	hosts		- List of hosts to distribute to.
//			  If this list is empty, use default set of
//			  hosts for this kind of distribution.
//
//////////////////////////////////////////////////////////////////
int
SADist::distribute(TextList &distributions, TextList &hosts)
{
    char fileName[PATH_MAX];	// Full path for a file.
    TextList *useList = 0;	// List of hosts to use.
    TextList defaultList;	// List of machines if none specified.
    TextList subList;		// List of subDistributions.
    char *required_user;	// Required user name if any.
    char *whoami;		// My current user name.
    int rc = 0;			// Return code.
    Pix dist;			// Index over distributions.
    Pix idx;			// Index over machines.
    int useDatabase;		// Flag if database shall be used.
    SADB db;

    //
    // Check database value.
    //
    if (dataBase == 0) {
	Tcl_SetResult(interp, "database: variable is not set in the sadist.tcl file", 0);
	return 1;
    }

    //
    // Loop through all distributions.
    //
    for(dist = distributions.first(); dist != 0; distributions.next(dist))
    {
	skip = 0;
	useDatabase = 1;

	//
	// Set the distribution variable.
	//
	Tcl_SetVar(interp, "SAdistribution", (char *)distributions(dist).name(),
		   TCL_GLOBAL_ONLY);

	//
	// Execute global sadist setup file.
	//
	if ((rc = source_setupfiles())) {
	    return rc;
	}

	//
	// Source the Control file for current distribution.
	//
	sprintf(fileName, "%s/%s/Control", directory, distribution);
	rc = Tcl_EvalFile(interp, fileName);
	if (rc != TCL_OK) {
	    return 1;
	}

	//
	// Check if distribution (true value of the skip variable) is to be
	// skipped as a result that Control called the skip function, if not
	// do the distribution.
	//
	if (skip == 0)
	{
	    if (strcmp(dataBase, "none") == 0) {
		useDatabase = 0;
	    }
	    else {
		db.open(dataBase);
		if (db.errno != 0) {
		    Tcl_SetResult(interp, db.errorMessage, 0);
		    return 1;
		}
	    }

	    //
	    // Get a list of all columns in the machines database.
	    //
	    TextList columns;
	    if (useDatabase) {
		columns = db.list_columns();
	    }
	
	    //
	    // Check that current user is the required user.
	    //
	    required_user = Tcl_GetVar(interp, "SArequired_user", TCL_GLOBAL_ONLY);
	    whoami = Tcl_GetVar(interp, "SAwhoami", TCL_GLOBAL_ONLY);
	    if (required_user && required_user[0] != 0 && whoami &&
	        (strcmp(required_user, whoami) != 0)) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp,
				 distributions(dist).name(),
				 ": you must be running as ",
				 required_user,
				 0);
		return 1;
	    }

	    //
	    // If hosts was specified as arguments use them, otherwise make a
	    // proper list of hosts to use as set by the "hosts" tcl variable.
	    // This variable is either set in sadist.tcl or the Control file.
	    // If no hosts are specifed the defaultList needs to be cleared
	    // first in the case when multiple distributions are specified. 
	    // This can happend either when the user specifies multiple 
	    // distributions or when subdistribution occurs.
	    //
	    if (!hosts.empty()) {
	       useList = &hosts;
	    }
	    else
	    {
	        defaultList.clear();
	        useList = &defaultList;
		char *listString;
		char *p;

		//
		// Run the command which generated the hosts list.
		//
		rc = Tcl_GlobalEval(interp, "make_host_list");
		if (rc != TCL_OK) {
		    return 1;
		}
		p = listString  = strdup(interp->result);

		//
		// Add the elements to the defaultList which is the hosts
		// to really use.
		//
		while((p = strtok(p, " \t\n")) != NULL)
		{
		    defaultList.append(TextEntry(p));
		    p = NULL;
		}
		free(listString);
	    }

	    //
	    // Get all data about machines.
	    //
	    ListList useListFull;
	    if (useDatabase) {
		useListFull = db.extract_columns(columns, *useList);
	    }
	    else {
	        for(idx = useList->first(); idx != 0; useList->next(idx)) {
		    useListFull.append(ListEntry((*useList)(idx).name()));
	        }
	    }

	    //
	    // Continue to update all hosts.
	    //
	    rc = update_hosts(useListFull, db, useDatabase);
	    if (rc != TCL_OK) {
		return rc;
	    }

	}
    
        //
        // Preform sub distributions.
        //
        if (!pOptions.is_local())
	{
	    subList = list_distributions(distributions(dist).name());

	    if (!subList.empty()) {
		rc = distribute(subList, hosts);
		if (rc != 0) {
		    return rc;
		}
	    }
	}
    }

    return 0;
}


//////////////////////////////////////////////////////////////////
//		S A D I S T : : U P D A T E _ H O S T S
//		---------------------------------------
// Description:
//	Distribute one distribution to hosts.
//
// Arguments:
//	hosts		- List of hosts to distribute to.
//			  If this list is empty, use default set of
//			  hosts for this kind of distribution.
//
//	db		- Database to get data from.
//
//	columns		- List of columns in database.
//
//////////////////////////////////////////////////////////////////
int
SADist::update_hosts(ListList &hosts, SADB &db, int useDatabase)
{
    char commandString[MAXNAME];// System command name.
    const char *p = 0;		// Temporary pointer.
    int rc = 0;			// Return code.
    Pix m;			// Index over hosts.
    Pix c;			// Index over columns.
    static int proc_count;	// Counter of number of parallel processes.
    const char *hostName, *columnName;	// Name of host and column.

    //
    // Loop through hosts, expand the hosts input file
    // and send the file to the host.
    //
    for (m = hosts.first(); m != 0; hosts.next(m))
    {
	hostName = hosts(m).name();
	//
	// Reset the flag for if a host should be skipped.
	//
	skip = 0;

	//
	// Set various variables for this particular machine.
	//
	if (useDatabase) {
	    for(c = hosts(m).list().first(); c != 0;  hosts(m).list().next(c))
	    {
		columnName = hosts(m).list()(c).name();
		p = db.fetch(hostName, columnName);
		if (db.errno) {
		    Tcl_SetResult(interp, db.errorMessage, 0);
		    return 1;
		}
		if (p == 0) {
		    p = "";
		}
		Tcl_SetVar(interp, (char *)columnName, (char *)p, TCL_GLOBAL_ONLY);
	    }
	}

	//
	// Set the host variable.
	//
	Tcl_SetVar(interp, "host", (char *)hosts(m).name(), TCL_GLOBAL_ONLY);

	//
	// Evaluate the preprocess_host command.
	//
	sprintf(commandString, "preprocess_host %s", hostName);
	rc = Tcl_GlobalEval(interp, commandString);
	if (rc != TCL_OK) {
	    return 1;
	}

	//
	// Check if host is to be skipped as a result that Control
	// or preprocess_host called the skip function.
	if (skip) {
	    continue;
	}

	//
	// Open temporary output file and expand either unique
	// name or the default file into the temporary output file.
	//
	(void)sprintf(outputFileName, "/tmp/sadist%d%d", getpid(), proc_count);
	output.open(outputFileName);
	if (!output) {
	    Tcl_AppendResult(interp, outputFileName, ": can not open file", 0);
	    return 1;
	}

	sprintf(commandString, "include %s default", hostName);
	rc = Tcl_GlobalEval(interp, commandString);
	output.close();

	if (rc != TCL_OK) {
	    unlink(outputFileName);
	    Tcl_AppendResult(interp, ": for host ", hostName, 0);
	    return rc;
	}

	//
	// Check if host is to be skipped, as a result from include got
	// a %error line some where.
	//
	if (skip) {
	    cout << hostName << ": skipped, reason: " << interp->result << endl;
	    unlink(outputFileName);
	    continue;
	}

	//
	// Run several parallell processes.
	//
	if (fork() == 0)
	{
	    //
	    // Ask tcl to perform the correct command, if not in debug mode
	    // the command is update_host otherwise it is debug_host.
	    //
	    if (pOptions.is_debug()) {
		sprintf(commandString, "debug_host {%s} {%s}",
			hostName, outputFileName);
	    }
	    else if (pOptions.is_compare()) {
		sprintf(commandString, "compare_host {%s} {%s}",
			hostName, outputFileName);
	    }
	    else {
		sprintf(commandString, "update_host {%s} {%s}",
			hostName, outputFileName);
	    }
	    rc = Tcl_Eval(interp, commandString);

	    //
	    // Each child is responsible for removing it's file.
	    //
	    unlink(outputFileName);

	    //
	    // Child dies.
	    //
	    exit(0);
	    _exit(0);
	}

	//
	// Wait for one child to die if maximum number of parallel processes
	// reached.
	//
	if (++proc_count >= pOptions.max_processes) {
	    proc_count = 0;
	    while(wait(0) >= 0)
	        ;
	}
    }

    //
    // Wait until all hosts are updated.
    //
    while(wait(0) >= 0)
        ;

    return 0;
}

//////////////////////////////////////////////////////////////////
//		S A D I S T : : L I S T _ D I S T R I B U T I O N S
//		---------------------------------------------------
// Description:
//	List available distributions.
//
// Arguments:
//	distName	- List distributions below distribution.
//			  if distName is NULL, list all top distributions.
//
//////////////////////////////////////////////////////////////////
TextList
SADist::list_distributions(const char *distName)
{
    TextList returnList;		// List of distributions to return.
    DIR *dp;				// Pointer to open directory.
    struct dirent *entry;		// Index over directory entries.
    char dir[PATH_MAX];			// Distribution directory.
    char fileName[PATH_MAX];		// Full name of directory entry.
    char entryName[PATH_MAX];		// Name of current entry.

    pStatus = 0;

    Tcl_ResetResult(interp);

    //
    // Get the name of the top directory.
    //
    if (directory == 0) {
    	Tcl_SetResult(interp, "the directory variable is not set by the sadist.tcl file", 0);
	pStatus = 1;
    	return returnList;
    }
    strcpy(dir, directory);
    if (distName) {
    	strcat(dir, "/");
    	strcat(dir, distName);
    }

    //
    // Open the sadist top directory.
    //
    if ((dp = opendir(dir)) == NULL) {
    	Tcl_AppendResult(interp, dir, ": can open directory", 0);
	pStatus = 1;
    	return returnList;
    }

    //
    // Read directory entries.
    //
    while((entry = readdir(dp)))
    {
    	//
    	// Get the entry name.
    	//
    	strncpy(entryName, entry->d_name, entry->d_reclen);
    	entryName[entry->d_reclen] = 0;

	//
	// Skip parent and current directory.
	//
	if (strcmp(entryName, ".") == 0 || strcmp(entryName, "..") == 0) {
	    continue;
	}

    	//
    	// Make up the full file name for a Control file in the entry 
    	// subdirectory.
    	//
    	strcpy(fileName, dir);
    	strcat(fileName, "/");
    	strcat(fileName, entryName);
    	strcat(fileName, "/Control");

    	//
    	// Check if its a directory.
    	//
    	if (access(fileName, F_OK) == 0) {
	    //
	    // Append distName if specified. If this is not done
	    // the list returned can not be used directly by calling
	    // function.
	    //
	    if (distName) {
		strcpy(fileName, distName);
		strcat(fileName, "/");
		strcat(fileName, entryName);
	    }
	    else {
		strcpy(fileName, entryName);
	    }
	    returnList.append(TextEntry(fileName));
	}
    }

    return returnList;
}

//
// Following procedures are only wrappers for Tcl registrated procedures
// in order to make it more C++ suitable, i.e. the tcl procedures are mapped
// to SADist member methods. The CliendData is assumed to be a pointer to 
// the SADist object itself.
//
int
SADist::priv_proc_include(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    SADist *p = (SADist *)clientData;
    return p->proc_include(argc, argv);
}

int
SADist::priv_proc_error(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    SADist *p = (SADist *)clientData;
    return p->proc_error(argc, argv);
}

int
SADist::priv_proc_text(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    SADist *p = (SADist *)clientData;
    return p->proc_text(argc, argv);
}

int
SADist::priv_proc_sadb(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    SADist *p = (SADist *)clientData;
    return p->proc_sadb(argc, argv);
}

int
SADist::priv_proc_sarand(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    SADist *p = (SADist *)clientData;
    return p->proc_sarand(argc, argv);
}

const char *
SADist::error_message(void)
{
    if (interp && interp->result) {
    	return interp->result;
    }
    else {
    	return "internal error";
    }
}

int
SADist::status(void)
{
    return pStatus;
}


//////////////////////////////////////////////////////////////////
//
//		S O U R C E S _ S E T U P F I L E S
//		-----------------------------------
// Description:
//	Execute global sadist setup file.
//
//////////////////////////////////////////////////////////////////
int
SADist::source_setupfiles(void)
{
    char fileName[PATH_MAX];
    char *SABinDir = getenv(SABINDIR);
    if (SABinDir == 0) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp,
			 SABINDIR,
			 ": environment not set",
			 0);
    	return 1;
    }

    Tcl_SetVar(interp, "SABINDIR", SABinDir, TCL_GLOBAL_ONLY);
    sprintf(fileName, "%s/lib/sadist.tcl", SABinDir);

    return Tcl_EvalFile(interp, fileName);
}


//////////////////////////////////////////////////////////////////
//		S A D I S T : : ~ S A D I S T
//		-----------------------------
// Description:
//	Destructor for SADist.
//
// Arguments:
//	none
//	
//////////////////////////////////////////////////////////////////
SADist::~SADist(void)
{
    //
    // Delete interpreter.
    //
    Tcl_DeleteInterp(interp);

    //
    // Make sure the temporary file is deleted.
    //
    unlink(outputFileName);
}

//
// History of changes:
// SADist.cc,v
// Revision 1.31  1996/09/14 18:33:34  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.30  1996/06/08 21:58:28  lasse
// *** empty log message ***
//
// Revision 1.29  1996/05/16 19:59:44  lasse
// possible to specify database none
//
// Revision 1.28  1996/05/01 21:56:40  lasse
// backup
//
// Revision 1.27  1996/03/12 19:42:31  lasse
// Checking in from remote.
//
// Revision 1.26  1995/11/18  11:57:48  lasse
// Bugfix with multiple distributions and corrected the sarand to allways
// use hostname as the seed value
//
// Revision 1.25  1995/11/18  11:33:39  lasse
// Removed bug with multiple distributions.
//
// Revision 1.24  1995/11/09  21:21:01  lasse
// To be 0.9
//
// Revision 1.23  1995/10/28  11:50:49  lasse
// Selecting how if to use tcl or tclX
//
// Revision 1.22  1995/09/24  17:40:08  lasse
// Really irritating bug with freeing a Tcl_SplitList vector.
//
// Revision 1.21  1995/09/24  08:06:41  lasse
// Backup
//
// Revision 1.20  1995/09/23  13:46:07  lasse
// Imported from remote
//
// Revision 1.3  1995/09/22  19:05:49  qdtlarb
// Now runs ins parallel
//
// Revision 1.2  1995/09/21  08:16:27  qdtlarb
// Added the preprocess option and changed hosts from being a list
// to a command to execute which in turn makes the list.
//
// Revision 1.1.1.1  1995/09/11  09:23:06  qdtlarb
// THis is version 0.6
//
// Revision 1.19  1995/09/10  20:43:23  lasse
// Added copyright everywhere
//
// Revision 1.18  1995/09/10  19:03:42  lasse
// Corrected removed Log keyword
//
// Revision 1.1.1.1  1995/07/17  07:51:35  qdtlarb
// Original V0_3
//
// Revision 1.13  1995/07/16  17:59:22  lasse
// The all distribution seems to work.
//
// Revision 1.12  1995/07/16  13:45:48  lasse
// merged differences
//
// Revision 1.11  1995/07/05  19:03:43  lasse
// backup
//
// Revision 1.10  1995/06/15  10:56:09  lasse
// Moved the registration of database command to SADB
//
// Revision 1.9  1995/06/11  19:44:06  lasse
// Added option -l
//
// Revision 1.8  1995/06/11  12:26:22  lasse
// backup
//
// Revision 1.7  1995/06/09  21:37:04  lasse
// Now both sadbcmd and sadist seems to work with G++ DLList classes
//
// Revision 1.6  1995/06/08  19:55:54  lasse
// backup
//
// Revision 1.5  1995/06/08  19:35:56  lasse
// backup
//
// Revision 1.4  1995/06/08  19:20:57  lasse
// backup
//
// Revision 1.3  1995/06/08  18:27:42  lasse
// backup
//
// Revision 1.2  1995/06/07  20:36:05  lasse
// Backup
//
// Revision 1.1  1995/06/07  20:29:43  lasse
// It seems as i managed to make the SADist object ok.
//
//
