//
// Copyright (C) 1995, 1996  Lars Berntzon
//
//////////////////////////////////////////////////////////////////
//
// Name:
//	SADBMK
//
// Synopsis:
//	sadbmk [-r] database-directory
//
// Description:
//	Creates columns withing a sadb database and fills it with
//	data.
//	It reads a keyed list of data on its stdinput, the first
//	column must be the column namne, next is te key, rest of
//	the line is data for the column.
//	The "column" argument is a full path for the dbm files
//	except the .dir and .pag extensions.
//
// Options:
//	-r	Don't clear the whole database before reading data.
//
//	-x	Remove all entries from the database that
//		is not given as input data.
//
// Ident:
//	sadbmk.cc,v 1.15 1997/04/15 19:56:24 lasse Exp
//
//////////////////////////////////////////////////////////////////
#include <sadblib.hh>
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <ctype.h>
#include <SetList.hh>
#include <Set.hh>

//
// Global options.
//
struct {
   int r;	// The r-option.
   int x;	// The x-option.
} opt;

//
// Result from getopt.
//
extern int optind;

//
// Name of the program.
//
char *prog;

//
// Local routines.
//
static void check_arguments(int args, char **argv);
static char **tokenize(char *line);
static void write_data(SetList &columnList, char *directory, int mode);
static void remove_extra_entries(Set &keySet, char *directory);


//////////////////////////////////////////////////////////////////
//		M A I N   R O U T I N E
//		-----------------------
// Description:
//	This is the main routine for sadbmk.
//
//////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
    char line[10000];		// Input line.
    int mode = 2;		// Mode of operation, 1 read/write, 2 overwrite.
    SetList columnList;		// List of colmumns.
    Set keySet;			// Set of keys.
    char *directory;		// Directory for database.
    char **tokens;		// Tokens on input.

    //
    // Check arguments.
    //
    check_arguments(argc, argv);

    //
    // If either -r or -x, don't overwrite the database.
    //
    if (opt.r || opt.x) {
    	mode = 1;
    }

    //
    // Name of database directory.
    //
    directory = argv[optind];

    //
    // Read all lines of data.
    //
    while(!feof(stdin))
    {
	gets(line);
	
	tokens = tokenize(line);
	if (tokens == NULL) continue;

	//
	// Add value to column set list.
	//
	columnList.set(tokens[0], TextEntry(tokens[1], tokens[2]));

	//
	// Add key to set. This will be used to remove all entries
	// that was not in input data.
	//
	keySet.set(tokens[1]);
    }

    //
    // Write the data to the database.
    //
    write_data(columnList, directory, mode);

    //
    // Remove all entries which was not on input (-x option).
    //
    if (opt.x)
    {
    	remove_extra_entries(keySet, directory);
    }

    return 0;
}


//////////////////////////////////////////////////////////////////
//		C H E C K _ A R G U M E N T S
//		-----------------------------
// Description:
//	Check command line arguments.
//
//////////////////////////////////////////////////////////////////
static void
check_arguments(int argc, char **argv)
{
    int ch;

    while((ch = getopt(argc, argv, "rx")) != EOF)
    {
	switch(ch)
	{
	case 'r':
	    opt.r = 1;
	    break;

	case 'x':
	    opt.x = 1;
	    break;

	default:
	    cerr << argv[0] << ": unknown option: -" << ch << endl;
	    exit(1);
	}
    }
    if ((argc - optind) < 1) {
	fprintf(stderr, "usage: %s [-r] dbm-name\n", argv[0]);
	exit(1);
    }
}


//////////////////////////////////////////////////////////////////
//		T O K E N I Z E
//		---------------
// Description:
//	Tokenize input string and return array of pointers.
//
//////////////////////////////////////////////////////////////////
static char **
tokenize(char *line)
{
    static char *tokens[3];
    char *p;

    //
    // Get Column name.
    //
    tokens[0] =  line + strspn(line, " \t");

    //
    // Get key.
    //
    if ((tokens[1] = strpbrk(tokens[0], " \t")) == NULL) return NULL;
    *(tokens[1])++ = 0;
    tokens[1] = tokens[1] + strspn(tokens[1], " \t");

    //
    // Get value, and trim off trailing blanks.
    //
    if ((tokens[2] = strpbrk(tokens[1], " \t")) == NULL) return NULL;
    *(tokens[2])++ = 0;
    tokens[2] += strspn(tokens[2], " \t");
    p = tokens[2] + strlen(tokens[2]) - 1;
    while((p > tokens[2]) && isspace(*p)) {
	p--;
    }
    p[1] = 0;

    return tokens;
}


//////////////////////////////////////////////////////////////////
//		W R I T E _ D A T A
//		-------------------
// Description:
//	Write data to the database.
//
//////////////////////////////////////////////////////////////////
static void
write_data(SetList &columnList, char *directory, int mode)
{
    Pix i;
    Pix e;
    char fileName[PATH_MAX];

    //
    // Write data to database.
    //
    for(i = columnList.first(); i != 0; columnList.next(i))
    {
	sprintf(fileName, "%s/%s", directory, columnList(i).name());
	DB_OF_CHOICE db(fileName, mode);

	if (!db.ok()) {
	    cerr << prog << ": " << fileName << ": failed to open database\n";
	    exit(1);
	}

    	TextList l = columnList(i).list();
    	for(e = l.first(); e; l.next(e))
    	{
	    db.store(l(e).name(),l(e).text(), DB::replace_t);
	    if (!db.ok()) {
		cerr << prog << ": failed to store data\n";
		exit(1);
	    }
    	}
    }
}


//////////////////////////////////////////////////////////////////
//		R E M O V E _ E X T R A _ E N T R I E S
//		---------------------------------------
// Description:
//	Remove all entries from the database key in directory
//	which are not in the keySet set.
//
//////////////////////////////////////////////////////////////////
static void
remove_extra_entries(Set &keySet, char *directory)
{
    const char *name;
    Pix i;
    Set removeSet;
    char fileName[PATH_MAX];

    //
    // Contruct the database.
    //
    sprintf(fileName, "%s/key", directory);
    DB_OF_CHOICE db(fileName, 1);
    if (!db.ok()) {
	cerr << prog << ": " << fileName << ": failed to open database\n";
	exit(1);
    }

    //
    // Loop through all keys in the database, the check the keySet
    // list to see if the record is there. If not, remove it from
    // the database, unless of course it is a description.-record.
    //
    for(name = db.firstkey(); name != NULL; name = db.nextkey())
    {
	for(i = keySet.first(); i != 0; keySet.next(i))
	{
	    if (strcmp(name, keySet(i).name()) == 0) {
		break;
	    }
	}
	if (i == 0 && strncmp(name, "description.", 12) != 0) {
	    removeSet.set(name);
	}
    }

    //
    // Loop through all keys to be removed and remove them.
    //
    for(i = removeSet.first(); i != 0; removeSet.next(i))
    {
	cout << "removing " << removeSet(i).name() << endl;
	db.remove(removeSet(i).name());
    }
}

//
// History of changes:
// sadbmk.cc,v
// Revision 1.15  1997/04/15 19:56:24  lasse
// Corrected bug with the description.* removal.
//
// Revision 1.14  1996/10/20 21:52:16  lasse
// To be 1.2 maybe
//
// Revision 1.13  1996/10/20 21:17:48  lasse
// Seems to work with the -x option.
//
// Revision 1.12  1996/09/21 22:53:29  lasse
// Don't print trace
//
// Revision 1.11  1996/09/14 18:33:32  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.10  1996/07/30 21:55:05  lasse
// New version of sadbmk
//
// Revision 1.9  1996/03/21 18:57:56  lasse
// Backup
//
// Revision 1.8  1995/09/23  13:46:06  lasse
// Imported from remote
//
// Revision 1.1.1.1  1995/09/11  09:23:05  qdtlarb
// THis is version 0.6
//
// Revision 1.7  1995/09/10  20:43:22  lasse
// Added copyright everywhere
//
// Revision 1.6  1995/09/10  19:03:41  lasse
// Corrected removed Log keyword
//
// Revision 1.1.1.1  1995/07/17  07:51:34  qdtlarb
// Original V0_3
//
// Revision 1.4  1995/07/16  16:19:33  lasse
// sahostinfo a whole lot better.
//
// Revision 1.3  1995/07/16  13:43:26  lasse
// merged differences
//
// Revision 1.2  1995/06/13  10:31:18  lasse
// fixed bug with store
//
// Revision 1.1  1995/06/11  10:35:05  lasse
// Created
//
