/* This is server.c, the "server" end of rfiledist 0.90.  This is GPL software, originally created by Tom Kunz, tkunz@fast.net */


/* Includes */

#include "header.h"
#include "io.h"
#include "node.h"
#include "comm.h"
#include "listlib.h"

/* Defines */

#define BUFFER 512
#define PATH_LIM 6

/* The "ACK" character, plus the command-line option values and some
   other important variables */

char *ACK = "A";
int INTERRUPT = 0,/* verbose = 0, */debug = 0;
char *client_addr;
pid_t self_pid;
list SECURE;

/* Function Prototypes used in this code */

int serv(conn *C, conn *client);
static void sig_int(int signo);
int bg(void);
void std_quit(char*);
void process_options(int arg_count, char *arg_list[]);
int read_fnode(fnode *F,int fd);
int handle_fnode(conn *C, int fd, conn *client, char *packagename);
int handle_pre(int clisock, conn *client);
int handle_post(int clisock, conn *client);
int last_package(char *packagename);
int if_get_cmd(char *cmd);
int if_pre_cmd(char *cmd);
int if_post_cmd(char *cmd);
int if_quit_cmd(char *cmd);
int if_waitfor(char *packagename);
char *stripspace(char *name);
int add_address(char *addr, list L);
const char *inet_ntop(int family, const void *addrptr, char *strptr, size_t len);
int inlist(list L, void *data);
int make_pkg_secure(int fd, list L);
int is_secure(char *packagename, char *IP, list master);

void main(int argc, char *argv[])
{
  int clilen,childpid=1,grandchildpid=1,i;
  struct sigaction INTintr,INTointr,ALRMintr,ALRMointr;
  conn srv, cli;
  int DEBUG = 1;
  struct sockaddr_in cli_addr,serv_addr,testserv_addr;
  struct passwd *pw, *thisproc;
  char *configpath[PATH_LIM];
  char *goodmachines[PATH_LIM];
  list namelist;
  int opened = 0,fd;
  struct stat statbuf;
  char *data,*nospcdata,*resolvedaddress;
  char *securitydb;
  char *self_uid;
  
  /* Inform the log file that you are starting up */
  logger(SRV,"main: Starting...");

  /* Set the umask to 0 and become the session leader */
  umask(0);
  setsid();
  
  /* Make all possible attempts to become root if we ain't already */
  thisproc = (struct passwd *)getpwuid(getuid());
  self_uid = strdup(thisproc->pw_name);
  pw = (struct passwd *)getpwnam("root");
  logger(SRV,"main: Got %s as my name, and %s as the root name",self_uid,pw->pw_name);
  if (strcmp(self_uid,"root") != 0) {
    logger(SRV,"main: Attempting to change ruid & euid to %d uid from %d",pw->pw_uid,getuid());
    if (setreuid(pw->pw_uid,pw->pw_uid) != 0) {
      logger(SRV,"main: I was not run as root, and I failed to set ruid & euid to %d.  You must run '%s' as uid = %d.  Exiting.",pw->pw_uid,argv[0],pw->pw_uid);
      exit(0);
    }
  }

  /* Define our possible config file locations */
  configpath[0] = "/usr/local/refdis/conf/refdis-server.conf";
  configpath[1] = "/etc/refdis-server.conf";
  configpath[2] = "/etc/refdis/refdis-server.conf";
  configpath[3] = "/usr/local/etc/refdis-server.conf";
  configpath[4] = "/usr/local/etc/refdis/refdis-server.conf";
  configpath[5] = "./refdis-server.conf";

  /* Define the possible locations of the good-machine list */
  goodmachines[0] = "/usr/local/refdis/conf/access.list";
  goodmachines[1] = "/etc/access.list";
  goodmachines[2] = "/etc/refdis/access.list";
  goodmachines[3] = "/usr/local/etc/access.list";
  goodmachines[4] = "/usr/local/etc/refdis/access.list";
  goodmachines[5] = "./access.list";
  namelist = lstcreate("Good Machine Names - Name & Numbers");

  for (i = 0; i < PATH_LIM; i++) {
    if (!opened) {
      if (stat(configpath[i],&statbuf) == 0) {
	if (DEBUG) { 
	  logger(SRV,"main: Opening %s",configpath[i]);
	}
	
	/* Open the config file */
	if ((fd = open(configpath[i],O_RDONLY)) < 0) {
	  logger(SRV,"main: Failed open of %s. Now dying.",configpath[i]);
	  exit(0);
	}
	opened = 1;
      }

      /* Let the log file know if we couldn't open a certain one, but
         only if we're in DEBUG mode.  Otherwise, failure to open some
         of them is normal... */
      else {
	if (DEBUG) logger(SRV,"main: Failed to stat %s",configpath[i]);
	opened = 0;
      }
    } /* if (!opened) */
  } /* for (i = 0; i < PATH_LIM; i++) */
  
  /* Since we couldn't get any config file to work, die peacefully */
  if (!opened) {
    logger(SRV,"main: Failed to open a valid config file!");
    logger(SRV,"main: Please configure me properly.  The following is my search path:");
    for (i = 0; i < PATH_LIM; i++) {
      logger(SRV,"main: %s",configpath[i]);
    }
    logger(SRV,"main: Exiting");
    exit(0);
  }
  
  /* Malloc piece for the initial block that data is read into.  Data
     will be massaged and modified and put in other locations */

  if ((data = malloc(BUFFER)) == NULL){
    logger(SRV,"main: Bad malloc for data chunk.  Exiting");
    exit(0);
  }

  /* Build the linked list of all the data in the config file */

  while (readstring(fd,data,BUFFER) > 0){
    /* strip out spaces in the string */
    nospcdata = stripspace(data);
    if (!(is_comment(nospcdata))){
      if (DEBUG) logger(SRV,"main: Got actual data, no comment data");
    } /* !(is_comment(nospcdata)) */
    else
      if (DEBUG) logger(SRV,"main: Got comment as \"%s\"",nospcdata); 
  }
  /* Done doing stuff with the config file */
  close(fd);
  opened = 0;
  
  for (i = 0; i < PATH_LIM; i++) {
    if (!opened) {
      if (stat(goodmachines[i],&statbuf) == 0) {
	if (DEBUG) { 
	  logger(SRV,"main: Opening %s",goodmachines[i]);
	}
	
	/* Open the access file */
	if ((fd = open(goodmachines[i],O_RDONLY)) < 0) {
	  logger(SRV,"main: Failed open of %s. Now dying.",goodmachines[i]);
	  exit(0);
	}
	opened = 1;
      }

      /* Let the log file know if we couldn't open a certain one, but
         only if we're in DEBUG mode.  Otherwise, failure to open some
         of them is normal... */
      else {
	if (DEBUG) logger(SRV,"main: Failed to stat %s",goodmachines[i]);
	opened = 0;
      }
    } /* if (!opened) */
  } /* for (i = 0; i < PATH_LIM; i++) */
  
  /* Since we couldn't get any access file to work, die peacefully */
  if (!opened) {
    logger(SRV,"main: Failed to open a valid access file!");
    logger(SRV,"main: Please configure me properly.  The following is my search path:");
    for (i = 0; i < PATH_LIM; i++) {
      logger(SRV,"main: %s",goodmachines[i]);
    }
    logger(SRV,"main: Exiting");
    exit(0);
  }
  
  /* Malloc piece for the initial block that data is read into.  Data
     will be massaged and modified and put in other locations */

  if ((data = malloc(BUFFER)) == NULL){
    logger(SRV,"main: Bad malloc for data chunk.  Exiting");
    exit(0);
  }

  /* Build the linked list of all the data in the access file */

  while (readstring(fd,data,BUFFER) > 0){
    /* strip out spaces in the string */
    nospcdata = stripspace(data);
    if (!(is_comment(nospcdata))){
      if (DEBUG) logger(SRV,"main: Got actual data, no comment data");
      if (add_address(nospcdata,namelist) < 0)
	logger(SRV,"main: Error adding addresses for %s to access list. Please check it out.",nospcdata);
      
    } /* !(is_comment(nospcdata)) */
    else
      if (DEBUG) logger(SRV,"main: Got comment as \"%s\"",nospcdata); 
  }
  /* Done doing stuff with the access file */
  close(fd);

  /* Now we need to read the file which tells us about accessibility
     of different packages to different machines. */

  securitydb = malloc(sizeof(PACKAGE_ROOT)+255);
  sprintf(securitydb,"%s/packages.secure",PACKAGE_ROOT);
  
  if (stat(securitydb,&statbuf) == 0) {
    logger(SRV,"main: Found security db at %s",securitydb);
    if ((fd = open(securitydb,O_RDONLY)) < 0)
      std_quit("main: Unable to open security db!  Shutting down to prevent mayhem!");
    SECURE = lstcreate("Secure Package List");
    if (make_pkg_secure(fd,SECURE) < 0)
      std_quit("main: Problem creating security db.  Shutting down to prevent illegal access.");
    close(fd);
  }
  else {
    logger(SRV,"main: No security db.  All packages are free & unsecured.");
  }
  
  /* Finished with security stuff */


  /* At this point, the list of machines which can access this server
     has been created, and we're ready for accepting client
     connections. */

  /* Allocate space for the hostname that will be connecting */
  client_addr = malloc(MAX_HOSTNAME_LENGTH);

  /* Find out our pid */
  self_pid = getpid();

  /* Nullify the sockaddr.sin_addr structure to all zeros */
  if (memset(&cli.sockaddr.sin_addr,0,sizeof(cli.sockaddr.sin_addr)) == NULL)
    std_quit("main: memset failed.  Exiting");

  /* Process the command-line arguments */  
  process_options(argc,argv);

  /* set the signal handlers */
  INTintr.sa_handler = sig_int;
  INTintr.sa_flags = 0;
  sigemptyset(&INTintr.sa_mask);
  if (sigaction(SIGINT,&INTintr,&INTointr) < 0){
    std_quit("main: problem setting sigaction(SIGINT)");
  }

  /* background the proc if -d was not on the command line */
  if (!debug) {

    /* Disconnect from controlling terminal (should be cleaned up to
     be POSIX.1 compliant, I know, but this'll work in virtually
     everything) */
    for (i = 0; i < MAX_FILES; i++)
      close(i);
    
    /* Background the daemon, if it is going to be, truly, a daemon */
    if (bg())
      std_quit("main: couldn't background self. Exiting");
  }

  /* Create a listening socket */
  if ((srv.lst_sock = make_sock(&srv,SERV_TCP_PORT)) < 0) {
    std_quit("main: make_sock failed. Exiting");
  }
  
  /* Set the size of the socket to be used */
  clilen = sizeof(cli.sockaddr);

RESET:  
  /* Nullify the client name to all nulls, also */
  if (memset(client_addr,0,sizeof(client_addr)) == NULL)
    std_quit("main: memset failed.  Exiting");

  /* While, we're not interrupted by a signal, do the work */
  while (!INTERRUPT){
    /* Notify logfile that we're ready for action */
    logger(SRV,"main: Ready to accept a client");
    
    /* Set up the accepting socket.  Block here. */
    srv.acc_sock = accept(srv.lst_sock,(struct sockaddr *)&cli.sockaddr,&clilen);

    /* If the strcpy of the client address goes ok... */
    if (strcpy(client_addr,inet_ntoa(cli.sockaddr.sin_addr)) != NULL) {
      printf("\"%s\"\n",client_addr);

      /* If it's in the list, great, if not, slam the door shut! */
      if (!inlist(namelist,client_addr)) {
	close(srv.acc_sock);
	srv.acc_sock = -1;
	logger(SRV,"main: Denied an attempt by %s to connect.",client_addr);
	logger(SRV,"main: Resetting and waiting for valid client");
	goto RESET;
      }
      
      /* ...but there's an error on the socket... */
      if (srv.acc_sock < 0 ) {

	/* ...which is caused by a SIGINT, then exit. */
	if (INTERRUPT) {
	  std_quit("main: SIGINT caused a socket creation error.  This is probably ok, if you are trying to kill the server");
	} 

	/* Otherwise, the error was caused by something else, and we
           should die appropriately. */
	else {
	  std_quit("main: accept error");
	}
      }
    }
    else {
      std_quit("main: strncpy failed!  Exiting!");
    }

    /* Start the forking, so we can be a background child */
    if ((childpid = fork()) < 0) {
      std_quit("main: error condition with first fork.  Exiting");
    }

    /* Inside parent server */
    if (childpid > 0) {
      close(srv.acc_sock);
      if (DEBUG) logger(SRV,"main: Inside parent, with srv.acc_sock = %d srv.lst_sock = %d",srv.acc_sock,srv.lst_sock);
    }
    /* Once we're inside the first child... */
    else if (childpid == 0) {

      /* ...fork a second child. */
      if ((grandchildpid = fork()) < 0) {

	/* Error on second fork */
	std_quit("main: error with second fork. Exiting");
      }
      else {

	/* No error on second fork.  Now we need to let the child
           exit, and the grandchild can continue to do the work. */
	if ((childpid == 0) && (grandchildpid != 0)) {
	  return;
	}
	if (grandchildpid == 0) {

	  /* Now inside the grandchild. */
	  if (DEBUG) logger(SRV,"main: Doing the serving");
	  close(srv.lst_sock);
	  serv(&srv,&cli);
	  close(srv.acc_sock);
	  if (DEBUG) logger(SRV,"main: Did the serving");
	  exit(0);
	  return;
	}
      } /* no error on second fork, grandchild. */
    } /* no error on child. childpid spawned ok */
    if (waitpid(childpid,NULL,0) != childpid) {
      std_quit("main: waitpid error");
    }
  } /* while (!INTERRUPT) */
  return;
}

static void sig_int(int signo)
{
  /* Handle the SIGINT's that come our way by just incrementing a
     global variable and checking for that being non-zero in our other
     functions. */
  if (signo == SIGINT)
    INTERRUPT++;
  if (getpid() != self_pid)
    kill(self_pid,signo);
  return;
}

int bg()
{
  /* Simple code to background ourselves */
  int err = 0;
  if((err = fork()))
    exit(0);
  return(err);
}

int serv(conn *C, conn *client)
{
  /* This is the function which handles the actual connection */
  char cmd[SMALL_BUF];
  char packagename[MAX_PKG_NAME];
  char pkgpath[MAX_PKG_NAME];
  int packfd = -1;
  int DEBUG = 1;
  int GOT_CMD = 0;
  int WRONG = 0;
  char *IP;
  
  if (INTERRUPT) {
    std_quit("serv: Interrupting server before it really starts");
  }
  while (!if_quit_cmd(cmd)) {

    /* Read command. */
    if ((GOT_CMD = recv_send(C->acc_sock,cmd)) < 0)
      logger(SRV,"serv: recv_send error on cmd");
    if (GOT_CMD > 0) {
      if (if_quit_cmd(cmd)) {
	if (DEBUG) logger(SRV,"serv: Breaking the if_quit_cmd loop since cmd is %s",cmd);
	break;
	if (DEBUG) logger(SRV,"serv: You should never see this");
      }
      if (DEBUG) logger(SRV,"serv: Got %s as cmd",cmd);
      
      /* If we want a pre-script to come over, do this. */
      if (if_pre_cmd(cmd)) {
	if (DEBUG) logger(SRV,"serv: got a request for a prescript");
	if (handle_pre(C->acc_sock,client) < 0) {
	  logger(SRV,"serv: problem with handle_pre");
	  WRONG = -1;
	}

	/* Put "empty" into the command buffer, so that it is not
           interpreted as something it shouldn't on the next time
           through */
	strcpy(cmd,"empty");

	/* And if we want a post-script to come over, do this. */
      } else if (if_post_cmd(cmd)) {
	if (DEBUG) logger(SRV,"serv: got a req for a postscript");
	if (handle_post(C->acc_sock,client) < 0) {
	  logger(SRV,"serv: problem with handle_post");
	  WRONG = -1;
	}
	
	/* Put "empty" into the command buffer, so that it is not
           interpreted as something it shouldn't on the next time
           through */
	strcpy(cmd,"empty");
	
	/* But if we want an actual package, do this. */
      } else if (if_get_cmd(cmd)) {

	/* If this isn't the last package to be gotten (ie a null
           package name), do this */
	if (!last_package(packagename)) {
	  
	  /* Get the package name */
	  if (DEBUG) logger(SRV,"serv: Getting packagename");
	  if (recv_send(C->acc_sock,packagename) < 0) {
	    logger(SRV,"serv: recv_send error on packagename");
	    WRONG = -1;
	  }
	  if (DEBUG) logger(SRV,"serv: Got %s as packagename",packagename);
	  /*
	  IP = malloc(SMALL_BUF);
	  if (strcpy(IP,inet_ntoa(client->sockaddr.sin_addr)) == NULL) {
	    logger(SRV,"serv: Got some problems with determining address of remote.");
	  }
	  if (!is_secure(packagename,IP,SECURE)) {
	    logger(SRV,"serv: About to inform the client that they don't have rights to \"%s\"",packagename);
	    if (send_recv(C->acc_sock,NONEXISTENT) < 0) {
	      logger(SRV,"serv: problem with filename send_recv()");
	      WRONG = -1;
	    }
	    
	    if (send_recv(C->acc_sock,NONEXISTENT) < 0) {
	      logger(SRV,"serv: problem with link send_recv()");
	      WRONG = -1;
	    }
	    */
	    /* Jump out of here if there's nothing else to be done */
	    /*	    return(WRONG);*/
	    /*	    break;*/
	    /*
	  }
	  */
	  
	  /* Exit loop if this is the last package name */
	  if (last_package(packagename)) {
	    if (DEBUG) logger(SRV,"serv: Breaking the loop since packagename is %s",packagename);
	    break;
	  }
	  if (DEBUG) logger(SRV,"serv: %s is not the last package",packagename);

	  /* Start building the actual pathname to look at */
	  if (strcpy(pkgpath,PACKAGE_ROOT) == NULL) {
	    logger(SRV,"serv: Bad strcpy of PACKAGE_ROOT");
	    WRONG = -1;
	  }
	  if (DEBUG) logger(SRV,"serv: Using \"%s\" as the package root",pkgpath);

	  /* Add the packagename onto the name of the package root */
	  if (strcat(pkgpath,packagename) == NULL) {
	    logger(SRV,"serv: Bad strcat of packagename");
	    WRONG = -1;
	  }
	  if (DEBUG) logger(SRV,"serv: Using \"%s\" as the full package path",pkgpath);

	  /* Open up the filename pointed to by pkgpath */
	  if ((packfd = open(pkgpath,O_RDONLY)) < 0) {
	    if (DEBUG) logger(SRV,"serv: No Package by name %s found in location %s",packagename,pkgpath);
	    WRONG = -1;
	  }

	  /* Handle the package.  handle_fnode is ok if packfd == -1. */
	  if (handle_fnode(C,packfd,client,packagename) < 0) {
	    logger(SRV,"serv: handle_fnode quit unexpectedly!");
	    WRONG = -1;
	  }
	  if (DEBUG) logger(SRV,"serv: finished handle_fnode");
	} /* if (!last_packagename(packagename)) */

	if (DEBUG) logger(SRV,"serv: After the break out from package handling");

	/* Put "empty" into the command buffer, so that it is not
           interpreted as something it shouldn't on the next time
           through */
	strcpy(cmd,"empty");
      } /* else if (if_get_cmd(cmd)) */
    } /* if (GOT_CMD > 0) */ 
    else
      break; /* This way, we will continue looping in circles until
    GOT_CMD > 0. */
    
  } /* while (!if_quit_cmd(cmd)) */
  return(WRONG);
}

void std_quit(char *text)
{
  /* This function is just to help us exit more gracefully than simply
     dying */
  int i;
  logger(SRV,text);
  logger(SRV,"std_quit: Raising SIGINT");
  raise(SIGINT);
  if (self_pid != getpid())
    kill(self_pid,SIGINT);

  /* Closing all descriptors manually */
  for (i = 0; i < MAX_FILES; i++){
    close(i);
  }

  /* Turn out the lights when you go... */
  logger(SRV,"std_quit: Exiting");
  sync();
  exit(0);
}

void process_options(int arg_count,char *arg_list[])
{
  /* This is to process the command-line option */
  arg_count--;
  arg_list++;
  while (arg_count > 0 && *arg_list[0] == '-') {
    while (*++arg_list[0]) {
      switch (*arg_list[0]) {
      case 'd':
	logger(SRV,"process_options: Got a debug flag, staying in foreground.");
	printf("process_options: Got a debug flag, staying in foreground.\n");
	debug++;
	break;
	/*    case 'v':
	      logger(SRV,"process_options: Got verbosity!");
	      verbose++;
	      break;*/
      default:
	logger(SRV,"process_options: Ignoring option \"%c\"",*arg_list[0]);
	break;
      } /* switch (*arg_list[0]) */
    } /* while (*++arg_list[0]) */
    arg_count--;
    arg_list++;
  } /* while (arg_count > 0 && *arg_list[0] == '-') */
}

ssize_t read_fnode(fnode *node,int fd)
{
  /* This function returns the total number of bytes read from the
     file for any particular fnode */
  int lim = 20,i;
  ssize_t r[lim],sum = 0;

  /* Initialize array to zero */  
  for (i = 0; i < lim; i++)
    r[i] = 0;

  /* Set all the elements of the fnode to NULL or zero */
  if (nullify_fnode(node) < 0)
    logger(SRV,"read_fnode: nullify_fnode failure");

  /* Read data from the file */
  /* The first two elements of the fnode are the name and link type,
     which are present in ALL fnodes, no matter what kind.  After
     these first two, we can make a determination about the type based
     on "link" content.  The results of the readstring() functions are
     placed in a result array, r[], so that any errors in reading (the
     presence of a -1 in the result array) can be found. */
  if ((r[0] = readstring(fd,node->fn,MAX_LENGTH)) < 0)
    logger(SRV,"Error reading package file");
  if ((r[1] = readstring(fd,node->link,MAX_LENGTH)) < 0)
    logger(SRV,"Error reading package file");

  /* If it's a FIFO, get the fields appropriate to FIFO's */
  if (is_fifo(node)) {
    if ((r[2] = readstring(fd,node->mode,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[3] = readstring(fd,node->uid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[4] = readstring(fd,node->gid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[5] = readstring(fd,node->actime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[6] = readstring(fd,node->modtime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");

    /* If it's a block or character device, get those fields */
  } else if (is_blockdev(node) || is_chardev(node)) {
    if ((r[2] = readstring(fd,node->major,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[3] = readstring(fd,node->minor,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[4] = readstring(fd,node->mode,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[5] = readstring(fd,node->uid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[6] = readstring(fd,node->gid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[7] = readstring(fd,node->actime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[8] = readstring(fd,node->modtime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");

    /* If it's a directory, get those fields */
  } else if (is_dir(node)) {
    if ((r[2] = readstring(fd,node->mode,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[3] = readstring(fd,node->uid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[4] = readstring(fd,node->gid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[5] = readstring(fd,node->actime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[6] = readstring(fd,node->modtime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");

    /* If it's a regular file, get the appropriate fields */
  } else if (is_regular(node)) {
    if ((r[2] = readstring(fd,node->local,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[3] = readstring(fd,node->sz,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[4] = readstring(fd,node->mode,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[5] = readstring(fd,node->uid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[6] = readstring(fd,node->gid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[7] = readstring(fd,node->actime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    if ((r[8] = readstring(fd,node->modtime,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");

    /* If it's a symbolic link, get the appropriate fields */
  } else if (is_symlink(node)) {
    if ((r[3] = readstring(fd,node->uid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    /*     logger(SRV,"Read uid to be %s",node->uid); */
    if ((r[4] = readstring(fd,node->gid,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
    /*     logger(SRV,"Read gid to be %s",node->gid); */
    if ((r[5] = readstring(fd,node->linkto,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");

    /* If it's hard link, you only have to get 1 field */
  } else if (is_hardlink(node)){
    if ((r[5] = readstring(fd,node->linkto,MAX_LENGTH)) < 0)
      logger(SRV,"Error reading package file");
  }

  /* Place the "finish" field into the last result entry (r[lim-1]) */
  if ((r[lim-1] = readstring(fd,node->finish,MAX_LENGTH)) < 0)
    logger(SRV,"Error reading package file");

  /* Total up the number of bytes read.  If any one element in the
     array is -1, we return -1.  Otherwise, we return the total number
     of bytes read. */
  for (i = 0; i < lim; i++) {
    sum += r[i];
    if (r[i] <0)
      return(r[i]);
  }
  return(sum);
}

int handle_fnode(conn *C, int packfd, conn *client, char *packagename)
{
  /* This function is meant to handle the fnode once it has been read
     by read_fnode() */
  fnode node;
  int filecount = 0;
  char cmd[SMALL_BUF];
  char fileACK[ONE_CHAR];
  int clisock;
  int DEBUG = 0;
  int WRONG = 0;
  
  char *IP;
  
  /* Since we depend upon the contents of the fnode to know what
     happens next, we want to zero out the contents */
  if (nullify_fnode(&node) < 0)
    logger(SRV,"handle_fnode: nullify_fnode failed");

  /* As long as we get a packfd != -1, which means the previous open
     failed, we want to do this. */
  if (packfd != -1) {
    IP = malloc(SMALL_BUF);
    if (strcpy(IP,inet_ntoa(client->sockaddr.sin_addr)) == NULL) {
      logger(SRV,"handle_pre: Got some problems with determining address of remote.");
    }
    if (!is_secure(packagename,IP,SECURE)) {
      logger(SRV,"handle_pre: About to inform the client that they don't have rights to \"%s\"",packagename);
      if (send_recv(C->acc_sock,NONEXISTENT) < 0) {
	logger(SRV,"handle_fnode: problem with filename send_recv()");
	WRONG = -1;
      }
      if (send_recv(C->acc_sock,NONEXISTENT) < 0) {
	logger(SRV,"handle_fnode: problem with link send_recv()");
	WRONG = -1;
      }
      /* Jump out of here if there's nothing else to be done */
      return(WRONG);
    }
  
    /* For as long as this is not the last node in the series... */
    while (!last_node(&node)) {
      if (DEBUG) logger(SRV,"handle_fnode: not last node");

      /* ...read the file entry for the fnode. */
      if(read_fnode(&node,packfd) < 0)
	logger(SRV,"handle_fnode: Incomplete read of package data");
      if (DEBUG) print_fnode(&node);

      /* Now write the two fields that are present in all fnodes to
         the client. */
      if (send_recv(C->acc_sock,node.fn) < 0)
	logger(SRV,"handle_fnode: problem with filename send_recv");
      if (send_recv(C->acc_sock,node.link) < 0)
	logger(SRV,"handle_fnode: problem with link send_recv");
      
      /* If it's a regular file... */
      if (is_regular(&node)) {

	/* ...send file size over to client... */
	if (send_recv(C->acc_sock,node.sz) < 0)
	  logger(SRV,"handle_fnode: problem with size send_recv()");
	
	/* ...and file mode... */
	if (send_recv(C->acc_sock,node.mode) < 0)
	  logger(SRV,"handle_fnode: problem with mode send_recv()");
	
	/* ...and uid... */
	if (send_recv(C->acc_sock,node.uid) < 0)
	  logger(SRV,"handle_fnode: problem with uid send_recv()");
	
	/* ...and gid... */
	if (send_recv(C->acc_sock,node.gid) < 0)
	  logger(SRV,"handle_fnode: problem with gid send_recv()");
	
	/* ...and last-access time... */
	if (send_recv(C->acc_sock,node.actime) < 0)
	  logger(SRV,"handle_fnode: problem with actime send_recv()");
	
	/* ...and last-modification time... */
	if (send_recv(C->acc_sock,node.modtime) < 0)
	  logger(SRV,"handle_fnode: problem with modtime send_recv()");
	
	/* ...and the finish marker. */
	if (send_recv(C->acc_sock,node.finish) < 0)
	  logger(SRV,"handle_fnode: problem with finish send_recv()");

	/* If it's a device file of some sort... */
      } else if ((is_blockdev(&node)) || (is_chardev(&node)) ) {

	/* ...send the major number... */
	if (send_recv(C->acc_sock,node.major) < 0)
	  logger(SRV,"handle_fnode: problem with major send_recv()");
	
	/* ...and minor number... */
	if (send_recv(C->acc_sock,node.minor) < 0)
	  logger(SRV,"handle_fnode: problem with minor send_recv()");
	
	/* ...and mode... */
	if (send_recv(C->acc_sock,node.mode) < 0)
	  logger(SRV,"handle_fnode: problem with mode send_recv()");
	
	/* ...and uid... */
	if (send_recv(C->acc_sock,node.uid) < 0)
	  logger(SRV,"handle_fnode: problem with uid send_recv()");
	
	/* ...and gid... */
	if (send_recv(C->acc_sock,node.gid) < 0)
	  logger(SRV,"handle_fnode: problem with gid send_recv()");
	
	/* ...and last-access time... */
	if (send_recv(C->acc_sock,node.actime) < 0)
	  logger(SRV,"handle_fnode: problem with actime send_recv()");
	
	/* ...and last-modification time... */
	if (send_recv(C->acc_sock,node.modtime) < 0)
	  logger(SRV,"handle_fnode: problem with modtime send_recv()");
	
	/* ...and the finish marker. */
	if (send_recv(C->acc_sock,node.finish) < 0)
	  logger(SRV,"handle_fnode: problem with finish send_recv()");

	/* If it's a symbolic link... */
      } else if (is_symlink(&node)) {

	/* ...send the uid... */
	if (send_recv(C->acc_sock,node.uid) < 0)
	  logger(SRV,"handle_fnode: problem with uid send_recv()");
	
	/* ...and the gid... */
	if (send_recv(C->acc_sock,node.gid) < 0)
	  logger(SRV,"handle_fnode: problem with gid send_recv()");
	
	/* ...and the file to link it to... */
	if (send_recv(C->acc_sock,node.linkto) < 0)
	  logger(SRV,"handle_fnode: problem with linkto send_recv()");
	
	/* ...and the finish marker. */
	if (send_recv(C->acc_sock,node.finish) < 0)
	  logger(SRV,"handle_fnode: problem with finish send_recv()");
	
	/* If this is a hard link... */
      } else if (is_hardlink(&node)) {
	
	/* ...send only the name of the file to link it to... */
	if (send_recv(C->acc_sock,node.linkto) < 0)
	  logger(SRV,"handle_fnode: problem with linkto send_recv()");
	
	/* ...and the finish marker. */
	if (send_recv(C->acc_sock,node.finish) < 0)
	  logger(SRV,"handle_fnode: problem with finish send_recv()");

	/* If it's a directory or a fifo... */
      } else if ((is_dir(&node)) || (is_fifo(&node))) {

	/* ...send the mode... */
	if (send_recv(C->acc_sock,node.mode) < 0)
	  logger(SRV,"handle_fnode: problem with mode send_recv()");
	
	/* ...and the uid... */
	if (send_recv(C->acc_sock,node.uid) < 0)
	  logger(SRV,"handle_fnode: problem with uid send_recv()");
	
	/* ...and the gid... */
	if (send_recv(C->acc_sock,node.gid) < 0)
	  logger(SRV,"handle_fnode: problem with gid send_recv()");
	
	/* ...and the last-access time... */
	if (send_recv(C->acc_sock,node.actime) < 0)
	  logger(SRV,"handle_fnode: problem with actime send_recv()");
	
	/* ...and the modification time... */
	if (send_recv(C->acc_sock,node.modtime) < 0)
	  logger(SRV,"handle_fnode: problem with modtime send_recv()");
	
	/* ...and the finish marker. */
	if (send_recv(C->acc_sock,node.finish) < 0)
	  logger(SRV,"handle_fnode: problem with finish send_recv()");
	
      }
      if (DEBUG) logger(SRV,"handle_fnode: Finished sending the node");

      /* Increase the file count */
      filecount++;

      /* Read the next command, find out what needs to be done with
         the thing now that we've read it and sent it to the
         client. */
      if (recv_send(C->acc_sock,cmd) < 0)
	logger(SRV,"handle_fnode: Couldn't read get/keep cmd");
      if (DEBUG) logger(SRV,"handle_fnode: Got cmd as %s",cmd);

      /* If it's a "get" command, then get the file. */
      if (if_get_cmd(cmd)) {
	if (DEBUG) logger(SRV,"handle_fnode: if_get_cmd is true");

	/* Open a port on the client to send over the file */
	if ((clisock = tcp_open(client_addr,NULL,CLI_TCP_PORT)) < 0)
	  if (DEBUG) logger(SRV,"handle_fnode: Client socket bad");
	
	/* Set a 40-minute timeout on sending any file */
	alarm(2400);
	if (DEBUG) logger(SRV,"handle_fnode: about to send file");
	if (sendfile(node.local,clisock) < 0)
	  logger(SRV,"handle_fnode: Bad sendfile operation");
	if (DEBUG) logger(SRV,"handle_fnode: sent the file");
	alarm(0);

	/* Close socket */
	close(clisock);

	/* Wait for the other side to acknowledge the file.  We had
           problems with this early-on, because over Ethernet, the
           file transfer goes by so quickly and the buffers are large
           enough to gobble up the data as soon as it gets on the
           wire.  However, over PPP (note: I highly DISCOURAGE using
           this in any kind of significant way over PPP!) the line
           speeds are significantly slower and the buffers never got a
           hold of important fnode data.  So we now have both server
           and client wait until everything is acknowledged before
           cramming more data down the wire. */
	logger(SRV,"handle_fnode: about to recv file-finish ACK");
	if (recv_send(C->acc_sock,fileACK) < 0) {
	  logger(SRV,"handle_fnode: Couldn't read file-finished ACK");
	} else {
	  if (DEBUG) logger(SRV,"handle_fnode: got file-finished ACK as %s",fileACK);
	}
	if (DEBUG) logger(SRV,"handle_fnode: done sending file");
      } /* if (if_get_cmd(cmd)) */
      if (DEBUG) logger(SRV,"handle_fnode: %d = filecount, Got cmd as %s",filecount,cmd);
      
    } /* while (!last_node(&node))  */
  } /* if (packfd != -1) */

  /* Now do something different if packfd == -1, because that means we
     couldn't open a package file, and the client needs to know about
     it. */
  else {
    if (send_recv(C->acc_sock,NONEXISTENT) < 0)
      logger(SRV,"handle_fnode: problem with filename send_recv()");
    if (send_recv(C->acc_sock,NONEXISTENT) < 0)
      logger(SRV,"handle_fnode: problem with link send_recv()");
  }

  /* Return the number of files handled in the transfer */
  if (DEBUG) logger(SRV,"handle_fnode: returning %d for filecount",filecount);
  return (filecount);
}

int last_package(char *packagename)
{
  /* If this is the last package, return 1, otherwise return 0 */
  if (strcmp(packagename,NULL_PACKAGE_NAME) == 0)
    return(1);
  else
    return(0);
}

int if_get_cmd(char *cmd)
{
  /* If this command is a "get" command, return 1, otherwise return 0 */
  if ((strcmp(cmd,GET_CMD) == 0))
    return(1);
  else
    return(0);
}

int if_pre_cmd(char *cmd)
{
  /* If the cmd is for a pre-script, return 1, otherwise 0 */
  if (strcmp(cmd,PRE_CMD) == 0)
    return(1);
  else
    return(0);
}

int if_post_cmd(char *cmd)
{
  /* If the cmd is for a post-script, return 1, otherwise 0 */
  if (strcmp(cmd,POST_CMD) == 0)
    return(1);
  else
    return(0);
}

int if_quit_cmd(char *cmd)
{
  /* If the cmd is to quit, return 1, otherwise 0 */
  if (strcmp(cmd,QUIT_CMD) == 0)
    return(1);
  else
    return(0);
}
int handle_pre(int clisock, conn *client)
{
  /* This function is designed to handle the request for a pre-script
     once the request has been made */
  int DEBUG = 1;
  int WRONG = 0;
  char packagename[MAX_PKG_NAME];
  char pkg[MAX_PKG_NAME];
  char scriptline[BIG_BUF];
  struct stat statbuf;
  char ready[SMALL_BUF];
  int scriptfd;
  char *IP;
  
  if (DEBUG) logger(SRV,"handle_pre: Starting.");
  if (recv_send(clisock,packagename) < 0) {
    logger(SRV,"handle_pre: problem recv packagename");
    WRONG = -1;
  }
  if (DEBUG) logger(SRV,"handle_pre: got %s as packagename",packagename);
  
  IP = malloc(SMALL_BUF);
  if (strcpy(IP,inet_ntoa(client->sockaddr.sin_addr)) == NULL) {
    logger(SRV,"handle_pre: Got some problems with determining address of remote.");
  }
  if (!is_secure(packagename,IP,SECURE)) {
    logger(SRV,"handle_pre: About to inform the client that they don't have rights to \"%s\"",packagename);
    if (send_recv(clisock,NO_PRE) < 0) {
      logger(SRV,"handle_pre: can't tell client that it can't have the pre-script.");
      WRONG = -1;
    }
    /* Jump out of here if there's nothing else to be done */
    return(WRONG);
  }
  
  /* Convert packagename to an actual script location */
  if (strcpy(pkg,PACKAGE_ROOT) == NULL) {
    logger(SRV,"handle_pre: Problem doing strcpy of PACKAGE_ROOT into pkg");
    WRONG = -1;
  }
  if (strcat(pkg,packagename) == NULL) {
    logger(SRV,"handle_pre: problem doing strcat of packagename onto pkg");
    WRONG = -1;
  }
  if (strcat(pkg,PRE_EXT) == NULL) {
    logger(SRV,"handle_pre: problem doing strcat of PRE_EXT onto pkg");
    WRONG = -1;
  }
  if (DEBUG) logger(SRV,"handle_pre: full script name is %s",pkg);
  /* Finished converting packagename to script location */

  /* stat() the completed package filename and let the client know if
     it doesn't exist. */
  if (stat(pkg,&statbuf) < 0) { /* Assuming errno = ENOENT, there is no prescript for this package */
    if (send_recv(clisock,NO_PRE) < 0) {
      logger(SRV,"handle_pre: can't tell client that the prescript doesn't exist.");
      WRONG = -1;
    }

    /* Jump out of here if there's nothing else to be done */
    return(WRONG);

    /* If stat() returns 0, then the file exists and we can send it over */
  } else { /* Assuming errno = 0, there is a prescript with this packagename */

    /* Acknowledge the pre-script's existence to the client */
    if (send_recv(clisock,ACK) < 0) {
      logger(SRV,"handle_pre: problem ACKnowledging pre-script.");
      WRONG = -1;
    }

    /* Let the client know if it should wait until the pre-script ends
       or just fork it off */
    if (if_waitfor(pkg)) {
      if (send_recv(clisock,WAIT_SCRIPT) < 0) {
	logger(SRV,"handle_pre: bad send_recv while sending WAIT_SCRIPT.");
	WRONG = -1;
      }
    } else {
      if (send_recv(clisock,NO_WAIT_SCRIPT) < 0) {
	logger(SRV,"handle_pre: bad send_recv while sending NO_WAIT_SCRIPT.");
	WRONG = -1;
      }
    }

    /* Open the pre-script read-only */
    if ((scriptfd = open(pkg,O_RDONLY)) < 0) {
      logger(SRV,"handle_pre: can't open pre-script. Sending END_DELIMITER");
      WRONG = -1;

      /* If scriptfd == -1, then just send down the end delimiter.
         This is non-executable inside a script, so it will have no
         effect */
      writeline(clisock,END_DELIMITER);
    }

    /* If scriptfd is greater than or equal to zero, we want to send
       the scriptfile down the wire. */
    if (!(scriptfd < 0)) {
      while (readstring(scriptfd,scriptline,BIG_BUF) > 0) {
	if (writeline(clisock,scriptline) < 0) {
	  logger(SRV,"handle_pre: can't send scriptline down");
	  WRONG = -1;
	}
      }
      writeline(clisock,END_DELIMITER);
      if (DEBUG) logger(SRV,"handle_pre: finished sending over the script");
    }
  } /* else [if (if_waitfor(pkg))] */
  
  /* Now return -1 if anything went wrong while doing this */
  if (DEBUG) logger(SRV,"handle_pre: Finishing.");
  return(WRONG);
}

int handle_post(int clisock, conn *client)
{
  /* This is intended to handle the requests for a post-script, should
     it be needed.  See handle_pre() for comments, works identically */
  int DEBUG = 1;
  int WRONG = 0;
  char packagename[MAX_PKG_NAME];
  char pkg[MAX_PKG_NAME];
  char scriptline[BIG_BUF];
  struct stat statbuf;
  char ready[SMALL_BUF];
  int scriptfd;
  char *IP;
  
  if (DEBUG) logger(SRV,"handle_post: Starting.");
  if (recv_send(clisock,packagename) < 0) {
    logger(SRV,"handle_post: problem recv packagename");
    WRONG = -1;
  }
  if (DEBUG) logger(SRV,"handle_post: got %s as packagename",packagename);

  IP = malloc(SMALL_BUF);
  if (strcpy(IP,inet_ntoa(client->sockaddr.sin_addr)) == NULL) {
    logger(SRV,"handle_post: Got some problems with determining address of remote.");
  }
  if (!is_secure(packagename,IP,SECURE)) {
    logger(SRV,"handle_post: About to inform the client that they don't have rights to \"%s\"",packagename);
    if (send_recv(clisock,NO_POST) < 0) {
      logger(SRV,"handle_post: can't tell client that it can't have the post-script.");
      WRONG = -1;
    }
    /* Jump out of here if there's nothing else to be done */
    return(WRONG);
  }

  if (strcpy(pkg,PACKAGE_ROOT) == NULL) {
    logger(SRV,"handle_post: Problem doing strcpy of PACKAGE_ROOT into pkg");
    WRONG = -1;
  }
  if (strcat(pkg,packagename) == NULL) {
    logger(SRV,"handle_post: problem doing strcat of packagename onto pkg");
    WRONG = -1;
  }
  if (strcat(pkg,POST_EXT) == NULL) {
    logger(SRV,"handle_post: problem doing strcat of POST_EXT onto pkg");
    WRONG = -1;
  }
  if (DEBUG) logger(SRV,"handle_post: full script name is %s",pkg);

  if (stat(pkg,&statbuf) == -1) {
    if (send_recv(clisock,NO_POST) < 0) {
      logger(SRV,"handle_post: can't tell client that the postscript doesn't exist.");
      WRONG = -1;
    }
    
    return(WRONG);
  } else {
    if (send_recv(clisock,ACK) < 0) {
      logger(SRV,"handle_post: bad ACK while getting read for post-script send.");
      WRONG = -1;
    }
    if (if_waitfor(pkg)) {
      if (send_recv(clisock,WAIT_SCRIPT) < 0) {
	logger(SRV,"handle_post: bad send_recv while sending WAIT_SCRIPT.");
	WRONG = -1;
      }
    } else {
      if (send_recv(clisock,NO_WAIT_SCRIPT) < 0) {
	logger(SRV,"handle_post: bad send_recv while sending NO_WAIT_SCRIPT.");
	WRONG = -1;
      }
    }
    if ((scriptfd = open(pkg,O_RDONLY)) < 0) {
      logger(SRV,"handle_post: can't open post-script for sending to client.");
      WRONG = -1;
      writeline(clisock,END_DELIMITER);
    }
    if (!(scriptfd < 0)) {
      while (readstring(scriptfd,scriptline,BIG_BUF) > 0) {
	if (writeline(clisock,scriptline) < 0) {
	  logger(SRV,"handle_post: can't send scriptline down");
	  WRONG = -1;
	}
      }
      writeline(clisock,END_DELIMITER);
      if (DEBUG) logger(SRV,"handle_post: finished sending over the script");
    }
  }
  if (DEBUG) logger(SRV,"handle_post: Finishing.");
  return(WRONG);
}


int if_waitfor(char *packagename) 
{
  /* This function returns 1 if the client is supposed to wait for it
     to complete, or 0 if the client can fork it off by itself */
  int DEBUG = 1;
  struct stat statbuf;
  char filename[MAX_PKG_NAME];
  
  if (DEBUG) logger(SRV,"if_waitfor: Starting");
  if (DEBUG) logger(SRV,"if_waitfor: got %s as argument",packagename);
  
  /* Create the filename that we want to test */
  if (strcpy(filename,packagename) == NULL) {
    logger(SRV,"if_waitfor: problem strcat'ing packagename to filename");
  }
  if (DEBUG) logger(SRV,"if_waitfor: %s is the filename so far",filename);
  if (strcat(filename,".NOWAIT") == NULL) {
    logger(SRV,"if_waitfor: problem strcat'ing \".NOWAIT\" to filename");
  }
  if (DEBUG) logger(SRV,"if_waitfor: %s is the filename so far",filename);
  /* Finished creating the filename */

  /* Now stat() the resulting filename... */
  if (DEBUG) logger(SRV,"if_waitfor: stat()'ing %s",filename);
  if (stat(filename,&statbuf) == 0) {
    if (DEBUG) logger(SRV,"if_waitfor: returning 0, client doesn't waitfor this script");
    return(0);
  } else {
    if (DEBUG) logger(SRV,"if_waitfor: returning 1, client must wait for this script");
    return(1);
  }
}

void printListData(char *data)  /* Will be called by display() in listlib.c */
{
        printf(" \"%x\" ", data);
}

void printListLabel(char *data)
{
  printf("List = \"%x\" ",data);
}
char *stripspace(char *data)
{
  int i=0,j=0;
  char *tmpdata;
  char *retdata;
  if ((tmpdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"stripspace: Bad malloc #1. Exiting");
    exit(0);
  }
  if ((retdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"stripspace: Bad malloc #2. Exiting");
    exit(0);
  }
  j=0;
  for (i=0;i<BUFFER;i++){
    if ((data[i] != ' ')){
      tmpdata[j] = data[i];
      j++;
    }
  }
  j=0;
  for (i=0;i<BUFFER;i++){
    if (tmpdata[i] != TAB) {
      retdata[j] = tmpdata[i];
      j++;
    }
  }
  return(retdata);
  
}

/*char *strip_PRE(char *data)
{
  return(strip_prefix(data,NO_PRE_PREFIX,sizeof(NO_PRE_PREFIX) - 1));
}
 
char *strip_POST(char *data)
{
  return(strip_prefix(data,NO_POST_PREFIX,sizeof(NO_POST_PREFIX) - 1));
}
*/
/*
char *strip_prefix(char *data, char *prefix, int length)
{
  int i=0,j=0,str;
  char *tmpdata;
  char *retdata;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"strip_prefix: args = %s, %s, %d",data,prefix,length);
  if ((retdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"strip_prefix: Bad malloc. Exiting");
    exit(0);
  }
  tmpdata = data;
  if (DEBUG) logger(CLI,"strip_prefix: Have %s for tmpdata",tmpdata);
  if ((str=strncmp(data,prefix,length)) == 0) {
    for (i=0;i<length;i++) {
      *tmpdata++;
    }
    if (DEBUG) logger(CLI,"strip_prefix: Now have %s for tmpdata",tmpdata);
    if (strcpy(retdata,tmpdata) == NULL) {
      logger(CLI,"strip_prefix: Bad strncpy!. Exiting");
      exit(0);
    }
    
  }
  if (DEBUG) logger(CLI,"strip_prefix: Finishing.");
  return(retdata);
}
*/
int inlist(list L,void *data)
{
  int retval=0;
  int DEBUG = 1;
  void *item;
  void *cmp;
  
  cmp = data;
  
  if (first(L) == NULL)
    return(retval);
  else {
    if (strcmp(first(L),cmp) == 0){
      if (DEBUG) logger(SRV,"inlist: found on first entry \"%s\"",cmp);
      return(1);
    }
    while ((item = next(L)) != NULL) {
      if (strcmp(item,cmp) == 0){
	if (DEBUG) logger(SRV,"inlist: Found it!  \"%s\"",cmp);
	return(1);
      }
    }
  }
  return(retval);
}

int is_comment(void *data)
{
  if (strncmp(data,"#",1) == 0)
    return(1);
  if (strncmp(data,"",1) == 0)
    return(1);
  else
    return(0);
}

int add_address(char *name, list L)
{
  int DEBUG = 0;
  struct hostent *hname;
  char **aliases;
  char **incr;
  char textaddress[INET_ADDRSTRLEN];
  char *newlocation[MAX_IP_COUNT];
  int i=0;
  /*  printf("ADD_ADDRESS:Got \"%s\" for hostname\n",name);*/
  
  if ((hname = gethostbyname(name)) == NULL) {
    logger(SRV,"add_address: gethostbyname returned NULL for \"%s\"",name);
    return(-1);
  }
  
  if (DEBUG) logger(SRV,"add_address: The canonical name of %s is %s",name,hname->h_name);
  for (aliases = hname->h_aliases; *aliases != NULL; aliases++)
    if (DEBUG) logger(SRV,"add_address: Got an alias of: %s",*aliases);
  for (incr = hname->h_addr_list;*incr != NULL;incr++) {
    if (DEBUG) logger(SRV,"add_address: Got an address of %s",inet_ntop(hname->h_addrtype,*incr,textaddress,INET_ADDRSTRLEN));
    newlocation[i] = malloc(INET_ADDRSTRLEN);
    if (strncpy(newlocation[i],inet_ntop(hname->h_addrtype,*incr,textaddress,INET_ADDRSTRLEN),INET_ADDRSTRLEN) == NULL) {
      logger(SRV,"add_address: Bad news for the strncpy!");
    }
    append(L,newlocation[i]);
    i++;
  }
  return(0);
}

const char *inet_ntop(int family, const void *addrptr, char *strptr, size_t len)
{
        const u_char *p = (const u_char *) addrptr;
	
        if (family == AF_INET) {
                char    temp[INET_ADDRSTRLEN];
		/*
                snprintf(temp, sizeof(temp), "%d.%d.%d.%d",
                                 (p[0]), p[1], (p[2]), p[3]);
				 */
		sprintf(temp,"%d.%d.%d.%d", p[0], p[1], p[2], p[3]);
                if (strlen(temp) >= len) {
                        errno = ENOSPC;
                        return (NULL);
                }
                strcpy(strptr, temp);
                return (strptr);
        }
        errno = EAFNOSUPPORT;
        return (NULL);
}

int make_pkg_secure(int fd, list L)
{
  int DEBUG = 0;
  char line[HUGE_BUF];
  char *tmppkgname,*tmphn;
  char *pkgname,*hn;
  int i;
  list floater;
  size_t length;
  
  while(readstring(fd,&line,HUGE_BUF) > 1){
    i = 0;
    length = strlen(line);
    tmppkgname=malloc(MAX_PKG_NAME);
    while((line[i] != SPACE)&&(line[i] != TAB)) {
      strncat(tmppkgname,&line[i],1);
      i++;
    }
    pkgname=strdup(tmppkgname);
    floater = lstcreate(pkgname);
    append(L,floater);
    i++;
    if (DEBUG) printf("Made \"%s\" as the floater list\n",pkgname);
    while (line[i] != EOL) {
      tmphn = malloc(MAX_HOSTNAME_LENGTH);
      while((line[i] != SPACE)&&(line[i] != TAB)) {
	if (DEBUG) printf("Got \"%s\" as hostname\n",tmphn);
	if (i >= length) break;
	strncat(tmphn,&line[i],1);
	i++;
      }
      hn=strdup(tmphn);
      if (DEBUG) printf("Final hostname is \"%s\"\n",hn);
      if (i > length) break;
      if (add_address(hn,floater) < 0)
	return(-1);
      i++;
    }
    display(floater);
  }
  return(0);
}

int is_secure(char *packagename,char *IP,list master)
{
  int DEBUG = 1;
  list lst;
  lst = first(master);
  while (strcmp((char*)getname(lst),packagename) != 0) {
    lst = next(master);
    if (lst == NULL) break;
  }
  if (lst != NULL) {
    if (inlist(lst,IP)) {
      if (DEBUG) logger(SRV,"is_secure: Determined that %s can access %s",IP,packagename);
      return(1);
    }
    else {
      logger(SRV,"is_secure: Attempt by %s to get %s",IP,packagename);
      return(0);
    }
  }
  return(1);
  
}
