/* This is part of the rfiledist package. */

/* Written by Tom Kunz <tkunz@fast.net> for the purpose of making a
system to distribute files/packages across a network, ***WAY*** more
powerful than the existing rdist(1) stuff. */

/* Copyright (C) 1999 by Tom Kunz */

/* GPL LICENSE */
/* This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

You can contact me by email at: tkunz@fast.net

You can contact me by snail mail at:
	Tom Kunz
	605 Harmony Brass Castle Rd
	Phillipsburg, NJ  08865

*/


/* This is client.c, the "client" end of rfiledist. */

/* Includes */

#include "header.h"
#include "io.h"
#include "node.h"
#include "comm.h"
#include "listlib.h"

/* Defines */

#define NO_PRE_PREFIX "NO_PRE"
#define NO_POST_PREFIX "NO_POST"
#define TAB 0x09
#define BUFFER 512 
#define PATH_LIM 6

/* The "ACK" character */

char *ACK = "A";

/* Function Prototypes used in this code */

int is_comment(void *data);
int is_NOPRE(void *data);
int is_NOPOST(void *data);
char *strip_prefix(char *data, char *prefix,int length);
char *strip_PRE(char *data);
char *strip_POST(char *data);
char *stripspace(char *data);
int inlist(list L,void *data);
void std_quit(char *);
int read_fnode(fnode *node, int fd);
int handle_regular(fnode *node, int fd, conn *C);
int handle_symlink(fnode *node, int fd);
int handle_hardlink(fnode *node, int fd);
int handle_dir(fnode *node, int fd);
int handle_blockdev(fnode *node, int fd);
int handle_chardev(fnode *node, int fd);
int handle_fifo(fnode *node, int fd);
int set_perms(fnode *node);
int make_link(fnode *node);
int req_pkg(int fd, conn *srv, char *packagename);
int req_pre(int fd, char *packagename);
int req_post(int fd, char *packagename);
int is_null_packagename(char *packagename);

/* The main() function.  Accepts a single parameter, which is the
   hostname or address of the server to connect to. */

main(int argc, char *argv[])
{
  int DEBUG = 1;
  int clisock;
  char *pkg;
  /*  int numfiles;*/
  fnode node;
  conn srv;
  int fd;
  char *data;
  list NO_PRE_LIST,NO_POST_LIST,packagelist;
  char *nospcdata, *noprefixdata;
  int i;
  char *configpath[PATH_LIM];
  struct stat statbuf;
  int opened = 0;
  uid_t this_uid;
  struct passwd *pw, *thisproc;
  char *self_uid;
  
  /* Inform the log file that we are starting now */
  logger(CLI,"main: Starting...");

  /* Set the umask to 0 and become the session leader */
  umask(0);
  setsid();
  
  /* Make all possible attempts to become root if we ain't already */
  thisproc = (struct passwd *)getpwuid(getuid());
  self_uid = strdup(thisproc->pw_name);
  pw = (struct passwd *)getpwnam("root");
  logger(CLI,"main: Got %s as my name, and %s as the root name",self_uid,pw->pw_name);
  if (strcmp(self_uid,"root") != 0) {
    logger(CLI,"main: Attempting to change ruid & euid to %d uid from %d",pw->pw_uid,getuid());
    if (setreuid(pw->pw_uid,pw->pw_uid) != 0) {
      logger(CLI,"main: Failed to set ruid & euid to %d.  You must run '%s' as uid = %d.  Exiting.",pw->pw_uid,argv[0],pw->pw_uid);
      exit(0);
    }
  }

  /* Define our possible config file locations */
  configpath[0] = "/usr/local/refdis/conf/refdis.conf";
  configpath[1] = "/etc/refdis.conf";
  configpath[2] = "/etc/refdis/refdis.conf";
  configpath[3] = "/usr/local/etc/refdis.conf";
  configpath[4] = "/usr/local/etc/refdis/refdis.conf";
  configpath[5] = "./refdis.conf";
  
  /* Create the lists */
  NO_PRE_LIST = lstcreate(NO_PRE_PREFIX);
  NO_POST_LIST = lstcreate(NO_POST_PREFIX);
  packagelist = lstcreate("package list");
  
  for (i = 0; i < PATH_LIM; i++) {
    if (!opened) {
      if (stat(configpath[i],&statbuf) == 0) {
	if (DEBUG) { 
	  logger(CLI,"main: Opening %s",configpath[i]);
	}
	
	/* Open the config file */
	if ((fd = open(configpath[i],O_RDONLY)) < 0) {
	  logger(CLI,"main: Failed open of %s. Now dying.",configpath[i]);
	  exit(0);
	}
	opened = 1;
      }

      /* Let the log file know if we couldn't open a certain one, but
         only if we're in DEBUG mode.  Otherwise, failure to open some
         of them is normal... */
      else {
	if (DEBUG) logger(CLI,"main: Failed to stat %s",configpath[i]);
	opened = 0;
      }
    } /* if (!opened) */
  } /* for (i = 0; i < PATH_LIM; i++) */
  
  /* Since we couldn't get any config file to work, die peacefully */
  if (!opened) {
    logger(CLI,"main: Failed to open a valid config file!");
    logger(CLI,"main: Please configure me properly.  The following is my search path:");
    for (i = 0; i < PATH_LIM; i++) {
      logger(CLI,"main: %s",configpath[i]);
    }
    logger(CLI,"main: Exiting");
    exit(0);
  }
  
  /* Malloc piece for the initial block that data is read into.  Data
     will be massaged and modified and put in other locations */

  if ((data = malloc(BUFFER)) == NULL){
    logger(CLI,"main: Bad malloc for data chunk.  Exiting");
    exit(0);
  }

  /* Build the linked list of all the data in the config file */

  while (readstring(fd,data,BUFFER) > 0){
    /* strip out spaces in the string */
    nospcdata = stripspace(data);
    if (!(is_comment(nospcdata))){
      if (DEBUG) logger(CLI,"main: Got actual data, no comment data");
      if (is_NOPRE(nospcdata)) {
	/* Strip out prefixes  */
	noprefixdata = strip_PRE(nospcdata);
	/* Throw away "spcdata" since it won't be used for anything now */
	free(nospcdata);
	if (DEBUG) logger(CLI,"main: Appending NOPRE data as \"%s\"",noprefixdata);
	/* We should just be appending a packagename at this point */
	append(NO_PRE_LIST,noprefixdata);
      }
      else if (is_NOPOST(nospcdata)) {
	/* Strip out prefixes  */
	noprefixdata = strip_POST(nospcdata);
	/* Throw away "spcdata" since it won't be used for anything now */
	free(nospcdata);
	if (DEBUG) logger(CLI,"main: Appending NOPOST data as \"%s\"",noprefixdata);
	/* We should just be appending a packagename at this point */
	append(NO_POST_LIST,noprefixdata);
      }
      else{
	if (DEBUG) logger(CLI,"main: Appending package data as \"%s\"",nospcdata);
	append(packagelist,nospcdata);
      }
    } /* !(is_comment(nospcdata)) */
    else
      if (DEBUG) logger(CLI,"main: Got comment as \"%s\"",nospcdata); 
  }
  /* Done doing stuff with the config file */
  close(fd);
  
  /* Now do the actual file transfers by initiating a connection */
  if (argc > 1) {
    if ((clisock = tcp_open(argv[1],NULL,SERV_TCP_PORT)) < 0) {
      logger(CLI,"main: tcp_open failed.  Dying peacefully");
      std_quit("main: tcp_open failed.  This is the second report. Bye.");
    }
  }
  else {
    /* Log this and exit */
    std_quit("main: Need hostname as the argument");
  }

  /* Get the listening socket ready */
  if (!(srv.lst_sock = make_sock(&srv,CLI_TCP_PORT))) {
    std_quit("main: make_sock failed");
  }
  /*  numfiles = 0;*/

  /* At this point, what I want to do is go throught the listing of
     packages.  First, get the name of a package from packagelist.
     Then check to see if it's in the NO_PRE list or NO_POST list.
     Then act accordingly by getting the pre-script, the package, and
     the post-script, as instructed in the config file.  */

  /* This is going to process the item in the package list */
  pkg = first(packagelist);
  if (pkg != NULL) {
    if (!inlist(NO_PRE_LIST,pkg)) {
      if (DEBUG) logger(CLI,"main: Getting PRE-SCRIPT for %s",pkg);
      if (req_pre(clisock,pkg) < 0) {
	if (DEBUG) logger(CLI,"main: req_pre failed for %s.",pkg);
      }
    }
    if (DEBUG) logger(CLI,"main: Getting main package for %s",pkg);
    if (req_pkg(clisock,&srv,pkg) < 0) {
      if (DEBUG) logger(CLI,"main: req_pkg problem with %s",pkg);
    }
    if (!inlist(NO_POST_LIST,pkg)){
      if (DEBUG) logger(CLI,"main: Getting POST-SCRIPT for %s",pkg);
      if (req_post(clisock,pkg) < 0) {
	if (DEBUG) logger(CLI,"main: req_post failed for %s.",pkg);	
      }
    }
  }
  if (DEBUG) logger(CLI,"Moving on to second package...");

  /* This is going to process the next items in the list */
  pkg = next(packagelist);
  while (pkg != NULL) {
    if (!inlist(NO_PRE_LIST,pkg)) {
      if (DEBUG) logger(CLI,"main: Getting PRE-SCRIPT for %s",pkg);
      if (req_pre(clisock,pkg) < 0) {
	logger(CLI,"main: req_pre failed for %s",pkg);
      }
    }
    if (DEBUG) logger(CLI,"main: Getting main package for %s",pkg);
    if (req_pkg(clisock,&srv,pkg) < 0) {
      logger(CLI,"main: req_pkg problem with %s",pkg);
    }
    if (!inlist(NO_POST_LIST,pkg)) {
      logger(CLI,"main: Getting POST-SCRIPT for %s",pkg);
      if (req_post(clisock,pkg) < 0) {
	logger(CLI,"main: req_post failed for %s",pkg);
      }
    }
    pkg = next(packagelist);
  }

  /* By this time, all of the packages have be obtained and everything
     should be ready for closing it all down. */
  if (req_pkg(clisock,&srv,NULL_PACKAGE_NAME) < 0) {
    logger(CLI,"main: send_recv() problem");
  }
  if (DEBUG) logger(CLI,"main: Sending quit cmd");
  if (send_recv(clisock,QUIT_CMD) < 0) {
    logger(CLI,"main: bad send_recv of quit cmd");
  }
  close(clisock);

  /* Display lists if debugging the code and get rid of the lists and
     terminate normally */
  if (DEBUG) display(packagelist);
  if (DEBUG) display(NO_PRE_LIST);
  if (DEBUG) display(NO_POST_LIST);
  lstdestroy(packagelist);
  lstdestroy(NO_PRE_LIST);
  lstdestroy(NO_POST_LIST);
  logger(CLI,"main: Done");
}

void std_quit(char *text)
{
  int i;
  /* Issuing the prescribed text to the logfile */
  logger(CLI,text);

  /* Closing all descriptors manually */
  for (i = 0; i < MAX_FILES; i++){
    close(i);
  }

  /* Turn out the lights when you go... */
  logger(CLI,"std_quit: Exiting");
  sync();
  exit(0);
}
    
ssize_t read_fnode(fnode *node,int fd)
{
  int DEBUG = 1;
  char tmp;
  ssize_t r[50],sum = 0;
  int lim = 10,i;
  int T = 15; /* number of seconds to timeout */
  
  for (i = 0;i < 50; i++)
    r[i] = 0;
  if (nullify_fnode(node) < 0) {
    logger(CLI,"read_fnode: Problem nullifying fnode!");
  }
  if ((r[0] = recv_send(fd,node->fn)) < 0)
    logger(CLI,"read_fnode: Error reading package file node->fn"); 
  if ((r[1] = recv_send(fd,node->link)) < 0)
    logger(CLI,"read_fnode: Error reading package file node->link");
  if (is_nonexistent(node)) {
    return(0);
  }
  if ((is_blockdev(node)) || (is_chardev(node)) ) {
    if ((r[2] = recv_send(fd,node->major)) < 0)
      logger(CLI,"read_fnode: Error reading package file node->major");
    if ((r[3] = recv_send(fd,node->minor)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[4] = recv_send(fd,node->mode)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[5] = recv_send(fd,node->uid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[6] = recv_send(fd,node->gid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[7] = recv_send(fd,node->actime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[8] = recv_send(fd,node->modtime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[9] = recv_send(fd,node->finish)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
  } else if (is_dir(node) || (is_fifo(node))) {
    if ((r[2] = recv_send(fd,node->mode)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[3] = recv_send(fd,node->uid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[4] = recv_send(fd,node->gid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[5] = recv_send(fd,node->actime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[6] = recv_send(fd,node->modtime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[7] = recv_send(fd,node->finish)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
  } else if (is_regular(node)) {
    if ((r[2] = recv_send(fd,node->sz)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[3] = recv_send(fd,node->mode)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[4] = recv_send(fd,node->uid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[5] = recv_send(fd,node->gid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[6] = recv_send(fd,node->actime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[7] = recv_send(fd,node->modtime)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[8] = recv_send(fd,node->finish)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
  } else if (is_symlink(node)) {
    if ((r[3] = recv_send(fd,node->uid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[4] = recv_send(fd,node->gid)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[5] = recv_send(fd,node->linkto)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[6] = recv_send(fd,node->finish)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
  } else if (is_hardlink(node)) {
    if ((r[2] = recv_send(fd,node->linkto)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
    if ((r[3] = recv_send(fd,node->finish)) < 0)
      logger(CLI,"read_fnode: Error reading package file");
  }
  if (DEBUG) logger(CLI,"read_fnode: full fnode");
  for (i = 0; i < lim; i++) {
    sum += r[i];
    if (r[i] < 0) {
      logger(CLI,"read_fnode: Got a negative return value");
      return(r[i]);
    }
  }
  return(sum);
}

int handle_regular(fnode *node,int clisock,conn *C)
{
  struct stat statbuf;
  struct sockaddr_in cli_addr;
  int clilen;
  int WRONG = 0;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"handle_regular: Got a REGULAR file");
  clilen = sizeof(C->sockaddr);
  if (conv_perms(node) < 0)
    logger(CLI,"handle_regular: Permission conversion problem");

  /* If it doesn't exist, get it */
  if (lstat(node->fn,&statbuf) < 0) {
    if (DEBUG) logger(CLI,"handle_regular: GETTING FILE - Sending G");
    if (send_recv(clisock,GET_CMD) < 0) {
      logger(CLI,"handle_regular: Problem sending Get or rec'ving ACK");
      WRONG = -1;
    }
    C->acc_sock = accept(C->lst_sock,(struct sockaddr *)&cli_addr,&clilen);
    if (recvfile(node->fn,C->acc_sock) < 0){
      logger(CLI,"handle_regular: Problem receiving file");
      WRONG = -1;
    }
    close(C->acc_sock);
    if (send_recv(clisock,ACK_CHAR) < 0) {
      logger(CLI,"handle_regular: Problem sending or rec'ving ACK");
      WRONG = -1;
    }
    else {
      if (DEBUG) logger(CLI,"handle_regular: Sent file-finish ACK as %s",ACK_CHAR);
    }
    
    if (set_perms(node) < 0)
      logger(CLI,"handle_regular: set_perms failed!");
    if (DEBUG) logger(CLI,"handle_regular: Done with file %s",node->fn);
  }

  /* Not same size, obviously different */
  else if (statbuf.st_size != node->real_sz) {
    if (DEBUG) logger(CLI,"handle_regular: Unlinking");
    if (unlink(node->fn) < 0)
      logger(CLI,"handle_regular: Problem removing %s",node->fn);
    if (DEBUG) logger(CLI,"handle_regular: FILE SIZE CHANGE - Sending G");
    if (send_recv(clisock,GET_CMD) < 0) {
      logger(CLI,"handle_regular: Problem sending Get or rec'ving ACK");
      WRONG = -1;
    }
    C->acc_sock = accept(C->lst_sock,(struct sockaddr *)&cli_addr,&clilen);
    if (recvfile(node->fn,C->acc_sock) < 0) {
      logger(CLI,"handle_regular: Problem receiving file");
      WRONG = -1;
    }
    
    close(C->acc_sock);
    if (send_recv(clisock,ACK_CHAR) < 0) {
      logger(CLI,"handle_regular: Problem sending Get or rec'ving ACK");
      WRONG = -1;
    }
    else {
      if (DEBUG) logger(CLI,"handle_regular: Sent file-finish ACK as %s",ACK_CHAR);
    }
    if (set_perms(node) < 0)
      logger(CLI,"handle_regular: set_perms failed!");
    if (DEBUG) logger(CLI,"handle_regular: Done with file %s",node->fn);
  } else if (statbuf.st_size == node->real_sz) { /* If size is the same... */
    if (statbuf.st_mode != node->real_mode ) {/* ...but mode if wrong... */
      if(set_perms(node)<0) {
	logger(CLI,"handle_regular: Problem changing mode via set_perms()");
      }
      WRONG = 1;
    }

    /* ...and/or uid/gid is wrong... */
    if ((statbuf.st_uid != node->real_uid) || (statbuf.st_gid != node->real_gid)) {
      if (DEBUG) logger(CLI,"handle_regular: Changing uid/gid");
      if (chown(node->fn,node->real_uid,node->real_gid) < 0){
	logger(CLI,"handle_regular: Couldn't chown %s to owner = %d and group = %d",node->fn,node->real_uid,node->real_gid);
      }
      WRONG = 1;
    }

    /* ...and/or timestamps are wrong. */
    if ((statbuf.st_atime != node->timebuf.actime) || (statbuf.st_mtime != node->timebuf.modtime)) {
      if (utime(node->fn,&node->timebuf) < 0)
	logger(CLI,"handle_regular: Error setting timestamp on %s to actime = %d and modtime = %d",node->fn,node->timebuf.actime,node->timebuf.modtime);
      WRONG = 1;
    }
    if (send_recv(clisock,KEEP_CMD) < 0) {
      logger(CLI,"handle_regular: Problem sending Get or rec'ving ACK");
      WRONG = -1;
    }
    if (DEBUG) logger(CLI,"handle_regular: Sending K");
  }
  if (DEBUG) logger(CLI,"handle_regular: Finishing handle_regular");
  return(WRONG);
}

int handle_symlink(fnode *node,int clisock)
{
  int WRONG = 0;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"handle_symlink: Got a SYMLINK");
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_symlink: Permission conversion problem");
    WRONG = -1;
  }
  if (make_link(node) < 0) {
    logger(CLI,"handle_symlink: make_link() for %s failed",node->fn);
    WRONG = -1;
  }
  if (set_perms(node) < 0) {
    logger(CLI,"handle_symlink: set_perms failed!");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_symlink: Done with file %s",node->fn);
  if (DEBUG) logger(CLI,"handle_symlink: Sending K");
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_symlink: Problem sending Get or rec'ving ACK");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_symlink: Finishing handle_symlink");
  return(WRONG);
}

int handle_hardlink(fnode *node,int clisock)
{
  int WRONG = 0;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"handle_hardlink: Got a HARDLINK");
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_hardlink: Permission conversion problem");
    WRONG = -1;
  }
  if (make_link(node) < 0) {
    logger(CLI,"handle_hardlink: make_link() for %s failed",node->fn);
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_hardlink: Done with file %s",node->fn);
  if (DEBUG) logger(CLI,"handle_hardlink: Sending K");
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_hardlink: Problem sending Get or rec'ving ACK");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_hardlink: Finishing handle_hardlink");
  return(WRONG);
}


int set_perms(fnode *node)
{
  int DEBUG = 1;
  errno = 0;

  /* All files but hardlinks can have the ownership changed meaningfully */
  if (DEBUG) logger(CLI,"set_perms: About to chown %s to %d/%d",node->fn,node->real_uid,node->real_gid);
  if (chown(node->fn,node->real_uid,node->real_gid) < 0) {
    logger(CLI,"set_perms: Problem changing %s to %d/%d",node->fn,node->real_uid,node->real_gid);
  }

  /* if the file is a symlink, we can return now.  Other types of files get timestamp and mode changes */
  if (is_symlink(node)) {
    if (errno != 0)
      return(-1);
    else
      return(0);
  }
  /* Regular files go on to have their time & mode changed */
  if (DEBUG) logger(CLI,"set_perms: chmodding %s to %o",node->fn,node->real_mode);
  if (chmod(node->fn,node->real_mode) < 0) {
    logger(CLI,"set_perms: Problem changing mode on %s to %o",node->fn,node->real_mode);
  }
  if (DEBUG) logger(CLI,"set_perms: Now changing %s to time %d and %d",node->fn,node->timebuf.actime,node->timebuf.modtime);
  if (utime(node->fn,&node->timebuf) < 0) {
    logger(CLI,"set_perms: Problem changing %s to actime = %d modtime = %d",node->fn,node->timebuf.actime,node->timebuf.modtime);
  }
  /* If errno got set by any of these items, then return -1.
     Otherwise, everything went ok (and it *should* be since this is
     supposed to be root!) */
  if (errno != 0)
    return(-1);
  else  
    return(0);
}

int make_link(fnode *node)
{
  int WRONG = 0;
  int DEBUG = 1;
  
  errno = 0;
  if (DEBUG) logger(CLI,"make_link: About to unlink %s",node->fn);
  if (unlink(node->fn) < 0) {
    logger(CLI,"make_link: failed to unlink pre-existing link");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"make_link: Did the unlink of %s",node->fn);
  if (is_symlink(node)) {
    WRONG = 1;
    if (symlink(node->linkto,node->fn) < 0) {
      logger(CLI,"make_link: failed to make symlink %s -> %s",node->linkto,node->fn);
      WRONG = -1;
    }
  } else {
    WRONG = 1;
    if (link(node->linkto,node->fn) < 0) {
      logger(CLI,"make_link: failed to make hardlink %s -> %s",node->linkto,node->fn);
      WRONG = -1;
    }
  }
  return(WRONG);
}

int req_pkg(int clisock, conn *srv, char *packagename)
{
  int DEBUG = 1;
  int WRONG = 0;
  int numfiles = 0;
  fnode node;

  if (nullify_fnode(&node) < 0) {
    logger(CLI,"req_pkg: bad nullify of the fnode");
  }
  if (DEBUG) logger(CLI,"req_pkg: Starting");
  logger(CLI,"req_pkg: acquiring package %s",packagename);
  
  if (is_null_packagename(packagename)) {
    if (DEBUG) logger(CLI,"req_pkg: is_null_packagename");
    if (send_recv(clisock,NULL_PACKAGE_NAME) < 0) {
      logger(CLI,"req_pkg: send_recv of quit cmd failed while finishing package");
      WRONG = -1;
    }
    return(WRONG);
  } else if (send_recv(clisock,GET_CMD) < 0) {
    logger(CLI,"req_pkg: send_recv of get cmd failed");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pkg: about to send packagename as %s",packagename);
  if (send_recv(clisock,packagename) < 0) {
    logger(CLI,"req_pkg: send_recv of packagename failed");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pkg: about to test for last_node");
  while (!last_node(&node)) {
    if (DEBUG) logger(CLI,"req_pkg: last_node := not last");
    numfiles++;
    if (DEBUG) logger(CLI,"req_pkg: Starting read_fnode for file # %d",numfiles);
    
    if (read_fnode(&node,clisock) < 0) {
      logger(CLI,"req_pkg: Something went wrong while reading fnode");
      WRONG = -1;
    }
    if (DEBUG) logger(CLI,"req_pkg: Done with read_fnode for file # %d",numfiles);
    if (is_nonexistent(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: Nonexistent package name");
      if (DEBUG) logger(CLI,"req_pkg: Created 0 files for package %s",packagename);
      return(0);
    }
    if (is_regular(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: REGULAR");
      if (handle_regular(&node,clisock,srv) < 0) {
	logger(CLI,"req_pkg: handle_regular problem");
	WRONG = -1;
      }
    } else if (is_symlink(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: SYMLINK");
      if (handle_symlink(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_symlink problem");
	WRONG = -1;
      } 
    } else if (is_hardlink(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: HARDLINK");
      if (handle_hardlink(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_hardlink problem");
	WRONG = -1;
      }
    } else if (is_dir(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: DIRECTORY");
      if (handle_dir(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_dir problem");
	WRONG = -1;
      }
    } else if (is_blockdev(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: BLOCKDEV");
      if (handle_blockdev(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_blockdev problem");
	WRONG = -1;
      }
      
    } else if (is_chardev(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: CHARDEV");
      if (handle_chardev(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_chardev problem");
	WRONG = -1;
      }
    } else if (is_fifo(&node)) {
      if (DEBUG) logger(CLI,"req_pkg: FIFO");
      if (handle_fifo(&node,clisock) < 0) {
	logger(CLI,"req_pkg: handle_fifo problem");
	WRONG = -1;
      }
    }
  }
  logger(CLI,"req_pkg: %d files in package %s",numfiles,packagename);
  logger(CLI,"req_pkg: Finished package %s",packagename);
  return(WRONG);
}

int handle_dir(fnode *node,int clisock)
{
  int WRONG = 0;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"handle_dir: Got a DIRECTORY");
  
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_dir: Permission conversion problem");
    WRONG = -1;
  }
  if (opendir(node->fn) == NULL) {
    if (mkdir(node->fn,node->real_mode) < 0)
      logger(CLI,"handle_dir: mkdir() failed");
    if (set_perms(node) < 0)
      logger(CLI,"handle_dir: set_perms() failed");
  }
  else {
    if (set_perms(node) < 0)
      logger(CLI,"handle_dir: set_perms() failed");
  }
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_dir: Problem sending Get or rec'ving ACK");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_dir: Finishing handle_dir");
  return(WRONG);
}

int handle_blockdev(fnode *node,int clisock)
{
  int DEBUG = 1;
  int WRONG = 0;
  struct stat statbuf;
  
  if (DEBUG) logger(CLI,"handle_blockdev: Got a BLOCK DEVICE");
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_blockdev: Permission conversion problem");
    WRONG = -1;
  }
  if (stat(node->fn,&statbuf) == 0) {
    if (unlink(node->fn) < 0) {
      logger(CLI,"handle_blockdev: Removal of old link failed");
      WRONG = -1;
    }
  }
  
  if (mknod(node->fn,S_IFBLK,node->real_dev) < 0){
    logger(CLI,"handle_blockdev: mknod() failed");
    WRONG = -1;
  }
  if (set_perms(node) < 0)
    logger(CLI,"handle_blockdev: set_perms() failed");
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_blockdev: Problem sending Get or rec'ving ACK");
    WRONG = -1;
  }

  if (DEBUG) logger(CLI,"handle_blockdev: Finishing handle_blockdev");
  return(WRONG);
  
}

int handle_chardev(fnode *node,int clisock)
{
  int DEBUG = 1;
  int WRONG = 0;
  struct stat statbuf;
  
  if (DEBUG) logger(CLI,"handle_chardev: Got a CHARACTER DEVICE");
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_chardev: Permission conversion problem");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_chardev: Removing character device %s",node->fn);
  if (stat(node->fn,&statbuf) == 0) {
    if (unlink(node->fn) < 0) {
      logger(CLI,"handle_chardev: Removal of character device %s failed",node->fn);
      WRONG = -1;
    }
    if (DEBUG) logger(CLI,"handle_chardev: Removed.  About to mknod it.");
  }
  if (mknod(node->fn,S_IFCHR,node->real_dev) < 0) {
    logger(CLI,"handle_chardev: mknod() failed");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_chardev: About to set_perms on %s",node->fn);
  
  if (set_perms(node) < 0) {
    logger(CLI,"handle_chardev: set_perms() failed");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_chardev: About to send KEEP_CMD");
  
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_chardev: Problem sending Get or rec'ving ACK");
    WRONG = -1;
  }
  
  if (DEBUG) logger(CLI,"handle_chardev: Finished with handle_chardev");
  return(WRONG);
}

int handle_fifo(fnode *node,int clisock)
{
  int DEBUG = 1;
  int WRONG = 0;
  struct stat statbuf;
  
  if (DEBUG) logger(CLI,"handle_fifo: Got a FIFO");
  if (conv_perms(node) < 0) {
    logger(CLI,"handle_fifo: Permission conversion problem");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_fifo: Unlinking and making!");
  if (stat(node->fn,&statbuf) == 0) {
    if (unlink(node->fn) < 0) {
      logger(CLI,"handle_fifo: Removal of fifo %s failed",node->fn);
      WRONG = -1;
    }
  }
  if (mkfifo(node->fn,node->real_mode) < 0) {
    logger(CLI,"handle_fifo: mknod() failed");
    WRONG = -1;
  }
  if (set_perms(node) < 0) {
    logger(CLI,"handle_fifo: set_perms() failed");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_fifo: After the creation");
  
  if (send_recv(clisock,KEEP_CMD) < 0) {
    logger(CLI,"handle_fifo: Problem sending CMD or rec'ving ACK");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"handle_fifo: Finishing handle_fifo");
  return(WRONG);
}

int is_null_packagename(char *packagename)
{
  if (strcmp(packagename,NULL_PACKAGE_NAME) == 0)
    return(1);
  else
    return(0);
}

int req_pre(int clisock, char *packagename)
{
  int DEBUG = 1;
  int WRONG = 0;
  char scriptline[BIG_BUF];
  char prestat[SMALL_BUF];
  char waitforscript[SMALL_BUF];
  char localscript[MAX_PKG_NAME];
  int scriptfd;
  pid_t childpid,grandchildpid;
  
  if (DEBUG) logger(CLI,"req_pre: Sending PRE_CMD");
  if (send_recv(clisock,PRE_CMD) < 0) {
    logger(CLI,"req_pkg: problem asking for PRE_CMD");
    WRONG = -1;
  }
  /* server now does handle_pre(), upon receipt of the above */
  if (DEBUG) logger(CLI,"req_pre: sending packagename as %s",packagename);
  if (send_recv(clisock,packagename) < 0) {
    logger(CLI,"req_pre: problem with packagename");
    WRONG = -1;
  }
  if (recv_send(clisock,prestat) < 0) {
    logger(CLI,"req_pre: problem asking for status of prescript (exist or not)");
    WRONG = -1;
  }
  if (strcmp(prestat,NO_PRE) == 0) {
    if (DEBUG) logger(CLI,"req_pre: No pre-script available");
    return(WRONG);
  } else if (strcmp(prestat,ACK_CHAR) == 0) {
    if (DEBUG) logger(CLI,"req_pre: Pre-script available on server");
  } 
  if (DEBUG) logger(CLI,"req_pre: about to get waitforscript value");
 
  if (recv_send(clisock,waitforscript) < 0) {
    logger(CLI,"req_pre: problem getting waitforscript");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pre: got waitforscript value as %s",waitforscript); 
  if (DEBUG) logger(CLI,"req_pre: creating localscript name");
 
  if (strcpy(localscript,TEMP_DIR) == NULL) {
    logger(CLI,"req_pre: bad strcpy of TEMP_DIR to localscript");
    WRONG = -1;
  }
  if (strcat(localscript,packagename) == NULL) {
    logger(CLI,"req_pre: bad strcat of packagename onto localscript");
    WRONG = -1;
  }
  if (strcat(localscript,PRE_EXT) == NULL) {
    logger(CLI,"req_pre: bad strcat of PRE_EXT onto localscript");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pre: finished creating localscript name");

  if ((scriptfd = creat(localscript,S_IRWXU)) < 0) {
    logger(CLI,"req_pre: can't open localscript for dumping into");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pre: %d is the descriptor for the localscript",scriptfd);
  /* Get the script via the clisock line */
  while (readline(clisock,scriptline,BIG_BUF) > 0) {
    /*    if (DEBUG) logger(CLI,"req_pre: READ: %s",scriptline);*/
    if (strncmp(scriptline,END_DELIMITER,sizeof(END_DELIMITER - 1)) == 0) {
      /*      if (DEBUG) logger(CLI,"req_pre: got a match on the END_DELIMITER");*/
      break;
    }
    if (writestring(scriptfd,scriptline) < 0) {
      logger(CLI,"req_pre: can't write scriptline into localscript");
      WRONG = -1;
    }
    /*    if (DEBUG) logger(CLI,"req_pre: WROTE: %s",scriptline);*/
  }
  /* Done getting script.  Now it closes. */
  if (close(scriptfd) < 0) {
    logger(CLI,"req_pre: problem with closing.  Not good");
    WRONG = -1;
  }
  /* Do the script, based on the waitforscript value */
  if (strcmp(waitforscript,WAIT_SCRIPT) == 0) {
    if (DEBUG) logger(CLI,"req_pre: about to system() the pre-script");
    system(localscript);
    if (unlink(localscript) < 0) {
      logger(CLI,"req_pre: failed to unlink %s",localscript);
    }
  }
  else if (strcmp(waitforscript,NO_WAIT_SCRIPT) == 0) {
    if (DEBUG) logger(CLI,"req_pre: about to do 2-fork trick and not wait for the script");
    if ((childpid = fork()) < 0) { /* error on first fork */
      std_quit("req_pre: error with first fork.  exiting");
    }
    if (childpid > 0) {
      if (DEBUG) logger(CLI,"req_pre: Inside parent");
    }
    else if (childpid == 0) {
      if ((grandchildpid = fork()) < 0) { /* error on second fork */
	std_quit("req_pre: error with second fork");
      }
      else { /* no error on second fork */
	if ((childpid == 0) && (grandchildpid != 0)) {
	  return;
	}
	if (grandchildpid == 0) {
	  if (DEBUG) logger(CLI,"req_pre: doing the script in no-wait mode");
	  system(localscript);
	  if (unlink(localscript) < 0) {
	    logger(CLI,"req_pre: failed to unlink %s",localscript);
	  }
	  if (DEBUG) logger(CLI,"req_pre: finished the script in no-wait mode");
	  exit(0);
	}
      } /* no error on second fork */
    } /* no error on child.  */
  }
  else {
    if (DEBUG) logger(CLI,"req_pre: I have no idea whether or not to wait for the script, so I'm not doing the script at all!");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_pre: Finishing, with return value as %d", WRONG);
  return(WRONG);
}

int req_post(int clisock, char *packagename)
{
  int DEBUG = 1;
  int WRONG = 0;
  char scriptline[BIG_BUF];
  char poststat[SMALL_BUF];
  char waitforscript[SMALL_BUF];
  char localscript[MAX_PKG_NAME];
  int scriptfd;
  pid_t childpid,grandchildpid;
  
  if (DEBUG) logger(CLI,"req_post: Sending POST_CMD");
  if (send_recv(clisock,POST_CMD) < 0) {
    logger(CLI,"req_pkg: problem asking for POST_CMD");
    WRONG = -1;
  }
  /* server now does handle_post(), upon receipt of the above */
  if (DEBUG) logger(CLI,"req_post: sending packagename as %s",packagename);
  if (send_recv(clisock,packagename) < 0) {
    logger(CLI,"req_post: problem with packagename");
    WRONG = -1;
  }
  if (recv_send(clisock,poststat) < 0) {
    logger(CLI,"req_post: problem asking for status of postscript (exist or not)");
    WRONG = -1;
  }
  if (strcmp(poststat,NO_POST) == 0) {
    if (DEBUG) logger(CLI,"req_post: No post-script available");
    return(WRONG);
  } else if (strcmp(poststat,ACK_CHAR) == 0) {
    if (DEBUG) logger(CLI,"req_post: Post-script available on server");
  } 
  if (DEBUG) logger(CLI,"req_post: about to get waitforscript value");
 
  if (recv_send(clisock,waitforscript) < 0) {
    logger(CLI,"req_post: problem getting waitforscript");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_post: got waitforscript value as %s",waitforscript); 
  if (DEBUG) logger(CLI,"req_post: creating localscript name");
 
  if (strcpy(localscript,TEMP_DIR) == NULL) {
    logger(CLI,"req_post: bad strcpy of TEMP_DIR to localscript");
    WRONG = -1;
  }
  if (strcat(localscript,packagename) == NULL) {
    logger(CLI,"req_post: bad strcat of packagename onto localscript");
    WRONG = -1;
  }
  if (strcat(localscript,POST_EXT) == NULL) {
    logger(CLI,"req_post: bad strcat of POST_EXT onto localscript");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_post: finished creating localscript name");

  if ((scriptfd = creat(localscript,S_IRWXU)) < 0) {
    logger(CLI,"req_post: can't open localscript for dumping into");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_post: %d is the descriptor for the localscript",scriptfd);
  /* Get the script via the clisock line */
  while (readline(clisock,scriptline,BIG_BUF) > 0) {
    if (strncmp(scriptline,END_DELIMITER,sizeof(END_DELIMITER - 1)) == 0) {
      if (DEBUG) logger(CLI,"req_post: got a match on the END_DELIMITER");
      break;
    }
    if (writestring(scriptfd,scriptline) < 0) {
      logger(CLI,"req_post: can't write scriptline into localscript");
      WRONG = -1;
    }
    /*    if (DEBUG) logger(CLI,"req_post: WROTE: %s",scriptline);*/
  }
  /* Done getting script.  Now it closes. */
  if (close(scriptfd) < 0) {
    logger(CLI,"req_post: problem with closing.  Not good");
    WRONG = -1;
  }
  /* Do the script, based on the waitforscript value */
  if (strcmp(waitforscript,WAIT_SCRIPT) == 0) {
    if (DEBUG) logger(CLI,"req_post: about to system() the post-script");
    system(localscript);
    if (unlink(localscript) < 0) {
      logger(CLI,"req_post: failed to unlink %s",localscript);
    }
  }
  else if (strcmp(waitforscript,NO_WAIT_SCRIPT) == 0) {
    if (DEBUG) logger(CLI,"req_post: about to do 2-fork trick and not wait for the script");
    if ((childpid = fork()) < 0) { /* error on first fork */
      std_quit("req_post: error with first fork.  exiting");
    }
    if (childpid > 0) {
      if (DEBUG) logger(CLI,"req_post: Inside parent");
    }
    else if (childpid == 0) {
      if ((grandchildpid = fork()) < 0) { /* error on second fork */
	std_quit("req_post: error with second fork");
      }
      else { /* no error on second fork */
	if ((childpid == 0) && (grandchildpid != 0)) {
	  return;
	}
	if (grandchildpid == 0) {
	  if (DEBUG) logger(CLI,"req_post: doing the script in no-wait mode");
	  system(localscript);
	  if (unlink(localscript) < 0) {
	    logger(CLI,"req_post: failed to unlink %s",localscript);
	  }
	  if (DEBUG) logger(CLI,"req_post: finished the script in no-wait mode");
	  exit(0);
	}
      } /* no error on second fork */
    } /* no error on child.  */
  }
  
  else {
    if (DEBUG) logger(CLI,"req_post: I have no idea whether or not to wait for the script, so I'm not doing the script at all!");
    WRONG = -1;
  }
  if (DEBUG) logger(CLI,"req_post: Finishing, with return value as %d", WRONG);
  return(WRONG);
}

void printListData(char *data)  /* Will be called by display() in listlib.c */
{
        printf(" \"%s\" ", data);
}

void printListLabel(char *data)
{
  printf("List = \"%s\" ",data);
}

char *stripspace(char *data)
{
  int i=0,j=0;
  char *tmpdata;
  char *retdata;
  if ((tmpdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"stripspace: Bad malloc #1. Exiting");
    exit(0);
  }
  if ((retdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"stripspace: Bad malloc #2. Exiting");
    exit(0);
  }
  j=0;
  for (i=0;i<BUFFER;i++){
    if ((data[i] != ' ')){
      tmpdata[j] = data[i];
      j++;
    }
  }
  j=0;
  for (i=0;i<BUFFER;i++){
    if (tmpdata[i] != TAB) {
      retdata[j] = tmpdata[i];
      j++;
    }
  }
  return(retdata);
  
}

char *strip_PRE(char *data)
{
  return(strip_prefix(data,NO_PRE_PREFIX,sizeof(NO_PRE_PREFIX) - 1));
}
 
char *strip_POST(char *data)
{
  return(strip_prefix(data,NO_POST_PREFIX,sizeof(NO_POST_PREFIX) - 1));
}


char *strip_prefix(char *data, char *prefix, int length)
{
  int i=0,j=0,str;
  char *tmpdata;
  char *retdata;
  int DEBUG = 1;
  
  if (DEBUG) logger(CLI,"strip_prefix: args = %s, %s, %d",data,prefix,length);
  if ((retdata = malloc(BUFFER)) == NULL) {
    logger(CLI,"strip_prefix: Bad malloc. Exiting");
    exit(0);
  }
  tmpdata = data;
  if (DEBUG) logger(CLI,"strip_prefix: Have %s for tmpdata",tmpdata);
  if ((str=strncmp(data,prefix,length)) == 0) {
    for (i=0;i<length;i++) {
      *tmpdata++;
    }
    if (DEBUG) logger(CLI,"strip_prefix: Now have %s for tmpdata",tmpdata);
    if (strcpy(retdata,tmpdata) == NULL) {
      logger(CLI,"strip_prefix: Bad strncpy!. Exiting");
      exit(0);
    }
    
  }
  if (DEBUG) logger(CLI,"strip_prefix: Finishing.");
  return(retdata);
  
}

int inlist(list L,void *data)
{
  int retval=0;
  int DEBUG = 1;
  void *item;
  void *cmp;
  
  cmp = data;
  
  if (first(L) == NULL)
    return(retval);
  else {
    if (strcmp(first(L),cmp) == 0){
      if (DEBUG) logger(CLI,"inlist: found on first entry \"%s\"",cmp);
      return(1);
    }
    while ((item = next(L)) != NULL) {
      if (strcmp(item,cmp) == 0){
	if (DEBUG) logger(CLI,"inlist: Found it!  \"%s\"",cmp);
	return(1);
      }
    }
  }
  return(retval);
}

int is_comment(void *data)
{
  if (strncmp(data,"#",1) == 0)
    return(1);
  if (strncmp(data,"",1) == 0)
    return(1);
  else
    return(0);
}

int is_NOPRE(void *data)
{
  if (strncmp(data,NO_PRE_PREFIX,sizeof(NO_PRE_PREFIX) - 1) == 0)
    return(1);
  else
    return(0);
}

int is_NOPOST(void *data)
{
  if (strncmp(data,NO_POST_PREFIX,sizeof(NO_POST_PREFIX) - 1) == 0)
    return(1);
  else
    return(0);
}
