/***************************************************************************
 *
 * Copyright (c) 1998 Balazs Scheidler
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

/* interface definitions, and interface required system headers */
#include "var.h"

/* system headers */
#include <stdio.h>
#include <assert.h>
#include <sys/syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

/* local headers */
#include "nsyslogd.h"
#include "ftable.h"
#include "hash.h"
#include "utils.h"
#include "htcp.h"
#include "cfgfile.h"
#include "y.tab.h"
#include "log.h"

#if !defined(lint)
static const char rcsid[] = "$Id: var.c,v 1.28 1998/12/01 18:10:00 bazsi Exp $";
#endif

#ifndef USE_POSIX_REGEX
#define	INIT		register char *sp = instring;
#define	GETC()		(*sp++)
#define	PEEKC()		(*sp)
#define	UNGETC(c)	(--sp)
#define	RETURN(c)	return;
#define	ERROR(c)	regerr(c)

#include <regexp.h>
#else
#include <regex.h>
#endif

typedef	struct	table	{
	char *name;
	int value;
} table_t;

var_t		*vlist = NULL;
filespec_t	*fslist = NULL;
unixspec_t	*xslist = NULL;
cmdspec_t	*cslist = NULL;
userspec_t	*uslist = NULL;
streamspec_t	*sslist = NULL;
doorspec_t	*dslist = NULL;
matchspec_t	*mslist = NULL;
matchfilespec_t	*mfslist = NULL;
matchprogspec_t	*mpslist = NULL;
netspec_t	*nslist = NULL;
facspec_t	*faclist = NULL;
prispec_t	*prilist = NULL;
facprispec_t	*fplist = NULL;
nmaskspec_t	*nmlist = NULL;
sdlist_t	*sdlist = NULL, *sdtail = NULL;

static	var_t		*current = NULL;
static	filespec_t	*cur_file = NULL;
static	int	do_raw = 0, do_hash = 0;
static	int	not_match = 0, save_not;

static	char 	*comp_toname __P((int));


void
set_not(val)
	int val;
{
	save_not = not_match;
	not_match = val;
}


int
get_not()
{
	int ret = not_match;

	not_match = save_not;
	return ret;
}


/*
 * set the do_raw flag so that the next network socket created picks it
 * up.
 */
void
set_raw()
{
	do_raw = VAR_RAW;
}

void
set_hashed()
{
	do_hash = 1;
}

/*
 * set a global option.
 */
void
set_option(opt, parm)
	int opt;
	char *parm;
{
	if (opt == IL_MARK)
		options.op_markfreq = strtol(parm, NULL, 0);
	else if (opt == IL_SYNC)
		options.op_syncfreq = strtol(parm, NULL, 0);
	else if (opt == IL_LOGDIR) {
		if (options.op_logdir && (options.op_logdir != NSYSLOG_LOGDIR))
			free(options.op_logdir);
		options.op_logdir = parm;
	} else if (opt == IL_DIRECTORY) {
		if (options.op_sysdir && (options.op_sysdir != NSYSLOG_DIR))
			free(options.op_sysdir);
		options.op_sysdir = parm;
	} else if (opt == IL_CHROOT) {
		if (options.op_chroot && strcmp(options.op_chroot, parm))
			free(options.op_chroot);
		options.op_chroot = parm;
	}
}

/*
 * set a hash option.
 */
void
set_hoption(opt, parm)
	int opt;
	char *parm;
{
	hashtype_t *ht;

	if (opt == IL_HASHALGO) {
		ht = hash_find(parm);
		if (ht) {
			options.op_hashtype = ht->ht_name;
			options.op_hashext = ht->ht_ext;
		} else
			log_printf(LOG_SYSLOG | LOG_ERR, "unknown hash algorithm %s", parm);
	} else if (opt == IL_DIRECTORY) {
		if (options.op_hashdir)
			free(options.op_hashdir);
		options.op_hashdir = parm;
	}
}

/*
 * get a pointer to a named variable.
 */
var_t *
get_var(name)
	char *name;
{
	var_t *v;

	for (v = vlist; v; v = v->v_next)
		if (!strcmp(v->v_name, name))
			return v;
	return NULL;
}


/*
 * create a new named variable and return its type.  if one already exists
 * with that name, return its type anyway.
 */
int
new_var(name, type)
char *name;
	int type;
{
	var_t *v;

	if ((v = get_var(name)) != 0L)
		return v->v_type;
	if (!(NEW(v)))
		return -1;
	v->v_name = (char *)strdup(name);
	v->v_list = new_filetype(type);
	v->v_type = type;
	v->v_next = vlist;
	vlist = v;
	return 0;
}


/*
 * create a new anonymous variable.  An anonymous variable isn't defined in
 * the configuration file and is an artifact of creating something without
 * a name.
 */
var_t *
new_anon(type)
	int type;
{
	static int anonvars = 0;
	char sbuf[20];
	var_t *v;

	if (!(NEW(v)))
		return NULL;
	snprintf(sbuf, sizeof(sbuf), "<%x>", anonvars++);
	v->v_type = type;
	v->v_name = (char *)strdup(sbuf);
	v->v_list = new_filetype(type);
	v->v_next = vlist;
	vlist = v;
	return v;
}


/*
 * set the working object to a known variable.
 */
void
new_obj(name)
	char *name;
{
	var_t *v;

	v = get_var(name);
	if (v) 
		NEW(v->v_list);
	current = v;
}


/*
 * associate an internal file type structure with a known variable.
 */
var_t *
add_obj(type, ptr)
	int type;
	void *ptr;
{
	var_t *v;

	assert(ptr);

	if (VAR_TYPE(type) == SRC_INET)
		type = ((netspec_t *)ptr)->ns_type;
	if (!current)
		switch (VAR_TYPE(type))
		{
		case DEST_FILE : case DEST_UNIX : case DEST_INET :
		case DEST_USER : case DEST_CMD:
			current = new_anon(VAR_DST);
			break;
		case SRC_FILE : case SRC_UNIX : case SRC_INET :
		case SRC_SUN_STREAM : case SRC_SUN_DOOR :
			current = new_anon(VAR_SRC);
			break;
		case FIL_FAC : case FIL_PRI : case FIL_FACPRI :
		case FIL_WRAP : case FIL_MATCH : case FIL_CMD :
		case FIL_MATCHFILE : case FIL_MATCHPROG :
			current = new_anon(VAR_FIL);
			break;
		default:
			return NULL;
			break;
		}

	if (!(v = current))
		return NULL;

	if (v->v_num > 0) {
		v->v_num++;
		v->v_list = (filetype_t *)realloc(v->v_list,
						sizeof(*v->v_list) * v->v_num);
	} else {
		v->v_num = 1;
		if (!(NEW(v->v_list))) {
			free(v);
			return NULL;
		}
	}
	memset(&v->v_list[v->v_num - 1], 0, sizeof(v->v_list[0]));
	v->v_list[v->v_num - 1].ft_owner = v;
	v->v_list[v->v_num - 1].ft_ndx = v->v_num - 1;
	v->v_list[v->v_num - 1].ft_ptr = ptr;
	v->v_list[v->v_num - 1].ft_type = type;
	return v;
}


/*
 * stop working with a variable.
 */
var_t *
end_obj()
{
	var_t *v;

	v = current;
	current = NULL;
	return v;
}


table_t	facs[] = {
	{ "kern", LOG_KERN },		{ "user", LOG_USER },
	{ "mail", LOG_MAIL },		{ "daemon", LOG_DAEMON },
	{ "auth", LOG_AUTH },		{ "syslog", LOG_SYSLOG },
	{ "lpr", LOG_LPR },		{ "news", LOG_NEWS },
	{ "uucp", LOG_UUCP },		{ "cron", LOG_CRON },
#ifdef LOG_FTP
	{ "ftp", LOG_FTP },
#endif
#ifdef LOG_AUTHPRIV
	{ "authpriv", LOG_AUTHPRIV },
#endif
	{ "local0", LOG_LOCAL0 },	{ "local1", LOG_LOCAL1 },
	{ "local2", LOG_LOCAL2 },	{ "local3", LOG_LOCAL3 },
	{ "local4", LOG_LOCAL4 },	{ "local5", LOG_LOCAL5 },
	{ "local6", LOG_LOCAL6 },	{ "local7", LOG_LOCAL7 },
	{ NULL, 0 }
};

table_t	pris[] = {
	{ "emerg", LOG_EMERG },		{ "alert", LOG_ALERT },
	{ "crit", LOG_CRIT },		{ "err", LOG_ERR },
	{ "warn", LOG_WARNING },	{ "notice", LOG_NOTICE },
	{ "info", LOG_INFO },		{ "debug", LOG_DEBUG },
	{ "none", NOPRI },		{ NULL, 0 }
};


/*
 * map a facility name to its number
 */
int
find_facname(name)
	char *name;
{
	int	i;

	for (i = 0; facs[i].name; i++)
		if (!strcmp(facs[i].name, name))
			return facs[i].value;
	return -1;
}


/*
 * map a priority name to its number
 */
int
find_priname(name)
	char *name;
{
	int	i;

	for (i = 0; pris[i].name; i++)
		if (!strcmp(pris[i].name, name))
			return pris[i].value;
	return -1;
}


/*
 * map a facility number to its name
 */
char *
fac_toname(facpri)
	int facpri;
{
	int	i, fac;

	for (i = 0, fac = facpri & LOG_FACMASK; facs[i].name; i++)
		if (fac == facs[i].value)
			return facs[i].name;
	return NULL;
}


/*
 * map a priority number to its name
 */
char *
pri_toname(facpri)
	int facpri;
{
	int	i, pri;

	for (i = 0, pri = facpri & LOG_PRIMASK; pris[i].name; i++)
		if (pri == pris[i].value)
			return pris[i].name;
	return NULL;
}


matchspec_t *
match_find(parm)
	char *parm;
{
	matchspec_t *ms;

	for (ms = mslist; ms; ms = ms->ms_next)
		if (!strcmp(ms->ms_match, parm))
			return ms;
	return NULL;
}


matchspec_t *
new_match(parm)
	char *parm;
{
	matchspec_t *ms;
	int len;

	if ((ms = match_find(parm))) {
		ms->ms_cnt++;
		return ms;
	}

	if (!(NEW(ms)))
		return NULL;
	len = strlen(parm);
	len *= 2;
#ifdef USE_POSIX_REGEX
	(void) regcomp(&ms->ms_buf, parm, REGCOMP_FLAGS);
#else
	ms->ms_buf = (char *)malloc(len);
	(void) compile(parm, ms->ms_buf, ms->ms_buf + len, '\0');
#endif
	ms->ms_match = strdup(parm);
	ms->ms_next = mslist;
	ms->ms_comp = get_not();
	ms->ms_cnt = 1;
	mslist = ms;
	return ms;
}


matchprogspec_t *
matchprog_find(parm)
	char *parm;
{
	matchprogspec_t *mp;

	for (mp = mpslist; mp; mp = mp->mp_next)
		if (!strcmp(mp->mp_prog, parm))
			return mp;
	return NULL;
}


matchprogspec_t *
new_matchprog(parm)
	char *parm;
{
	matchprogspec_t *mp;
	int len;

	if ((mp = matchprog_find(parm))) {
		mp->mp_cnt++;
		return mp;
	}

	if (!(NEW(mp)))
		return NULL;
	len = strlen(parm);
	len *= 2;
	mp->mp_prog = strdup(parm);
	mp->mp_next = mpslist;
	mp->mp_comp = get_not();
	mp->mp_cnt = 1;
	mpslist = mp;
	return mp;
}


matchfilespec_t *
matchfile_find(parm)
char *parm;
{
	matchfilespec_t *mf;

	for (mf = mfslist; mf; mf = mf->mf_next)
		if (!strcmp(mf->mf_file, parm))
			return mf;
	return NULL;
}


matchfilespec_t *
new_matchfile(parm)
	char *parm;
{
	matchfilespec_t *mf;
	char line[MATCHSZ + 1];
	int i;
	FILE *fp;

	if ((mf = matchfile_find(parm))) {
		mf->mf_cnt++;
		return mf;
	}

	if (!(NEW(mf)))
		return NULL;

	if (!(fp = fopen(parm, "r"))) {
		log_printf(LOG_SYSLOG | LOG_ERR, "error opening matchfile %s: %s\n", parm, strerror(errno));
		free(mf);
		return NULL;
	}
	for (i = 0; (fgets(line, sizeof(line) - 1, fp)); i++)
		;
	rewind(fp);
	mf->mf_nlines = i;
	mf->mf_bufs = (void *)malloc(i * sizeof(*mf->mf_bufs));

	for (i = 0; (fgets(line, sizeof(line) - 1, fp)); i++) {
		line[sizeof(line) - 1] = '\0';
#ifdef USE_POSIX_REGEX
		(void) regcomp(mf->mf_bufs[i], parm, REGCOMP_FLAGS);
#else
		len = strlen(line) * 2;
		mf->mf_bufs[i] = (char *)calloc(1, len);
		(void) compile(line, mf->mf_bufs[i], mf->mf_bufs[i] + len,'\n');
#endif
	}
	fclose(fp);
	mf->mf_file = strdup(parm);
	mf->mf_next = mfslist;
	mf->mf_comp = get_not();
	mf->mf_cnt = 1;
	mfslist = mf;
	return mf;
}

/*
 * stream handling code
 */
streamspec_t *
sunstream_find(mode, parm)
	int mode;
	char *parm;
{
	streamspec_t *ss;

	for (ss = sslist; ss; ss = ss->ss_next)
		if (((ss->ss_mode & ~VAR_OLD) == mode) &&
		    !strcmp(ss->ss_path, parm))
			return ss;
	return NULL;
}

#ifdef USE_SUN_STREAMS

streamspec_t *
new_sunstream(mode, parm)
	int mode;
	char *parm;
{
	streamspec_t *ss;

	if ((ss = sunstream_find(mode, parm))) {
		ss->ss_mode &= ~VAR_OLD;
		ss->ss_cnt++;
		return ss;
	}

	if (!(NEW(ss)))
		return NULL;
	ss->ss_mode = mode;
	ss->ss_path = strdup(parm);
	ss->ss_cnt = 1;
	ss->ss_next = sslist;
	sslist = ss;
	afstreams_open_stream(ss);
	return ss;
}

#else

streamspec_t *
new_sunstream(mode, parm)
	int mode;
	char *parm;
{
	log_printf(LOG_SYSLOG | LOG_ERR, "sunstream support is not compiled in");
	return NULL;
}

#endif

static void
del_sunstream(ss)
	streamspec_t *ss;
{
	close(ss->ss_fd);
	free(ss->ss_path);
	free(ss);
}

/*
 * door handling code
 */
doorspec_t *
sundoor_find(mode, parm)
	int mode;
	char *parm;
{
	doorspec_t *ds;

	for (ds = dslist; ds; ds = ds->ds_next)
		if (((ds->ds_mode & ~VAR_OLD) == mode) &&
		    !strcmp(ds->ds_path, parm))
			return ds;
	return NULL;
}

#ifdef USE_SUN_DOORS

doorspec_t *
new_sundoor(mode, parm)
	int mode;
	char *parm;
{
	doorspec_t *ds;

	if ((ds = sundoor_find(mode, parm))) {
		ds->ds_mode &= ~VAR_OLD;
		ds->ds_cnt++;
		return ds;
	}

	if (!(NEW(ds)))
		return NULL;
	ds->ds_mode = mode;
	ds->ds_path = strdup(parm);
	ds->ds_cnt = 1;
	ds->ds_next = dslist;
	dslist = ds;
	afdoor_open_door(ds);
	return ds;
}

#else

doorspec_t *
new_sundoor(mode, parm)
	int mode;
	char *parm;
{
	log_printf(LOG_SYSLOG | LOG_ERR, "sundoor support is not compiled in");
	return NULL;
}

#endif

static void
del_sundoor(ds)
	doorspec_t *ds;
{
#ifdef USE_SUN_DOORS
	afdoor_close_door(ds);
#endif
}

/*
 * file handling code
 */
filespec_t *
file_find(mode, parm)
	int mode;
	char *parm;
{
	filespec_t *fs;

	for (fs = fslist; fs; fs = fs->fs_next)
		if (((fs->fs_mode & ~VAR_OLD) == mode) &&
		    !strcmp(fs->fs_path, parm))
			return fs;
	return NULL;
}

filespec_t *
new_file(mode, parm)
	int mode;
	char *parm;
{
	filespec_t *fs;

	if ((fs = file_find(mode, parm))) {
		fs->fs_mode &= ~VAR_OLD;
		fs->fs_cnt++;
		return fs;
	}

	if (!(NEW(fs)))
		return NULL;
	cur_file = fs;
	fs->fs_mode = mode;
	fs->fs_path = strdup(parm);
	fs->fs_syncfreq = options.op_syncfreq;
	fs->fs_markfreq = options.op_markfreq;
	fs->fs_next = fslist;
	fs->fs_cnt = 1;
	fs->fs_phash = NULL;
	fslist = fs;
	if (do_hash) {
		fs->fs_mode |= VAR_HASH;
		do_hash = 0;
	}
	affile_open_file(fs);
	if (VAR_TYPE(fs->fs_mode) == DEST_FILE) {
		ftable_add_fd(fs->fs_fd, current, current ? current->v_num : 0, DEST_FILE, fs);
	}
	return fs;
}

static void
del_file(fs)
	filespec_t *fs;
{
		free(fs->fs_path);
		free(fs->fs_phash);
		close(fs->fs_fd);
		free(fs);
}

void
end_file()
{
	cur_file = NULL;
}


void
set_sync(syncfreq)
	int syncfreq;
{
	cur_file->fs_syncfreq = syncfreq;
}


void
set_mark(markfreq)
	int markfreq;
{
	cur_file->fs_markfreq = markfreq;
}


/*
 * unix handling code
 */
unixspec_t *
unix_find(type, proto, parm)
	int type;
	int proto;
	char *parm;
{
	unixspec_t *xs;

	for (xs = xslist; xs; xs = xs->xs_next)
		if ((xs->xs_type == type) &&
		    (xs->xs_proto == proto) &&
		    !strncmp(xs->xs_un.sun_path, parm,
			     sizeof(xs->xs_un.sun_path)))
			return xs;
	return NULL;
}


unixspec_t *
new_unix(type, proto, parm)
	int type;
	int proto;
	char *parm;
{
	unixspec_t *xs;

	if ((xs = unix_find(type, proto, parm))) {
		xs->xs_mode &= ~VAR_OLD;
		xs->xs_cnt++;
		return xs;
	}

	if (!(NEW(xs)))
		return NULL;
	xs->xs_type = type;
	if (do_raw) {
		xs->xs_mode |= do_raw;
		do_raw = 0;
	}
	xs->xs_next = xslist;
	xs->xs_un.sun_family = AF_UNIX;
	strncpy(xs->xs_un.sun_path, parm, sizeof(xs->xs_un.sun_path));
	xs->xs_un.sun_path[sizeof(xs->xs_un.sun_path) - 1] = '\0';
	xs->xs_cnt = 1;
	xs->xs_proto = proto;
	afunix_open_socket(xs);
	if (xs->xs_type == DEST_UNIX) {
		ftable_add_fd(xs->xs_fd, current, current ? current->v_num : 0, xs->xs_type, xs);
	}
	xslist = xs;
	return xs;
}

static void
del_unix(xs)
	unixspec_t *xs;
{
	close(xs->xs_fd);
	free(xs);
}

/*
 * user handling code
 */
userspec_t *
user_find(parm)
	char *parm;
{
	userspec_t *us;

	for (us = uslist; us; us = us->us_next)
		if (!strncmp(us->us_user, parm, 9))
			return us;
	return NULL;
}


userspec_t *
new_user(parm)
	char *parm;
{
	userspec_t *us;

	if ((us = user_find(parm))) {
		us->us_cnt++;
		return us;
	}

	if (!(NEW(us)))
		return NULL;
	strncpy(us->us_user, parm, sizeof(us->us_user));
	us->us_user[sizeof(us->us_user) - 1] = '\0';
	us->us_next = uslist;
	us->us_cnt = 1;
	uslist = us;
	return us;
}

/*
 * network handling code
 */
netspec_t *
net_find(type, proto, ip, port)
	int type;
	int proto;
	struct in_addr ip;
	u_short port;
{
	netspec_t *ns;

	for (ns = nslist; ns; ns = ns->ns_next)
		if ((ns->ns_type == type) && (ns->ns_proto == proto) &&
		    (ns->ns_sin.sin_port == port) &&
		    !memcmp(&ns->ns_sin.sin_addr, &ip, sizeof(ip)))
			return ns;
	return NULL;
}


netspec_t *
new_net(type, proto, parm1, parm2)
	int type;
	int proto;
	char *parm1;
	char *parm2;
{
	netspec_t *ns;
	struct in_addr ip;
	u_short port;

	ip = getipv4addr(parm1);
	port = getportnum(NULL, parm2);
	if ((ns = net_find(type, proto, ip, port))) {
		ns->ns_mode &= ~VAR_OLD;
		ns->ns_cnt++;
		return ns;
	}

	if (!(NEW(ns)))
		return NULL;
	ns->ns_cnt = 1;
	ns->ns_type = type;
	ns->ns_proto = proto;
	ns->ns_sin.sin_family = AF_INET;
	ns->ns_sin.sin_addr = ip;
	ns->ns_sin.sin_port = port;
	afinet_open_socket(ns);
	if (ns->ns_type == DEST_INET) {
		ftable_add_fd(ns->ns_fd, current, current ? current->v_num : 0, ns->ns_type, ns);
	}

	if (do_raw) {
		ns->ns_mode |= do_raw;
		do_raw = 0;
	}
	ns->ns_next = nslist;
	nslist = ns;
	return ns;
}

static void
del_net(ns)
	netspec_t *ns;
{
	close(ns->ns_fd);
	free(ns);
}

/*
 * command handling
 */
cmdspec_t *
cmd_find(parm)
	char *parm;
{
	cmdspec_t *c;

	for (c = cslist; c; c = c->cm_next)
		if (!strcmp(c->cm_path, parm))
			return c;
	return NULL;
}

cmdspec_t *
new_cmd(parm)
	char *parm;
{
	cmdspec_t *c;

	if ((c = cmd_find(parm))) {
		c->cm_cnt++;
		return c;
	}

	if (!(NEW(c)))
		return NULL;
	c->cm_cnt = 1;
	c->cm_path = strdup(parm);
	c->cm_next = cslist;
	afpipe_open_pipe(c);
	ftable_add_fd(c->cm_fd, current, current ? current->v_num : 0, DEST_CMD, c);
	cslist = c;
	return c;
}

static void
del_cmd(cm)
	cmdspec_t *cm;
{
	close(cm->cm_fd);
	free(cm->cm_path);
	free(cm);
}

nmaskspec_t *
nmask_find(parm)
	char *parm;
{
	nmaskspec_t *nm;

	for (nm = nmlist; nm; nm = nm->nm_next)
		if (!strcmp(nm->nm_parm, parm))
			return nm;
	return NULL;
}


nmaskspec_t *
new_nmask(parm)
	char *parm;
{
	nmaskspec_t *nm;
	char *s;

	if ((nm = nmask_find(parm))) {
		nm->nm_cnt++;
		return nm;
	}

	if (!(NEW(nm)))
		return NULL;
	nm->nm_parm = strdup(parm);
	if ((s = strchr(parm, '/'))) {
		*s++ = '\0';
		nm->nm_addr = getipv4addr(parm);
		nm->nm_mask = getipmask(s);
	} else {
		nm->nm_addr = getipv4addr(s);
		nm->nm_mask.s_addr = inet_addr("255.255.255.255");
	}
	nm->nm_next = nmlist;
	nm->nm_comp = get_not();
	nm->nm_cnt = 1;
	nmlist = nm;
	return nm;
}


facspec_t *
fac_find(fac)
	int fac;
{
	facspec_t *fs;

	for (fs = faclist; fs; fs = fs->fs_next)
		if (fs->fs_fac == fac)
			return fs;
	return NULL;
}


facspec_t *
new_fac(parm)
	char *parm;
{
	facspec_t *fs;
	int fac;

	if ((fac = find_facname(parm)) < 0)
		return NULL;

	if ((fs = fac_find(fac))) {
		fs->fs_cnt++;
		return fs;
	}

	if (!(NEW(fs)))
		return NULL;
	fs->fs_fac = fac;
	fs->fs_next = faclist;
	fs->fs_comp = get_not();
	fs->fs_cnt = 1;
	faclist = fs;
	return fs;
}


prispec_t *
pri_find(pri, comp)
	int pri;
	int comp;
{
	prispec_t *ps;

	for (ps = prilist; ps; ps = ps->ps_next)
		if (ps->ps_pri == pri && ps->ps_comp == comp)
			return ps;
	return NULL;
}


prispec_t *
new_pri(comp, parm)
	int comp;
	char *parm;
{
	prispec_t *ps;
	int pri;

	if ((pri = find_priname(parm)) < 0)
		return NULL;

	if ((ps = pri_find(comp, pri))) {
		ps->ps_cnt++;
		return ps;
	}

	if (!(NEW(ps)))
		return NULL;
	ps->ps_pri = pri;
	ps->ps_next = prilist;
	ps->ps_not = get_not();
	ps->ps_comp = comp;
	ps->ps_cnt = 1;
	prilist = ps;
	return ps;
}


facprispec_t *
facpri_find(facpri)
	int facpri;
{
	facprispec_t *fp;

	for (fp = fplist; fp; fp = fp->fp_next)
		if (fp->fp_facpri == facpri)
			return fp;
	return NULL;
}


facprispec_t *
new_facpri(parm1, comp, parm2)
	char *parm1;
	int comp;
	char *parm2;
{
	facprispec_t *fp;
	int fac, pri;

	if ((fac = find_facname(parm1)) < 0) {
		fprintf(stderr, "unknown facility %s\n", parm1);
		return NULL;
	}
	if ((pri = find_priname(parm2)) < 0) {
		fprintf(stderr, "unknown priority %s\n", parm2);
		return NULL;
	}

	if ((fp = facpri_find(fac|pri))) {
		fp->fp_cnt++;
		return fp;
	}

	if (!(NEW(fp)))
		return NULL;
	fp->fp_facpri = fac|pri;
	fp->fp_next = fplist;
	fp->fp_not = get_not();
	fp->fp_comp = comp;
	fp->fp_cnt = 1;
	fplist = fp;
	return fp;
}


prispec_t *
new_prival(pri, comp)
	int pri, comp;
{
	prispec_t *ps;

	if ((ps = pri_find(pri, 0))) {
		ps->ps_cnt++;
		return ps;
	}

	if (!(NEW(ps)))
		return NULL;
	ps->ps_pri = pri;
	ps->ps_next = prilist;
	ps->ps_cnt = 1;
	ps->ps_not = 0;
	ps->ps_comp = comp;
	prilist = ps;
	return ps;
}


facspec_t *
new_facval(fac)
	int fac;
{
	facspec_t *fs;

	if ((fs = fac_find(fac))) {
		fs->fs_cnt++;
		return fs;
	}

	if (!(NEW(fs)))
		return NULL;
	fs->fs_fac = fac;
	fs->fs_next = faclist;
	fs->fs_cnt = 1;
	fs->fs_comp = 0;
	faclist = fs;
	return fs;
}


facprispec_t *
new_facprival(facpri, comp)
	int facpri, comp;
{
	facprispec_t *fp;

	if ((fp = facpri_find(facpri))) {
		fp->fp_cnt++;
		return fp;
	}

	if (!(NEW(fp)))
		return NULL;
	fp->fp_facpri = facpri;
	fp->fp_next = fplist;
	fp->fp_cnt = 1;
	fp->fp_not = 0;
	fp->fp_comp = comp;
	fplist = fp;
	return fp;
}


/* links a source, filter and a destination together */
void
link_sfd(src, fil, dst)
	var_t *src;
	fillist_t *fil;
	var_t *dst;
{
	sdlist_t *sd;

	if (!(NEW(sd)))
		return;
	sd->sd_src = src;
	src->v_cnt++;
	sd->sd_dst = dst;
	dst->v_cnt++;
	sd->sd_fil = fil;
	if (!sdlist)
		sdlist = sd;
	else
		sdtail->sd_next = sd;
	sdtail = sd;
}


/* link two filters, connecting them in "and" logical op */
fillist_t *
link_filters(name, tail)
	char *name;
	fillist_t *tail;
{
	fillist_t *fl;
	var_t *fil;

	fil = get_var(name);
	if (!fil || !(NEW(fl)))
		return NULL;
	fl->fl_fil = fil;
	fl->fl_comp = get_not();
	fil->v_cnt++;
	fl->fl_and = tail;
	return fl;
}


fillist_t *
add_filter(f1, f2)
	fillist_t *f1, *f2;
{
	f1->fl_or = f2;
	return f1;
}


void
print_not(isnot)
	int isnot;
{
	if (isnot)
		printf(" !");
}


table_t	comparators[] = {
	{ "= ",		IL_EQ },
	{ "!= ",	IL_NEQ },
	{ "<= ",	IL_LE },
	{ "< ",		IL_LT },
	{ "",		IL_GE },
	{ "> ",		IL_GT },
	{ NULL, 0 }
};


char *
comp_toname(comp)
	int comp;
{
	int i;

	for (i = 0; comparators[i].name; i++)
		if (comparators[i].value == comp)
			return comparators[i].name;
	return NULL;
}


void
print_type(ft)
	filetype_t *ft;
{
	filespec_t	*fs;
	unixspec_t	*xs;
	cmdspec_t	*cm;
	userspec_t	*us;
	streamspec_t	*ss;
	doorspec_t	*ds;
	matchspec_t	*ms;
	matchfilespec_t	*mf;
	matchprogspec_t	*mp;
	netspec_t	*ns;
	nmaskspec_t	*nm;
	facspec_t	*fa;
	prispec_t	*ps;
	facprispec_t	*fp;

	switch (VAR_TYPE(ft->ft_type))
	{
	case DEST_FILE : case SRC_FILE :
		fs = ft->ft_ptr;
		printf(" file %s ", fs->fs_path);
		break;
	case DEST_UNIX : case SRC_UNIX :
		xs = ft->ft_ptr;
		printf(" unix-%s %s ", xs->xs_proto == NET_UNIX_STREAM ? "stream" : "dgram", xs->xs_un.sun_path);
		break;
	case DEST_INET : case SRC_INET : 
		ns = ft->ft_ptr;
		if (ns->ns_proto == NET_TCP)
			printf(" tcp");
		else if (ns->ns_proto == NET_UDP)
			printf(" udp");
		else
			printf(" <%d>", ns->ns_proto);
		printf(" %s,%d ", inet_ntoa(ns->ns_sin.sin_addr),
			ns->ns_sin.sin_port);
		break;
	case SRC_SUN_STREAM:
		ss = ft->ft_ptr;
		printf(" sun-stream %s ", ss->ss_path);
		break;
	case SRC_SUN_DOOR:
		ds = ft->ft_ptr;
		printf(" sun-door %s ", ds->ds_path);
		break;
	case DEST_USER :
		us = ft->ft_ptr;
		printf(" user %s ", us->us_user); 
		break;
	case DEST_CMD :
		cm = ft->ft_ptr;
		printf(" command |%s\n ", cm->cm_path);
		break;
	case FIL_FAC :
		fa = ft->ft_ptr;
		print_not(fa->fs_comp);
		printf(" facility %s ", fac_toname(fa->fs_fac));
		break;
	case FIL_PRI :
		ps = ft->ft_ptr;
		print_not(ps->ps_not);
		printf(" priority %s%s", comp_toname(ps->ps_comp),
		       pri_toname(ps->ps_pri));
		break;
	case FIL_FACPRI :
		fp = ft->ft_ptr;
		print_not(fp->fp_not);
		printf(" facpri %s, %s%s", fac_toname(fp->fp_facpri),
		       comp_toname(fp->fp_comp), pri_toname(fp->fp_facpri));
		break;
	case FIL_MATCH :
		ms = ft->ft_ptr;
		print_not(ms->ms_comp);
		printf(" match %s ", ms->ms_match);
		break;
	case FIL_MATCHFILE :
		mf = ft->ft_ptr;
		print_not(mf->mf_comp);
		printf(" match-file %s ", mf->mf_file);
		break;
	case FIL_MATCHPROG :
		mp = ft->ft_ptr;
		print_not(mp->mp_comp);
		printf(" match-prog %s ", mp->mp_prog);
		break;
	case FIL_WRAP :
		nm = ft->ft_ptr;
		print_not(nm->nm_comp);
		printf(" netmask %s/", inet_ntoa(nm->nm_addr));
		printf("%s ", inet_ntoa(nm->nm_mask));
		break;
#if 0
	case FIL_CMD :
		c = ft->ft_ptr;
		printf(" command |%s\n ", c->path);
		break;
#endif
	default:
		break;
	}
}


void
dump_all()
{
	fillist_t *f1, *f2;
	sdlist_t *s;
	var_t *v;
	int i;

	for (v = vlist; v; v = v->v_next) {
		if (v->v_type == VAR_SRC || v->v_type == IL_SRCNAME)
			printf("source ");
		else if (v->v_type == VAR_DST || v->v_type == IL_DESTNAME)
			printf("destination ");
		else if (v->v_type == VAR_FIL || v->v_type == IL_FILNAME)
			printf("filter ");
		printf("%s {", v->v_name);
		for (i = 0; i < v->v_num; i++) {
			print_type(v->v_list + i);
			printf(";");
		}
		printf("}; # %d, %d\n", v->v_num, v->v_cnt);
	}

	for (s = sdlist; s; s = s->sd_next) {
		printf("log { source %s;\n", s->sd_src->v_name);
		for (f1 = s->sd_fil; f1; f1 = f1->fl_or) {
			printf("\tfilter ");
			for (f2 = f1; f2; f2 = f2->fl_and) {
				printf("%s%c", f2->fl_fil->v_name,
					f2->fl_and ? ',' : ';');
				printf(" # %d\n", f2->fl_fil->v_cnt);
			}
		}
		printf("\tdestination %s; # %d\n};\n", s->sd_dst->v_name,
			s->sd_dst->v_cnt);
	}
}


/* walk through source file descriptors */
int
next_sfd(ptr, cnt)
	void **ptr;
	int *cnt;
{
	sdlist_t *s = NULL;
	int i = 0, fd = -1;
	var_t *v;

	if (ptr)
		s = *((sdlist_t **)ptr);

	if (!s)
		s = sdlist;
	else {
		if (*cnt >= s->sd_src->v_num)
			s = s->sd_next;
		else
			i = *cnt;
	}
	*ptr = s;

	if (!s)
		return -2;

	v = s->sd_src;
	switch (VAR_TYPE(v->v_list[i].ft_type))
	{
	case SRC_FILE :
		fd = ((filespec_t *)v->v_list[i].ft_ptr)->fs_fd;
		break;
	case SRC_INET : /* case SRC_LISTEN : case SRC_TCP : */
		fd = ((netspec_t *)v->v_list[i].ft_ptr)->ns_fd;
		break;
	case SRC_UNIX :
		fd = ((unixspec_t *)v->v_list[i].ft_ptr)->xs_fd;
		break;
	case SRC_SUN_STREAM:
		fd = ((streamspec_t *) v->v_list[i].ft_ptr)->ss_fd;
	default :
		break;
	}
	i++;
	*cnt = i;
	return fd;
}


/* walk through destination filedescriptors */
int
next_dfd(ptr, cnt)
	void **ptr;
	int *cnt;
{
	sdlist_t *s = NULL;
	int i = 0, fd = -1;
	var_t *v;

	if (ptr)
		s = *((sdlist_t **)ptr);

	if (!s)
		s = sdlist;
	else {
		if (*cnt >= s->sd_dst->v_num)
			s = s->sd_next;
		else
			i = *cnt;
	}
	*ptr = s;

	if (!s)
		return -2;

	v = s->sd_dst;
	switch (VAR_TYPE(v->v_list[i].ft_type))
	{
	case DEST_FILE :
		fd = ((filespec_t *)v->v_list[i].ft_ptr)->fs_fd;
		break;
	case DEST_INET :
		fd = ((netspec_t *)v->v_list[i].ft_ptr)->ns_fd;
		break;
	case DEST_UNIX :
		fd = ((unixspec_t *)v->v_list[i].ft_ptr)->xs_fd;
		break;
	case DEST_CMD :
		fd = ((cmdspec_t *)v->v_list[i].ft_ptr)->cm_fd;
		break;
	case DEST_USER :
		break;
	default :
		break;
	}
	i++;
	*cnt = i;
	return fd;
}


void
regerr(err)
	int err;
{
	fprintf(stderr, "Regular expression error: ");
	if (err == 11)
		fprintf(stderr, "Range endpoint too large.\n");
	else if (err == 16)
		fprintf(stderr, "Bad number.\n");
	else if (err == 25)
		fprintf(stderr, "``\\ digit'' out of range.\n");
	else if (err == 36)
		fprintf(stderr, "Illegal or missing delimiter.\n");
	else if (err == 41)
		fprintf(stderr, "No remembered search string.\n");
	else if (err == 42)
		fprintf(stderr, "\\( \\) imbalance.\n");
	else if (err == 43)
		fprintf(stderr, "Too many \\(.\n");
	else if (err == 44)
		fprintf(stderr, "More than 2 numbers  given  in \\{ \\}.\n");
	else if (err == 45)
		fprintf(stderr, "} expected after \\.\n");
	else if (err == 46)
		fprintf(stderr, "First number exceeds second in \\{ \\}.\n");
	else if (err == 49)
		fprintf(stderr, "[] imbalance.\n");
	else if (err == 50)
		fprintf(stderr, "Regular expression too long.\n");
	else
		fprintf(stderr, "unknown error %d\n", err);
	exit(1);
}


void
free_var(v)
	var_t *v;
{
	var_t **vp;
	int i;
	for (vp = &vlist; (*vp); ) {
		if (*vp == v) {
			v->v_cnt--;
			assert(v->v_cnt >= 0);
			if (v->v_cnt) {
				v = NULL;
				break;
			}
			*vp = v->v_next;
			for (i = v->v_num - 1; i >= 0; i--)
				free_type(v->v_list + i);
			free(v->v_list);
			free(v->v_name);
			free(v);
			v = NULL;
			break;
		} else
			vp = &(*vp)->v_next;
	}
	if (v && (opts & OPT_DEBUG))
		fprintf(stderr, "Didn't find %p [%s]\n", v, v->v_name);
}


void
free_filter(fil)
	fillist_t *fil;
{
	fillist_t *fl;

	while ((fl = fil)) {
		fil = fl->fl_and;
		free_var(fl->fl_fil);
		if (fl->fl_or)
			free_filter(fl->fl_or);
		free(fl);
	}
}


void
expire_all()
{
	filespec_t	*fs, **fsp;
	unixspec_t	*xs, **xsp;
	netspec_t	*ns, **nsp;
	cmdspec_t	*cs;
	userspec_t	*us;
	streamspec_t	*ss;
	doorspec_t	*ds;
	matchspec_t	*ms;
	matchfilespec_t	*mfs;
	matchprogspec_t	*mps;
	facspec_t	*fcs;
	prispec_t	*pri;
	facprispec_t	*fp;
	nmaskspec_t	*nm;
	sdlist_t	*sd;
	int	i;

	while ((sd = sdlist)) {
		sdlist = sd->sd_next;
		free_var(sd->sd_src);
		free_var(sd->sd_dst);
		free_filter(sd->sd_fil);
		free(sd);
	}
	sdtail = sdlist;
	assert(sdlist == NULL);

	for (fsp = &fslist; (fs = *fsp); ) {
		assert(fs->fs_cnt == 0);
		if (VAR_TYPE(fs->fs_mode) == SRC_FILE) {
			fs->fs_mode |= VAR_OLD;
			fsp = &fs->fs_next;
			continue;
		}
		*fsp = fs->fs_next;
		del_file(fs);
	}

	for (xsp = &xslist; (xs = *xsp); ) {
		assert(xs->xs_cnt == 0);
		if (VAR_TYPE(xs->xs_mode) == SRC_UNIX) {
			xs->xs_mode |= VAR_OLD;
			xsp = &xs->xs_next;
			continue;
		}
		*xsp = xs->xs_next;
		del_unix(xs);
	}

	for (nsp = &nslist; (ns = *nsp); ) {
		assert(ns->ns_cnt == 0);
		if (VAR_TYPE(ns->ns_mode) == SRC_INET) {
			if (opts & OPT_DEBUG)
				fprintf(stderr, "keep net: %s,%d\n",
					inet_ntoa(ns->ns_sin.sin_addr),
					ntohs(ns->ns_sin.sin_port));
			ns->ns_mode |= VAR_OLD;
			nsp = &ns->ns_next;
			continue;
		}
		*nsp = ns->ns_next;
		del_net(ns);
	}

	for (ss = sslist; ss; ss = ss->ss_next) {
		assert(ss->ss_cnt == 0);
		ss->ss_mode |= VAR_OLD;
	}
	
	for (ds = dslist; ds; ds = ds->ds_next) {
		assert(ds->ds_cnt == 0);
		ds->ds_mode |= VAR_OLD;
	}

	while ((cs = cslist)) {
		assert(cs->cm_cnt == 0);
		cslist = cs->cm_next;
		del_cmd(cs);
	}

	while ((us = uslist)) {
		assert(us->us_cnt == 0);
		uslist = us->us_next;
		free(us);
	}

	while ((ms = mslist)) {
		assert(ms->ms_cnt == 0);
		mslist = ms->ms_next;
#ifdef USE_POSIX_REGEX
		regfree(&ms->ms_buf);
#else
		if (ms->ms_buf)
			free(ms->ms_buf);
#endif
		free(ms->ms_match);
		free(ms);
	}

	while ((mfs = mfslist)) {
		assert(mfs->mf_cnt == 0);
		mfslist = mfs->mf_next;

		for (i = 0; i < mfs->mf_nlines; i++)
#ifdef USE_POSIX_REGEX
			regfree(mfs->mf_bufs[i]);
#else
			free(mfs->mf_bufs[i]);
#endif
		free(mfs->mf_bufs);
		free(mfs->mf_file);
		free(mfs);
	}

	while ((mps = mpslist)) {
		assert(mps->mp_cnt == 0);
		mpslist = mps->mp_next;
		free(mps->mp_prog);
		free(mps);
	}

	while ((fcs = faclist)) {
		assert(fcs->fs_cnt == 0);
		faclist = fcs->fs_next;
		free(fcs);
	}

	while ((pri = prilist)) {
		assert(pri->ps_cnt == 0);
		prilist = pri->ps_next;
		free(pri);
	}

	while ((fp = fplist)) {
		assert(fp->fp_cnt == 0);
		fplist = fp->fp_next;
		free(fp);
	}

	while ((nm = nmlist)) {
		assert(nm->nm_cnt == 0);
		nmlist = nm->nm_next;
		free(nm->nm_parm);
		free(nm);
	}
	options.op_markfreq = MARKFREQ;
	options.op_syncfreq = SYNCFREQ;
}


void
expire_old()
{
	filespec_t	*fs, **fsp;
	unixspec_t	*xs, **xsp;
	netspec_t	*ns, **nsp;
	streamspec_t	*ss, **ssp;
	doorspec_t	*ds, **dsp;

	for (fsp = &fslist; (fs = *fsp); ) {
		if (!(fs->fs_mode & VAR_OLD)) {
			fsp = &fs->fs_next;
			continue;
		}
		assert(fs->fs_cnt == 0);
		*fsp = fs->fs_next;
		del_file(fs);
	}

	for (xsp = &xslist; (xs = *xsp); ) {
		if (!(xs->xs_mode & VAR_OLD)) {
			xsp = &xs->xs_next;
			continue;
		}
		assert(xs->xs_cnt == 0);
		*xsp = xs->xs_next;
		del_unix(xs);
	}

	for (nsp = &nslist; (ns = *nsp); ) {
		if (!(ns->ns_mode & VAR_OLD)) {
			nsp = &ns->ns_next;
			continue;
		}
		if (opts & OPT_DEBUG)
			fprintf(stderr, "close net: %s,%d\n",
				inet_ntoa(ns->ns_sin.sin_addr),
				ntohs(ns->ns_sin.sin_port));
		assert(ns->ns_cnt == 0);
		*nsp = ns->ns_next;
		del_net(ns);
	}

	for (ssp = &sslist; (ss = *ssp); ) {
		if (!(ss->ss_mode & VAR_OLD)) {
			ssp = &ss->ss_next;
			continue;
		}
		if (opts & OPT_DEBUG)
			fprintf(stderr, "close sunstream: %s\n",
				ss->ss_path);
		assert(ss->ss_cnt == 0);
		*ssp = ss->ss_next;
		del_sunstream(ss);
	}

	for (dsp = &dslist; (ds = *dsp); ) {
		if (!(ds->ds_mode & VAR_OLD)) {
			dsp = &ds->ds_next;
			continue;
		}
		if (opts & OPT_DEBUG)
			fprintf(stderr, "close sundoor: %s\n",
				ss->ss_path);
		assert(ds->ds_cnt == 0);
		*dsp = ds->ds_next;
		del_sundoor(ds);
	}

}

void
setup_filetable()
{
	int cnt = 0, fd;
	void *ptr = NULL;
	sdlist_t *s;

	/*
	 * populate fttable with open message sources.  store a pointer to
	 * the source variable, the type of file and a pointer to its parent
	 * data structure.
	 */
	while ((fd = next_sfd(&ptr, &cnt)) != -2) {
		if (fd >= 0) {
			s = (sdlist_t *)ptr;
			ftable_add_fd(fd, s->sd_src, cnt-1, s->sd_src->v_list[cnt-1].ft_type, s->sd_src->v_list[cnt-1].ft_ptr);
			ftable_add_read_fd(fd);
		}
	}

	ptr = NULL;
	cnt = 0;

#if 0
	/* 
	 * destination fds are set up when opened
	 */
	while ((fd = next_dfd(&ptr, &cnt)) != -2) {
		if (fd >= 0) {
			s = (sdlist_t *)ptr;
			ftable_add_fd(fd, s->sd_dst, cnt-1, s->sd_dst->v_list[cnt - 1].ft_type, s->sd_dst->v_list[cnt - 1].ft_ptr, 0);
		}
	}
#endif
}

/*
 * allocate a new filetype structure and initialize the type to the one
 * passed.
 */
struct filetype *
new_filetype(type)
	int type;
{
	filetype_t *ft;

	if (!(NEW(ft)))
		return NULL;
	ft->ft_type = type;
	return ft;
}

void
free_type(ft)
	filetype_t *ft;
{
	int *cp = NULL;

	switch (VAR_TYPE(ft->ft_type))
	{
	case DEST_FILE : case SRC_FILE :
		cp = &((filespec_t *)ft->ft_ptr)->fs_cnt;
		break;
	case DEST_UNIX : case SRC_UNIX :
		cp = &((unixspec_t *)ft->ft_ptr)->xs_cnt;
		break;
	case DEST_INET : case SRC_INET : 
	/* case SRC_LISTEN : case SRC_TCP : */
		cp = &((netspec_t *)ft->ft_ptr)->ns_cnt;
		break;
	case DEST_USER :
		cp = &((userspec_t *)ft->ft_ptr)->us_cnt;
		break;
	case DEST_CMD :
		cp = &((cmdspec_t *)ft->ft_ptr)->cm_cnt;
		break;
	case SRC_SUN_STREAM:
		cp = &((streamspec_t *)ft->ft_ptr)->ss_cnt;
		break;
	case SRC_SUN_DOOR:
		cp = &((doorspec_t *)ft->ft_ptr)->ds_cnt;
		break;
	case FIL_FAC :
		cp = &((facspec_t *) ft->ft_ptr)->fs_cnt;
		break;
	case FIL_PRI :
		cp = &((prispec_t *)ft->ft_ptr)->ps_cnt;
		break;
	case FIL_FACPRI :
		cp = &((facprispec_t *)ft->ft_ptr)->fp_cnt;
		break;
	case FIL_MATCH :
		cp = &((matchspec_t *)ft->ft_ptr)->ms_cnt;
		break;
	case FIL_MATCHFILE :
		cp = &((matchfilespec_t *)ft->ft_ptr)->mf_cnt;
		break;
	case FIL_MATCHPROG :
		cp = &((matchprogspec_t *)ft->ft_ptr)->mp_cnt;
		break;
#if 0
	case FIL_CMD :
		cp = &((cmdspec_t *)ft->ft_ptr)->cm_cnt;
		break;
#endif
	default:
		log_printf(LOG_SYSLOG | LOG_ERR, "unknown filetype\n");
		break;
	}

	if (cp)
		(*cp)--;
}

/*
 * close all open files, fsync destination ones first.
 */
void
sync_all()
{
	int fd, cnt = 0;
	void *ptr = NULL;

	while ((fd = next_dfd(&ptr, &cnt)) != -2) {
		fsync(fd);
	}

}
