/***************************************************************************
 *
 * Copyright (c) 1998 Balazs Scheidler
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <syslog.h>
#include <time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "nsyslogd.h"
#include "log.h"
#include "var.h"
#include "hash.h"
#include "htcp.h"
#include "utils.h"
#include "oldcf.h"
#include "cfgfile.h"
#include "ftable.h"
#include "affile.h"

#if !defined(lint)
static const char rcsid[] = "$Id: main.c,v 1.28 1998/12/01 18:10:00 bazsi Exp $";
#endif

options_t	options = { MARKFREQ,		/* op_markfreq */
			    SYNCFREQ,		/* op_syncfreq */
			    NSYSLOG_LOGDIR,	/* op_logdir */
			    NSYSLOG_DIR,	/* op_sysir */
			    NULL,		/* op_chroot */
			    HASH_NONE,		/* op_hashtype */
			    NULL,		/* op_hashdir */
			    NULL };		/* op_hashext */

int	opts = 0,
	do_reopen = 0,
	do_terminate = 0,
	dofork = 1,
	started = 0;
char	*progname = NULL;
char	*cfgfile = NSYSLOG_CONF,
	*pidfile = NSYSLOG_PID;
time_t	nextmark;

static	void	startup_msg	__P((void));
static	void	hup_msg		__P((void));
static	void	end_msg		__P((void));
static	void	handle_hup	__P((int));
static	void	handle_term	__P((int));

static void
handle_hup(sig)
	int sig;
{
	do_reopen = 1;
}

static void
handle_term(sig)
	int sig;
{
	do_terminate = 1;
}

static void
end_msg()
{
	log_printf(LOG_SYSLOG | LOG_INFO, "%s, going down\n", progname, getpid());
}

static void
startup_msg()
{
	log_printf(LOG_SYSLOG | LOG_INFO, "%s, started\n", progname, getpid());
}


static void
hup_msg()
{
	log_printf(LOG_SYSLOG | LOG_INFO, "%s, received HUP.\n",
		progname, getpid());
}

int
load_config()
{
	int rc;

	if (opts & OPT_OLDSYSLOG)
		rc = read_oldconfig(cfgfile);
	else {
		rc = read_config(cfgfile);
		if (rc) {
			log_printf(LOG_SYSLOG | LOG_ERR, "Parse error reading config file: %s\n", cfgfile);
		}
	}
	if (rc == 0) 
		setup_filetable();
	return rc;
}

void
reload_config()
{
	expire_all();
	ftable_reinit();
	started = 0;
	if (load_config()) {
		log_printf(LOG_SYSLOG | LOG_ERR, "No configuration data, exiting\n");
		do_terminate = 1;
	}
	else {
		started = 1;
	}
}

int
setup_dir()
{
	struct stat sb;

	if (!options.op_sysdir)
		options.op_sysdir = strdup(NSYSLOG_DIR);
	if (stat(options.op_sysdir, &sb)) {
		log_printf(LOG_SYSLOG | LOG_ERR, "config directory (%s) not found: %s", options.op_sysdir, strerror(errno));
		return -1;
	}
	return 0;
}

static void
do_chroot(char *root)
{
	if (root) {
		if (chdir(root))
			log_printf(LOG_SYSLOG | LOG_ERR, "chdir(%s) failed: %s", root, strerror(errno));
		if (chroot(root))
			log_printf(LOG_SYSLOG | LOG_ERR, "chroot(%s) failed: %s", root, strerror(errno));
	}
}

int
main(argc, argv)
	int argc;
	char *argv[];
{
	int ch,i;
	time_t now;
	FILE *fp;

	tzset();
	progname = argv[0];
	if (progname[0] != '/') {
		fprintf(stderr, "Run the program with full path spec (e.g. /usr/local/sbin/nsyslogd)\n");
		exit(1);
	}

	while ((ch = getopt(argc, argv, "C:df:Op:vV")) != -1)
		switch (ch)
		{
		case 'C' :
			options.op_chroot = strdup(optarg);
			break;
		case 'd' :
			opts |= OPT_DEBUG;
			dofork--;
			break;
		case 'f' :
			cfgfile = optarg;
			break;
		case 'O' :
			opts |= OPT_OLDSYSLOG;
			cfgfile = SYSLOG_CONF;
			break;
		case 'p' :
			pidfile = optarg;
			break;
		case 'v' :
			opts |= OPT_VERBOSE;
			dofork--;
			break;
		case 'V' :
			printf("Version %s\n", NSYSLOG_VERSION);
			exit(0);
		}

	if (dofork < 0)
		yydebug = 1;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);
	signal(SIGALRM, SIG_IGN);
	signal(SIGHUP, handle_hup);
	signal(SIGTERM, handle_term);
	signal(SIGINT, handle_term);
	
	do_chroot(options.op_chroot);
	if (setup_dir()) {
		return 1;
	}

	if (dofork > 0) {
		if (fork() > 0)
			exit(0);
		setsid();
		close(0);
		close(1);
		/* fd #2 is kept for stderr */
	}
	if ((fp = fopen(pidfile, "w"))) {
		fprintf(fp, "%d\n", (int) getpid());
		fflush(fp);
		fclose(fp);
	}

	ftable_init();
	if (load_config()) {
		log_printf(LOG_SYSLOG | LOG_ERR, "No configuration data, exiting...\n");
		return 1;
	}

	started = 1;
	startup_msg();

	nextmark = time(NULL) + options.op_markfreq;
	now = time(NULL);
	
	while (1) {
		ftable_check_fds(MIN(60, nextmark - now));
		if (do_reopen) {
			do_reopen = 0;
			hup_msg();
			log_timeout_marks();
			ftable_check_fds(0);
			affile_hash_flush(1);
			sync_all();
			expire_all();
			expire_old();
			for (i = getdtablesize(); i; i--) {
				if (i != 2)
					close(i);
			}
			execv(progname, argv);
			/*do_reopen = 0;
			hup_msg();
			reload_config();
			signal(SIGHUP, handle_hup);*/
		}
		if (do_terminate) {
			end_msg();
			log_timeout_marks();
			ftable_check_fds(0); 	/* send any output we might have */
			affile_hash_flush(1);	/* write modified H0s out */
			sync_all();
			expire_all();
			expire_old();
			break;
		}

		now = time(NULL);
		if (now >= nextmark) {
			log_timeout_marks();
			nextmark = now + options.op_markfreq;
		}
	}
	return 0;
}
