/***************************************************************************
 *
 * Copyright (c) 1998 Balazs Scheidler
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "afinet.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <netdb.h>

#include "nsyslogd.h"
#include "log.h"
#include "utils.h"
#include "ftable.h"
#include "htcp.h"

#ifndef lint
static const char rcsid[]="$Id: afinet.c,v 1.18 1998/12/01 18:09:59 bazsi Exp $";
#endif

void
afinet_add_hostname(lmp, addr)
	logmsgtype_t *lmp;
	struct in_addr *addr;
{
	char buf[2*MAXHOSTNAMELEN+1];
	struct hostent *hp;
	char *hname, *p;
	
	hp = gethostbyaddr((char *) addr, sizeof(struct in_addr), AF_INET);
        if (!hp)
        	hname = inet_ntoa(*addr);
	else {
	        hname = hp->h_name;
	        p = strchr(hname, '.');
	        if (p) *p = 0;
	}
	if (lmp->lm_host) {
		snprintf(buf, sizeof(buf), "%s/%s", lmp->lm_host, hname);
	        free(lmp->lm_host);
	}
	else {
		snprintf(buf, sizeof(buf), "%s/%s", hname, hname);
	}
       	lmp->lm_host = strdup(buf);
}

int
afinet_input(filetable_entry_t *fte, struct sockaddr *addr, char *line)
{
	logmsgtype_t lm;
	
	lm.lm_log = new_logmsg(line);
	afinet_add_hostname(&lm, &((struct sockaddr_in *) addr)->sin_addr);
	lm.lm_fte = fte;
	lm.lm_flags = FLG_ALL;
	lm.lm_recvd = time(NULL);
	logmsg(&lm);
	return 0;
}

int 
afinet_output(lg, fte)
	logmsg_t *lg;
	filetable_entry_t *fte;
{
	char line[MAXLINE];
	netspec_t *ns = fte->fte_spec;
	int nw;
	
	snprintf(line, sizeof(line), "%s %s %s\n", lg->lg_time, lg->lg_host, lg->lg_line);
	if (opts & OPT_DEBUG)
		log_debug("%smsg to host %s\n",
                	(ns->ns_mode & VAR_RAW) ? "raw " : "",
                        inet_ntoa(ns->ns_sin.sin_addr));
                       
        if (ns->ns_fd >= 0) {
		if (ns->ns_proto == NET_UDP) {
			nw = write(ns->ns_fd, line, strlen(line));
		}
		else {
			nw = htcp_sendlog(ns->ns_fd, ns->ns_proto_state, line);
		}
	}
	if (nw == -1) {
		log_printf(LOG_SYSLOG | LOG_ERR, "error sending inet(%s) socket %s: %s, trying to reopen", 
			ns->ns_proto == NET_UDP ? "udp" : "tcp", 
			inet_ntoa(ns->ns_sin.sin_addr),
			strerror(errno));
		close(ns->ns_fd);
		ftable_del_fd(ns->ns_fd);
		ns->ns_fd = -1;
		return -1;	/* indicate unsuccessful completition */
	}
	else if (nw == -2) {
		return -1;
	}
	return 0;
}

void
afinet_open_socket(ns)
	netspec_t *ns;
{
	int fd = -1, opt = 1;
	int type = ns->ns_type & VAR_DSF;

	if (ns->ns_proto == NET_TCP)
		fd = socket(AF_INET, SOCK_STREAM, 0);
	else if (ns->ns_proto == NET_UDP)
		fd = socket(AF_INET, SOCK_DGRAM, 0);
	else
		log_printf(LOG_SYSLOG | LOG_ERR, "unknown network protocol");

	if (fd == -1)
		log_printf(LOG_SYSLOG | LOG_ERR, "error creating inet socket: %s", strerror(errno));
	else {
		setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&opt,
			   sizeof(opt));
		(void) fcntl(fd, F_SETFD, fcntl(fd, F_GETFD, 0) | FD_CLOEXEC);
		if (type == SRC_INET) {
			if (bind(fd, (struct sockaddr *) &ns->ns_sin, sizeof(ns->ns_sin))) {
				log_printf(LOG_SYSLOG | LOG_ERR, "inet bind failed: %s", strerror(errno));
				close(fd);
				fd = -1;
			}
			if (ns->ns_proto == NET_TCP) 
				listen(fd, 5);
		} else if (type == DEST_INET) {
			/*
			 * XXX - this can potentially hang for TCP connections.
			 */
			alarm(CONNECT_TIMEOUT);
			if (connect(fd, (struct sockaddr *) &ns->ns_sin, sizeof(ns->ns_sin))) {
				close(fd);
				fd = -1;
			}
			alarm(0);
			if (!ns->ns_proto_state)
				ns->ns_proto_state = malloc(sizeof(htcp_state_t));
			memset(ns->ns_proto_state, 0, sizeof(htcp_state_t));
			if ((ns->ns_proto == NET_TCP) && (fd >= 0)) {
				char *secret;
				int slen;
				
				if (afinet_getsecret(&ns->ns_sin, &secret, &slen) == 0) {
					htcp_init_client(ns->ns_proto_state, secret);
					free(secret);
					htcp_client(fd, ns->ns_proto_state);
				}
				else {
					close(fd);
					fd = -1;
				}
			} 
		}
	}
	ns->ns_fd = fd;
}

int
afinet_getsecret(sin, secret, secretlen)
	struct sockaddr_in *sin;
	char **secret;
	size_t *secretlen;
{
	char file[128];
	FILE *fd;

	snprintf(file, sizeof(file), NSYSLOG_SECRET ".%s", inet_ntoa(sin->sin_addr));

	*secret = NULL;
	*secretlen = 0;
	fd = fopen(file, "r");
	if (fd == NULL) {
		log_printf(LOG_SYSLOG | LOG_ERR, "can't open syslog secret file(%s): %s", file, strerror(errno));
		return -1;
	}
	fgets(file, sizeof(file), fd);
	chomp(file);
	*secret = strdup(file);
	fclose(fd);
	return 0;
}

int 
afinet_reconnect(ft)
	filetype_t *ft;
{
	netspec_t *ns = ft->ft_ptr;
	afinet_open_socket(ns);
	ftable_add_fd(ns->ns_fd, ft->ft_owner, ft->ft_ndx, ft->ft_type, ft->ft_ptr);
	return 0;
}


int
afinet_check_fd(fd, fte)
	int fd;
	filetable_entry_t *fte;
{
	struct sockaddr_in sin;
	int slen, len, nfd;
	char buf[MAXLINE];
	netspec_t *ns = (netspec_t *) fte->fte_spec;
	filetype_t *ftype;
	
	switch (VAR_TYPE(fte->fte_type)) {
	case SRC_INET:
		switch (ns->ns_proto) {
		case NET_UDP:
			slen = sizeof(sin);
			len = recvfrom(ns->ns_fd, buf, MAXLINE, 0, (struct sockaddr *)&sin, &slen);
			if (len <= 0) {
				log_printf(LOG_SYSLOG | LOG_ERR, "recvfrom(%d,%s,%d):%s", ns->ns_fd,
					inet_ntoa(sin.sin_addr), slen, strerror(errno));
				return -1;
			}
			buf[len] = '\0';
			afinet_input(fte, (struct sockaddr *) &sin, buf);
			break;
		case NET_TCP:
			slen = sizeof(sin);
			nfd = accept(ns->ns_fd, (struct sockaddr *) &sin, &slen);
			if (nfd == -1)
				log_printf(LOG_SYSLOG | LOG_ERR, "accept(%d,%s) failed: %s", ns->ns_fd,
					inet_ntoa(sin.sin_addr), strerror(errno));
			else {
				char *secret;
				int secretlen;
				
				log_printf(LOG_SYSLOG | LOG_INFO, "htcp client connection accepted from %s", inet_ntoa(sin.sin_addr));
				ftable_add_fd(nfd, fte->fte_owner, fte->fte_ndx, SRC_INET_ACCEPTED, fte->fte_spec);
				if (!ns->ns_proto_state) {
					if ((ns->ns_proto_state = malloc(sizeof(htcp_state_t))) == NULL) {
						return -1;
					}
				}
				memset(ns->ns_proto_state, 0, sizeof(htcp_state_t));
				if (afinet_getsecret(&sin, &secret, &secretlen) == 0) {
					htcp_init_server((htcp_state_t *) ns->ns_proto_state, secret, (struct sockaddr *) &sin, slen, afinet_input);
					free(secret);
					htcp_server(nfd, ns->ns_proto_state);
				}
				else {
					close(nfd);
					ftable_del_fd(nfd);
				}
			}
			break;
		}
		break;
	case SRC_INET_ACCEPTED:
		if (htcp_server(fd, ns->ns_proto_state) == -1) {
			log_printf(LOG_SYSLOG | LOG_ERR, "htcp client %s dropped connection\n", 
				inet_ntoa(((struct sockaddr_in *)((htcp_state_t *) ns->ns_proto_state)->hs_clientaddr)->sin_addr));
			htcp_close(fd, ns->ns_proto_state);
		}
		break;
	case DEST_INET:
		if (ns->ns_proto == NET_UDP || 
		     (ns->ns_proto_state && 
		     ((((htcp_state_t *) ns->ns_proto_state)->hs_conn_flags & HTCP_F_LOGALLOWED) == HTCP_F_LOGALLOWED))) {
			ftype = &fte->fte_owner->v_list[fte->fte_ndx];
			if (log_queue_run(ftype, (log_output_func_t) afinet_output, fte)) {
				ftable_del_write_fd(((netspec_t *) fte->fte_spec)->ns_fd);
			}
		}
		else
			if (htcp_client(fd, ns->ns_proto_state) == -1) {
				log_printf(LOG_SYSLOG | LOG_ERR, "htcp server %s dropped connection during authorization\n", inet_ntoa(ns->ns_sin.sin_addr));
				htcp_close(fd, ns->ns_proto_state);
				ns->ns_fd = -1;
			}
		break;
	}
	return 0;
}
