/***************************************************************************
 *
 * Copyright (c) 1998 Balazs Scheidler
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "affile.h"

#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>

#include "nsyslogd.h"
#include "log.h"
#include "utils.h"

static int
affile_hash_readH0(fs)
	filespec_t *fs;
{
	char buf[256];
	int fd;
	
	if (*fs->fs_path == '/') {
		snprintf(buf, sizeof(buf), "%s.H0", fs->fs_path);
	}
	else {
		snprintf(buf, sizeof(buf), "%s/%s.H0", options.op_hashdir ? options.op_hashdir : options.op_logdir, fs->fs_path);
	}
	fd = open(buf, O_RDONLY|O_NOCTTY);
	if (fd != -1) {
		read(fd, &fs->fs_lcnt, sizeof(fs->fs_lcnt));
		fs->fs_phlen = read(fd, buf, MAXHASH);
		fs->fs_phash = malloc(fs->fs_phlen);
		memcpy(fs->fs_phash, buf, fs->fs_phlen);
		close(fd);
		return 0;
	}
	else {
		return -1;
	}
}

static void
affile_hash_writeH0(fs)
	filespec_t *fs;
{
	char hpath[256];
	int fd;
	
	if (*fs->fs_path == '/') {
		snprintf(hpath, sizeof(hpath), "%s.H0", fs->fs_path);
	}
	else {
		snprintf(hpath, sizeof(hpath), "%s/%s.H0", options.op_hashdir ? options.op_hashdir : options.op_logdir, fs->fs_path);
	}
	fd = open(hpath, O_WRONLY | O_CREAT | O_NOCTTY, 0600);
	if (fd != -1) {
		write(fd, &fs->fs_lcnt, sizeof(fs->fs_lcnt));
		write(fd, fs->fs_phash, fs->fs_phlen);
		close(fd);
	}
}

void
affile_hash_flush(force)
	int force;
{
	filespec_t *fs;
	static last_flush = 0;
	
	if (!last_flush || force || last_flush - time(NULL) > HASH_WRITE_INTERVAL) {
		fs = fslist;
		while (fs) {
			if (fs->fs_h0_dirty) {
				affile_hash_writeH0(fs);
				fs->fs_h0_dirty = 0;
			}
			fs = fs->fs_next;
		}
		last_flush = time(NULL);
	}
}

static void
affile_hash_output_rec(fs, line)
	filespec_t *fs;
	char *line;
{
	hashrec_t hrec;
	char *digest, *p;
	int dlen;
	
	memset(&hrec, 0, sizeof(hrec));
	hrec.hr_lcnt = fs->fs_lcnt++;
	hash_init(fs->fs_hashing);
	hash_update(fs->fs_hashing, fs->fs_phash, fs->fs_phlen);
	hash_update(fs->fs_hashing, line, strlen(line));
	hash_update(fs->fs_hashing, (char *) &hrec, sizeof(hrec));
	hash_finalize(fs->fs_hashing, &digest, &dlen);
	if (dlen > sizeof(hrec.hr_mac)) {
		log_printf(LOG_SYSLOG | LOG_ERR, "too long digest size max %d, dlen = %d\n", sizeof(hrec.hr_mac), dlen);
		free(digest);
 		return;
	}
	else {
		memcpy(hrec.hr_mac, digest, dlen);
		free(digest);
	}
	write(fs->fs_hfd, &hrec, sizeof(hrec));

	p = fs->fs_phash;
	hash_init(fs->fs_hashing);
	hash_update(fs->fs_hashing, fs->fs_phash, fs->fs_phlen);
	hash_update(fs->fs_hashing, hrec.hr_mac, dlen);
	hash_finalize(fs->fs_hashing, &fs->fs_phash, &fs->fs_phlen);
	free(p);
	fs->fs_h0_dirty = 1;
}

int
affile_open_hash(fs)
	filespec_t *fs;
{
	char hpath[256];

	fs->fs_hfd = -1;
	if (fs->fs_fd == -1)
		return -1;

	if (affile_hash_readH0(fs) == 0) {
		if (*fs->fs_path == '/') {
			snprintf(hpath, sizeof(hpath), "%s%s", fs->fs_path, options.op_hashext);
		}
		else {
			snprintf(hpath, sizeof(hpath), "%s/%s%s", options.op_hashdir ? options.op_hashdir : options.op_logdir, fs->fs_path, options.op_hashext);
		}
		fs->fs_hfd = open(hpath, O_WRONLY|O_APPEND|O_CREAT|O_NOCTTY, 0600);
		fs->fs_h0_dirty = 0;
		if (fs->fs_hfd != -1) {
			(void) fcntl(fs->fs_hfd, F_SETFD, fcntl(fs->fs_hfd, F_GETFD, 0) | FD_CLOEXEC);
			return 0;
		}
		else {
			return -1;
		}
	}
	else {
		return -1;
	}
}

int
affile_open_file(fs)
	filespec_t *fs;
{
	char *fullpath;
	char *path = fs->fs_path;
	int mode = fs->fs_mode;

	if (*path == '/')
		fullpath = path;
	else {
		fullpath = (char *)malloc(strlen(path) +
					  strlen(options.op_logdir) + 2);
		strcpy(fullpath, options.op_logdir);
		strcat(fullpath, "/");
		strcat(fullpath, path);
	}

	if (VAR_TYPE(mode) == DEST_FILE)
		fs->fs_fd = open(fullpath, O_WRONLY|O_APPEND|O_NOCTTY);
	else
		fs->fs_fd = open(fullpath, O_RDONLY|O_NOCTTY);
	if (fs->fs_fd != -1) {
		fcntl(fs->fs_fd, F_SETFD, fcntl(fs->fs_fd, F_GETFD, 0) | FD_CLOEXEC);
	}
	
	if (fullpath != path)
		free(fullpath);
	return 0;
}

static int 
affile_output(lg, fte)
	logmsg_t *lg;
	filetable_entry_t *fte;
{
	char line[MAXLINE];
	filespec_t *f = fte->fte_spec;
	int nw;
	
	if ((f->fs_mode & VAR_HASH) && f->fs_hashing == NULL) {
		f->fs_hashing = hash_create(options.op_hashtype);
		affile_open_hash(f);
	}
	 
	snprintf(line, sizeof(line), "%s %s %s\n", lg->lg_time, lg->lg_host, lg->lg_line);
	if (opts & OPT_DEBUG)
		log_debug("msg to file %s\n", f->fs_path);
	if (f->fs_fd >= 0)
		nw = write(f->fs_fd, line, strlen(line));
	if (nw > 0) {
		if ((f->fs_hfd >= 0) && f->fs_hashing)
			affile_hash_output_rec(f, line);
		f->fs_written++;
		if (f->fs_written >= f->fs_syncfreq) {
			fsync(f->fs_fd);
			fsync(f->fs_hfd);
			f->fs_written = 0;
		}
	} else if (nw < 0) {
		log_printf(LOG_SYSLOG | LOG_ERR, "error writing to %s: %s, reopening file", f->fs_path, strerror(errno));
		close(f->fs_fd);
		ftable_del_fd(f->fs_fd);
		f->fs_fd = -1;
		return -1;	/* indicate unsuccessful completition */
	}
	return 0;
}


int
affile_reconnect(ft)
	filetype_t *ft;
{
	filespec_t *f = ft->ft_ptr;

	affile_open_file(f);
	ftable_add_fd(f->fs_fd, ft->ft_owner, ft->ft_ndx, ft->ft_type, ft->ft_ptr);
	return 0;
}

int
affile_check_fd(fd, fte)
	int fd;
	filetable_entry_t *fte;
{
	int len;
	logmsgtype_t lm;
	char buf[MAXLINE];
	filetype_t *ftype;

	switch (VAR_TYPE(fte->fte_type)) {
	case SRC_FILE:
		len = read(fd, buf, sizeof(buf));
		if (len <= 0)
			return -1;
		buf[len] = '\0';
		lm.lm_flags = FLG_SYS;
		lm.lm_log = new_logmsg(buf);
		lm.lm_fte = fte;
		lm.lm_recvd = time(NULL);
		/* lm.lm_saddr = NULL; */ 
		logmsg(&lm);
		break;
	case DEST_FILE:
		ftype = &fte->fte_owner->v_list[fte->fte_ndx];
		if (log_queue_run(ftype, (log_output_func_t) affile_output, fte)) {
			ftable_del_write_fd(((filespec_t *) fte->fte_spec)->fs_fd);
		}
		break;
	}
	affile_hash_flush(0);
	return 0;
}
