
/*

    File: logcat/logcat.c
    
    Copyright (C) 1999 by Wolfgang Zekoll  <wzk@quietsche-entchen.de>

    This source is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This source is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sys/stat.h>
#include <unistd.h>
#include <time.h>

#include "lib.h"


char	*program =		NULL;

int	readonly =		0;
int	syslogtime =		0;

char	recfile[200] =		"";

char	logfile[200] =		"";
int	device =		0;
int	inode =			0;
long	lastpos =		0;


char *getrecfile(void)
{
	char	*p;

	if ((p = getenv("HOME")) == NULL  ||  *p == 0) {
		fprintf (stderr, "%s: no home directory set\n", program);
		exit (1);
		}

	snprintf (recfile, sizeof(recfile) - 2, "%s/%s", p, ".logcat");
	return (recfile);
}


long getlastpos(char *filename, char *logfile)
{
	int	found;
	char	*p, word[200], line[500];
	long	pos;
	struct stat sbuf;
	FILE	*fp;

	if (stat(logfile, &sbuf) != 0) {
		fprintf (stderr, "%s: can't stat file: %s\n", program, logfile);
		return (-1);
		}

	found = 0;
	pos = -1;
	if ((fp = fopen(filename, "r")) != NULL) {
		while (fgets(line, sizeof(line), fp) != NULL) {
			p = noctrl(line);
			if (*get_word(&p, word, sizeof(word)) == 0)
				continue;

			if (strcmp(word, logfile) == 0) {
				found = 1;

				get_word(&p, word, sizeof(word));
				if (sbuf.st_dev != atoi(word))
					break;

				get_word(&p, word, sizeof(word));
				if (sbuf.st_ino != atoi(word))
					break;

				get_word(&p, word, sizeof(word));
				pos = atol(word);

				break;
				}
			}

		fclose (fp);
		}

	if (found == 0) {
		fprintf (stderr, "previously unseen: %s\n", logfile);
		}

	device = sbuf.st_dev;
	inode  = sbuf.st_ino;
	if (pos < 0)
		pos = 0;

	return (pos);
}

long setlastpos(char *filename, char *logfile, long pos)
{
	char	*p, word[200], line[500], newfile[200];
	FILE	*fp, *new;

	snprintf (newfile, sizeof(newfile) - 2, "%s.new", filename);
	if ((new = fopen(newfile, "w")) == NULL) {
		fprintf (stderr, "%s: can't create file: %s\n", program, newfile);
		exit (1);
		}

	if ((fp = fopen(filename, "r")) != NULL) {
		while (fgets(line, sizeof(line), fp) != NULL) {
			p = noctrl(line);

			get_word(&p, word, sizeof(word));
			if (strcmp(word, logfile) != 0)
				fprintf (new, "%s\n", line);
			}

		fclose (fp);
		}

	fprintf (new, "%s %d %d %ld\n", logfile, device, inode, pos);
	fclose (new);

	if (rename(newfile, filename) != 0) {
		fprintf (stderr, "%s: can't replace file: %s\n", program, filename);
		exit (1);
		}
		
	return (0);
}


unsigned long getsyslogtime(char *line)
{
	int	i;
	char	*p, word[80];
	unsigned long when;
	static struct tm *now = NULL, *then = NULL;
	static char month[][4] = { "jan", "feb", "mar", "apr", "may", "jun", "jul", "aug", "sep", "oct", "nov", "dec", "" };

	if (now == NULL) {
		then = allocate(sizeof(struct tm));
		when = time(NULL);
		now  = allocate(sizeof(struct tm));
		memmove(now, localtime(&when), sizeof(struct tm));
		}

	p = line;
	if (*get_word(&p, word, sizeof(word)) == 0)
		return (0);

	strlwr(word);
	then->tm_mon = -1;
	for (i=0; month[i][0] != 0; i++) {
		if (strcmp(word, month[i]) == 0) {
			then->tm_mon = i;
			break;
			}
		}

	if (then->tm_mon < 0)
		return (0);

	then->tm_mday = strtol(p, &p, 10);
	p = skip_ws(p);
	then->tm_hour = atoi(get_quoted(&p, ':', word, sizeof(word)));
	then->tm_min  = atoi(get_quoted(&p, ':', word, sizeof(word)));
	then->tm_sec  = atoi(get_quoted(&p, ' ', word, sizeof(word)));

	if (then->tm_mon > now->tm_mon)
		then->tm_year = now->tm_year - 1;
	else
		then->tm_year = now->tm_year;

	if ((when = mktime(then)) == (time_t) -1)
		when = 0;
	
	return (when);
}

long catlog(char *logfile, long pos)
{
	char	line[4096];
	unsigned long serial;
	FILE	*fp;

	if ((fp = fopen(logfile, "r")) == NULL) {
		fprintf (stderr, "%s: can't open logfile: %s\n", program, logfile);
		exit (1);
		}

	fseek(fp, pos, SEEK_SET);
	if (ftell(fp) != pos) {
		fprintf (stderr, "%s: can't locate postion: %s\n", program, logfile);
		fseek(fp, 0L, SEEK_SET);
		}

	while (fgets(line, sizeof(line), fp) != NULL) {
		noctrl(line);
		if (syslogtime != 0) {
			if ((serial = getsyslogtime(line)) == 0)
				printf ("- %s\n", line); 
			else
				printf ("%lu %s\n", serial, line);
			}
		else
			printf ("%s\n", line);
		}

	pos = ftell(fp);
	return (pos);
}


void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k;
	char	*p, option[80];

	program = argv[0];
	k = 1;

	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'r')
				readonly = 1;
			else if (c == 's')
				syslogtime = 1;
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (1);
				}
			}
		}

	if (k + 1 != argc) {
		fprintf (stderr, "usage: %s [<options>] <logfile>\n", program);
		exit (1);
		}

	copy_string(logfile, argv[k++], sizeof(logfile));
	getrecfile();
	if ((lastpos = getlastpos(recfile, logfile)) < 0)
		exit (1);

	lastpos = catlog(logfile, lastpos);
	if (lastpos >= 0  &&  readonly == 0)
		setlastpos(recfile, logfile, lastpos);

	exit (0);
}

