/*
 * config.cc: configuration file handling for Olympus
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "config.h"
#include "exceptions.h"
#include "regexx.hh"
#include "split.hh"
#include "olympus.h"
#include "serverconfig.h"

#define CONFIG_BUFFER_SIZE          4096

using namespace regexx;

static const char *rexxConfig = "^([a-zA-Z0-9]+) \"(.+)\";";

olympusConfig::olympusConfig(void)
{
    configContext = NULL;
    configFile = NULL;

    openConfig();
    readConfig();
    closeConfig();
}

olympusConfig::~olympusConfig(void)
{
    olympusConfigContext *temp, *next;

    for (temp = configContext; temp; temp = next)
    {
        next = temp->next;
        delete temp;
    }
}

olympusConfig::olympusConfigContext::olympusConfigContext(char *command, char *value, olympusConfigContext *next)
{
    this->command = strdup(command);
    this->value = strdup(value);
    this->next = next;
}

olympusConfig::olympusConfigContext::~olympusConfigContext(void)
{
    if (command != NULL)
        delete command;
    if (value != NULL)
        delete value;
}

void olympusConfig::openConfig(void)
{
    string tmp(*configPath);
    
    tmp.append(SERVER_CONFIG_FILE);

    if (!*(configFile = new ifstream(tmp.c_str(), ios::in)))
    {
        closeConfig();
        throw fileOpenError(tmp.c_str());
    }
}

void olympusConfig::closeConfig(void)
{
    if (configFile != NULL)
    {
        delete configFile;
        configFile = NULL;
    }
}

int olympusConfig::readConfig(void)
{
    Regexx *rexx = new Regexx;
    char buffer[CONFIG_BUFFER_SIZE];
    int line(0);

    while (!(configFile->eof()))
    {
        configFile->getline(buffer, CONFIG_BUFFER_SIZE, '\n');
        line++;

        if (buffer[0] != '#')
        {
            if (rexx->exec(buffer, rexxConfig))
            {
                char command[rexx->match[0].atom[0].length() + 1];
                char value[rexx->match[0].atom[1].length() + 1];

                strncpy(command, rexx->match[0].atom[0].str().c_str(), rexx->match[0].atom[0].length());
                command[rexx->match[0].atom[0].length()] = '\0';

                strncpy(value, rexx->match[0].atom[1].str().c_str(), rexx->match[0].atom[1].length());
                value[rexx->match[0].atom[1].length()] = '\0';

                if (insertCommand(command, value) == 0)
                {
                }
            }
        }
    }
    delete rexx;
}

inline int olympusConfig::strtolower(char *str)
{
    while (*str)
    {
        if (*str > 0x40 && *str < 0x5b)
            *str += 32;
        *(str++);
    }
}

int olympusConfig::insertCommand(char *command, char *value)
{
    olympusConfigContext *temp(configContext), *last(NULL);
    int retc;

    strtolower(command);

    while (temp)
    {
        if ((retc = strcmp(command, temp->command)) < 1)
        {
            if (retc == 0)
                return 0;
            break;
        }
        last = temp;
        temp = temp->next;
    }

    if (last == NULL)
    {
        temp = new olympusConfigContext(command, value, temp);
        configContext = temp;
    }
    else
    {
        last->next = new olympusConfigContext(command, value, temp);
    }
    return 1;
}

char *olympusConfig::getCharValue(char *command)
{
    olympusConfigContext *temp;

    strtolower(command);

    for (temp = configContext; temp; temp = temp->next)
    {
        if (strcmp(command, temp->command) == 0)
            return strdup(temp->value);
    }
    throw configNoValue();
}

int olympusConfig::getIntValue(char *command)
{
    olympusConfigContext *temp;

    strtolower(command);

    for (temp = configContext; temp; temp = temp->next)
    {
        if (strcmp(command, temp->command) == 0)
            return strtol(temp->value, NULL, 10);
    }
    throw configNoValue();
}

string *olympusConfig::getStringValue(char *command)
{
    olympusConfigContext *temp;

    strtolower(command);

    for (temp = configContext; temp; temp = temp->next)
    {
        if (strcmp(command, temp->command) == 0)
            return new string(temp->value);
    }
    throw configNoValue();
}
