/*
 * olympus.cc
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "config.h"
#include "defs.h"
#include "transport.h"
#include "exceptions.h"
#include "channel.h"
#include "processmanager.h"
#include "pipemanager.h"
#include "serverconfig.h"
#include "logstream.h"
#include "manager.h"
#include "sysbackup.h"
#include "usermanager.h"
#include "packagemanager.h"
#include "olympus.h"

olympus::olympus(void)
{
    config = NULL;
    sysManager = NULL;
    sysUsers = NULL;
    backupManager = NULL;
    socket = NULL;
    clients = NULL;
}

void olympus::checkPath(string *path)
{
    /* make sure there is a trailing '/' in the path */
    if ((*path)[path->length() - 1] != '/')
        path->append("/");
}

void olympus::setBasePath(void)
{
    baseOlympusPath = new string(OLYMPUS_BASE_PATH);
}

void olympus::setConfigPath(void)
{
    if (configPath == NULL)
        configPath = new string(SERVER_CONFIG_PATH);
    checkPath(configPath);
}

void olympus::setDataPath(void)
{
    try {
        dataPath = config->getStringValue("datapath");
    }

    catch (configNoValue)
    {
        dataPath = new string(SERVER_DATA_PATH);
    }
    checkPath(dataPath);
}

void olympus::setDescriptorPath(void)
{
    try {
        descriptorPath = config->getStringValue("descriptor");
    }

    catch (configNoValue)
    {
        descriptorPath = new string(SERVER_CONFIG_PATH);
    }
    checkPath(descriptorPath);
}

void olympus::setTmpDataPath(void)
{
    try {
        tmpDataPath = config->getStringValue("tmpdatapath");
    }

    catch (configNoValue)
    {
        tmpDataPath = new string(SERVER_TMPDATA_PATH);
    }
    checkPath(tmpDataPath);
}

char *olympus::getServerHost(void)
{
    char *hostname;

    try {
        hostname = config->getCharValue("listenaddress");
    }

    catch (configNoValue)
    {
        hostname = NULL;
    }
    return hostname;
}

int olympus::getServerPort(void)
{
    int port;

    try {
        port = config->getIntValue("port");
    }

    catch (configNoValue)
    {
        port = SERVER_PORT;
    }
    return port;
}

void olympus::initDatabases(void)
{
    sysManager = new manager(dataPath);
    sysUsers = new userManager(dataPath);
    backupManager = new sysBackup(dataPath);
    sysPackages = new packageManager(dataPath);
    
    /* check for entries in the system database, read descriptor if not */
    if ((sysManager->hasEntries()) == false)
    {
        try {
            sysManager->readDescriptor(descriptorPath);
        }

        catch (fileOpenError &fileError)
        {
            errorLog << "Unable to open descriptor file " << fileError.filename << endl;
            exit(-1);
        }
    }
}

void olympus::addClient(int sd)
{
    channel *temp;

    temp = new channel(sd);
    temp->next = clients;
    clients = temp;
}

void olympus::removeClient(channel *client)
{
    channel *temp, *last(NULL);

    for (temp = clients; temp; temp = temp->next)
    {
        if (temp == client)
        {
            if (last)
                last->next = temp->next;
            else clients = temp->next;

            delete client;
            break;
        }
        last = temp;
    }
}

int olympus::setPipes(channel* client, fd_set* fd_in)
{
    pipeManager::pipeElement* temp;
    int fd_max(-1), tfd;

    for (temp = client->pipeList->elements; temp; temp = temp->nextElement)
    {
        if (temp->queued == false)
        {
            if ((tfd = temp->getfd()) > fd_max)
            {
                fd_max = tfd;
            }
            FD_SET(tfd, fd_in);
        }
    }
    return fd_max;
}

void olympus::serverLoop(void)
{
    channel *temp, *next;
    fd_set fd_in;
    int fd_max(-1), nsd, tfd;

    while (!restartPending && !shutdownPending)
    {
        FD_ZERO(&fd_in);

        fd_max = socket->getsd();
        FD_SET(fd_max, &fd_in);

        for (temp = clients; temp; temp = temp->next)
        {
            if (temp->pipeList != NULL)
            {
                if ((tfd = setPipes(temp, &fd_in)) > fd_max)
                {
                    fd_max = tfd;
                }
            }
            if ((tfd = temp->getsd()) > fd_max)
            {
                fd_max = tfd;
            }

            FD_SET(tfd, &fd_in);
        }

        if (select(fd_max + 1, &fd_in, NULL, NULL, NULL) < 0)
        {
            if (errno != EINTR)
            {
            }
            continue;
        }

        if (FD_ISSET(socket->getsd(), &fd_in))
        {
            try {
                nsd = socket->accept();
            }

            catch (socketError &socketError)
            {
                if (socketError.error != EINTR)
                {
                }
                continue;
            }
            addClient(nsd);
        }

        for (temp = clients; temp; temp = next)
        {
            next = temp->next;

            if (FD_ISSET(temp->getsd(), &fd_in))
            {
                if (temp->processChannel() < 1)
                {
                    removeClient(temp);
                    continue;
                }
            }

            if (temp->pipeList != NULL)
            {
                temp->processPipes(&fd_in);
            }
        }
    }
}

void olympus::serverMain(void)
{
    char *hostname;
    int port;

    /* determine the path of the configuration file */
    setConfigPath();

    while (!shutdownPending)
    {
        try {
            config = new olympusConfig;
        }

        catch (fileOpenError &fileError) {
            errorLog << "Unable to open configuration file " << fileError.filename << endl;
            exit(-1);
        }

        /* determine other paths which may be changed by configuration values */
        setBasePath();
        setDataPath();
        setDescriptorPath();
        setTmpDataPath();

        /* get configuration values */
        hostname = getServerHost();
        port = getServerPort();

        /* initialize the system and user databases */
        initDatabases();

        try {
            socket = new transport;
        }

        catch (socketCreationError)
        {
        }

        catch (socketSettingError &socketError)
        {
        }

        if (socket->bind(hostname, port) == -1)
        {
            errorLog << "Unable to bind to ";

            if (hostname != NULL)
                errorLog << hostname;
            else errorLog << "local";

            errorLog << " port " << port << endl;

            exit(-1);
        }

        if (socket->listen(getdtablesize()) == -1)
        {
            errorLog << "Unable to listen for connections on ";

            if (hostname != NULL)
                errorLog << hostname << endl;
            else errorLog << "local address" << endl;

            exit(-1);
        }

        /* enter the inner server loop */
        serverLoop();
    }
}
