/*****************************************************************
 *
 * Copyright (c) 2000 Mount Linux Inc.
 *
 * nmSourceExpand.h
 *
 * Receive a tarball from a client server and get status
 * on when it reaches the server.
 *****************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <iostream>
#include <strstream>
#include <string>
#include <vector>

#include "config.h"
#include "defs.h"
#include "regexx.hh"
#include "olympus.h"
#include "manager.h"
#include "processmanager.h"
#include "nmStrStream.h"
#include "nmSourceExpand.h"

#define DEBUG_SOURCEEXPAND

// constructor
nmSourceExpand::nmSourceExpand(transport *connection, unsigned long int sessionID, unsigned long int netmessageID) :
    nmProcess(connection, sessionID, netmessageID)
{
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::nmSourceExpand: enter constructor" << endl;
    cerr << "RECV:\tcharstarType" << endl
         << "\tcharstarType" << endl
         << "\tfileType" << endl;
#endif
    tmpFilePath = 0;
    members->add(&filename, nmMemberList::charstarType, 0);
    members->add(&tarname, nmMemberList::charstarType, 0);
    members->add(&tmpFilePath, nmMemberList::fileType, 0);
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::nmSourceExpand: leaving constructor" << endl;
#endif
}

nmSourceExpand::~nmSourceExpand()
{
    if (tmpFilePath)
    {
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "nmSourceExpand::~nmSourceExpand: deleting "
             << tmpFilePath << endl;
#endif
        remove(tmpFilePath);
        delete[] tmpFilePath;
    }
}

// set the lengths on the member list.
void nmSourceExpand::prepareToSend()
{
    struct stat st = {0};
    int size;

    // get the file length
    if (stat(tmpFilePath, &st) == 0)
    {
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "nmSourceExpand::prepareToSend: stat==0.  "
             << "st.st_size=" << st.st_size << endl
             << "nmSourceExpand::prepareToSend: tmpFilePath="
             << tmpFilePath << endl;
#endif
        size = st.st_size;
    }
    else
    {
        size = 0;
    }

#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::prepareToSend:" << endl;
    cerr << "SEND:\tintType" << endl
         << "\tfileType" << endl;
#endif
    // new data to send back
    delete members->nextMember;
    members->nextMember = NULL;

    members->add(&filename, nmMemberList::charstarType, strlen(filename) + 1);
    members->add(&tarname, nmMemberList::charstarType, strlen(tarname) + 1);
    members->add(&processStatus, nmMemberList::intType, 0);
    members->add(&numberOfFiles, nmMemberList::intType, 0);
    members->add(&seconds, nmMemberList::intType, 0);
    members->add(&useconds, nmMemberList::intType, 0);
    members->add(&tmpFilePath, nmMemberList::fileType, st.st_size);
}

// process the tarball.  call olyexec to unpack it and send back
// the output and status.
processManager* nmSourceExpand::newProcess()
{
    processManager* procmgr;
    struct stat st;
    string command;
    char **arglist;
    char buff[256];
    string tarpath;
    string tmppath;
    int idx;
    int length;
    int rc;

#if defined(DEBUG_SOURCEEXPAND)
    cerr << "newProcess: tarname=" << tarname << endl;
#endif
    tmppath = *baseOlympusPath + "tmp" + '/' + tarname;
    tarpath  = tmppath + '/' + filename;
    rc = stat(tmppath.c_str(), &st);
    if (rc)
    {
        rc = mkdir(tmppath.c_str(), 0755);
        if (rc)
        {
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "unable to create " << tmppath << endl;
#endif
            return (0);
        }
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "created directory '" << tmppath << "'" << endl;
#endif
    }

    // copy the file to the data directory
    if (copyfile(&tarpath))
    {
#if defined(DEBUG_SOURCEBUILD)
        cerr << "copy file succeeded" << endl;
#endif
        // call execcmd
        procmgr = untarfile();
    }
    else
    {
        int count;

        // call olyexec with error status
        command = *baseOlympusPath + "bin/olyexec";
        arglist = new char*[6];
        count = 0;
        arglist[count] = new char[8];
        strcpy(arglist[count++], "olyexec");

        intToString(commandID, buff);
        length = strlen(buff) + 1;
        arglist[count] = new char[length];
        strcpy(arglist[count++], buff);

        intToString(sessionID, buff);
        length = strlen(buff) + 1;
        arglist[count] = new char[length];
        strcpy(arglist[count++], buff);

        processStatus = -1;
        strcpy(buff, "ERROR");
        length = strlen(buff) + 1;
        arglist[count] = new char[length];
        strcpy(arglist[count++], buff);

        intToString(processStatus, buff);
        length = strlen(buff) + 1;
        arglist[count] = new char[length];
        strcpy(arglist[count++], buff);
        arglist[count] = 0;
        procmgr = new processManager(command.c_str(), arglist);
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "ARGUMENTS:" << endl;
#endif
        for (idx = 0; idx < count ; idx++)
        {
            delete[] arglist[idx];
        }
        delete[] arglist;
    }
    return (procmgr);
}

// reimplement activate
int nmSourceExpand::activate()
{
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::activate: commandID=" << commandID << endl;
    cerr << "nmSourceExpand::activate: sessionID=" << sessionID << endl;
#endif
    return 1;
}

// pack the data from the process
int nmSourceExpand::activate(processManager *pm)
{
    string pkgpath;
    char line[PATH_MAX];
    int msglength;
    unsigned int tmp;

#if defined(DEBUG_COMMANDID)
    cerr << "nmSourceExpand::activate(pm): commandID=" << commandID << endl
         << "nmSourceExpand::activate(pm): sessionID=" << sessionID << endl;
#endif
    pm->read((char *) &seconds, sizeof(int));
    pm->read((char *) &useconds, sizeof(int));
    pm->read((char *) &processStatus, sizeof(int));
    pm->read((char *) &msglength, sizeof(int));

    // the data file path is the remaining contents of the pipe.  Get this
    // file send it back to the client.
    tmpFilePath = new char[msglength + 1];
    pm->read(tmpFilePath, msglength);
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::activate: tmpFilePath=" << tmpFilePath << endl;
#endif

    // count the number of lines in the file - set numberOfFiles
    ifstream* istr = new ifstream(tmpFilePath);
    numberOfFiles = 0;
    while (istr->getline(line, PATH_MAX))
    {
        numberOfFiles++;
    }
    delete istr;

    pkgpath = *baseOlympusPath + "tmp" + '/' + tarname;
    sendDocFiles(pkgpath);

    return 1;
}

// private methods

// copy the tempfile to a real file in the $OLYMPUS/tmp directory
int nmSourceExpand::copyfile(string *tarpath)
{
    char buffer[4096];
    int fdin;
    int fdout;
    int len0;
    int len1;
    int retn = 1;
#if defined(DEBUG_SOURCEEXPAND)
    int total = 0;
#endif

    // do the copy
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::copyfile: tmpFilePath=" << tmpFilePath << endl;
#endif
    fdin = open(tmpFilePath, O_RDONLY);
    fdout = open(tarpath->c_str(), O_CREAT|O_RDWR, 0600);
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::copyfile() fdin=" << fdin
         << ". fdout=" << fdout << '.' << endl;
#endif
    if (fdin >= 0 && fdout >= 0)
    {
        while ((len0 = read(fdin, buffer, 4096)) > 0)
        {
            len1 = write(fdout, buffer, len0);
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "nmSourceExpand::copyfile() read " << len0 << '.' << endl
                 << "writing " << len1 << " bytes to output" << endl;
            total += len1;
#endif
            if (len1 < len0)
            {
                retn = 0;
                break;
            }
        }
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "nmSourceExpand::copyfile() total bytes=" << total << endl;
#endif
        close(fdin);
        close(fdout);
        remove(tmpFilePath);
        delete[] tmpFilePath;
        tmpFilePath = 0;
    }
#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::copyfile: commandID=" << commandID << endl;
    cerr << "nmSourceExpand::copyfile: sessionID=" << sessionID << endl;
#endif
    return (retn);
}

// getExtension - get the commands that process according to the file
// extension.  compression programs are always used in a pipe and
// archive programs are always used at the pipe's endpoint.
vector<string>* nmSourceExpand::getExtension(const string *filename)
{
    vector<string> *rtnvalue;
    string ext;
    char* arg;
    string key1;
    string key2;
    std::string::size_type begin;
    std::string::size_type end;
    std::string::size_type pos;
    std::string::size_type size;

    rtnvalue = new vector<string>;
    begin = end = pos = filename->length() - 1;
    while ((pos = filename->rfind('.', end)) != std::string::npos)
    {
        begin = pos + 1;
        size = end - begin + 1;
        ext = filename->substr(begin, size);
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "getExtension: begin=" << begin << ". end=" << end
             << ". pos=" << pos << ". size=" << size << endl;
        cerr << "filename->substr(" << begin << ',' << size << ")="
             << filename->substr(begin, size) << endl;
#endif
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "getExtension: ext=" << ext << endl;
#endif
        // when you find cpio/tar you are finished
        if (ext == "tar")
        {
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "getExtension: cpio/tar ext=" << ext << endl;
#endif
            key1 = "archive";
            arg = sysManager->filename(key1.c_str(), ext.c_str());
            rtnvalue->push_back(arg);
            delete arg;

            key2 = ext + "expand";
            arg = sysManager->filename(key1.c_str(), key2.c_str());
            rtnvalue->push_back(arg);
            rtnvalue->push_back("-");
            delete arg;

            break;
        }
        else if (ext == "bz2" || ext == "gz" || ext == "Z" || ext == "z")
        {
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "getExtension: gzip etc. ext=" << ext << endl;
#endif
            key1 = "compression";
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "key1=" << key1 << ". ext=" << ext << endl;
#endif
            arg = sysManager->filename(key1.c_str(), ext.c_str());
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "arg=" << hex << arg << dec << endl;
#endif
            rtnvalue->push_back(arg);
            delete arg;

            key2 = ext + "expstdout";
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "key1=" << key1 << ". ext=" << ext << endl;
#endif
            arg = sysManager->filename(key1.c_str(), key2.c_str());
#if defined(DEBUG_SOURCEEXPAND)
            cerr << "arg=" << hex << arg << dec << endl;
#endif
            rtnvalue->push_back(arg);
            delete arg;

            rtnvalue->push_back(*filename);
            rtnvalue->push_back("|");
        }
        end = pos - 1;
    }

    return (rtnvalue);
}

// make an char*[] from a vector
char** nmSourceExpand::makearglist(vector<string>* argv)
{
    vector<string>::iterator ele;
    char **args;
    int idx;
    int length;

    length = argv->size() + 1;
    args = new char*[length];
    for (ele = argv->begin(), idx = 0; ele != argv->end(); ele++, idx++)
    {
        length = ele->size() + 1;
        args[idx] = new char[length];
        ele->copy(args[idx], ele->size());
    }
    args[idx] = NULL;

    return (args);
}

// send a StrStream message to the request endpoint
void nmSourceExpand::sendDocFiles(string& pkgpath)
{
    using namespace regexx;
    DIR* dirp;
    nmStrStream* nm = 0;
    struct dirent* dp;
    struct stat sbuf;
    Regexx* re = new Regexx;
    int count = 0;

    if ((dirp = opendir(pkgpath.c_str())) != NULL)
    {
        while ((dp = readdir(dirp)) != NULL)
        {
            stat(dp->d_name, &sbuf);
            if (! S_ISREG(sbuf.st_mode))
            {
                continue;
            }
            if (count == 0)
            {
                nm = new nmStrStream(socket, sessionID, commandID);
                nm->setSequence(1);
            }
            // add the name to the nmStrStream here if it matches one
            // of the patterns
            re->str(dp->d_name);

            if (re->exec("^read", Regexx::nocase) > 0 ||
                re->exec("^install$", Regexx::nocase) > 0 ||
                re->exec("^changes$", Regexx::nocase) > 0 ||
                re->exec("^FAQ$") > 0 ||
                re->exec("^NEWS$") ||
                re->exec("^ANNOUNCE$") > 0 ||
                re->exec("^TODO$") ||
                re->exec("trouble", Regexx::nocase) > 0 ||
                re->exec("\\PORTING$", Regexx::nocase) > 0 ||
                re->exec("upgrade", Regexx::nocase) > 0 ||
                re->exec("manifest", Regexx::nocase) > 0 ||
                re->exec("^licenc", Regexx::nocase) > 0 ||
                re->exec("\\.readme", Regexx::nocase) > 0 ||
                re->exec("\\.faq", Regexx::nocase) > 0 ||
                re->exec("\\.install", Regexx::nocase) > 0 ||
                re->exec("\\.licence", Regexx::nocase) > 0 ||
                re->exec("\\.changes$", Regexx::nocase) > 0 ||
                re->exec("\\.txt", Regexx::nocase) > 0 ||
                re->exec("\\.man", Regexx::nocase) > 0 ||
                re->exec("\\.[1-9][a-z]*") > 0)
            {
                nm->addStr(dp->d_name);
            }
            if (count > 9)
            {
                ((netmessage*)nm)->send(socket, sessionID);
                nm->clear();
                count = 0;
            }
        }
    }
}

// untar the file into the tmppath directory
processManager* nmSourceExpand::untarfile()
{
    processManager* mgr = 0;
    vector<string> arg;
    vector<string> *suffix;
    vector<string>::iterator idx;
    string command;
    string filenamestr;
    string tmpstr;
    char buff[20];
    char **arglist;
    int j;

#if defined(DEBUG_SOURCEEXPAND)
    cerr << "nmSourceExpand::untarfile: untarfile entry" << endl;
#endif
    command = *baseOlympusPath + "bin/olyexec";
    arg.push_back(string("olyexec")); // main command
    intToString(sessionID, buff);
    arg.push_back(buff);
    intToString(COMMAND_EXPANDSRC, buff);
    arg.push_back(buff);
    tmpstr = *baseOlympusPath + "bin";
    arg.push_back(tmpstr);
    tmpstr = *tmpDataPath + "tmp" + '/' + tarname;
    arg.push_back(tmpstr);
    filenamestr = filename;
    suffix = getExtension(&filenamestr);
    for (idx = suffix->begin(); idx != suffix->end(); idx++)
    {
        arg.push_back(*idx);
    }
    try
    {
        arglist = makearglist(&arg);
        mgr = new processManager(command.c_str(), arglist);
        j = 0;
        while (arglist[j])
        {
            delete[] arglist[j++];
        }
        delete[] arglist;
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "nmSourceExpand::untarfile: returning new processmanager"
             << endl;
#endif
    }
    catch (...)
    {
        mgr = 0;
        processStatus = -1;
#if defined(DEBUG_SOURCEEXPAND)
        cerr << "nmSourceExpand::untarfile: null processmanager" << endl;
#endif
    }

    return (mgr);
}
