/*
 *  This code was writen by Chris Schlaeger for ktop
 *  Sebastien Taylor made a few minor changes to get
 *  it working as a netmessage in Olympus.
 */

#include <stdlib.h>
#include <dirent.h>

#include "OSStatus.h"

// uncomment this line to fake SMP reporting on non SMP systems
// #define FAKE_SMP 1

#ifdef linux

// Code for Linux 2.x

OSStatus::OSStatus()
{    
	error = false;
	cpuCount = -1;

	if ((stat = fopen("/proc/stat", "r")) == NULL)
	{
		error = true;
		errMessage = PROC_FILEOPENERROR;
		return;
	}

    setvbuf(stat, NULL, _IONBF, 0);

	if (!readCpuInfo("cpu", &userTicks, &sysTicks, &niceTicks,
					 &idleTicks))
		return;

	/*
	 * Contrary to /proc/stat the /proc/meminfo file cannot be left open.
	 * If done the rewind and read functions will use significant amounts
	 * of system time. Therefore we open and close it each time we have to
	 * access it.
	 */
	FILE* meminfo;
	if ((meminfo = fopen("/proc/meminfo", "r")) == NULL)
	{
		error = true;
		errMessage = PROC_FILEOPENERROR;
		return;
	}
	fclose(meminfo);

	userTicksX = sysTicksX = niceTicksX = idleTicksX = 0;
}

OSStatus::~OSStatus()
{
	delete [] userTicksX;
	delete [] sysTicksX;
	delete [] niceTicksX;
	delete [] idleTicksX;

	if (stat)
		fclose(stat);
}

/*
bool
OSStatus::getCpuLoad(int& user, int& sys, int& nice, int& idle)
{
	/*
	 * The CPU load is calculated from the values in /proc/stat. The cpu
	 * entry contains 4 counters. These counters count the number of ticks
	 * the system has spend on user processes, system processes, nice
	 * processes and idle time.
	 *
	 * SMP systems will have cpu1 to cpuN lines right after the cpu info. The
	 * format is identical to cpu and reports the information for each cpu.
	 * Linux kernels <= 2.0 do not provide this information!
	 *
	 * The /proc/stat file looks like this:
	 *
	 * cpu  1586 4 808 36274
	 * disk 7797 0 0 0
	 * disk_rio 6889 0 0 0
	 * disk_wio 908 0 0 0
	 * disk_rblk 13775 0 0 0
	 * disk_wblk 1816 0 0 0
	 * page 27575 1330
	 * swap 1 0
	 * intr 50444 38672 2557 0 0 0 0 2 0 2 0 0 3 1429 1 7778 0
	 * ctxt 54155
	 * btime 917379184
	 * processes 347 
	 *

	int currUserTicks, currSysTicks, currNiceTicks, currIdleTicks;

	if (!readCpuInfo("cpu", &currUserTicks, &currSysTicks,
					 &currNiceTicks, &currIdleTicks))
		return (false);
		
	int totalTicks = ((currUserTicks - userTicks) +
					  (currSysTicks - sysTicks) +
					  (currNiceTicks - niceTicks) +
					  (currIdleTicks - idleTicks));

	if (totalTicks > 0)
	{
		user = (100 * (currUserTicks - userTicks)) / totalTicks;
		sys = (100 * (currSysTicks - sysTicks)) / totalTicks;
		nice = (100 * (currNiceTicks - niceTicks)) / totalTicks;
		idle = (100 - (user + sys + nice));
	}
	else
	{    
		user = sys = nice = idle = 0;
    }
	userTicks = currUserTicks;
	sysTicks = currSysTicks;
	niceTicks = currNiceTicks;
	idleTicks = currIdleTicks;

	return (true);
}
*/
int
OSStatus::getCpuCount(void)
{
	/*
	 * The number of CPUs is only determined once. Initially the cpuCount
	 * is set to -1. The first call of this function then checks for the
	 * availability of cpu load information from /proc/stat.
	 */
	if (cpuCount < 0)
	{
		int dum;
		string cpuName;
		for (cpuCount = 1; ; cpuCount++)
		{
		    cpuName.replace(0, cpuName.length(), "");
			cpuName.append("cpu");
			cpuName.append((char *)cpuCount);
			if (!readCpuInfo(cpuName.c_str(), &dum, &dum, &dum, &dum))
			{
				/*
				 * Clear the error flag since this situation was caused on
				 * purpose.
				 */
				error = false;
				errMessage = 0;

				if (cpuCount > 1)
				{
					/*
					 * We only need the TicksX arrays on SMP systems. So we
					 * don't allocate them on single CPU systems.
					 */
					userTicksX = new int[cpuCount];
					sysTicksX = new int[cpuCount];
					niceTicksX = new int[cpuCount];
					idleTicksX = new int[cpuCount];
				}
				return (cpuCount);
			}
		}
	}

	return (cpuCount);
}

/*
bool
OSStatus::getCpuXLoad(int cpu, int& user, int& sys, int& nice, int& idle)
{
	int currUserTicks, currSysTicks, currNiceTicks, currIdleTicks;

	string cpuName;
	cpuName.replace(0, cpuName.length(), "");
	cpuName.append("cpu");
	cpuName.append((char*)cpu);

	if (!readCpuInfo(cpuName.c_str(), &currUserTicks, &currSysTicks,
					 &currNiceTicks, &currIdleTicks))
		return (false);

	int totalTicks = ((currUserTicks - userTicksX[cpu]) +
					  (currSysTicks - sysTicksX[cpu]) +
					  (currNiceTicks - niceTicksX[cpu]) +
					  (currIdleTicks - idleTicksX[cpu]));

	if (totalTicks > 0)
	{
		user = (100 * (currUserTicks - userTicksX[cpu])) / totalTicks;
		sys = (100 * (currSysTicks - sysTicksX[cpu])) / totalTicks;
		nice = (100 * (currNiceTicks - niceTicksX[cpu])) / totalTicks;
		idle = (100 - (user + sys + nice));
	}
	else
		user = sys = nice = idle = 0;

	userTicksX[cpu] = currUserTicks;
	sysTicksX[cpu] = currSysTicks;
	niceTicksX[cpu] = currNiceTicks;
	idleTicksX[cpu] = currIdleTicks;

	return (true);
}
*/
bool
OSStatus::getMemoryInfo(int& total, int& mfree, int& used, int& buffers,
						int& cached)
{
	/*
	 * The amount of total and used memory is read from the /proc/meminfo.
	 * It also contains the information about the swap space.
	 * The 'file' looks like this:
	 *
	 *         total:    used:    free:  shared: buffers:  cached:
	 * Mem:  64593920 60219392  4374528 49426432  6213632 33689600
	 * Swap: 69636096   761856 68874240
	 * MemTotal:     63080 kB
	 * MemFree:       4272 kB
	 * MemShared:    48268 kB
	 * Buffers:       6068 kB
	 * Cached:       32900 kB
	 * SwapTotal:    68004 kB
	 * SwapFree:     67260 kB
	 */

	FILE* meminfo;

	if ((meminfo = fopen("/proc/meminfo", "r")) == NULL)
	{
		error = true;
		errMessage = PROC_FILEOPENERROR;
		return (false);
	}
	if (fscanf(meminfo, "%*[^\n]\n") == EOF)
	{
		error = true;
		errMessage = MEMINFO_READERROR;
		return (false);
	}
	/*
	 * The following works only on systems with 4GB or less. Currently this
	 * is no problem but what happens if Linus changes his mind?
	 */
	fscanf(meminfo, "%*s %d %d %d %*d %d %d\n",
		   &total, &used, &mfree, &buffers, &cached);

	total /= 1024;
	mfree /= 1024;
	used /= 1024;
	buffers /= 1024;
	cached /= 1024;

	fclose(meminfo);

	return (true);
}

bool
OSStatus::readCpuInfo(const char* cpu, int* u, int* s, int* n, int* i)
{
#ifdef FAKE_SMP
	/*
	 * This code redirects inquieries for cpu0 and cpu1 to cpu to fake a
	 * 2 processor SMP system for test purposes.
	 */
	char cpuFake[10] = "cpu";
	if (strcmp(cpu, "cpu0") != 0 && strcmp(cpu, "cpu1") != 0)
		strcpy(cpuFake, cpu);
#define cpu cpuFake
#endif
	char tag[32];

	rewind(stat);

	do
	{
		if (fscanf(stat, "%32s %d %d %d %d", tag, u, n, s, i) != 5)
		{
			error = true;
			errMessage = PROC_READERROR;
			return (false);
		}
	} while (strcmp(tag, cpu));

#ifdef FAKE_SMP
#undef cpu
#endif

	return (true);
}

bool
OSStatus::getSwapInfo(int& stotal, int& sfree)
{
	FILE* meminfo;

	if ((meminfo = fopen("/proc/meminfo", "r")) == NULL)
	{
		error = true;
		errMessage = PROC_FILEOPENERROR;
		return (false);
	}
	if (fscanf(meminfo, "%*[^\n]\n") == EOF)
	{
		error = true;
		errMessage = MEMINFO_READERROR;
		return (false);
	}
	fscanf(meminfo, "%*[^\n]\n");
	fscanf(meminfo, "%*s %d %*d %d\n",
		   &stotal, &sfree);
	fclose(meminfo);

	stotal /= 1024;
	sfree /= 1024;

	return (true);
}

int 
isProcDir(const struct dirent* dir)
{
	return (atoi(dir->d_name));
}

int
OSStatus::getNoProcesses(void)
{
	int processes;
	struct dirent** namelist;


	processes = scandir("/proc", &namelist, isProcDir, alphasort);

	for (int i = 0; i < processes; i++)
		free(namelist[i]);
	free(namelist);

	return (processes);
}

#elif __FreeBSD__
/* Port to FreeBSD by Hans Petter Bieker <zerium@webindex.no>.
 *
 * Copyright 1999 Hans Petter Bieker <zerium@webindex.no>.
 */

#include <stdlib.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <vm/vm_param.h>
#include <sys/vmmeter.h>
#include <sys/user.h>
#include <unistd.h>

OSStatus::OSStatus()
{
	cpuCount = -1;
	error = false;
}

OSStatus::~OSStatus ()
{
}

bool
OSStatus::getCpuLoad(int& user, int& sys, int& nice, int& idle)
{
	user = 0; // FIXME
	sys = 0; // FIXME
	nice = 0; // FIXME

	return (true);
}

int
OSStatus::getCpuCount(void)
{
	if (cpuCount < 0)
	{
		size_t len = sizeof (cpuCount);
		if (sysctlbyname("hw.ncpu", &cpuCount, &len, NULL, 0) == -1 || !len)
			cpuCount = 1; // default to 1;
	}

	return (cpuCount);
}

bool
OSStatus::getCpuXLoad(int, int& user, int& sys, int& nice, int& idle)
{
	// FIXME
	// SMP support not yet implemented!
	return (getCpuLoad(user, sys, nice, idle));
}

bool
OSStatus::readCpuInfo(const char*, int*, int*, int*, int*)
{
	// FIXME
	return (false);
}

bool
OSStatus::getMemoryInfo(int& total, int& mfree, int& used, int& buffers,
						int& cached)
{
	int mib[2];
	mib[0] = CTL_VM;
	mib[1] = VM_METER;

	size_t len = sizeof (vmtotal);
	struct vmtotal p;
	sysctl(mib, 2, &p, &len, NULL, 0);

	mib[0] = CTL_HW;
	mib[1] = HW_PHYSMEM;
	len = sizeof (total);

	// total
	sysctl(mib, 2, &total, &len, NULL, 0);
	total /= 1024;

	// mfree
	mfree = p.t_free * getpagesize() / 1024;

	// used
	used = p.t_arm * getpagesize() / 1024;

	// buffers
	len = sizeof (buffers);
	if ((sysctlbyname("vfs.bufspace", &buffers, &len, NULL, 0) == -1) || !len)
		buffers = 0; // Doesn't work under FreeBSD v2.2.x

	// cached
	len = sizeof (cached);
	if ((sysctlbyname("vm.stats.vm.v_cache_count", &cached, &len, NULL, 0) == -1) || !len)
		cached = 0; // Doesn't work under FreeBSD v2.2.x
	cached *= getpagesize() / 1024;

	

	return (true);
}

bool
OSStatus::getSwapInfo(int& stotal, int& sfree)
{
	FILE *file;
	char buf[256];

	/* Q&D hack for swap display. Borrowed from xsysinfo-1.4 */
	if ((file = popen("/usr/sbin/pstat -ks", "r")) == NULL) {
		stotal = sfree = 0;
	return (true);
	}

	fgets(buf, sizeof(buf), file);
	fgets(buf, sizeof(buf), file);
	fgets(buf, sizeof(buf), file);
	fgets(buf, sizeof(buf), file);
	pclose(file);

	char *total_str, *free_str;
	strtok(buf, " ");
	total_str = strtok(NULL, " ");
	strtok(NULL, " ");
	free_str = strtok(NULL, " ");

	stotal = atoi(total_str);
	sfree = atoi(free_str);

	return (true);
}

int
OSStatus::getNoProcesses(void)
{
	int mib[3];
	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC;
	mib[2] = KERN_PROC_ALL;

	size_t len;
	sysctl(mib, 3, NULL, &len, NULL, 0);

	return (len / sizeof (struct kinfo_proc));
}

#elif __osf__
/* Port to Digital Unix by Wolfram Klaus <Wolfram.Klaus@physik.fu-berlin.de>.
 *
 * Copyright 1999 Wolfram Klaus <Wolfram.Klaus@physik.fu-berlin.de>.
 */
#include <sys/sysinfo.h>
#include <machine/hal_sysinfo.h>
#include <sys/table.h>
#include <unistd.h>
OSStatus::OSStatus()
{
	/* 
	 * Under Digital Unix ktop has to be suid root in order to be
	 * able to ioctl() on the /proc/... files. The calls in
	 * OSProcess.h are wrapped with seteuid(0) / seteuid(getuid())
	 * pairs so only these calls are made under root id. Now we
	 * need a place to set the EUID initially to the RUID (we dont
	 * want users to renice or kill other users' tasks, do we?).
	 * The safest place for this would be main(), but I wanted to
	 * keep it in some of the OS* files. This constructor is
	 * called in main by OSStatus priv;
  	 */

	seteuid(getuid());

	error = false;
	cpuCount = -1;

	if (!readCpuInfo("cpu", &userTicks, &sysTicks, &niceTicks,
					 &idleTicks))
		return;
	userTicksX = sysTicksX = niceTicksX = idleTicksX = 0;
	si=0;
}

OSStatus::~OSStatus()
{
	delete[] userTicksX;
	delete[] sysTicksX;
	delete[] niceTicksX;
	delete[] idleTicksX;
	delete[] si;
}

bool
OSStatus::getCpuLoad(int& user, int& sys, int& nice, int& idle)
{

	int currUserTicks, currSysTicks, currNiceTicks, currIdleTicks;

	if (!readCpuInfo("cpu", &currUserTicks, &currSysTicks,
					 &currNiceTicks, &currIdleTicks))
		return (false);
		
	int totalTicks = ((currUserTicks - userTicks) +
					  (currSysTicks - sysTicks) +
					  (currNiceTicks - niceTicks) +
					  (currIdleTicks - idleTicks));

	if (totalTicks > 0)
	{
		user = (100 * (currUserTicks - userTicks)) / totalTicks;
		sys = (100 * (currSysTicks - sysTicks)) / totalTicks;
		nice = (100 * (currNiceTicks - niceTicks)) / totalTicks;
		idle = (100 - (user + sys + nice));
	}
	else
		user = sys = nice = idle = 0;

	userTicks = currUserTicks;
	sysTicks = currSysTicks;
	niceTicks = currNiceTicks;
	idleTicks = currIdleTicks;

	return (true);
}

int
OSStatus::getCpuCount(void)
{
	/*
	 * The number of CPUs is only determined once. Initially the cpuCount
	 * is set to -1. The first call of this function then checks for the
	 * number of CPUs using getsysinfo().
	 */
	if (cpuCount < 0)
	  {
	    int dum;

	    getsysinfo(GSI_CPUS_IN_BOX, (caddr_t )&cpuCount, sizeof(cpuCount), &dum);
	    if (cpuCount > 1)
	      {
		/*
		 * We only need the TicksX arrays on SMP systems. So we
		 * don't allocate them on single CPU systems.
		 */
		userTicksX = new int[cpuCount];
		sysTicksX = new int[cpuCount];
		niceTicksX = new int[cpuCount];
		idleTicksX = new int[cpuCount];
		si=new struct tbl_sysinfo[cpuCount];
	      }
	    else 
	      cpuCount=1; // This should never be reached (well, only if getsysinfo fails)
	  }
	return (cpuCount);
}



bool
OSStatus::getCpuXLoad(int cpu, int& user, int& sys, int& nice, int& idle)
{
	int currUserTicks, currSysTicks, currNiceTicks, currIdleTicks;

	if (cpu==0) // Only call table the first time. The returned 
	            // stuct-array contains information on all cpuCount 
	            // cpu's. As a nice side effect the info on all CPUs 
	            // refers to the same point in time.
	  if (table(TBL_SYSINFO_MP, 0L, si, 1L, sizeof(struct tbl_sysinfo)*cpuCount)==-1)
	    return (false);

	currUserTicks=si[cpu].si_user;
	currSysTicks= si[cpu].si_sys;
	currNiceTicks=si[cpu].si_nice;
	currIdleTicks=si[cpu].si_idle;

	int totalTicks = ((currUserTicks - userTicksX[cpu]) +
					  (currSysTicks - sysTicksX[cpu]) +
					  (currNiceTicks - niceTicksX[cpu]) +
					  (currIdleTicks - idleTicksX[cpu]));

	if (totalTicks > 0)
	{
		user = (100 * (currUserTicks - userTicksX[cpu])) / totalTicks;
		sys = (100 * (currSysTicks - sysTicksX[cpu])) / totalTicks;
		nice = (100 * (currNiceTicks - niceTicksX[cpu])) / totalTicks;
		idle = (100 - (user + sys + nice));
	}
	else
		user = sys = nice = idle = 0;

	userTicksX[cpu] = currUserTicks;
	sysTicksX[cpu] = currSysTicks;
	niceTicksX[cpu] = currNiceTicks;
	idleTicksX[cpu] = currIdleTicks;

	return (true);
}

bool
OSStatus::getMemoryInfo(int& total, int& mfree, int& used, int& buffers,
						int& cached)
{
  struct tbl_vmstats vm;
  if (table(TBL_VMSTATS, 0L, &vm, 1L, sizeof(vm))==-1)
    return (false);

  total=vm.active_count+vm.wire_count+vm.free_count+vm.inactive_count+vm.ubc_pages;
  used=vm.active_count+vm.wire_count+vm.inactive_count+vm.ubc_pages;
  mfree=vm.free_count;
  buffers=0;
  cached=vm.ubc_pages;

  total *= vm.pagesize/1024;
  mfree *= vm.pagesize/1024;
  used *= vm.pagesize/1024;
  buffers *= vm.pagesize/1024;
  cached *= vm.pagesize/1024;


  return (true);
}

bool
OSStatus::readCpuInfo(const char* cpu, int* u, int* s, int* n, int* i)
{
  struct tbl_sysinfo si;
  if (table(TBL_SYSINFO, 0L, &si, 1L, sizeof(si))==-1)
    return (false);
  *u=(int)si.si_user;
  *s=(int)si.si_sys;
  *n=(int)si.si_nice;
  *i=(int)si.si_idle;
  
  return (true);
}

bool
OSStatus::getSwapInfo(int& stotal, int& sfree)
{
  struct tbl_swapinfo sw;
  if (table(TBL_SWAPINFO, -1L, &sw, 1L, sizeof(sw))==-1)
    return (false);
  stotal=sw.size*getpagesize()/1024;
  sfree =sw.free*getpagesize()/1024;
  
  return (true);
}


int
OSStatus::getNoProcesses(void)
{
  struct tbl_tblstats ts;
  if (table(TBL_TBLSTATS, 0L, &ts, 1L, sizeof(ts))==-1)
    return (0);
  return (int) ts.tb_procu;
}

#elif __sun__

#include <sys/param.h>
#include <unistd.h>
#include <sys/sysinfo.h>
#include <sys/stat.h>
#include <sys/swap.h>


// Solaris

OSStatus::OSStatus()
{
  error = false;
  errMessage = "unknown error";
  
  if(!(kstat_control = kstat_open())) {
    error = true;
    errMessage = KSTAT_HANDLERROR;
    return;
  }
  
  cpuCount = sysconf(_SC_NPROCESSORS_CONF);
  pagefactor = sysconf(_SC_PAGESIZE)/1024;

  int dum;
  getSwapInfo(dum, dum);
  sleep(1);

  cpu_info = new unsigned long[cpuCount*5];
  last_user = new int[cpuCount];
  last_sys = new int[cpuCount];
  last_idle = new int[cpuCount];
  last_wait = new int[cpuCount];
}

OSStatus::~OSStatus()
{
  if (kstat_control)
    kstat_close(kstat_control);
  delete[] cpu_info;
  delete[] last_user;
  delete[] last_sys;
  delete[] last_idle;
  delete[] last_wait;
}

bool
OSStatus::getCpuLoad(int& user, int& sys, int& nice, int& idle)
{
  if (cpuCount == 1) {
    return(getCpuXLoad(0, user, sys, nice, idle));
  } else {
    user = sys = idle = nice = 0;
    for (int cpu=0; cpu<cpuCount; cpu++) {
      int u, s, i, n;
      getCpuXLoad(cpu, u, s, n, i);
      user += u;
      sys += s;
      idle += i;
      nice += n;
    }
    int cpu_online = sysconf(_SC_NPROCESSORS_ONLN);
    user /= cpu_online;
    sys /= cpu_online;
    idle /= cpu_online;
    nice /= cpu_online;
    return true;
  }
}

int
OSStatus::getCpuCount(void)
{
  return sysconf(_SC_NPROCESSORS_CONF);
}

bool
OSStatus::getCpuXLoad(int cpu, int& user, int& sys, int& nice, int& idle)
{
  while (kstat_chain_update(kstat_control)==-1);
  
  char name[] = "cpu_statX";
  name[8] = cpu + '0';
  
  kstat_t *ksp = kstat_lookup(kstat_control, "cpu_stat", cpu, name);
  if (ksp && kstat_read(kstat_control, ksp, 0) != -1) {
    cpu_stat_t *info = static_cast<cpu_stat_t *>(ksp->ks_data);
    int div_user = info->cpu_sysinfo.cpu[CPU_USER]-cpu_info[5*cpu+CPU_USER];
    int div_sys = info->cpu_sysinfo.cpu[CPU_KERNEL]-cpu_info[5*cpu+CPU_KERNEL];
    int div_idle = info->cpu_sysinfo.cpu[CPU_IDLE]-cpu_info[5*cpu+CPU_IDLE];
    int div_wait = info->cpu_sysinfo.cpu[CPU_WAIT]-cpu_info[5*cpu+CPU_WAIT];
    int interval = (div_user+div_sys+div_idle+div_wait+5)/100; // +5 wg. Rundung
    if (interval) {
      last_user[cpu] = user = div_user/interval;
      last_sys[cpu] = sys = div_sys/interval;
      last_idle[cpu] = idle = div_idle/interval;
      last_wait[cpu] = nice = div_wait/interval;
      for (int i=0; i<5; i++) {
	cpu_info[5*cpu+i] = info->cpu_sysinfo.cpu[i];
      }
      return true;
    } 
    user = last_user[cpu];
    sys = last_sys[cpu];
    idle = last_idle[cpu];
    nice = last_wait[cpu];
  } else {
    user = sys = idle = nice = 0;
  }
  return true;
}

bool
OSStatus::getMemoryInfo(int& total, int& mfree, int& used, int& buffers,
						int& cached)
{
  int phys, free, avail;
  phys = free = avail = 0;

  while (kstat_chain_update(kstat_control)==-1);

  kstat_t *ksp = kstat_lookup(kstat_control, "unix", 0, "system_pages");
  if (ksp && kstat_read(kstat_control, ksp, 0) != -1) {

    /* physmem */
    kstat_named_t *ksn = 
      static_cast<kstat_named_t *>(kstat_data_lookup(ksp, "physmem"));
    if (ksn) {
      phys = ksn->value.ul*pagefactor;
    } else {
      phys = 0;
    }
    
    /* freemem */
    ksn = static_cast<kstat_named_t *>(kstat_data_lookup(ksp, "freemem"));
    if (ksn) {
      free = ksn->value.ul*pagefactor;
    } else {
      free = 0;
    }

    /* availrmem */
    ksn = static_cast<kstat_named_t *>(kstat_data_lookup(ksp, "availrmem"));
    if (ksn) {
      avail = ksn->value.ul*pagefactor;
    } else {
      avail = 0;
    }
  }

  total = sysconf( _SC_PHYS_PAGES ) * pagefactor;
  used = total - free;
  mfree = free;
  buffers = 0;  // FIXME
  cached = 0;   // FIXME

  return true;
}

bool
OSStatus::readCpuInfo(const char* cpu, int* u, int* s, int* n, int* i)
{
  // FIXME
  *u = 10;
  *s = 10;
  *n = 10;
  *i = 10;
  return (false);
}

bool
OSStatus::getSwapInfo(int& stotal, int& sfree)
{
	struct swaptable *swt;
	struct swapent	*ste;
	int		i, n;
	static char	dummy[128];

	/*
	 *  allocate enough memory to hold the info for all
	 *  swap devices
	 */
	n = swapctl( SC_GETNSWP, NULL );
	swt = (struct swaptable *) 
		malloc( sizeof( int ) + n * sizeof( struct swapent ));
	if( swt == NULL ) {
		error = true;
		errMessage = SWAPINFO_ERROR;
		return( false );
	}

	/*
	 *  fill in the required fields
	 */
	swt->swt_n = n;
	ste = swt->swt_ent;
	for( i = 0; i < n; i++ ) {
		/*
		*  we're not interested in the paths, so we
		*  reuse the same buffer for all of them
		*/
		ste->ste_path = dummy;
		ste++;
	}
	/*
	 *  get the swapinfo
	 */
	swapctl( SC_LIST, swt );

	/*
	 *  walk thru the entries and sum up total/free pages
	 */
	stotal = sfree = 0;
	ste = swt->swt_ent;
	for( i = 0; i < n; i++ ) {
		/*
		 *  don't count space that's being deleted
		 */
		if( ! (ste->ste_flags & ST_INDEL) ) {
			stotal += (int) ste->ste_pages;
			sfree += (int) ste->ste_free;
		}
	}
	free( swt );

	stotal *= (int) pagefactor;
	sfree *= (int) pagefactor;

	return( true );
}

int
OSStatus::getNoProcesses(void)
{
  while(kstat_chain_update(kstat_control)==-1);

  kstat_t *ksp = kstat_lookup(kstat_control, "unix", 0, "system_misc");
  if (ksp && kstat_read(kstat_control, ksp, 0) != -1) {
    kstat_named_t *ksn = 
      static_cast<kstat_named_t *>(kstat_data_lookup(ksp, "nproc"));
    if (ksn) {
      return ksn->value.ul;
    }
  }
 
  return 0;
}

#else

OSStatus::OSStatus()
{
	error = true;
	errMessage = SYSTEM_UNSUPPORTED;;
	return;
}

OSStatus::~OSStatus ()
{
}

bool
OSStatus::getCpuLoad(int &, int &, int &, int &)
{
	error = true;
	errMessage = SYSTEM_UNSUPPORTED;
			  
	return false;
}

int
OSStatus::getCpuCount(void)
{
	return (1);
}

bool
OSStatus::getCpuXLoad(int, int& user, int& sys, int& nice, int& idle)
{
	return (getCpuLoad(user, sys, nice, idle));
}

bool
OSStatus::readCpuInfo(const char*, int*, int*, int*, int*)
{
	return (false);
}

bool
OSStatus::getMemoryInfo(int &, int &, int &, int &, int &)
{
	error = true;
	errMessage = SYSTEM_UNSUPPORTED;
			  
	return false;
}

bool
OSStatus::getSwapInfo(int &, int &)
{
	error = true;
	errMessage = SYSTEM_UNSUPPORTED;
			  
	return false;
}

int
OSStatus::getNoProcesses(void)
{
	return (0);
}

#endif
