//****************************************************************
// exec.h
//
// Copyright (c) 1999-2000 Mount Linux Inc.
// Licensed under the terms of the GPL
//****************************************************************

#include "exec.h"

//#define DEBUG_EXEC

// CONSTRUCTORS

// Construct the object with these set.
Exec::Exec(char *sessidStr, char *netmsgIDStr, char *execPath, char *tmpPath)
{
    if (CheckArguments(sessidStr, netmsgIDStr, execPath, tmpPath, 0))
    {
        this->sessionID = atoi(sessidStr);
        this->commandID = atoi(netmsgIDStr);
        this->execpath = string(execPath);
        this->temppath = string(tmpPath);
        if (temppath.length() > 0 && temppath[temppath.length() - 1] != '/')
        {
            temppath += '/';
        }
        // change directories
        if (chdir(tmpPath))
        {
            this->status = -1;
            EmitIncomplete();
            exit(1);
        }
    }
    else
    {
        // don't go any farther!  specifically do not call EmitIncomplete
        // 'cause sessionID and commandID have garbage in them at this
        // point.
        exit(1);
    }
    this->pipedata = "";
}

// Delete all the allocated space.
Exec::~Exec()
{
    if (pidlist)
    {
        delete pidlist;
    }
}

// Process the command line here.  Only return true if all child processes
// are created successfully.
void Exec::ExecChain(int argc, char **cmdlist)
{
    char **cmd = new char*[argc + 1];
    pid_t pid;
    int fds[2];
    int idx0;
    int idx1;
    int k;
    int rd;                     // from the reference of the parent
    int wr;                     // ditto
    bool first;

    check_the_arguments(argc, cmd, cmdlist);
    MakeEnvironment();
    gettimeofday(&start, NULL);
#if defined(DEBUG_EXEC)
    cerr << "start sec=" << start.tv_sec << endl
         << "start usec=" << start.tv_usec << endl;
#endif

    // execute commands for the parts of the pipe
    first = true;
    if (argc > 0)
    {
        pipe(fds);
        rd = fds[0];
        wr = fds[1];
#if 0
        cerr << "Pipe fds[0]=" << fds[0] << ". fds[1]="
             << fds[1] << endl;
#endif
        for (idx0 = 0, idx1 = 0; idx1 < argc; idx1++)
        {
            if (cmd[idx1][0] == '|' && cmd[idx1][1] == '\0')
            {
                // only if this argument is a '|'
                delete[] cmd[idx1];
                cmd[idx1] = 0;
                pipe(fds);
#if 0
                cerr << "Pipe fds[0]=" << fds[0] << ". fds[1]="
                     << fds[1] << endl;
#endif
                pid = ExecChild(rd, wr, fds[0], fds[1], &cmd[idx0]);
                rd = fds[0];
#if 0
                cerr << "reader=" << rd << endl;
#endif
                if (first)
                {
                    pidlist = new pidList(pid);
                    first = false;
                }
                else
                {
                    pidlist->add(pid);
                }
                for (k = idx0; cmd[k]; k++)
                {
                    delete[] cmd[k];
                    cmd[k] = 0;
                }
                idx0 = idx1 + 1;
            }
        }
    }
    pipe(fds);
#if 0
    cerr << "Pipe: fds[0]=" << fds[0] << " fds[1]=" << fds[1] << endl;
#endif
    pid = ExecChild(rd, wr, fds[0], fds[1], &cmd[idx0]);
    if (first)
    {
        pidlist = new pidList(pid);
        first = false;
    }
    else
    {
        pidlist->add(pid);
    }
    pidlist->add(pid);
    this->filedesc = fds[0];
}

// massage the arguments for callin execvp
int Exec::ExecChild(int rd0, int wr0, int rd1, int wr1, char **cmd)
{
    string strpath;
    string strtemp;
    int idx;
    int tmp;
    bool rc;

    // if the command is an absolute path then we want argv[0] to be the
    // actual file name not the whole path
    strpath = cmd[0];
    idx = strpath.size() - 1;
    // find the last '/'
    while (idx > 0)
    {
        char c = strpath.at(idx);
        if (c == '/')
        {
            idx++;              // position to the char after
            break;
        }
        idx--;
    }
    strtemp = strpath.substr(idx);
    delete[] cmd[0];
    tmp = strtemp.size();
    cmd[0] = new char[tmp + 1];
    strtemp.copy(cmd[0], tmp);
    cmd[0][tmp] = '\0';
#if 0
    cerr << "calling MakeChild: " << strpath << endl;
    cerr << "argument list:" << endl;
    tmp = 0;
    while (cmd[tmp])
    {
        cerr << cmd[tmp++] << ' ' << endl;
    }
#endif
    rc = ExecChild1(rd0, wr0, rd1, wr1, strpath.c_str(), cmd);
    return (rc);
}

// Make the process and change the i/o so we get stdout on a different fd
// than stderr.
pid_t
Exec::ExecChild1(int r0, int w0, int r1, int w1, const char *path, char **cmd)
{
    int rv;

#if 0
    cerr << path << ": closing " << w0 << " and " << r1 << ". "
         << "stdin=" << r0 << " stdout=" << w1 << endl;
#endif
    // make a child and close what you don't need
    pid_t pid = fork();
    switch (pid)
    {
    case -1:
        // no child process
        // !!! need to return something here through stdout
        perror("fork");
        exit(1);
        break;
    case 0: // child
        // make both stdout and stderr write to the same pipe
        close(0);               // close stdin
        close(1);               // close stdout
        close(2);               // close stderr
        dup(r0);
        dup(w1);
        dup(w1);
        close(w0);
        close(r1);
        rv = execvp(path, cmd);
        status = -1;
        EmitIncomplete();
        exit(1);
        break;
    default:
        // parent process
#if 0
    cerr << "parent closing " << r0 << " and " << w1 << endl;
#endif
        close(r0);
        close(w1);
        break;
    }
    return (pid);
}

// Get at most "size" characters from the output.  The return value is the
// number of bytes of data actually in the buffer.
void Exec::Output()
{
    string *datapath;
    pidList* pidptr;
    char* text;
    char *p;
    int length;
    int tmp;
    int fd = fileno(stdout);

    // get any output from the process pipe
    datapath = CaptureOutput();
    length = 24 + datapath->length() + 1; // header 24; path; null
    text = new char[length];

    // if any status is non-zero then report that status
    pidlist->reap();
    for (pidptr = pidlist; pidptr; pidptr = pidptr->nextpid)
    {
        if ((status = pidptr->getStatus()) != 0)
        {
            break;
        }
    }
    delete pidlist;
    pidlist = 0;

    // get the final timestamp
    gettimeofday(&end, NULL);
#if defined(DEBUG_EXEC)
    cerr << "end sec=" << end.tv_sec << endl
         << "end usec=" << end.tv_usec << endl;
#endif

    // calculate the difference between start and end
    // perform the carry for the subtraction by updating start
    if (end.tv_usec < start.tv_usec)
    {
        tmp = (start.tv_usec - end.tv_usec) / 1000000 + 1;
        start.tv_usec -= 1000000 * tmp;
        start.tv_sec += tmp;
    }
    if (end.tv_usec - start.tv_usec > 1000000)
    {
        tmp = (end.tv_usec - start.tv_usec) / 1000000;
        start.tv_usec += 1000000 * tmp;
        start.tv_sec -= tmp;
    }
    // calculate the difference; it'll always be positive since start
    // precedes end
    elapsedTime.tv_sec = end.tv_sec - start.tv_sec;
    elapsedTime.tv_usec = end.tv_usec - start.tv_usec;
#if defined(DEBUG_EXEC)
    cerr << "elapsed time sec=" << elapsedTime.tv_sec << endl
         << "elapsed time usec=" << elapsedTime.tv_usec << endl;
#endif

    // form the header
    p = text;

    memcpy(p, (char *) &sessionID, sizeof(int)); // 8 bytes
    p += sizeof(int);

    memcpy(p, (char *) &commandID, sizeof(int)); // 4 bytes
    p += sizeof(int);

    memcpy(p, (char *) &elapsedTime.tv_sec, sizeof(long int)); // 12 bytes
    p += sizeof(long int);

    memcpy(p, (char *) &elapsedTime.tv_usec, sizeof(long int)); // 16 bytes
    p += sizeof(long int);

    memcpy(p, (char *) &status, sizeof(int)); //  20 bytes
    p += sizeof(int);

    tmp = htonl(datapath->length() + 1);
    memcpy(p, (char *) &tmp, sizeof(int)); //  24 bytes
    p += sizeof(int);

    tmp = datapath->length();
    datapath->copy(p, length);
    p += tmp;
    p[0] = '\0';
    p++;
    write(fd, text, length);
    delete text;
}

// Execute the command and put stdout in temporary file.  Return the
// name of the temporary file.
string* Exec::CaptureOutput()
{
    string *datapath;
    char buffer[1024];
    int fd;
    int size;

    datapath = 0;
    temppath.copy(buffer, 1024);
    buffer[temppath.length()] = '\0';
    strncat(buffer, "execXXXXXX", 10);
    buffer[temppath.length() + 10] = '\0';
    if ((fd = mkstemp(buffer)) >= 0)
    {
        datapath = new string(buffer);
        while ((size = read(filedesc, buffer, 1024)) > 0)
        {
            write(fd, buffer, size);
        }
        close(fd);
    }

    return (datapath);
}

// check for any arguments that are longer than 1024 bytes and only
// allow characters that match [-_./0-9A-Za-z]
bool Exec::CheckArguments(char *arg, ...)
{
    using namespace regexx;
    Regexx re;
    char *p = arg;
    bool good = true;
    re.expr("[^-_./0-9A-Za-z]");
    va_list ap;
    va_start(ap, arg);
    while (good && p)
    {
        if (strlen(p) > 1024)
        {
            return (false);
        }
        re.str(p);
        if (re.exec() > 0)
        {
            return (false);
        }
        p = va_arg(ap, char *);
    }
    return (good);
}

void Exec::check_the_arguments(int argc, char **arglist, char **cmdlist)
{
    char **ptr = cmdlist;
    int idx;

    // check the arguments
    for (idx = 0; idx < argc; idx++)
    {
        if (CheckArguments(*ptr, 0) || (ptr[0][0] == '|' && ptr[0][1] == '\0'))
        {
            arglist[idx] = new char[strlen(*ptr) + 1];
            strcpy(arglist[idx], *ptr);
            ptr++;
        }
        else
        {
            // return with nothing
            status = -1;
            EmitIncomplete();
            exit(1);
        }
    }
    arglist[idx] = 0;
}

// Tell the server that the action never happened
void Exec::EmitIncomplete()
{
    char text[256];
    char *p = text;
    int fd = fileno(stdout);
    int tmp = 0;
    *p++ = commandID >> 24;
    *p++ = commandID >> 16;
    *p++ = commandID >> 8;
    *p++ = commandID;
    *p++ = sessionID >> 24;
    *p++ = sessionID >> 16;
    *p++ = sessionID >> 8;
    *p++ = sessionID;
    *p++ = status >> 24;
    *p++ = status >> 16;
    *p++ = status >> 8;
    *p++ = status;
    tmp = 0;
    *p++ = tmp >> 24;
    *p++ = tmp >> 16;
    *p++ = tmp >> 8;
    *p++ = tmp;
    write(fd, text, sizeof(int) << 2);
}

// The new environment has the execpath as the first element in the path
void Exec::MakeEnvironment()
{
    char *cpath;
    int tmp;

    if ((cpath = getenv("PATH")) != 0)
    {
        string path = string(cpath);
        path = execpath + ':' + path;
        setenv("PATH", path.c_str(), 1);
        tmp = chdir(temppath.c_str());
    }
    else
    {
        exit(1);
    }
}
