/*
 * channel.cc: object to handle client connections
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "config.h"
#include "defs.h"
#include "cipher.h"
#include "transport.h"
#include "netmessage.h"
#include "commid.h"
#include "msgs.h"
#include "olympus.h"
#include "pipemanager.h"
#include "processmanager.h"
#include "channel.h"

/* netmessages */
#include "nmAlgInit.h"
#include "nmExec.h"
#include "nmFetchConfig.h"
#include "nmKexInit.h"
#include "nmKexReply.h"
#include "nmPing.h"
#include "nmSourceExpand.h"
#include "nmSourceList.h"
#include "nmSysInfo.h"
#include "nmUpdateConfig.h"
#include "nmUsers.h"
#include "nmUserAuth.h"
#include "nmVersion.h"

/* channel states */
#define CHANNEL_NEW             0
#define CHANNEL_ALGINIT         1
#define CHANNEL_KEXINIT         2
#define CHANNEL_KEXREPLY        3
#define CHANNEL_USERAUTH        4
#define CHANNEL_PROCESS         5

channel::channel(int nsd)
{
    socket = new transport(nsd);
    pipeList = NULL;
    hlen = 0;
    state = CHANNEL_NEW;
    opcode = commandID = sessionID = 0;
    request = NULL;
    pipeQueue = 0;
    next = NULL;
}

channel::~channel(void)
{
    if (request != NULL)
    {
        delete request;
    }

    if (pipeList != NULL)
    {
        delete pipeList;
    }

    delete socket;
}

int channel::getsd(void)
{
    return socket->getsd();
}

void channel::addProcess(processManager *process)
{
    if (pipeList == NULL)
    {
        pipeList = new pipeManager(process);
    }
    else
    {
        pipeList->add(process);
    }
}

int channel::processGeneric(void)
{
    switch (opcode)
    {
        case CHANNEL_CLOSE:
        case CHANNEL_ERROR:
        default:
            return -1;
    }
    return 1;
}

int channel::processConnect(void)
{
    int rv, nmresult;

    if (request == NULL)
    {
        if (commandID == COMMAND_VERSION)
        {
            request = new nmVersion(socket, sessionID, commandID);
        }
        else return -1;
    }

    switch (opcode)
    {
        case PROTOCOL_VERSION:
            if ((rv = request->receive()) == 0)
            {
                nmresult = request->activate();
                delete request, request = NULL;
            }
            return rv;
        default:
            return -1;
    }
    return 1;
}

int channel::processAex(void)
{
    int rv, nmresult;

    if (request == NULL)
    {
        if (commandID == COMMAND_ALGINIT)
        {
            request = new nmAlgInit(socket, sessionID, commandID);
        }
        else return -1;
    }

    switch (opcode)
    {
        case ALG_INIT:
            if ((rv = request->receive()) == 0)
            {
                nmresult = request->activate();
                delete request, request = NULL;
            }
            return rv;
        default:
            return -1;
    }
    return 1;
}

int channel::processKex(void)
{
    int rv, nmresult;

    if (request == NULL)
    {
        if (commandID == COMMAND_KEXINIT)
        {
            if (state == CHANNEL_KEXINIT)
            {
                request = new nmKexInit(socket, sessionID, commandID);
            }
            else return -1;
        }
        else if (commandID == COMMAND_KEXREPLY)
        {
            if (state == CHANNEL_KEXREPLY)
            {
                request = new nmKexReply(socket, sessionID, commandID);
            }
            else return -1;
        }
    }

    switch (opcode)
    {
        case KEX_DH_INIT:
            if ((rv = request->receive()) == 0)
            {
                nmresult = request->activate();
                delete request, request = NULL;
            }
            return rv;
        case KEX_DH_REPLY:
            if ((rv = request->receive()) == 0)
            {
                nmresult = request->activate();

                socket->setCipher(CIPHER_3DES, false);
                delete request, request = NULL;
            }
            return rv;
        default:
            return -1;
    }
    return 1;
}

int channel::processAuth(void)
{
    int retc, nmresult;

    if (request == NULL)
    {
        if (commandID == COMMAND_USERAUTH)
        {
            if (opcode == USERAUTH_REQUEST)
            {
                request = new nmUserAuth(socket, sessionID, commandID);
            }
            else return -1;
        }
        else return -1;
    }

    switch (opcode)
    {
        case USERAUTH_REQUEST:
            if ((retc = request->receive()) == 0)
            {
                nmresult = request->activate();
                delete request, request = NULL;

                if (nmresult == 0)
                    return -1;
            }
            return retc;
        case USERAUTH_SUCCESS:
        case USERAUTH_FAILURE:
        default:
            return -1;
    }
    return 1;
}

int channel::processRequest(void)
{
    int nmresult, retc;
    
    if (request == NULL)
    {
        if (commandID == COMMAND_PING)
        {
            request = new nmPing(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_FETCHCONFIG)
        {
            request = new nmFetchConfig(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_SYSINFO)
        {
            request = new nmSysInfo(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_UPDATECONFIG)
        {
            request = new nmUpdateConfig(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_USERS)
        {
            request = new nmUsers(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_EXEC)
        {
            request = new nmExec(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_LISTSRC)
        {
            request = new nmSourceList(socket, sessionID, commandID);
        }
        else if (commandID == COMMAND_EXPANDSRC)
        {
            request = new nmSourceExpand(socket, sessionID, commandID);
        }
        else return -1;
    }

    switch (commandID)
    {
        case COMMAND_PING:
        case COMMAND_FETCHCONFIG:
        case COMMAND_SYSINFO:
        case COMMAND_UPDATECONFIG:
        case COMMAND_USERS:
        case COMMAND_EXEC:
        case COMMAND_LISTSRC:
            if ((retc = request->receive()) == 0)
            {
                nmresult = request->activate();
                delete request, request = NULL;
            }
            return retc;
        case COMMAND_EXPANDSRC:
            if ((retc = request->receive()) == 0)
            {
                if ((nmresult = request->activate()) > 0)
                {
                    processManager *nouveau;

                    nouveau = ((nmSourceExpand *) request)->newProcess();
                    addProcess(nouveau);
                }
            }
            return retc;
        default:
            return -1;
    }
    return 1;
}

int channel::pipeMessage(void)
{
    int retc, nmresult;

    if (request == NULL)
    {
    }

    switch (commandID)
    {
        default:
            return -1;
    }
    return 1;
}

int channel::processPipes(fd_set* fd_in = NULL)
{
    pipeManager::pipeElement* temp, *next;
    int rv;

    for (temp = pipeList->elements; temp; temp = next)
    {
        next = temp->nextElement;

        if (temp->queued || (fd_in && FD_ISSET(temp->getfd(), fd_in)))
        {
            if (request == NULL)
            {
                temp->process->read(&sessionID, 4);
                temp->process->read(&commandID, 4);

                if ((rv = pipeMessage()) < 0)
                {
                    pipeList->remove(temp->getfd());
                    return rv;
                }
                pipeList->remove(temp->getfd());
            }
            else
            {
                if (temp->queued == false)
                {
                    temp->queued = true;
                    pipeQueue += 1;
                }
            }
        }
    }
    return 1;
}

int channel::processChannel(void)
{
    int rv;

    if (opcode == 0)
    {
        if ((rv = socket->recv(&header[hlen], HEADER_SIZE - hlen)) < 1)
        {
            if (errno == EINTR || errno == EAGAIN)
            {
                return 1;
            }
            return rv;
        }

        if (rv + hlen == HEADER_SIZE)
        {
            memcpy(&opcode, &header[0], sizeof(opcode));
            opcode = ntohl(opcode);
            memcpy(&sessionID, &header[8], sizeof(sessionID));
            sessionID = ntohl(sessionID);
            memcpy(&commandID, &header[12], sizeof(commandID));
            commandID = ntohl(commandID);
            hlen = 0;
        }
        else
        {
            hlen += rv;
            return 1;
        }
    }

    if (opcode > 19)
    {
        if (state == CHANNEL_NEW)
        {
            if (opcode > 19 && opcode < 40)
            {
                if ((rv = processConnect()) == 0)
                {
                    request = new nmAlgInit(socket, 1, COMMAND_ALGINIT);
                    request->send(socket, ALG_INIT);
                    delete request, request = NULL;

                    opcode = sessionID = commandID = 0;
                    state = CHANNEL_ALGINIT;
                }
            }
            else return -1;
        }
        else if (state == CHANNEL_ALGINIT)
        {
            if (opcode > 39 && opcode < 50)
            {
                if ((rv = processAex()) == 0)
                {
                    opcode = sessionID = commandID = 0;
                    state = CHANNEL_KEXINIT;
                }
            }
            else return -1;
        }
        else if (state == CHANNEL_KEXINIT)
        {
            if (opcode > 49 && opcode < 60)
            {
                if ((rv = processKex()) == 0)
                {
                    opcode = sessionID = commandID = 0;
                    state = CHANNEL_KEXREPLY;
                }
            }
            else return -1;
        }
        else if (state == CHANNEL_KEXREPLY)
        {
            if (opcode > 49 && opcode < 60)
            {
                if ((rv = processKex()) == 0)
                {
                    opcode = sessionID = commandID = 0;
                    state = CHANNEL_USERAUTH;
                }
            }
            else return -1;
        }
        else if (state == CHANNEL_USERAUTH)
        {
            if (opcode > 59 && opcode < 70)
            {
                if ((rv = processAuth()) == 0)
                {
                    opcode = sessionID = commandID = 0;
                    state = CHANNEL_PROCESS;
                }
            }
            else return -1;
        }
        else if (state == CHANNEL_PROCESS)
        {
            if (opcode > 69 && opcode < 90)
            {
                if ((rv = processRequest()) == 0)
                {
                    opcode = sessionID = commandID = 0;
                }
            }
            else return -1;
        }
    }
    else
    {
    }

    if (pipeQueue != 0 && request == NULL)
    {
        processPipes();
    }

    return rv + 1;
}
